/*
 *=============================================================================
 *                                  tSippMisc.c
 *-----------------------------------------------------------------------------
 * Miscellaneous commands.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippMisc.c,v 5.13 1996/12/09 07:04:05 markd Exp $
 *============================================================================
 */

#include "tSippInt.h"
#include "patchlevel.h"

/*
 * Runtime directory.  Set here rather than passed directly to Tcl_VarEval to
 * help detect non-string values.
 */
static char  *runtimeLibrary = TSIPP_LIBRARY;

/*
 * Internal prototypes.
 */
static void
ReturnColor _ANSI_ARGS_((tSippGlob_t   *tSippGlobPtr,
                         Color         *colorPtr));

static void
UpdateSignalHandler _ANSI_ARGS_((void_pt  clientData));

static void
TSippCleanUp _ANSI_ARGS_((ClientData   clientData,
                          Tcl_Interp  *interp));

/*=============================================================================
 * ReturnColor --
 *   Return a color into the interp->result with no lose of precision.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o colorPtr (I) - Pointer to the color to format.
 *-----------------------------------------------------------------------------
 */
static void
ReturnColor (tSippGlobPtr, colorPtr)
    tSippGlob_t   *tSippGlobPtr;
    Color         *colorPtr;
{
    char redBuf [TCL_DOUBLE_SPACE];
    char grnBuf [TCL_DOUBLE_SPACE];
    char bluBuf [TCL_DOUBLE_SPACE];


    Tcl_PrintDouble (tSippGlobPtr->interp,
                     colorPtr->red,
                     redBuf);
                     
    Tcl_PrintDouble (tSippGlobPtr->interp,
                     colorPtr->grn,
                     grnBuf);
                     
    Tcl_PrintDouble (tSippGlobPtr->interp,
                     colorPtr->blu,
                     bluBuf);

    Tcl_AppendResult (tSippGlobPtr->interp,
                      redBuf, " ",
                      grnBuf, " ",
                      bluBuf, (char *) NULL);
}

/*=============================================================================
 * UpdateSignalHandler -- 
 *   This routine is called during rendering at the specified update period
 * to handle signals (any other async events) if they have occured.  When
 * linked with Tk, another routine is used that handles events as well as
 * signals.  If an error occurs in signal handling, then it is noted in the
 * tSippGlob and the rendering is aborted.  The error will be reported once
 * the rendering terminates.
 *
 * Parameters:
 *   o clientData (I) - Actually a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
UpdateSignalHandler (clientData)
    void_pt clientData;
{
    if (Tcl_AsyncReady ()) {
        tSippGlob_t  *tSippGlobPtr = (tSippGlob_t *) clientData;

        if (Tcl_AsyncInvoke (tSippGlobPtr->interp, TCL_OK) == TCL_ERROR)
            TSippHandleRenderingError (tSippGlobPtr);
    }
}

/*=============================================================================
 * SippShowBackFaces --
 *   Implements the command:
 *     SippShowBackFaces [flag]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippShowBackFaces (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t  *tSippGlobPtr = (tSippGlob_t *) clientData;
    int           flag;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc > 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " [flag]", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (argc == 1) {
        interp->result = tSippGlobPtr->showBackFaces ? "1" : "0";
        return TCL_OK;
    }

    if (Tcl_GetBoolean (interp, argv [1], &flag) != TCL_OK)
        return TCL_ERROR;
    tSippGlobPtr->showBackFaces = flag;
    sipp_show_backfaces (flag);
    return TCL_OK;
}

/*=============================================================================
 * SippBackground --
 *   Implements the command:
 *     SippBackground [color]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippBackground (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t  *tSippGlobPtr = (tSippGlob_t *) clientData;
    Color         color;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc > 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " [color]", (char *) NULL);
        return TCL_ERROR;
    }
    if (argc == 1) {
        ReturnColor (tSippGlobPtr, &tSippGlobPtr->backgroundColor);
        return TCL_OK;
    }

    if (!TSippConvertColor (tSippGlobPtr, argv [1], &color))
        return TCL_ERROR;
    sipp_background (color.red, color.grn, color.blu);
    tSippGlobPtr->backgroundColor = color;

    return TCL_OK;
}

/*=============================================================================
 * SippLineColor --
 *   Implements the command:
 *     SippLineColor [color]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippLineColor (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t  *tSippGlobPtr = (tSippGlob_t *) clientData;
    Color         color;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc > 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " [color]", (char *) NULL);
        return TCL_ERROR;
    }
    if (argc == 1) {
        ReturnColor (tSippGlobPtr, &tSippGlobPtr->lineColor);
        return TCL_OK;
    }

    if (!TSippConvertColor (tSippGlobPtr, argv [1], &color))
        return TCL_ERROR;
    tSippGlobPtr->lineColor = color;
    return TCL_OK;
}

/*=============================================================================
 * SippInfo --
 *   Implements the command:
 *     SippInfo attribute
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippInfo (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t  *tSippGlobPtr = (tSippGlob_t *) clientData;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " attribute", (char *) NULL);
        return TCL_ERROR;
    }                     
    if (STREQU ("VERSION", argv [1])) {
        interp->result = TSIPP_FULL_VERSION;
        return TCL_OK;
    }
    if (STREQU ("SIPPVERSION", argv [1])) {
        interp->result = SIPP_VERSION;
        return TCL_OK;
    }
    if (STREQU ("TSIPPVERSION", argv [1])) {
        interp->result = TSIPP_VERSION;
        return TCL_OK;
    }
    if (STREQU ("TSIPPPATCHLEVEL", argv [1])) {
        sprintf (interp->result, "%d", PATCHLEVEL);
        return TCL_OK;
    }
    if (STREQU ("RLE", argv [1])) {
#ifdef TSIPP_HAVE_RLE
        interp->result = "1";
#else
        interp->result = "0";
#endif
        return TCL_OK;
    }
    if (STREQU ("RENDERING", argv [1])) {
        interp->result = tSippGlobPtr->rendering ? "1" : "0";
        return TCL_OK;
    }

    if (STREQU ("TK", argv [1])) {
        interp->result = (tSippGlobPtr->tkMainWindow != NULL) ? "1" : "0";
        return TCL_OK;
    }

    Tcl_AppendResult (interp, "expected an attribute of \"VERSION\", ",
                      "\"SIPPVERSION\", \"TSIPPVERSION\", ",
                      "\"TSIPPPATCHLEVEL\", \"RLE\", \"RENDERING\", or",
                      " \"TK\", got \"", argv [1], "\"", (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * SippColor --
 *   Implements the command:
 *     SippColor subcommand ...
 * Notes:
 *   This procedure calls subcommands that are only present in a Tk
 * environment.  These are called through prointers in the global structure.
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippColor (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t  *tSippGlobPtr = (tSippGlob_t *) clientData;

    if (argc < 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " subcommand ...", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (STREQU ("tox", argv [1])) {
        if (tSippGlobPtr->sippColorToXColor == NULL)
            goto needTk;
        return tSippGlobPtr->sippColorToXColor (tSippGlobPtr,
                                                argc,
                                                argv);
    }

    if (STREQU ("tosipp", argv [1])) {
        if (tSippGlobPtr->xColorToSippColor == NULL)
            goto needTk;
        return tSippGlobPtr->xColorToSippColor (tSippGlobPtr,
                                                argc,
                                                argv);
    }

    Tcl_AppendResult (interp, "expected a subcommand of \"tox\", or ",
                      "\"tosipp\", got \"", argv [1], "\"", (char *) NULL);
    return TCL_ERROR;

  needTk:
    Tcl_AppendResult (interp, "the subcommand ", argv [1],
                      " is only valid when TSIPP is linked with Tk",
                      (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * SippAbortRender --
 *   Implements the command:
 *     SippAbortRender
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippAbortRender (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], (char *) NULL);
        return TCL_ERROR;
    }                     

    sipp_render_terminate ();
    return TCL_OK;
}

/*=============================================================================
 * TSippCleanUp -- 
 *   Called a interpreter deletion to clean up Tcl SIPP and release all
 * resources.
 *
 * Parameters:
 *   o clientData (I) - A pointer to the Tcl SIPP global structure.
 *   o interp (I) - Pointer to the Tcl interpreter.
 *-----------------------------------------------------------------------------
 */
static void
TSippCleanUp (clientData, interp)
    ClientData   clientData;
    Tcl_Interp  *interp;
{
    tSippGlob_t *tSippGlobPtr = (tSippGlob_t *) clientData;

    Tcl_DeleteAssocData (interp, "TSIPP-GLOB");
    
    TSippPolyCleanUp (tSippGlobPtr);
    TSippObjectCleanUp (tSippGlobPtr);
    TSippShaderCleanUp (tSippGlobPtr);
    TSippLightCleanUp (tSippGlobPtr);
    TSippCameraCleanUp (tSippGlobPtr);
    if (tSippGlobPtr->rleCleanUpProc != NULL)
        (*tSippGlobPtr->rleCleanUpProc) (tSippGlobPtr);
    TSippPBMCleanUp (tSippGlobPtr);
    TSippPixMapCleanUp (tSippGlobPtr);

    sfree (tSippGlobPtr);
}

/*=============================================================================
 * Tsipp_Init -- 
 *   Initialize the Tcl/Sipp environment.  This follows the Tcl init function
 * calling conventions.
 *
 * Parameters:
 *   o interp (I) - Pointer to the Tcl interpreter.
 *
 * Returns:
 *   TCL_OK.
 *-----------------------------------------------------------------------------
 */
int
Tsipp_Init (interp)
    Tcl_Interp *interp;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippShowBackFaces", (Tcl_CmdProc *) SippShowBackFaces},
        {"SippBackground",    (Tcl_CmdProc *) SippBackground},
        {"SippLineColor",     (Tcl_CmdProc *) SippLineColor},
        {"SippInfo",          (Tcl_CmdProc *) SippInfo},
        {"SippColor",         (Tcl_CmdProc *) SippColor},
        {"SippAbortRender",   (Tcl_CmdProc *) SippAbortRender},
        {NULL,                NULL}
    };
    tSippGlob_t *tSippGlobPtr;
    char         numBuf [32];

    if (Tcl_PkgRequire (interp, "Tclx", TCLX_VERSION, 0) == NULL) {
 	return TCL_ERROR;
    }

   /*
    * Define info to be returned by the infox and SippInfo commands.
    */
    TclX_SetAppInfo (FALSE, /* defaultValues */
                     "TSIPP",
                     "Tcl/SIPP",
                     TSIPP_VERSION,
                     PATCHLEVEL);


    /*
     * Set the floating point precision to be the max -1.  User can adjust
     * later.  The max is not used, as this leads to round off problems.
     */
    sprintf (numBuf, "%d", TCL_MAX_PREC-1);
    if (Tcl_SetVar (interp,
                    "tcl_precision",
                    numBuf,
                    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
        return NULL;
                

    /*
     * Initialize SIPP.  We associate headers with surface descriptors and
     * manage the object and surface reference counts.
     */
    sipp_init ();
    sipp_surface_desc_headers (TRUE);
    sipp_user_refcount (TRUE);

    /*
     * Set up the TSIPP global structure.
     */
    tSippGlobPtr = (tSippGlob_t *) smalloc (sizeof (tSippGlob_t));
    Tcl_SetAssocData (interp, "TSIPP-GLOB", TSippCleanUp,
                      tSippGlobPtr);

    tSippGlobPtr->interp        = interp;
    tSippGlobPtr->showBackFaces = FALSE;
    tSippGlobPtr->rendering     = FALSE;

    /*
     * Set update signal handler.  This is replaced if linked with Tk.
     */
    tSippGlobPtr->updateProc   = UpdateSignalHandler;
    tSippGlobPtr->delayedError = NULL;

    tSippGlobPtr->storageClassPtrs [0] = NULL;

    tSippGlobPtr->backgroundColor.red = 0.0;
    tSippGlobPtr->backgroundColor.grn = 0.0;
    tSippGlobPtr->backgroundColor.blu = 0.0;

    tSippGlobPtr->lineColor.red = 1.0;
    tSippGlobPtr->lineColor.grn = 1.0;
    tSippGlobPtr->lineColor.blu = 1.0;

    tSippGlobPtr->tkMainWindow = NULL;
    tSippGlobPtr->sippColorToXColor = NULL;
    tSippGlobPtr->xColorToSippColor = NULL;

    TSippInitCmds (tSippGlobPtr, cmdTable);

    TSippErrorHandlingInit (tSippGlobPtr);

    TSippBezierInit (tSippGlobPtr);
    TSippCameraInit (tSippGlobPtr);
    TSippCommentInit (tSippGlobPtr);
    TSippGeomInit (tSippGlobPtr);
    TSippPixMapInit (tSippGlobPtr);
    TSippLightInit (tSippGlobPtr);
    TSippObjectInit (tSippGlobPtr);
    TSippPBMInit (tSippGlobPtr);
    TSippPolyInit (tSippGlobPtr);
    TSippPrimInit (tSippGlobPtr);
    TSippRenderInit (tSippGlobPtr);
    TSippRLEInit (tSippGlobPtr);
    TSippShaderInit (tSippGlobPtr);
    TSippShadowsInit (tSippGlobPtr);

    /*
     * If we are debugging memory leaks, force the TclX shell to delete the
     * interpreter instead of going through the exit command.
     */
#ifdef TSIPP_MEM_CHECK
    tclDeleteInterpAtEnd = TRUE;
#endif

    if (Tcl_SetVar (interp, "tsipp_library", runtimeLibrary,
                    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
        return TCL_ERROR;

    if (Tcl_SetVar (interp, "auto_path", runtimeLibrary,
                    TCL_GLOBAL_ONLY | TCL_APPEND_VALUE |
                    TCL_LIST_ELEMENT | TCL_LEAVE_ERR_MSG) == NULL)
        return TCL_ERROR;
    Tcl_ResetResult (interp);

    if (Tcl_PkgProvide (interp, "Tsipp", TSIPP_VERSION) != TCL_OK) {
 	return TCL_ERROR;
    }
    return TCL_OK;
}

