/* 
 * tkCombo.c --
 *
 *	This module implements combo widgets for the Tk
 *	toolkit.  An combo displays a string and allows
 *	the string to be edited.
 *
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1997 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: @(#) $Id: tkCombobox.c,v 1.3 2002/09/26 01:45:05 hobbs Exp $
 */

//#include "tkInt.h"
//#include "default.h"
#include "pkg.h"

/*
 * Defaults for entries:
 */
#if defined(WIN32) || defined(__WIN32__)

#define DEF_COMBO_BG_COLOR		"SystemWindow"
#define DEF_COMBO_FONT			CTL_FONT
#define DEF_COMBO_FG			TEXT_FG
#define DEF_COMBO_HIGHLIGHT		HIGHLIGHT
#define DEF_COMBO_HIGHLIGHT_WIDTH	"0"
#define DEF_COMBO_INSERT_BG		TEXT_FG
#define DEF_COMBO_INSERT_WIDTH		"2"
#define DEF_COMBO_RELIEF		"sunken"
#define DEF_COMBO_SELECT_BD_COLOR	"0"

#elif defined(MAC_TCL) || defined(MAC_OSX_TK)

#define DEF_COMBO_BG_COLOR		NORMAL_BG
#define DEF_COMBO_FONT			"Helvetica 12"
#define DEF_COMBO_FG			BLACK
#define DEF_COMBO_HIGHLIGHT		BLACK
#define DEF_COMBO_HIGHLIGHT_WIDTH	"0"
#define DEF_COMBO_INSERT_BG		BLACK
#define DEF_COMBO_INSERT_WIDTH		"1"
#define DEF_COMBO_RELIEF		"solid"
#define DEF_COMBO_SELECT_BD_COLOR	"1"

#else

#define DEF_COMBO_BG_COLOR		NORMAL_BG
#define DEF_COMBO_FONT			"Helvetica -12"
#define DEF_COMBO_FG			BLACK
#define DEF_COMBO_HIGHLIGHT		BLACK
#define DEF_COMBO_HIGHLIGHT_WIDTH	"1"
#define DEF_COMBO_INSERT_BG		BLACK
#define DEF_COMBO_INSERT_WIDTH		"2"
#define DEF_COMBO_RELIEF		"sunken"
#define DEF_COMBO_SELECT_BD_COLOR	"1"

#endif

#define DEF_COMBO_BG_MONO		WHITE
#define DEF_COMBO_BORDER_WIDTH		"2"	/* mac should be 1 */
#define DEF_COMBO_CURSOR		"xterm"
#define DEF_COMBO_EXPORT_SELECTION	"1"

#define DEF_COMBO_DISABLED_FG_COLOR	DISABLED
#define DEF_COMBO_DISABLED_FG_MONO	BLACK
#define DEF_COMBO_HIGHLIGHT_BG		NORMAL_BG

#define DEF_COMBO_INSERT_BD_COLOR	"0"
#define DEF_COMBO_INSERT_BD_MONO	"0"
#define DEF_COMBO_INSERT_OFF_TIME	"300"
#define DEF_COMBO_INSERT_ON_TIME	"600"
#define DEF_COMBO_JUSTIFY		"left"
#define DEF_COMBO_SCROLL_COMMAND	""
#define DEF_COMBO_SELECT_COLOR		SELECT_BG
#define DEF_COMBO_SELECT_MONO		BLACK
#define DEF_COMBO_SELECT_BD_MONO	"0"
#define DEF_COMBO_SELECT_FG_COLOR	SELECT_FG
#define DEF_COMBO_SELECT_FG_MONO	WHITE
#define DEF_COMBO_SHOW			(char *) NULL
#define DEF_COMBO_STATE			"normal"
#define DEF_COMBO_TAKE_FOCUS		(char *) NULL
#define DEF_COMBO_TEXT_VARIABLE		""
#define DEF_COMBO_WIDTH			"20"

#define DEF_COMBO_REPEAT_DELAY		"400"
#define DEF_COMBO_REPEAT_INTERVAL	"100"

#define DEF_COMBO_SPINCMD		"tkComboSpinCmd %W %S"
#define DEF_COMBO_DROPCMD		"tkComboDropCmd %W"

#define DEF_COMBO_SPINUP_CURSOR		""
#define DEF_COMBO_SPINDOWN_CURSOR	""
#define DEF_COMBO_COMBO_CURSOR		""

#define DEF_COMBO_FROM			"0"
#define DEF_COMBO_TO			"100.0"
#define DEF_COMBO_RESOLUTION		"1.0"
#define DEF_COMBO_DIGITS		"0"

#define DEF_COMBO_MODE			"range"

#define DEF_COMBO_LIST_VARIABLE		(char *) NULL

#define COMBO_MULT 1.5

/*
 * A data structure of the following type is kept for each combo
 * widget managed by this file:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the combo. NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with combo. */
    Tcl_Command widgetCmd;	/* Token for combo's widget command. */
    Tk_OptionTable optionTable;	/* Table that defines configuration options
				 * available for this widget. */


    /*
     * Fields that are set by widget commands other than "configure".
     */
     
    char *string;		/* Pointer to storage for string;
				 * NULL-terminated;  malloc-ed. */
    int insertPos;		/* Character index before which next typed
				 * character will be inserted. */

    /*
     * Information about what's selected, if any.
     */

    int selectFirst;		/* Character index of first selected
				 * character (-1 means nothing selected. */
    int selectLast;		/* Character index just after last selected
				 * character (-1 means nothing selected. */
    int selectAnchor;		/* Fixed end of selection (i.e. "select to"
				 * operation will use this as one end of the
				 * selection). */

    /*
     * Information for scanning:
     */

    int scanMarkX;		/* X-position at which scan started (e.g.
				 * button was pressed here). */
    int scanMarkIndex;		/* Character index of character that was at
				 * left of window when scan started. */

    /*
     * Configuration settings that are updated by Tk_ConfigureWidget.
     */

    Tk_3DBorder normalBorder;	/* Used for drawing border around whole
				 * window, plus used for background. */
    int borderWidth;		/* Width of 3-D border around window. */
    Tk_Cursor cursor;		/* Current cursor for window, or None. */
    Tk_Cursor dCursor;		/* Current cursor for window, or None. */
    Tk_Cursor suCursor;		/* Current cursor for window, or None. */
    Tk_Cursor sdCursor;		/* Current cursor for window, or None. */
    int exportSelection;	/* Non-zero means tie internal combo selection
				 * to X selection. */
    Tk_Font tkfont;		/* Information about text font, or NULL. */
    XColor *fgColorPtr;		/* Text color in normal mode. */
    XColor *disabledFgColorPtr;	/* Text color in disabled mode. */
    XColor *highlightBgColorPtr;/* Color for drawing traversal highlight
				 * area when highlight is off. */
    XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
    int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */
    Tk_3DBorder insertBorder;	/* Used to draw vertical bar for insertion
				 * cursor. */
    int insertBorderWidth;	/* Width of 3-D border around insert cursor. */
    int insertOffTime;		/* Number of milliseconds cursor should spend
				 * in "off" state for each blink. */
    int insertOnTime;		/* Number of milliseconds cursor should spend
				 * in "on" state for each blink. */
    int insertWidth;		/* Total width of insert cursor. */
    Tk_Justify justify;		/* Justification to use for text within
				 * window. */
    int relief;			/* 3-D effect: TK_RELIEF_RAISED, etc. */
    Tk_3DBorder selBorder;	/* Border and background for selected
				 * characters. */
    int selBorderWidth;		/* Width of border around selection. */
    XColor *selFgColorPtr;	/* Foreground color for selected text. */
    char *showChar;		/* Value of -show option.  If non-NULL, first
				 * character is used for displaying all
				 * characters in combo.  Malloc'ed. */
    int state;		        /* Normal or disabled.  Combo is read-only
				 * when disabled. */
    char *textVarName;		/* Name of variable (malloc'ed) or NULL.
				 * If non-NULL, combo's string tracks the
				 * contents of this variable and vice versa. */
    char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */
    int prefWidth;		/* Desired width of window, measured in
				 * average characters. */
    char *scrollCmd;		/* Command prefix for communicating with
				 * scrollbar(s).  Malloc'ed.  NULL means
				 * no command to issue. */
    char *spinUpCmd;		/* Command to invoke for spinup button.
				 * NULL means no command to issue. */
    char *spinDownCmd;		/* Command to invoke for spindown button.
				 * NULL means no command to issue. */
    char *dropCmd;		/* Command to invoke for combo button.
				 * NULL means no command to issue. */

    /*
     * Fields whose values are derived from the current values of the
     * configuration settings above.
     */

    int numBytes;		/* Length of string in bytes. */
    int numChars;		/* Length of string in characters.  Both
				 * string and displayString have the same
				 * character length, but may have different
				 * byte lengths due to being made from
				 * different UTF-8 characters. */
    char *displayString;	/* String to use when displaying.  This may
				 * be a pointer to string, or a pointer to
				 * malloced memory with the same character
				 * length as string but whose characters
				 * are all equal to showChar. */
    int numDisplayBytes;	/* Length of displayString in bytes. */
    int inset;			/* Number of pixels on the left and right
				 * sides that are taken up by XPAD, borderWidth
				 * (if any), and highlightWidth (if any). */
    Tk_TextLayout textLayout;	/* Cached text layout information. */
    int layoutX, layoutY;	/* Origin for layout. */
    int leftX;			/* X position at which character at leftIndex
				 * is drawn (varies depending on justify). */
    int leftIndex;		/* Character index of left-most character
				 * visible in window. */
    Tcl_TimerToken insertBlinkHandler;
				/* Timer handler used to blink cursor on and
				 * off. */
    GC textGC;			/* For drawing normal text. */
    GC selTextGC;		/* For drawing selected text. */
    GC highlightGC;		/* For drawing traversal highlight. */
    int avgWidth;		/* Width of average character. */
    int bWidth;			/* Width of a button control. */
    int flags;			/* Miscellaneous flags;  see below for
				 * definitions. */
    Tk_TSOffset tsoffset;

    char *validateCmd;          /* Command prefix to use when invoking
				 * validate command.  NULL means don't
				 * invoke commands.  Malloc'ed. */
    int validate;               /* Non-zero means try to validate */
    char *invalidCmd;		/* Command called when a validation returns 0
				 * (successfully fails), defaults to {}. */
    int controls;		/* Type of controls to show */
    int selElement;		/* currently selected control */
    int curElement;		/* currently selected control */

    int repeatDelay;		/* repeat delay */
    int repeatInterval;		/* repeat interval */

    double fromValue;		/* Value corresponding to left/top of dial */
    double toValue;		/* Value corresponding to right/bottom
				 * of dial */
    double resolution;		/* If > 0, all values are rounded to an
				 * even multiple of this value. */
    int digits;			/* Number of significant digits to print
				 * in values.  0 means we get to choose the
				 * number based on resolution and/or the
				 * range of the dial. */
    char valueFormat[10];	/* Sprintf conversion specifier computed from
				 * digits and other information; used for
				 * the value. */

    char *mode;			/* mode */

    char *listVarName;		/* List variable name */
    Tcl_Obj *listObj;		/* Pointer to the list object being used */
    int nElements;		/* Holds the current count of elements */
} Combo;

/*
 * Assigned bits of "flags" fields of Combo structures, and what those
 * bits mean:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler has
 *				already been queued to redisplay the combo.
 * BORDER_NEEDED:		Non-zero means 3-D border must be redrawn
 *				around window during redisplay.  Normally
 *				only text portion needs to be redrawn.
 * CURSOR_ON:			Non-zero means insert cursor is displayed at
 *				present.  0 means it isn't displayed.
 * GOT_FOCUS:			Non-zero means this window has the input
 *				focus.
 * UPDATE_SCROLLBAR:		Non-zero means scrollbar should be updated
 *				during next redisplay operation.
 * GOT_SELECTION:		Non-zero means we've claimed the selection.
 * COMBO_DELETED:		This combo has been effectively destroyed.
 * VALIDATING:                  Non-zero means we are in a validateCmd
 * VALIDATE_VAR:                Non-zero means we are attempting to validate
 *                              the combo's textvariable with validateCmd
 * VALIDATE_ABORT:              Non-zero if validatecommand signals an abort
 *                              for current procedure and make no changes
 */

#define REDRAW_PENDING		1
#define BORDER_NEEDED		2
#define CURSOR_ON		4
#define GOT_FOCUS		8
#define UPDATE_SCROLLBAR	0x10
#define GOT_SELECTION		0x20
#define COMBO_DELETED           0x40
#define VALIDATING              0x80
#define VALIDATE_VAR            0x100
#define VALIDATE_ABORT          0x200

/*
 * The following macro defines how many extra pixels to leave on each
 * side of the text in the combo.
 */

#define XPAD 1
#define YPAD 1

/*
 * The following enum is used to define a type for the -state option
 * of the Combo widget.  These values are used as indices into the 
 * string table below.
 */

enum state {
    STATE_DISABLED, STATE_NORMAL
};

static char *stateStrings[] = {
    "disabled", "normal", (char *) NULL
};

/*
 * Definitions for -validate option values:
 */

static char *validateStrings[] = {
    "all", "key", "focus", "focusin", "focusout", "none", (char *) NULL
};
enum validateType {
    VALIDATE_ALL, VALIDATE_KEY, VALIDATE_FOCUS,
    VALIDATE_FOCUSIN, VALIDATE_FOCUSOUT, VALIDATE_NONE,
    /*
     * These extra enums are for use with ComboValidateChange
     */
    VALIDATE_FORCED, VALIDATE_DELETE, VALIDATE_INSERT, VALIDATE_BUTTON
};
#define DEF_COMBO_VALIDATE	"none"
#define DEF_COMBO_INVALIDCMD	""

/*
 * Definitions for -controls option values:
 */

static char *controlsStrings[] = {
    "all", "drop", "none", "spin", (char *) NULL
};
enum controlsType {
    CONTROLS_ALL, CONTROLS_COMBO, CONTROLS_NONE, CONTROLS_SPIN
};
#define DEF_COMBO_CONTROLS	"none"

/*
 * Information used for argv parsing.
 */

static Tk_OptionSpec optionSpecs[] = {
    {TK_OPTION_BORDER, "-background", "background", "Background",
	DEF_COMBO_BG_COLOR, -1, Tk_Offset(Combo, normalBorder),
	0, (ClientData) DEF_COMBO_BG_MONO, 0},
    {TK_OPTION_SYNONYM, "-bd", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-borderwidth", 0},
    {TK_OPTION_SYNONYM, "-bg", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-background", 0},
    {TK_OPTION_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_COMBO_BORDER_WIDTH, -1, Tk_Offset(Combo, borderWidth), 
        0, 0, 0},
    {TK_OPTION_STRING_TABLE, "-controls", "controls", "Controls",
       DEF_COMBO_CONTROLS, -1, Tk_Offset(Combo, controls),
       0, (ClientData) controlsStrings, 0},
    {TK_OPTION_CURSOR, "-cursor", "cursor", "Cursor",
	DEF_COMBO_CURSOR, -1, Tk_Offset(Combo, cursor),
	TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_INT, "-digits", "digits", "Digits",
	DEF_COMBO_DIGITS, -1, Tk_Offset(Combo, digits), 0, 0, 0},
    {TK_OPTION_SYNONYM, "-disabledfg", "disabledForeground", (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-disabledforeground", 0},
    {TK_OPTION_COLOR, "-disabledforeground", "disabledForeground",
	"DisabledForeground", DEF_COMBO_DISABLED_FG_COLOR, -1,
	Tk_Offset(Combo, disabledFgColorPtr),
	0, (ClientData) DEF_COMBO_DISABLED_FG_MONO, 0},
    {TK_OPTION_STRING, "-dropcommand", "dropCommand", "DropCommand",
	DEF_COMBO_INVALIDCMD, -1, Tk_Offset(Combo, dropCmd),
	0, 0, 0},
    {TK_OPTION_CURSOR, "-dropcursor", "dropCursor", "ButtonCursor",
	DEF_COMBO_COMBO_CURSOR, -1, Tk_Offset(Combo, dCursor),
	TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_BOOLEAN, "-exportselection", "exportSelection",
        "ExportSelection", DEF_COMBO_EXPORT_SELECTION, -1, 
        Tk_Offset(Combo, exportSelection), 0, 0, 0},
    {TK_OPTION_SYNONYM, "-fg", "foreground", (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-foreground", 0},
    {TK_OPTION_FONT, "-font", "font", "Font",
	DEF_COMBO_FONT, -1, Tk_Offset(Combo, tkfont), 0, 0, 0},
    {TK_OPTION_COLOR, "-foreground", "foreground", "Foreground",
	DEF_COMBO_FG, -1, Tk_Offset(Combo, fgColorPtr), 0, 
        0, 0},
    {TK_OPTION_DOUBLE, "-from", "from", "From",
	DEF_COMBO_FROM, -1, Tk_Offset(Combo, fromValue), 0, 0, 0},
    {TK_OPTION_COLOR, "-highlightbackground", "highlightBackground",
	"HighlightBackground", DEF_COMBO_HIGHLIGHT_BG,
	-1, Tk_Offset(Combo, highlightBgColorPtr), 
        0, 0, 0},
    {TK_OPTION_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
	DEF_COMBO_HIGHLIGHT, -1, Tk_Offset(Combo, highlightColorPtr),
	0, 0, 0},
    {TK_OPTION_PIXELS, "-highlightthickness", "highlightThickness",
	"HighlightThickness", DEF_COMBO_HIGHLIGHT_WIDTH, -1, 
	Tk_Offset(Combo, highlightWidth), 0, 0, 0},
    {TK_OPTION_BORDER, "-insertbackground", "insertBackground", "Foreground",
	DEF_COMBO_INSERT_BG, -1, Tk_Offset(Combo, insertBorder), 
        0, 0, 0},
    {TK_OPTION_PIXELS, "-insertborderwidth", "insertBorderWidth", 
        "BorderWidth", DEF_COMBO_INSERT_BD_COLOR, -1, 
        Tk_Offset(Combo, insertBorderWidth), 0, 
        (ClientData) DEF_COMBO_INSERT_BD_MONO, 0},
    {TK_OPTION_INT, "-insertofftime", "insertOffTime", "OffTime",
        DEF_COMBO_INSERT_OFF_TIME, -1, Tk_Offset(Combo, insertOffTime), 
        0, 0, 0},
    {TK_OPTION_INT, "-insertontime", "insertOnTime", "OnTime",
        DEF_COMBO_INSERT_ON_TIME, -1, Tk_Offset(Combo, insertOnTime), 
        0, 0, 0},
    {TK_OPTION_PIXELS, "-insertwidth", "insertWidth", "InsertWidth",
	DEF_COMBO_INSERT_WIDTH, -1, Tk_Offset(Combo, insertWidth), 
        0, 0, 0},
    {TK_OPTION_STRING, "-invalidcommand", "invalidCommand", "InvalidCommand",
	DEF_COMBO_INVALIDCMD, -1, Tk_Offset(Combo, invalidCmd),
	0, 0, 0},
    {TK_OPTION_SYNONYM, "-invcmd", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-invalidcommand", 0},
    {TK_OPTION_JUSTIFY, "-justify", "justify", "Justify",
	DEF_COMBO_JUSTIFY, -1, Tk_Offset(Combo, justify), 0, 0, 0},
    {TK_OPTION_STRING, "-listvariable", "listVariable", "Variable",
	 DEF_COMBO_LIST_VARIABLE, -1, Tk_Offset(Combo, listVarName),
	 TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_STRING, "-mode", "mode", "Mode",
	 DEF_COMBO_MODE, -1, Tk_Offset(Combo, mode),
	 TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_RELIEF, "-relief", "relief", "Relief",
	DEF_COMBO_RELIEF, -1, Tk_Offset(Combo, relief), 
        0, 0, 0},
    {TK_OPTION_INT, "-repeatdelay", "repeatDelay", "RepeatDelay",
        DEF_COMBO_REPEAT_DELAY, -1, Tk_Offset(Combo, repeatDelay), 
        0, 0, 0},
    {TK_OPTION_INT, "-repeatinterval", "repeatInterval", "RepeatInterval",
        DEF_COMBO_REPEAT_INTERVAL, -1, Tk_Offset(Combo, repeatInterval), 
        0, 0, 0},
    {TK_OPTION_DOUBLE, "-resolution", "resolution", "Resolution",
	DEF_COMBO_RESOLUTION, -1, Tk_Offset(Combo, resolution), 0, 0, 0},
    {TK_OPTION_BORDER, "-selectbackground", "selectBackground", "Foreground",
        DEF_COMBO_SELECT_COLOR, -1, Tk_Offset(Combo, selBorder),
        0, (ClientData) DEF_COMBO_SELECT_MONO, 0},
    {TK_OPTION_PIXELS, "-selectborderwidth", "selectBorderWidth", 
        "BorderWidth", DEF_COMBO_SELECT_BD_COLOR, -1, 
        Tk_Offset(Combo, selBorderWidth), 
        0, (ClientData) DEF_COMBO_SELECT_BD_MONO, 0},
    {TK_OPTION_COLOR, "-selectforeground", "selectForeground", "Background",
	DEF_COMBO_SELECT_FG_COLOR, -1, Tk_Offset(Combo, selFgColorPtr),
	0, (ClientData) DEF_COMBO_SELECT_FG_MONO, 0},
    {TK_OPTION_STRING, "-show", "show", "Show",
        DEF_COMBO_SHOW, -1, Tk_Offset(Combo, showChar), 
        TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_STRING, "-spincommand", "spinCommand", "SpinCommand",
	DEF_COMBO_SPINCMD, -1, Tk_Offset(Combo, spinDownCmd),
	0, 0, 0},
    {TK_OPTION_CURSOR, "-spindowncursor", "spinDownCursor", "ButtonCursor",
	DEF_COMBO_SPINDOWN_CURSOR, -1, Tk_Offset(Combo, sdCursor),
	TK_OPTION_NULL_OK, 0, 0},
#if 0
    {TK_OPTION_STRING, "-spinupcommand", "spinUpCommand", "SpinUpCommand",
	DEF_COMBO_SPINCMD, -1, Tk_Offset(Combo, spinUpCmd),
	0, 0, 0},
#endif
    {TK_OPTION_CURSOR, "-spinupcursor", "spinUpCursor", "ButtonCursor",
	DEF_COMBO_SPINUP_CURSOR, -1, Tk_Offset(Combo, suCursor),
	TK_OPTION_NULL_OK, 0, 0},
    {TK_OPTION_STRING_TABLE, "-state", "state", "State",
	DEF_COMBO_STATE, -1, Tk_Offset(Combo, state), 
        0, (ClientData) stateStrings, 0},
    {TK_OPTION_STRING, "-takefocus", "takeFocus", "TakeFocus",
	DEF_COMBO_TAKE_FOCUS, -1, Tk_Offset(Combo, takeFocus), 
        TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_STRING, "-textvariable", "textVariable", "Variable",
	DEF_COMBO_TEXT_VARIABLE, -1, Tk_Offset(Combo, textVarName),
	TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_DOUBLE, "-to", "to", "To",
	DEF_COMBO_TO, -1, Tk_Offset(Combo, toValue), 0, 0, 0},
    {TK_OPTION_STRING_TABLE, "-validate", "validate", "Validate",
       DEF_COMBO_VALIDATE, -1, Tk_Offset(Combo, validate),
       0, (ClientData) validateStrings, 0},
    {TK_OPTION_STRING, "-validatecommand", "validateCommand", "ValidateCommand",
       (char *) NULL, -1, Tk_Offset(Combo, validateCmd),
       TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_SYNONYM, "-vcmd", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, (ClientData) "-validatecommand", 0},
    {TK_OPTION_INT, "-width", "width", "Width",
	DEF_COMBO_WIDTH, -1, Tk_Offset(Combo, prefWidth), 0, 0, 0},
    {TK_OPTION_STRING, "-xscrollcommand", "xScrollCommand", "ScrollCommand",
	DEF_COMBO_SCROLL_COMMAND, -1, Tk_Offset(Combo, scrollCmd),
	TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, -1, 0, 0, 0}
};

/*
 * Flags for GetComboIndex procedure:
 */

#define ZERO_OK			1
#define LAST_PLUS_ONE_OK	2

/*
 * The following tables define the combo widget commands (and sub-
 * commands) and map the indexes into the string tables into 
 * enumerated types used to dispatch the combo widget command.
 */

static char *commandNames[] = {
    "bbox", "cget", "configure", "delete", "get", "icursor", "identify",
    "index", "insert", "invoke", "scan", "selection",
    "validate", "xview", (char *) NULL
};

enum command {
    COMMAND_BBOX, COMMAND_CGET, COMMAND_CONFIGURE, COMMAND_DELETE, 
    COMMAND_GET, COMMAND_ICURSOR, COMMAND_IDENTIFY, COMMAND_INDEX,
    COMMAND_INSERT, COMMAND_INVOKE, COMMAND_SCAN, COMMAND_SELECTION,
    COMMAND_VALIDATE, COMMAND_XVIEW
};

static char *selCommandNames[] = {
    "adjust", "clear", "element", "from", "present", "range", "to",
    (char *) NULL
};

enum selcommand {
    SELECTION_ADJUST, SELECTION_CLEAR, SELECTION_ELEMENT, SELECTION_FROM, 
    SELECTION_PRESENT, SELECTION_RANGE, SELECTION_TO
};

/*
 * Extra for selection of elements
 */

static char *selElementNames[] = {
    "drop", "none", "spindown", "spinup", (char *) NULL, "entry"
};
enum selelement {
    SEL_DROP, SEL_NONE, SEL_SPINDOWN, SEL_SPINUP, SEL_NULL, SEL_ENTRY
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int		ConfigureCombo _ANSI_ARGS_((Tcl_Interp *interp,
			    Combo *comboPtr, int objc, 
                            Tcl_Obj *CONST objv[], int flags));
static void		DeleteChars _ANSI_ARGS_((Combo *comboPtr, int index,
			    int count));
static void		DestroyCombo _ANSI_ARGS_((char *memPtr));
static void		DisplayCombo _ANSI_ARGS_((ClientData clientData));
static void		ComboBlinkProc _ANSI_ARGS_((ClientData clientData));
static void		ComboCmdDeletedProc _ANSI_ARGS_((
			    ClientData clientData));
static void		ComboComputeGeometry _ANSI_ARGS_((Combo *comboPtr));
static void		ComboEventProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static void		ComboFocusProc _ANSI_ARGS_ ((Combo *comboPtr,
			    int gotFocus));
static int		ComboFetchSelection _ANSI_ARGS_((ClientData clientData,
			    int offset, char *buffer, int maxBytes));
static void		ComboLostSelection _ANSI_ARGS_((
			    ClientData clientData));
static void		EventuallyRedraw _ANSI_ARGS_((Combo *comboPtr));
static void		ComboScanTo _ANSI_ARGS_((Combo *comboPtr, int y));
static void		ComboSetValue _ANSI_ARGS_((Combo *comboPtr,
			    char *value));
static void		ComboSelectTo _ANSI_ARGS_((
			    Combo *comboPtr, int index));
static char *		ComboTextVarProc _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, char *name1, char *name2,
			    int flags));
static char *		ComboListVarProc _ANSI_ARGS_ ((ClientData clientData,
	                    Tcl_Interp *interp, char *name1, char *name2,
 	                    int flags));
static void		ComboUpdateScrollbar _ANSI_ARGS_((Combo *comboPtr));
static int		ComboValidate _ANSI_ARGS_((Combo *comboPtr,
			    char *cmd));
static int		ComboValidateChange _ANSI_ARGS_((Combo *comboPtr,
			    char *change, char *new, int index, int type));
static void		ExpandPercents _ANSI_ARGS_((Combo *comboPtr,
			    char *before, char *change, char *new, int index,
			    int type, Tcl_DString *dsPtr));
static void		ComboValueChanged _ANSI_ARGS_((Combo *comboPtr));
static void		ComboVisibleRange _ANSI_ARGS_((Combo *comboPtr,
			    double *firstPtr, double *lastPtr));
static int		ComboWidgetObjCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int objc, 
			    Tcl_Obj *CONST objv[]));
static void		ComboWorldChanged _ANSI_ARGS_((
			    ClientData instanceData));
static int		GetComboIndex _ANSI_ARGS_((Tcl_Interp *interp,
			    Combo *comboPtr, char *string, int *indexPtr));
static int		GetComboElement _ANSI_ARGS_((Combo *comboPtr,
			    int x, int y));
static void		InsertChars _ANSI_ARGS_((Combo *comboPtr, int index,
			    char *string));
static int		ComboInvoke _ANSI_ARGS_((Tcl_Interp *interp,
			    Combo *comboPtr, int element));

/*
 * The structure below defines combo class behavior by means of procedures
 * that can be invoked from generic window code.
 */
#ifdef TkClassProcs
static TkClassProcs comboClass = {
    NULL,			/* createProc. */
    ComboWorldChanged,		/* geometryProc. */
    NULL			/* modalProc. */
};
#endif

/*
 *--------------------------------------------------------------
 *
 * Tk_ComboObjCmd --
 *
 *	This procedure is invoked to process the "combo" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
Tk_ComboObjCmd(clientData, interp, objc, objv)
    ClientData clientData;	/* Either NULL or pointer to option table. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];      /* Argument objects. */
{
    register Combo *comboPtr;
    Tk_OptionTable optionTable;
    Tk_Window tkwin;

    optionTable = (Tk_OptionTable) clientData;
    if (optionTable == NULL) {
	Tcl_CmdInfo info;
	char *name;

	/*
	 * We haven't created the option table for this widget class
	 * yet.  Do it now and save the table as the clientData for
	 * the command, so we'll have access to it in future
	 * invocations of the command.
	 */

	optionTable = Tk_CreateOptionTable(interp, optionSpecs);
	name = Tcl_GetString(objv[0]);
	Tcl_GetCommandInfo(interp, name, &info);
	info.objClientData = (ClientData) optionTable;
	Tcl_SetCommandInfo(interp, name, &info);
    }

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "pathName ?options?");
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp),
            Tcl_GetString(objv[1]), (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }

    /*
     * Initialize the fields of the structure that won't be initialized
     * by ConfigureCombo, or that ConfigureCombo requires to be
     * initialized already (e.g. resource pointers).
     */

    comboPtr			= (Combo *) ckalloc(sizeof(Combo));
    comboPtr->tkwin		= tkwin;
    comboPtr->display		= Tk_Display(tkwin);
    comboPtr->interp		= interp;
    comboPtr->widgetCmd		= Tcl_CreateObjCommand(interp,
	    Tk_PathName(comboPtr->tkwin), ComboWidgetObjCmd,
	    (ClientData) comboPtr, ComboCmdDeletedProc);
    comboPtr->optionTable	= optionTable;
    comboPtr->string		= (char *) ckalloc(1);
    comboPtr->string[0]		= '\0';
    comboPtr->insertPos		= 0;
    comboPtr->selectFirst	= -1;
    comboPtr->selectLast	= -1;
    comboPtr->selectAnchor	= 0;
    comboPtr->scanMarkX		= 0;
    comboPtr->scanMarkIndex	= 0;

    comboPtr->normalBorder	= NULL;
    comboPtr->borderWidth	= 0;
    comboPtr->cursor		= None;
    comboPtr->exportSelection	= 1;
    comboPtr->tkfont		= NULL;
    comboPtr->fgColorPtr	= NULL;
    comboPtr->disabledFgColorPtr	= NULL;
    comboPtr->highlightBgColorPtr	= NULL;
    comboPtr->highlightColorPtr	= NULL;
    comboPtr->highlightWidth	= 0;
    comboPtr->insertBorder	= NULL;
    comboPtr->insertBorderWidth	= 0;
    comboPtr->insertOffTime	= 0;
    comboPtr->insertOnTime	= 0;
    comboPtr->insertWidth	= 0;
    comboPtr->justify		= TK_JUSTIFY_LEFT;
    comboPtr->relief		= TK_RELIEF_FLAT;
    comboPtr->selBorder		= NULL;
    comboPtr->selBorderWidth	= 0;
    comboPtr->selFgColorPtr	= NULL;
    comboPtr->showChar		= NULL;
    comboPtr->state		= STATE_NORMAL;
    comboPtr->textVarName	= NULL;
    comboPtr->takeFocus		= NULL;
    comboPtr->prefWidth		= 0;
    comboPtr->scrollCmd		= NULL;
    comboPtr->numBytes		= 0;
    comboPtr->numChars		= 0;
    comboPtr->displayString	= comboPtr->string;
    comboPtr->numDisplayBytes	= 0;
    comboPtr->inset		= XPAD;
    comboPtr->textLayout	= NULL;
    comboPtr->layoutX		= 0;
    comboPtr->layoutY		= 0;
    comboPtr->leftX		= 0;
    comboPtr->leftIndex		= 0;
    comboPtr->insertBlinkHandler	= (Tcl_TimerToken) NULL;
    comboPtr->textGC		= None;
    comboPtr->selTextGC		= None;
    comboPtr->highlightGC	= None;
    comboPtr->avgWidth		= 1;
    comboPtr->flags		= 0;
    comboPtr->validateCmd	= NULL;
    comboPtr->validate		= VALIDATE_NONE;
    comboPtr->invalidCmd	= NULL;
    comboPtr->controls		= CONTROLS_NONE;
    comboPtr->selElement	= SEL_NONE;
    comboPtr->curElement	= SEL_NONE;
    comboPtr->dropCmd		= NULL;
    comboPtr->spinUpCmd		= NULL;
    comboPtr->spinDownCmd	= NULL;
    comboPtr->dCursor		= None;
    comboPtr->sdCursor		= None;
    comboPtr->suCursor		= None;
    comboPtr->repeatDelay	= 400;
    comboPtr->repeatInterval	= 100;
    comboPtr->fromValue		= 0.0;
    comboPtr->toValue		= 100.0;
    comboPtr->resolution	= 1.0;
    comboPtr->listVarName	= NULL;
    comboPtr->listObj		= NULL;
    comboPtr->nElements		= 0;
    comboPtr->mode		= NULL;

    Tk_SetClass(comboPtr->tkwin, "Combo");
#ifdef TkClassProcs
    TkSetClassProcs(comboPtr->tkwin, &comboClass, (ClientData) comboPtr);
#endif
    Tk_CreateEventHandler(comboPtr->tkwin,
	    PointerMotionMask|ExposureMask|StructureNotifyMask|FocusChangeMask,
	    ComboEventProc, (ClientData) comboPtr);
    Tk_CreateSelHandler(comboPtr->tkwin, XA_PRIMARY, XA_STRING,
	    ComboFetchSelection, (ClientData) comboPtr, XA_STRING);

    if (Tk_InitOptions(interp, (char *) comboPtr, optionTable, tkwin)
	    != TCL_OK) {
	Tk_DestroyWindow(comboPtr->tkwin);
	return TCL_ERROR;
    }
    if (ConfigureCombo(interp, comboPtr, objc-2, objv+2, 0) != TCL_OK) {
	goto error;
    }
    
    Tcl_SetResult(interp, Tk_PathName(comboPtr->tkwin), TCL_STATIC);
    return TCL_OK;

    error:
    Tk_DestroyWindow(comboPtr->tkwin);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ComboWidgetObjCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
ComboWidgetObjCmd(clientData, interp, objc, objv)
    ClientData clientData;	/* Information about combo widget. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    Combo *comboPtr = (Combo *) clientData;
    int cmdIndex, selIndex, result;
    Tcl_Obj *objPtr;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg ...?");
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) comboPtr);

    /* 
     * Parse the widget command by looking up the second token in
     * the list of valid command names. 
     */

    result = Tcl_GetIndexFromObj(interp, objv[1], commandNames,
	    "option", 0, &cmdIndex);
    if (result != TCL_OK) {
	return result;
    }

    switch (cmdIndex) {
        case COMMAND_BBOX: {
	    int index, x, y, width, height;
	    char buf[TCL_INTEGER_SPACE * 4];

	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "index");
		goto error;
	    }
	    if (GetComboIndex(interp, comboPtr, Tcl_GetString(objv[2]), 
                    &index) != TCL_OK) {
	        goto error;
	    }
	    if ((index == comboPtr->numChars) && (index > 0)) {
	        index--;
	    }
	    Tk_CharBbox(comboPtr->textLayout, index, &x, &y, 
                    &width, &height);
	    sprintf(buf, "%d %d %d %d", x + comboPtr->layoutX,
		    y + comboPtr->layoutY, width, height);
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	    break;
	} 
	
        case COMMAND_CGET: {
	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "option");
		goto error;
	    }
	    
	    objPtr = Tk_GetOptionValue(interp, (char *) comboPtr,
		    comboPtr->optionTable, objv[2], comboPtr->tkwin);
	    if (objPtr == NULL) {
		 goto error;
	    } else {
		Tcl_SetObjResult(interp, objPtr);
	    }
	    break;
	}

        case COMMAND_CONFIGURE: {
	    if (objc <= 3) {
		objPtr = Tk_GetOptionInfo(interp, (char *) comboPtr,
			comboPtr->optionTable,
			(objc == 3) ? objv[2] : (Tcl_Obj *) NULL,
			comboPtr->tkwin);
		if (objPtr == NULL) {
		    goto error;
		} else {
		    Tcl_SetObjResult(interp, objPtr);
		}
	    } else {
		result = ConfigureCombo(interp, comboPtr, objc-2, objv+2, 0);
	    }
	    break;
	}

        case COMMAND_DELETE: {
	    int first, last;

	    if ((objc < 3) || (objc > 4)) {
	        Tcl_WrongNumArgs(interp, 2, objv, "firstIndex ?lastIndex?");
		goto error;
	    }
	    if (GetComboIndex(interp, comboPtr, Tcl_GetString(objv[2]), 
                    &first) != TCL_OK) {
	        goto error;
	    }
	    if (objc == 3) {
	        last = first + 1;
	    } else {
	        if (GetComboIndex(interp, comboPtr, Tcl_GetString(objv[3]), 
                        &last) != TCL_OK) {
		    goto error;
		}
	    }
	    if ((last >= first) && (comboPtr->state == STATE_NORMAL)) {
	        DeleteChars(comboPtr, first, last - first);
	    }
	    break;
	}

        case COMMAND_GET: {
	    if (objc != 2) {
	        Tcl_WrongNumArgs(interp, 2, objv, (char *) NULL);
		goto error;
	    }
	    Tcl_SetResult(interp, comboPtr->string, TCL_STATIC);
	    break;
	}

        case COMMAND_ICURSOR: {
	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "pos");
		goto error;
	    }
	    if (GetComboIndex(interp, comboPtr, Tcl_GetString(objv[2]),
                    &comboPtr->insertPos) != TCL_OK) {
	        goto error;
	    }
	    EventuallyRedraw(comboPtr);
	    break;
	}
	
        case COMMAND_IDENTIFY: {
	    int x, y, elem;

	    if (objc != 4) {
	        Tcl_WrongNumArgs(interp, 2, objv, "x y");
		goto error;
	    }
	    if ((Tcl_GetIntFromObj(interp, objv[2], &x) != TCL_OK) ||
		    (Tcl_GetIntFromObj(interp, objv[3], &y) != TCL_OK)) {
		goto error;
	    }
	    elem = GetComboElement(comboPtr, x, y);
	    if (elem != SEL_NONE) {
		Tcl_SetResult(interp, selElementNames[elem], TCL_VOLATILE);
	    }
	    break;
	}
	
        case COMMAND_INDEX: {
	    int index;

	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "string");
		goto error;
	    }
	    if (GetComboIndex(interp, comboPtr, Tcl_GetString(objv[2]), 
                    &index) != TCL_OK) {
	        goto error;
	    }
	    Tcl_SetObjResult(interp, Tcl_NewIntObj(index));
	    break;
	}

        case COMMAND_INSERT: {
	    int index;

	    if (objc != 4) {
	        Tcl_WrongNumArgs(interp, 2, objv, "index text");
		goto error;
	    }
	    if (GetComboIndex(interp, comboPtr, Tcl_GetString(objv[2]), 
                    &index) != TCL_OK) {
	        goto error;
	    }
	    if (comboPtr->state == STATE_NORMAL) {
	        InsertChars(comboPtr, index, Tcl_GetString(objv[3]));
	    }
	    break;
	}

        case COMMAND_INVOKE: {
	    if (objc != 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "elemName");
		goto error;
	    }
	    result = Tcl_GetIndexFromObj(interp, objv[2],
		    selElementNames, "element", 0, &cmdIndex);
	    if (result != TCL_OK) {
		goto error;
	    }
	    if (comboPtr->state != STATE_DISABLED) {
		if (ComboInvoke(interp, comboPtr, cmdIndex) != TCL_OK) {
		    goto error;
		}
	    }
	    break;
	}

        case COMMAND_SCAN: {
	    int x;
	    char *minorCmd;

	    if (objc != 4) {
	        Tcl_WrongNumArgs(interp, 2, objv, "mark|dragto x");
		goto error;
	    }
	    if (Tcl_GetIntFromObj(interp, objv[3], &x) != TCL_OK) {
	        goto error;
	    }

	    minorCmd = Tcl_GetString(objv[2]);
	    if (minorCmd[0] == 'm' 
                    && (strncmp(minorCmd, "mark", strlen(minorCmd)) == 0)) {
	        comboPtr->scanMarkX = x;
		comboPtr->scanMarkIndex = comboPtr->leftIndex;
	    } else if ((minorCmd[0] == 'd')
		&& (strncmp(minorCmd, "dragto", strlen(minorCmd)) == 0)) {
	        ComboScanTo(comboPtr, x);
	    } else {
	        Tcl_AppendResult(interp, "bad scan option \"", 
                        Tcl_GetString(objv[2]), "\": must be mark or dragto", 
                        (char *) NULL);
		goto error;
	    }
	    break;
	}
	    
	case COMMAND_SELECTION: {
	    int index, index2;

	    if (objc < 3) {
	        Tcl_WrongNumArgs(interp, 2, objv, "option ?index?");
		goto error;
	    }

	    /* 
	     * Parse the selection sub-command, using the command
	     * table "selCommandNames" defined above.
	     */
	    
	    result = Tcl_GetIndexFromObj(interp, objv[2], selCommandNames,
                    "selection option", 0, &selIndex);
	    if (result != TCL_OK) {
	        goto error;
	    }

	    switch(selIndex) {
	        case SELECTION_ADJUST: {
		    if (objc != 4) {
		        Tcl_WrongNumArgs(interp, 3, objv, "index");
			goto error;
		    }
		    if (GetComboIndex(interp, comboPtr, 
                            Tcl_GetString(objv[3]), &index) != TCL_OK) {
		        goto error;
		    }
		    if (comboPtr->selectFirst >= 0) {
		        int half1, half2;
		
			half1 = (comboPtr->selectFirst 
			        + comboPtr->selectLast)/2;
			half2 = (comboPtr->selectFirst 
				+ comboPtr->selectLast + 1)/2;
			if (index < half1) {
			    comboPtr->selectAnchor = comboPtr->selectLast;
			} else if (index > half2) {
			    comboPtr->selectAnchor = comboPtr->selectFirst;
			} else {
			  /*
			   * We're at about the halfway point in the 
			   * selection; just keep the existing anchor.
			   */
			}
		    }
		    ComboSelectTo(comboPtr, index);
		    break;
		}

	        case SELECTION_CLEAR: {
		    if (objc != 3) {
		        Tcl_WrongNumArgs(interp, 3, objv, (char *) NULL);
			goto error;
		    }
		    if (comboPtr->selectFirst >= 0) {
		        comboPtr->selectFirst = -1;
			comboPtr->selectLast = -1;
			EventuallyRedraw(comboPtr);
		    }
		    goto done;
		}

	        case SELECTION_FROM: {
		    if (objc != 4) {
		        Tcl_WrongNumArgs(interp, 3, objv, "index");
			goto error;
		    }
		    if (GetComboIndex(interp, comboPtr, 
                            Tcl_GetString(objv[3]), &index) != TCL_OK) {
		        goto error;
		    }
		    comboPtr->selectAnchor = index;
		    break;
		}

	        case SELECTION_PRESENT: {
		    if (objc != 3) {
		        Tcl_WrongNumArgs(interp, 3, objv, (char *) NULL);
			goto error;
		    }
		    if (comboPtr->selectFirst < 0) {
		        Tcl_SetResult(interp, "0", TCL_STATIC);
		    } else {
		        Tcl_SetResult(interp, "1", TCL_STATIC);
		    }
		    goto done;
		}

	        case SELECTION_RANGE: {
		    if (objc != 5) {
		        Tcl_WrongNumArgs(interp, 3, objv, "start end");
			goto error;
		    }
		    if (GetComboIndex(interp, comboPtr, 
                            Tcl_GetString(objv[3]), &index) != TCL_OK) {
		        goto error;
		    }
		    if (GetComboIndex(interp, comboPtr, 
                            Tcl_GetString(objv[4]),& index2) != TCL_OK) {
		        goto error;
		    }
		    if (index >= index2) {
		        comboPtr->selectFirst = -1;
			comboPtr->selectLast = -1;
		    } else {
		        comboPtr->selectFirst = index;
			comboPtr->selectLast = index2;
		    }
		    if (!(comboPtr->flags & GOT_SELECTION)
			    && (comboPtr->exportSelection)) {
		        Tk_OwnSelection(comboPtr->tkwin, XA_PRIMARY, 
			        ComboLostSelection, (ClientData) comboPtr);
			comboPtr->flags |= GOT_SELECTION;
		    }
		    EventuallyRedraw(comboPtr);
		    break;
		}
		
	        case SELECTION_TO: {
		    if (objc != 4) {
		        Tcl_WrongNumArgs(interp, 3, objv, "index");
			goto error;
		    }
		    if (GetComboIndex(interp, comboPtr, 
                            Tcl_GetString(objv[3]), &index) != TCL_OK) {
		        goto error;
		    }
		    ComboSelectTo(comboPtr, index);
		    break;
		}

	        case SELECTION_ELEMENT: {
		    if ((objc < 3) || (objc > 4)) {
		        Tcl_WrongNumArgs(interp, 3, objv, "?elemName?");
			goto error;
		    }
		    if (objc == 3) {
			Tcl_SetResult(interp,
				selElementNames[comboPtr->selElement],
				TCL_VOLATILE);
		    } else {
			int lastElement = comboPtr->selElement;

			result = Tcl_GetIndexFromObj(interp, objv[3],
				selElementNames, "selection element", 0,
				&(comboPtr->selElement));
			if (result != TCL_OK) {
			    goto error;
			}
			if (lastElement != comboPtr->selElement) {
			    EventuallyRedraw(comboPtr);
			}
		    }
		    break;
		}
	    }
	    break;
	}

        case COMMAND_VALIDATE: {
	    int code;

	    if (objc != 2) {
		Tcl_WrongNumArgs(interp, 2, objv, (char *) NULL);
		goto error;
	    }
	    selIndex = comboPtr->validate;
	    comboPtr->validate = VALIDATE_ALL;
	    code = ComboValidateChange(comboPtr, (char *) NULL,
		    comboPtr->string, -1, VALIDATE_FORCED);
	    if (comboPtr->validate != VALIDATE_NONE) {
		comboPtr->validate = selIndex;
	    }
	    Tcl_SetObjResult(interp, Tcl_NewBooleanObj((code == TCL_OK)));
	    break;
	}

        case COMMAND_XVIEW: {
	    int index;

	    if (objc == 2) {
	        double first, last;
		char buf[TCL_DOUBLE_SPACE * 2];
	    
		ComboVisibleRange(comboPtr, &first, &last);
		sprintf(buf, "%g %g", first, last);
		Tcl_SetResult(interp, buf, TCL_VOLATILE);
		goto done;
	    } else if (objc == 3) {
	        if (GetComboIndex(interp, comboPtr, Tcl_GetString(objv[2]), 
                        &index) != TCL_OK) {
		    goto error;
		}
	    } else {
	        double fraction;
		int count;

		index = comboPtr->leftIndex;
		switch (Tk_GetScrollInfoObj(interp, objc, objv, &fraction, 
                        &count)) {
		    case TK_SCROLL_ERROR: {
		        goto error;
		    }
		    case TK_SCROLL_MOVETO: {
		        index = (int) ((fraction * comboPtr->numChars) + 0.5);
			break;
		    }
		    case TK_SCROLL_PAGES: {
		        int charsPerPage;
		    
			charsPerPage = ((Tk_Width(comboPtr->tkwin)
    			        - 2 * comboPtr->inset) 
                                / comboPtr->avgWidth) - 2;
			if (charsPerPage < 1) {
			    charsPerPage = 1;
			}
			index += count * charsPerPage;
			break;
		    }
		    case TK_SCROLL_UNITS: {
		        index += count;
			break;
		    }
		}
	    }
	    if (index >= comboPtr->numChars) {
	        index = comboPtr->numChars - 1;
	    }
	    if (index < 0) {
	        index = 0;
	    }
	    comboPtr->leftIndex = index;
	    comboPtr->flags |= UPDATE_SCROLLBAR;
	    ComboComputeGeometry(comboPtr);
	    EventuallyRedraw(comboPtr);
	    break;
	}
    }

    done:
    Tcl_Release((ClientData) comboPtr);
    return result;

    error:
    Tcl_Release((ClientData) comboPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyCombo --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of an combo at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the combo is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyCombo(memPtr)
    char *memPtr;		/* Info about combo widget. */
{
    Combo *comboPtr = (Combo *) memPtr;
    comboPtr->flags |= COMBO_DELETED;

    Tcl_DeleteCommandFromToken(comboPtr->interp, comboPtr->widgetCmd);
    if (comboPtr->flags & REDRAW_PENDING) {
        Tcl_CancelIdleCall(DisplayCombo, (ClientData) comboPtr);
    }

    /*
     * Free up all the stuff that requires special handling, then
     * let Tk_FreeOptions handle all the standard option-related
     * stuff.
     */

    ckfree(comboPtr->string);
    if (comboPtr->textVarName != NULL) {
	Tcl_UntraceVar(comboPtr->interp, comboPtr->textVarName,
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		ComboTextVarProc, (ClientData) comboPtr);
    }
    /* If we have an internal list object, free it */
    if (comboPtr->listObj != NULL) {
	Tcl_DecrRefCount(comboPtr->listObj);
	comboPtr->listObj = NULL;
    }
    if (comboPtr->listVarName != NULL) {
	Tcl_UntraceVar(comboPtr->interp, comboPtr->listVarName,
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		ComboListVarProc, (ClientData) comboPtr);
    }
    if (comboPtr->textGC != None) {
	Tk_FreeGC(comboPtr->display, comboPtr->textGC);
    }
    if (comboPtr->selTextGC != None) {
	Tk_FreeGC(comboPtr->display, comboPtr->selTextGC);
    }
    Tcl_DeleteTimerHandler(comboPtr->insertBlinkHandler);
    if (comboPtr->displayString != comboPtr->string) {
	ckfree(comboPtr->displayString);
    }
    Tk_FreeTextLayout(comboPtr->textLayout);
    Tk_FreeConfigOptions((char *) comboPtr, comboPtr->optionTable,
	    comboPtr->tkwin);
    comboPtr->tkwin = NULL;
    ckfree((char *) comboPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureCombo --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or reconfigure)
 *	an combo widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then the interp's result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for comboPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureCombo(interp, comboPtr, objc, objv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Combo *comboPtr;		/* Information about widget; may or may not
				 * already have values for some fields. */
    int objc;			/* Number of valid entries in argv. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    Tk_SavedOptions savedOptions;
    Tcl_Obj *errorResult = NULL;
    Tcl_Obj *oldListObj = NULL;
    int error;
    int oldExport;

    /*
     * Eliminate any existing trace on a variable monitored by the combo.
     */

    if (comboPtr->textVarName != NULL) {
	Tcl_UntraceVar(interp, comboPtr->textVarName, 
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		ComboTextVarProc, (ClientData) comboPtr);
    }
    if (comboPtr->listVarName != NULL) {
	Tcl_UntraceVar(comboPtr->interp, comboPtr->listVarName,
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		ComboListVarProc, (ClientData) comboPtr);
    }

    oldExport = comboPtr->exportSelection;

    for (error = 0; error <= 1; error++) {
	if (!error) {
	    /*
	     * First pass: set options to new values.
	     */

	    if (Tk_SetOptions(interp, (char *) comboPtr,
		    comboPtr->optionTable, objc, objv,
		    comboPtr->tkwin, &savedOptions, (int *) NULL) != TCL_OK) {
		continue;
	    }
	} else {
	    /*
	     * Second pass: restore options to old values.
	     */

	    errorResult = Tcl_GetObjResult(interp);
	    Tcl_IncrRefCount(errorResult);
	    Tk_RestoreSavedOptions(&savedOptions);
	}

	/*
	 * A few other options also need special processing, such as parsing
	 * the geometry and setting the background from a 3-D border.
	 */

	Tk_SetBackgroundFromBorder(comboPtr->tkwin, comboPtr->normalBorder);

	if (comboPtr->insertWidth <= 0) {
	    comboPtr->insertWidth = 2;
	}
	if (comboPtr->insertBorderWidth > comboPtr->insertWidth/2) {
	    comboPtr->insertBorderWidth = comboPtr->insertWidth/2;
	}

	/* Verify the current status of the list var.
	 * PREVIOUS STATE  | NEW STATE   | ACTION
	 * ----------------+-------------+----------------------------------
	 * no listvar      | listvar     | If listvar does not exist, create
	 *                                 it and copy the internal list obj's
	 *                                 content to the new var.  If it does
	 *                                 exist, toss the internal list obj.
	 *
	 * listvar         | no listvar  | Copy old listvar content to the
	 *                                 internal list obj
	 *
	 * listvar         | listvar     | no special action
	 *
	 * no listvar      | no listvar  | no special action
	 */
	oldListObj = comboPtr->listObj;
	if (comboPtr->listVarName != NULL) {
	    Tcl_Obj *listVarObj = Tcl_GetVar2Ex(interp, comboPtr->listVarName,
		    (char *)NULL, TCL_GLOBAL_ONLY);
	    int dummy;
	    if (listVarObj == NULL) {
		if (comboPtr->listObj != NULL) {
		    listVarObj = comboPtr->listObj;
		} else {
		    listVarObj = Tcl_NewObj();
		}
		if (Tcl_SetVar2Ex(interp, comboPtr->listVarName, (char *)NULL,
			listVarObj, TCL_GLOBAL_ONLY) == NULL) {
		    Tcl_DecrRefCount(listVarObj);
		    Tk_RestoreSavedOptions(&savedOptions);
		    continue;
		}
	    }
	    /* Make sure the object is a good list object */
	    if (Tcl_ListObjLength(comboPtr->interp, listVarObj, &dummy)
		    != TCL_OK) {
		Tk_RestoreSavedOptions(&savedOptions);
		Tcl_AppendResult(comboPtr->interp, ": invalid listvar value",
			(char *)NULL);
		continue;
	    }
    
	    comboPtr->listObj = listVarObj;
	    Tcl_TraceVar(comboPtr->interp, comboPtr->listVarName,
		    TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		    ComboListVarProc, (ClientData) comboPtr);
	} else {
	    if (comboPtr->listObj == NULL) {
		comboPtr->listObj = Tcl_NewObj();
	    }
	}
	Tcl_IncrRefCount(comboPtr->listObj);
	if (oldListObj != NULL) {
	    Tcl_DecrRefCount(oldListObj);
	}

	/* Make sure that the list length is correct */
	Tcl_ListObjLength(comboPtr->interp, comboPtr->listObj,
		&comboPtr->nElements);

	/*
	 * Restart the cursor timing sequence in case the on-time or 
	 * off-time just changed.  Set validate temporarily to none,
	 * so the configure doesn't cause it to be triggered.
	 */

	if (comboPtr->flags & GOT_FOCUS) {
	    int validate = comboPtr->validate;
	    comboPtr->validate = VALIDATE_NONE;
	    ComboFocusProc(comboPtr, 1);
	    comboPtr->validate = validate;
	}

	/*
	 * Claim the selection if we've suddenly started exporting it.
	 */

	if (comboPtr->exportSelection && (!oldExport)
	        && (comboPtr->selectFirst != -1)
	        && !(comboPtr->flags & GOT_SELECTION)) {
	    Tk_OwnSelection(comboPtr->tkwin, XA_PRIMARY, ComboLostSelection,
		    (ClientData) comboPtr);
	    comboPtr->flags |= GOT_SELECTION;
	}

	/*
	 * Recompute the window's geometry and arrange for it to be
	 * redisplayed.
	 */

	Tk_SetInternalBorder(comboPtr->tkwin,
	        comboPtr->borderWidth + comboPtr->highlightWidth);
	if (comboPtr->highlightWidth <= 0) {
	    comboPtr->highlightWidth = 0;
	}
	comboPtr->inset = comboPtr->highlightWidth 
	        + comboPtr->borderWidth + XPAD;
	break;
    }
    if (!error) {
	Tk_FreeSavedOptions(&savedOptions);
    }

    /*
     * If the combo is tied to the value of a variable, then set up
     * a trace on the variable's value, create the variable if it doesn't
     * exist, and set the combo's value from the variable's value.
     */

    if (comboPtr->textVarName != NULL) {
	char *value;

	value = Tcl_GetVar(interp, comboPtr->textVarName, TCL_GLOBAL_ONLY);
	if (value == NULL) {
	    ComboValueChanged(comboPtr);
	} else {
	    ComboSetValue(comboPtr, value);
	}
	Tcl_TraceVar(interp, comboPtr->textVarName,
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		ComboTextVarProc, (ClientData) comboPtr);
    }

    ComboWorldChanged((ClientData) comboPtr);
    if (error) {
        Tcl_SetObjResult(interp, errorResult);
	Tcl_DecrRefCount(errorResult);
	return TCL_ERROR;
    } else {
        return TCL_OK;
    }
}

/*
 *---------------------------------------------------------------------------
 *
 * ComboWorldChanged --
 *
 *      This procedure is called when the world has changed in some
 *      way and the widget needs to recompute all its graphics contexts
 *	and determine its new geometry.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Combo will be relayed out and redisplayed.
 *
 *---------------------------------------------------------------------------
 */
 
static void
ComboWorldChanged(instanceData)
    ClientData instanceData;	/* Information about widget. */
{
    XGCValues gcValues;
    GC gc = None;
    unsigned long mask;
    Combo *comboPtr;

    comboPtr = (Combo *) instanceData;

    comboPtr->avgWidth = Tk_TextWidth(comboPtr->tkfont, "0", 1);
    if (comboPtr->avgWidth == 0) {
	comboPtr->avgWidth = 1;
    }

    comboPtr->bWidth = comboPtr->avgWidth + 2 * (1+XPAD);
    if (comboPtr->bWidth < 11) {
	comboPtr->bWidth = 11; /* we want a min visible size */
    }

    if (comboPtr->normalBorder != NULL) {
	Tk_SetBackgroundFromBorder(comboPtr->tkwin, comboPtr->normalBorder);
    }

    if (comboPtr->state == STATE_DISABLED) {
	gcValues.foreground = comboPtr->disabledFgColorPtr->pixel;
    } else {
	gcValues.foreground = comboPtr->fgColorPtr->pixel;
    }
    gcValues.font = Tk_FontId(comboPtr->tkfont);
    gcValues.graphics_exposures = False;
    mask = GCForeground | GCFont | GCGraphicsExposures;
    gc = Tk_GetGC(comboPtr->tkwin, mask, &gcValues);
    if (comboPtr->textGC != None) {
	Tk_FreeGC(comboPtr->display, comboPtr->textGC);
    }
    comboPtr->textGC = gc;

    gcValues.foreground = comboPtr->selFgColorPtr->pixel;
    gcValues.font = Tk_FontId(comboPtr->tkfont);
    mask = GCForeground | GCFont;
    gc = Tk_GetGC(comboPtr->tkwin, mask, &gcValues);
    if (comboPtr->selTextGC != None) {
	Tk_FreeGC(comboPtr->display, comboPtr->selTextGC);
    }
    comboPtr->selTextGC = gc;

    /*
     * Recompute the window's geometry and arrange for it to be
     * redisplayed.
     */

    ComboComputeGeometry(comboPtr);
    comboPtr->flags |= UPDATE_SCROLLBAR;
    EventuallyRedraw(comboPtr);
}

/*
 *--------------------------------------------------------------
 *
 * DisplayCombo --
 *
 *	This procedure redraws the contents of an combo window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
DisplayCombo(clientData)
    ClientData clientData;	/* Information about window. */
{
    Combo *comboPtr = (Combo *) clientData;
    Tk_Window tkwin = comboPtr->tkwin;
    int baseY, selStartX, selEndX, cursorX;
    int xBound;
    Tk_FontMetrics fm;
    Pixmap pixmap;
    int showSelection;

    comboPtr->flags &= ~REDRAW_PENDING;
    if ((comboPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	return;
    }

    Tk_GetFontMetrics(comboPtr->tkfont, &fm);

    /*
     * Update the scrollbar if that's needed.
     */

    if (comboPtr->flags & UPDATE_SCROLLBAR) {
	comboPtr->flags &= ~UPDATE_SCROLLBAR;
	ComboUpdateScrollbar(comboPtr);
    }

    /*
     * In order to avoid screen flashes, this procedure redraws the
     * textual area of the combo into off-screen memory, then copies
     * it back on-screen in a single operation.  This means there's
     * no point in time where the on-screen image has been cleared.
     */

    pixmap = Tk_GetPixmap(comboPtr->display, Tk_WindowId(tkwin),
	    Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));

    /*
     * Compute x-coordinate of the pixel just after last visible
     * one, plus vertical position of baseline of text.
     */

    xBound = Tk_Width(tkwin) - comboPtr->inset;
    baseY = (Tk_Height(tkwin) + fm.ascent - fm.descent) / 2;

    if ((comboPtr->controls == CONTROLS_SPIN) ||
	    (comboPtr->controls == CONTROLS_ALL)) {
	xBound -= comboPtr->bWidth;
    }
    if ((comboPtr->controls == CONTROLS_COMBO) ||
	    (comboPtr->controls == CONTROLS_ALL)) {
	xBound -= COMBO_MULT*comboPtr->bWidth;
    }

    /*
     * On Windows and Mac, we need to hide the selection whenever we
     * don't have the focus.
     */

#if defined(ALWAYS_SHOW_SELECTION) || defined(WIN32) || defined(__WIN32__) || defined(MAC_TCL) || defined(MAC_OSX_TK)
    showSelection = (comboPtr->flags & GOT_FOCUS);
#else
    showSelection = 1;
#endif

    /*
     * Draw the background in three layers.  From bottom to top the
     * layers are:  normal background, selection background, and
     * insertion cursor background.
     */

    Tk_Fill3DRectangle(tkwin, pixmap, comboPtr->normalBorder,
		0, 0, Tk_Width(tkwin), Tk_Height(tkwin), 0, TK_RELIEF_FLAT);

    if (showSelection
	    && (comboPtr->selectLast > comboPtr->leftIndex)) {
	if (comboPtr->selectFirst <= comboPtr->leftIndex) {
	    selStartX = comboPtr->leftX;
	} else {
	    Tk_CharBbox(comboPtr->textLayout, comboPtr->selectFirst,
		    &selStartX, NULL, NULL, NULL);
	    selStartX += comboPtr->layoutX;
	}
	if ((selStartX - comboPtr->selBorderWidth) < xBound) {
	    Tk_CharBbox(comboPtr->textLayout, comboPtr->selectLast,
		    &selEndX, NULL, NULL, NULL);
	    selEndX += comboPtr->layoutX;
	    Tk_Fill3DRectangle(tkwin, pixmap, comboPtr->selBorder,
		    selStartX - comboPtr->selBorderWidth,
		    baseY - fm.ascent - comboPtr->selBorderWidth,
		    (selEndX - selStartX) + 2*comboPtr->selBorderWidth,
		    (fm.ascent + fm.descent) + 2*comboPtr->selBorderWidth,
		    comboPtr->selBorderWidth, TK_RELIEF_RAISED);
	} 
    }

    /*
     * Draw a special background for the insertion cursor, overriding
     * even the selection background.  As a special hack to keep the
     * cursor visible when the insertion cursor color is the same as
     * the color for selected text (e.g., on mono displays), write
     * background in the cursor area (instead of nothing) when the
     * cursor isn't on.  Otherwise the selection would hide the cursor.
     */

    if ((comboPtr->insertPos >= comboPtr->leftIndex)
	    && (comboPtr->state == STATE_NORMAL)
	    && (comboPtr->flags & GOT_FOCUS)) {
	Tk_CharBbox(comboPtr->textLayout, comboPtr->insertPos, &cursorX, NULL,
		NULL, NULL);
	cursorX += comboPtr->layoutX;
	cursorX -= (comboPtr->insertWidth)/2;
	if (cursorX < xBound) {
	    if (comboPtr->flags & CURSOR_ON) {
		Tk_Fill3DRectangle(tkwin, pixmap, comboPtr->insertBorder,
			cursorX, baseY - fm.ascent, comboPtr->insertWidth,
			fm.ascent + fm.descent, comboPtr->insertBorderWidth,
			TK_RELIEF_RAISED);
	    } else if (comboPtr->insertBorder == comboPtr->selBorder) {
		Tk_Fill3DRectangle(tkwin, pixmap, comboPtr->normalBorder,
			cursorX, baseY - fm.ascent, comboPtr->insertWidth,
			fm.ascent + fm.descent, 0, TK_RELIEF_FLAT);
	    }
	}
    }

    /*
     * Draw the text in two pieces:  first the unselected portion, then the
     * selected portion on top of it.
     */

    Tk_DrawTextLayout(comboPtr->display, pixmap, comboPtr->textGC,
	    comboPtr->textLayout, comboPtr->layoutX, comboPtr->layoutY,
	    comboPtr->leftIndex, comboPtr->numChars);

    if (showSelection
	    && (comboPtr->selTextGC != comboPtr->textGC)
	    && (comboPtr->selectFirst < comboPtr->selectLast)) {
	int selFirst;

	if (comboPtr->selectFirst < comboPtr->leftIndex) {
	    selFirst = comboPtr->leftIndex;
	} else {
	    selFirst = comboPtr->selectFirst;
	}
	Tk_DrawTextLayout(comboPtr->display, pixmap, comboPtr->selTextGC,
		comboPtr->textLayout, comboPtr->layoutX, comboPtr->layoutY,
		selFirst, comboPtr->selectLast);
    }

    /*
     * Draw the spin button controls.
     */
    if ((comboPtr->controls == CONTROLS_SPIN) ||
	    (comboPtr->controls == CONTROLS_ALL)) {
	XPoint points[3];
	int startx, height, inset, pad, bWidth, tHeight;

	bWidth = comboPtr->bWidth;
	pad = XPAD + 1;
	inset = comboPtr->inset - XPAD;
	startx = Tk_Width(tkwin) - (bWidth + inset);
	height = (Tk_Height(tkwin) - 2*inset)/2;
	Tk_Fill3DRectangle(tkwin, pixmap, comboPtr->normalBorder,
		startx, inset, comboPtr->bWidth, height, 1,
		(comboPtr->selElement == SEL_SPINUP) ?
		TK_RELIEF_SUNKEN : TK_RELIEF_RAISED);
	Tk_Fill3DRectangle(tkwin, pixmap, comboPtr->normalBorder,
		startx, inset+height, comboPtr->bWidth, height, 1,
		(comboPtr->selElement == SEL_SPINDOWN) ?
		TK_RELIEF_SUNKEN : TK_RELIEF_RAISED);

	startx += pad;
	bWidth -= 2*pad;
	tHeight  = (int) (0.866 * (double) bWidth);
	if (tHeight > (height - 2*pad)) {
	    tHeight = (height - 2*pad);
	}
	inset  += tHeight + pad;
	if ((height > 1) && (bWidth > 1)) {
	    points[0].x = startx;
	    points[0].y = inset;
	    points[1].x = startx + bWidth/2;
	    points[1].y = inset - tHeight;
	    points[2].x = startx + bWidth;
	    points[2].y = points[0].y;
	    XFillPolygon(comboPtr->display, pixmap, comboPtr->textGC,
		    points, 3, Convex, CoordModeOrigin);

	    inset += 2*pad;
	    points[0].y = inset;
	    points[1].y = inset + tHeight;
	    points[2].y = points[0].y;
	    XFillPolygon(comboPtr->display, pixmap, comboPtr->textGC,
		    points, 3, Convex, CoordModeOrigin);
	}
    }

    /*
     * Draw the drop button control.
     */
    if ((comboPtr->controls == CONTROLS_COMBO) ||
	    (comboPtr->controls == CONTROLS_ALL)) {
	XPoint points[3];
	int startx, height, inset, pad, bWidth, tHeight;

	bWidth = COMBO_MULT*comboPtr->bWidth;
	pad = XPAD + 2;
	inset = comboPtr->inset - XPAD;
	startx = Tk_Width(tkwin) - (bWidth + inset);
	if (comboPtr->controls == CONTROLS_ALL) {
	    /*
	     * Account for spinbox
	     */
	    startx -= comboPtr->bWidth;
	}
	height = Tk_Height(tkwin) - 2*inset;

	Tk_Fill3DRectangle(tkwin, pixmap, comboPtr->normalBorder,
		startx, inset, bWidth, height, 1,
		(comboPtr->selElement == SEL_DROP) ?
		TK_RELIEF_SUNKEN : TK_RELIEF_RAISED);

	startx += pad;
	bWidth -= 2*pad;
	tHeight = (int) (0.866 * (double) bWidth);
	height -= 2*pad;
	if (tHeight > height) {
	    tHeight = height;
	}
	inset  += pad;
	if ((height > 1) && (bWidth > 1)) {
	    points[0].x = startx;
	    points[0].y = inset;
	    points[1].x = startx + bWidth/2;
	    points[1].y = inset + tHeight;
	    points[2].x = startx + bWidth;
	    points[2].y = points[0].y;
	    XFillPolygon(comboPtr->display, pixmap, comboPtr->textGC,
		    points, 3, Convex, CoordModeOrigin);
	}
    }

    /*
     * Draw the border and focus highlight last, so they will overwrite
     * any text that extends past the viewable part of the window.
     */

    if (comboPtr->relief != TK_RELIEF_FLAT) {
	Tk_Draw3DRectangle(tkwin, pixmap, comboPtr->normalBorder,
		comboPtr->highlightWidth, comboPtr->highlightWidth,
		Tk_Width(tkwin) - 2 * comboPtr->highlightWidth,
		Tk_Height(tkwin) - 2 * comboPtr->highlightWidth,
		comboPtr->borderWidth, comboPtr->relief);
    }
    if (comboPtr->highlightWidth != 0) {
	GC fgGC, bgGC;

	bgGC = Tk_GCForColor(comboPtr->highlightBgColorPtr, pixmap);
	if (comboPtr->flags & GOT_FOCUS) {
	    fgGC = Tk_GCForColor(comboPtr->highlightColorPtr, pixmap);
	    TkpDrawHighlightBorder(tkwin, fgGC, bgGC, 
	            comboPtr->highlightWidth, pixmap);
	} else {
	    TkpDrawHighlightBorder(tkwin, bgGC, bgGC, 
	            comboPtr->highlightWidth, pixmap);
	}
    }

    /*
     * Everything's been redisplayed;  now copy the pixmap onto the screen
     * and free up the pixmap.
     */

    XCopyArea(comboPtr->display, pixmap, Tk_WindowId(tkwin), comboPtr->textGC,
	    0, 0, (unsigned) Tk_Width(tkwin), (unsigned) Tk_Height(tkwin),
	    0, 0);
    Tk_FreePixmap(comboPtr->display, pixmap);
    comboPtr->flags &= ~BORDER_NEEDED;
}

/*
 *----------------------------------------------------------------------
 *
 * ComboComputeGeometry --
 *
 *	This procedure is invoked to recompute information about where
 *	in its window an combo's string will be displayed.  It also
 *	computes the requested size for the window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The leftX and tabOrigin fields are recomputed for comboPtr,
 *	and leftIndex may be adjusted.  Tk_GeometryRequest is called
 *	to register the desired dimensions for the window.
 *
 *----------------------------------------------------------------------
 */

static void
ComboComputeGeometry(comboPtr)
    Combo *comboPtr;		/* Widget record for combo. */
{
    int totalLength, overflow, maxOffScreen, rightX;
    int height, width, i;
    Tk_FontMetrics fm;
    char *p;

    if (comboPtr->displayString != comboPtr->string) {
	ckfree(comboPtr->displayString);
	comboPtr->displayString = comboPtr->string;
	comboPtr->numDisplayBytes = comboPtr->numBytes;
    }

    /*
     * If we're displaying a special character instead of the value of
     * the combo, recompute the displayString.
     */

    if (comboPtr->showChar != NULL) {
	Tcl_UniChar ch;
	char buf[TCL_UTF_MAX];
	int size;

	/*
	 * Normalize the special character so we can safely duplicate it
	 * in the display string.  If we didn't do this, then two malformed
	 * characters might end up looking like one valid UTF character in
	 * the resulting string.
	 */

	Tcl_UtfToUniChar(comboPtr->showChar, &ch);
	size = Tcl_UniCharToUtf(ch, buf);

	comboPtr->numDisplayBytes = comboPtr->numChars * size;
	comboPtr->displayString =
		(char *) ckalloc((unsigned) (comboPtr->numDisplayBytes + 1));

	p = comboPtr->displayString;
	for (i = comboPtr->numChars; --i >= 0; ) {
	    p += Tcl_UniCharToUtf(ch, p);
	}
	*p = '\0';
    }
    Tk_FreeTextLayout(comboPtr->textLayout);
    comboPtr->textLayout = Tk_ComputeTextLayout(comboPtr->tkfont,
	    comboPtr->displayString, comboPtr->numChars, 0,
	    comboPtr->justify, TK_IGNORE_NEWLINES, &totalLength, &height);

    comboPtr->layoutY = (Tk_Height(comboPtr->tkwin) - height) / 2;

    /*
     * Recompute where the leftmost character on the display will
     * be drawn (comboPtr->leftX) and adjust leftIndex if necessary
     * so that we don't let characters hang off the edge of the
     * window unless the entire window is full.
     */

    overflow = totalLength - (Tk_Width(comboPtr->tkwin) - 2*comboPtr->inset);
    if (overflow <= 0) {
	comboPtr->leftIndex = 0;
	if (comboPtr->justify == TK_JUSTIFY_LEFT) {
	    comboPtr->leftX = comboPtr->inset;
	} else if (comboPtr->justify == TK_JUSTIFY_RIGHT) {
	    comboPtr->leftX = Tk_Width(comboPtr->tkwin) - comboPtr->inset
		    - totalLength;
	} else {
	    comboPtr->leftX = (Tk_Width(comboPtr->tkwin) - totalLength)/2;
	}
	comboPtr->layoutX = comboPtr->leftX;
    } else {
	/*
	 * The whole string can't fit in the window.  Compute the
	 * maximum number of characters that may be off-screen to
	 * the left without leaving empty space on the right of the
	 * window, then don't let leftIndex be any greater than that.
	 */

	maxOffScreen = Tk_PointToChar(comboPtr->textLayout, overflow, 0);
	Tk_CharBbox(comboPtr->textLayout, maxOffScreen,
		&rightX, NULL, NULL, NULL);
	if (rightX < overflow) {
	    maxOffScreen++;
	}
	if (comboPtr->leftIndex > maxOffScreen) {
	    comboPtr->leftIndex = maxOffScreen;
	}
	Tk_CharBbox(comboPtr->textLayout, comboPtr->leftIndex, &rightX,
		NULL, NULL, NULL);
	comboPtr->leftX = comboPtr->inset;
	comboPtr->layoutX = comboPtr->leftX - rightX;
    }

    Tk_GetFontMetrics(comboPtr->tkfont, &fm);
    height = fm.linespace + 2*comboPtr->inset + 2*(YPAD-XPAD);
    if (comboPtr->prefWidth > 0) {
	width = comboPtr->prefWidth*comboPtr->avgWidth + 2*comboPtr->inset;
    } else {
	if (totalLength == 0) {
	    width = comboPtr->avgWidth + 2*comboPtr->inset;
	} else {
	    width = totalLength + 2*comboPtr->inset;
	}
    }

    /*
     * Add one extra length for the spin buttons
     */
    if ((comboPtr->controls == CONTROLS_SPIN) ||
	    (comboPtr->controls == CONTROLS_ALL)) {
	width += comboPtr->bWidth;
    }

    /*
     * Add two extra lengths for the combo drop button
     */
    if ((comboPtr->controls == CONTROLS_COMBO) ||
	    (comboPtr->controls == CONTROLS_ALL)) {
	width += COMBO_MULT*comboPtr->bWidth;
    }

    Tk_GeometryRequest(comboPtr->tkwin, width, height);
}

#if 0
/*
 *----------------------------------------------------------------------
 *
 * DisplayComboButton --
 *
 *	This procedure is invoked to display a combo widget.  It is
 *	normally invoked as an idle handler.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Commands are output to X to display the combo in its
 *	current mode.  The REDRAW_PENDING flag is cleared.
 *
 *----------------------------------------------------------------------
 */

static void
DisplayComboButton(clientData)
    ClientData clientData;	/* Information about widget. */
{
    register Combo *comboPtr = (Combo *) clientData;
    GC gc;
    Tk_3DBorder border;
    Pixmap pixmap;
    int x = 0;			/* Initialization only needed to stop
				 * compiler warning. */
    int y, relief;
    Tk_Window tkwin = comboPtr->tkwin;
    int width, height;
    int offset;			/* 1 means this is a combo widget, so we
				 * offset the text to make the combo appear
				 * to move up and down as the relief changes. */

    comboPtr->flags &= ~REDRAW_PENDING;
    if ((comboPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	return;
    }

    border = comboPtr->normalBorder;
    if ((comboPtr->state == STATE_DISABLED) && (comboPtr->disabledFg != NULL)) {
	gc = comboPtr->disabledGC;
    } else if ((comboPtr->state == STATE_ACTIVE)
	    && !Tk_StrictMotif(comboPtr->tkwin)) {
	gc = comboPtr->activeTextGC;
	border = comboPtr->activeBorder;
    } else {
	gc = comboPtr->normalTextGC;
    }

    /*
     * In order to avoid screen flashes, this procedure redraws
     * the combo in a pixmap, then copies the pixmap to the
     * screen in a single operation.  This means that there's no
     * point in time where the on-sreen image has been cleared.
     */

    pixmap = Tk_GetPixmap(comboPtr->display, Tk_WindowId(tkwin),
	    Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));
    Tk_Fill3DRectangle(tkwin, pixmap, border, 0, 0, Tk_Width(tkwin),
	    Tk_Height(tkwin), 0, TK_RELIEF_FLAT);

    TkComputeAnchor(comboPtr->anchor, tkwin, comboPtr->padX, comboPtr->padY,
	    comboPtr->indicatorSpace + comboPtr->textWidth, comboPtr->textHeight,
	    &x, &y);
    
    x += comboPtr->indicatorSpace;

    x += offset;
    y += offset;
    if (relief == TK_RELIEF_RAISED) {
	x -= offset;
	y -= offset;
    } else if (relief == TK_RELIEF_SUNKEN) {
	x += offset;
	y += offset;
    }

    y += comboPtr->textHeight/2;

    /*
     * If the combo is disabled with a stipple rather than a special
     * foreground color, generate the stippled effect.  If the widget
     * is selected and we use a different background color when selected,
     * must temporarily modify the GC.
     */

    if ((comboPtr->state == STATE_DISABLED)
	    && (comboPtr->disabledFg == NULL)) {
	if ((comboPtr->flags & SELECTED) && !comboPtr->indicatorOn
		&& (comboPtr->selectBorder != NULL)) {
	    XSetForeground(comboPtr->display, comboPtr->disabledGC,
		    Tk_3DBorderColor(comboPtr->selectBorder)->pixel);
	}
	XFillRectangle(comboPtr->display, pixmap, comboPtr->disabledGC,
		comboPtr->inset, comboPtr->inset,
		(unsigned) (Tk_Width(tkwin) - 2*comboPtr->inset),
		(unsigned) (Tk_Height(tkwin) - 2*comboPtr->inset));
	if ((comboPtr->flags & SELECTED) && !comboPtr->indicatorOn
		&& (comboPtr->selectBorder != NULL)) {
	    XSetForeground(comboPtr->display, comboPtr->disabledGC,
		    Tk_3DBorderColor(comboPtr->normalBorder)->pixel);
	}
    }

    /*
     * Draw the border and traversal highlight last.  This way, if the
     * combo's contents overflow they'll be covered up by the border.
     * This code is complicated by the possible combinations of focus
     * highlight and default rings.  We draw the focus and highlight rings
     * using the highlight border and highlight foreground color.
     */

    if (relief != TK_RELIEF_FLAT) {
	int inset = comboPtr->highlightWidth;

	if (comboPtr->defaultState == DEFAULT_ACTIVE) {
	    /*
	     * Draw the default ring with 2 pixels of space between the
	     * default ring and the combo and the default ring and the
	     * focus ring.  Note that we need to explicitly draw the space
	     * in the highlightBorder color to ensure that we overwrite any
	     * overflow text and/or a different combo background color.
	     */

	    Tk_Draw3DRectangle(tkwin, pixmap, comboPtr->highlightBorder, inset,
		    inset, Tk_Width(tkwin) - 2*inset,
		    Tk_Height(tkwin) - 2*inset, 2, TK_RELIEF_FLAT);
	    inset += 2;
	    Tk_Draw3DRectangle(tkwin, pixmap, comboPtr->highlightBorder, inset,
		    inset, Tk_Width(tkwin) - 2*inset,
		    Tk_Height(tkwin) - 2*inset, 1, TK_RELIEF_SUNKEN);
	    inset++;
	    Tk_Draw3DRectangle(tkwin, pixmap, comboPtr->highlightBorder, inset,
		    inset, Tk_Width(tkwin) - 2*inset,
		    Tk_Height(tkwin) - 2*inset, 2, TK_RELIEF_FLAT);

	    inset += 2;
	} else if (comboPtr->defaultState == DEFAULT_NORMAL) {
	    /*
	     * Leave room for the default ring and write over any text or
	     * background color.
	     */

	    Tk_Draw3DRectangle(tkwin, pixmap, comboPtr->highlightBorder, 0,
		    0, Tk_Width(tkwin), Tk_Height(tkwin), 5, TK_RELIEF_FLAT);
	    inset += 5;
	}

	/*
	 * Draw the combo border.
	 */

	Tk_Draw3DRectangle(tkwin, pixmap, border, inset, inset,
		Tk_Width(tkwin) - 2*inset, Tk_Height(tkwin) - 2*inset,
		comboPtr->borderWidth, relief);
    }
    if (comboPtr->highlightWidth > 0) {
	GC gc;

	if (comboPtr->flags & GOT_FOCUS) {
	    gc = Tk_GCForColor(comboPtr->highlightColorPtr, pixmap);
	} else {
	    gc = Tk_GCForColor(Tk_3DBorderColor(comboPtr->highlightBorder),
		    pixmap);
	}

	/*
	 * Make sure the focus ring shrink-wraps the actual combo, not the
	 * padding space left for a default ring.
	 */

	if (comboPtr->defaultState == DEFAULT_NORMAL) {
	    TkDrawInsetFocusHighlight(tkwin, gc, comboPtr->highlightWidth,
		    pixmap, 5);
	} else {
	    Tk_DrawFocusHighlight(tkwin, gc, comboPtr->highlightWidth, pixmap);
	}
    }

    /*
     * Copy the information from the off-screen pixmap onto the screen,
     * then delete the pixmap.
     */

    XCopyArea(comboPtr->display, pixmap, Tk_WindowId(tkwin),
	    comboPtr->copyGC, 0, 0, (unsigned) Tk_Width(tkwin),
	    (unsigned) Tk_Height(tkwin), 0, 0);
    Tk_FreePixmap(comboPtr->display, pixmap);
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * InsertChars --
 *
 *	Add new characters to an combo widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	New information gets added to comboPtr;  it will be redisplayed
 *	soon, but not necessarily immediately.
 *
 *----------------------------------------------------------------------
 */

static void
InsertChars(comboPtr, index, value)
    Combo *comboPtr;		/* Combo that is to get the new elements. */
    int index;			/* Add the new elements before this
				 * character index. */
    char *value;		/* New characters to add (NULL-terminated
				 * string). */
{
    int byteIndex, byteCount, oldChars, charsAdded, newByteCount;
    char *new, *string;

    string = comboPtr->string;
    byteIndex = Tcl_UtfAtIndex(string, index) - string;
    byteCount = strlen(value);
    if (byteCount == 0) {
	return;
    }

    newByteCount = comboPtr->numBytes + byteCount + 1;
    new = (char *) ckalloc((unsigned) newByteCount);
    memcpy(new, string, (size_t) byteIndex);
    strcpy(new + byteIndex, value);
    strcpy(new + byteIndex + byteCount, string + byteIndex);

    if ((comboPtr->validate == VALIDATE_KEY ||
	 comboPtr->validate == VALIDATE_ALL) &&
	ComboValidateChange(comboPtr, value, new, index,
		VALIDATE_INSERT) != TCL_OK) {
	ckfree(new);
	return;
    }

    ckfree(string);
    comboPtr->string = new;

    /*
     * The following construction is used because inserting improperly
     * formed UTF-8 sequences between other improperly formed UTF-8
     * sequences could result in actually forming valid UTF-8 sequences;
     * the number of characters added may not be Tcl_NumUtfChars(string, -1),
     * because of context.  The actual number of characters added is how
     * many characters are in the string now minus the number that
     * used to be there.
     */

    oldChars = comboPtr->numChars;
    comboPtr->numChars = Tcl_NumUtfChars(new, -1);
    charsAdded = comboPtr->numChars - oldChars;
    comboPtr->numBytes += byteCount;

    if (comboPtr->displayString == string) {
	comboPtr->displayString = new;
	comboPtr->numDisplayBytes = comboPtr->numBytes;
    }

    /*
     * Inserting characters invalidates all indexes into the string.
     * Touch up the indexes so that they still refer to the same
     * characters (at new positions).  When updating the selection
     * end-points, don't include the new text in the selection unless
     * it was completely surrounded by the selection.
     */

    if (comboPtr->selectFirst >= index) {
	comboPtr->selectFirst += charsAdded;
    }
    if (comboPtr->selectLast > index) {
	comboPtr->selectLast += charsAdded;
    }
    if ((comboPtr->selectAnchor > index)
	    || (comboPtr->selectFirst >= index)) {
	comboPtr->selectAnchor += charsAdded;
    }
    if (comboPtr->leftIndex > index) {
	comboPtr->leftIndex += charsAdded;
    }
    if (comboPtr->insertPos >= index) {
	comboPtr->insertPos += charsAdded;
    }
    ComboValueChanged(comboPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * DeleteChars --
 *
 *	Remove one or more characters from an combo widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory gets freed, the combo gets modified and (eventually)
 *	redisplayed.
 *
 *----------------------------------------------------------------------
 */

static void
DeleteChars(comboPtr, index, count)
    Combo *comboPtr;		/* Combo widget to modify. */
    int index;			/* Index of first character to delete. */
    int count;			/* How many characters to delete. */
{
    int byteIndex, byteCount, newByteCount;
    char *new, *string;
    char *todelete;

    if ((index + count) > comboPtr->numChars) {
	count = comboPtr->numChars - index;
    }
    if (count <= 0) {
	return;
    }

    string = comboPtr->string;
    byteIndex = Tcl_UtfAtIndex(string, index) - string;
    byteCount = Tcl_UtfAtIndex(string + byteIndex, count) - (string + byteIndex);

    newByteCount = comboPtr->numBytes + 1 - byteCount;
    new = (char *) ckalloc((unsigned) newByteCount);
    memcpy(new, string, (size_t) byteIndex);
    strcpy(new + byteIndex, string + byteIndex + byteCount);

    todelete = (char *) ckalloc((unsigned) (byteCount + 1));
    memcpy(todelete, string + byteIndex, (size_t) byteCount);
    todelete[byteCount] = '\0';

    if ((comboPtr->validate == VALIDATE_KEY ||
	 comboPtr->validate == VALIDATE_ALL) &&
	ComboValidateChange(comboPtr, todelete, new, index,
		VALIDATE_DELETE) != TCL_OK) {
	ckfree(new);
	ckfree(todelete);
	return;
    }

    ckfree(todelete);
    ckfree(comboPtr->string);
    comboPtr->string = new;
    comboPtr->numChars -= count;
    comboPtr->numBytes -= byteCount;

    if (comboPtr->displayString == string) {
	comboPtr->displayString = new;
	comboPtr->numDisplayBytes = comboPtr->numBytes;
    }

    /*
     * Deleting characters results in the remaining characters being
     * renumbered.  Update the various indexes into the string to reflect
     * this change.
     */

    if (comboPtr->selectFirst >= index) {
	if (comboPtr->selectFirst >= (index + count)) {
	    comboPtr->selectFirst -= count;
	} else {
	    comboPtr->selectFirst = index;
	}
    }
    if (comboPtr->selectLast >= index) {
	if (comboPtr->selectLast >= (index + count)) {
	    comboPtr->selectLast -= count;
	} else {
	    comboPtr->selectLast = index;
	}
    }
    if (comboPtr->selectLast <= comboPtr->selectFirst) {
	comboPtr->selectFirst = -1;
	comboPtr->selectLast = -1;
    }
    if (comboPtr->selectAnchor >= index) {
	if (comboPtr->selectAnchor >= (index+count)) {
	    comboPtr->selectAnchor -= count;
	} else {
	    comboPtr->selectAnchor = index;
	}
    }
    if (comboPtr->leftIndex > index) {
	if (comboPtr->leftIndex >= (index + count)) {
	    comboPtr->leftIndex -= count;
	} else {
	    comboPtr->leftIndex = index;
	}
    }
    if (comboPtr->insertPos >= index) {
	if (comboPtr->insertPos >= (index + count)) {
	    comboPtr->insertPos -= count;
	} else {
	    comboPtr->insertPos = index;
	}
    }
    ComboValueChanged(comboPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ComboValueChanged --
 *
 *	This procedure is invoked when characters are inserted into
 *	an combo or deleted from it.  It updates the combo's associated
 *	variable, if there is one, and does other bookkeeping such
 *	as arranging for redisplay.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
ComboValueChanged(comboPtr)
    Combo *comboPtr;		/* Combo whose value just changed. */
{
    char *newValue;

    if (comboPtr->textVarName == NULL) {
	newValue = NULL;
    } else {
	newValue = Tcl_SetVar(comboPtr->interp, comboPtr->textVarName,
		comboPtr->string, TCL_GLOBAL_ONLY);
    }

    if ((newValue != NULL) && (strcmp(newValue, comboPtr->string) != 0)) {
	/*
	 * The value of the variable is different than what we asked for.
	 * This means that a trace on the variable modified it.  In this
	 * case our trace procedure wasn't invoked since the modification
	 * came while a trace was already active on the variable.  So,
	 * update our value to reflect the variable's latest value.
	 */

	ComboSetValue(comboPtr, newValue);
    } else {
	/*
	 * Arrange for redisplay.
	 */

	comboPtr->flags |= UPDATE_SCROLLBAR;
	ComboComputeGeometry(comboPtr);
	EventuallyRedraw(comboPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComboSetValue --
 *
 *	Replace the contents of a text combo with a given value.  This
 *	procedure is invoked when updating the combo from the combo's
 *	associated variable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The string displayed in the combo will change.  The selection,
 *	insertion point, and view may have to be adjusted to keep them
 *	within the bounds of the new string.  Note: this procedure does
 *	*not* update the combo's associated variable, since that could
 *	result in an infinite loop.
 *
 *----------------------------------------------------------------------
 */

static void
ComboSetValue(comboPtr, value)
    Combo *comboPtr;		/* Combo whose value is to be changed. */
    char *value;		/* New text to display in combo. */
{
    char *oldSource;
    int code, valueLen, malloced = 0;

    if (strcmp(value, comboPtr->string) == 0) {
	return;
    }
    valueLen = strlen(value);

    if (comboPtr->flags & VALIDATE_VAR) {
	comboPtr->flags |= VALIDATE_ABORT;
    } else {
	/*
	 * If we validate, we create a copy of the value, as it may
	 * point to volatile memory, like the value of the -textvar
	 * which may get freed during validation
	 */
	oldSource = (char *) ckalloc((unsigned) (valueLen + 1));
	strcpy(oldSource, value);
	value = oldSource;
	malloced = 1;
 
	comboPtr->flags |= VALIDATE_VAR;
	code = ComboValidateChange(comboPtr, (char *) NULL, value, -1,
		VALIDATE_FORCED);
	comboPtr->flags &= ~VALIDATE_VAR;
	/*
	 * If VALIDATE_ABORT has been set, then this operation should be
	 * aborted because the validatecommand did something else instead
	 */
	if (comboPtr->flags & VALIDATE_ABORT) {
	    comboPtr->flags &= ~VALIDATE_ABORT;
	    ckfree(value);
	    return;
	}
    }

    oldSource = comboPtr->string;
    ckfree(comboPtr->string);

    if (malloced) {
	comboPtr->string = value;
    } else {
	comboPtr->string = (char *) ckalloc((unsigned) (valueLen + 1));
	strcpy(comboPtr->string, value);
    }
    comboPtr->numBytes = valueLen;
    comboPtr->numChars = Tcl_NumUtfChars(value, valueLen);

    if (comboPtr->displayString == oldSource) {
	comboPtr->displayString = comboPtr->string;
	comboPtr->numDisplayBytes = comboPtr->numBytes;
    }

    if (comboPtr->selectFirst >= 0) {
	if (comboPtr->selectFirst >= comboPtr->numChars) {
	    comboPtr->selectFirst = -1;
	    comboPtr->selectLast = -1;
	} else if (comboPtr->selectLast > comboPtr->numChars) {
	    comboPtr->selectLast = comboPtr->numChars;
	}
    }
    if (comboPtr->leftIndex >= comboPtr->numChars) {
	if (comboPtr->numChars > 0) {
	    comboPtr->leftIndex = comboPtr->numChars - 1;
	} else {
	    comboPtr->leftIndex = 0;
	}
    }
    if (comboPtr->insertPos > comboPtr->numChars) {
	comboPtr->insertPos = comboPtr->numChars;
    }

    comboPtr->flags |= UPDATE_SCROLLBAR;
    ComboComputeGeometry(comboPtr);
    EventuallyRedraw(comboPtr);
}

/*
 *--------------------------------------------------------------
 *
 * ComboEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on comboes.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
ComboEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    Combo *comboPtr = (Combo *) clientData;

    switch (eventPtr->type) {
	case MotionNotify: {
	    int elem;

	    elem = GetComboElement(comboPtr, eventPtr->xmotion.x,
		    eventPtr->xmotion.y);
	    if (elem != comboPtr->curElement) {
		Tk_Cursor cursor;

		comboPtr->curElement = elem;
		if (elem == SEL_ENTRY) {
		    cursor = comboPtr->cursor;
		} else if (elem == SEL_DROP) {
		    cursor = comboPtr->dCursor;
		} else if (elem == SEL_SPINDOWN) {
		    cursor = comboPtr->sdCursor;
		} else if (elem == SEL_SPINUP) {
		    cursor = comboPtr->suCursor;
		} else {
		    cursor = None;
		}
		if (cursor != None) {
		    Tk_DefineCursor(comboPtr->tkwin, cursor);
		} else {
		    Tk_UndefineCursor(comboPtr->tkwin);
		}
	    }
	    break;
	}
	case Expose:
	    EventuallyRedraw(comboPtr);
	    comboPtr->flags |= BORDER_NEEDED;
	    break;
	case DestroyNotify:
	    DestroyCombo((char *) clientData);
	    break;
	case ConfigureNotify:
	    Tcl_Preserve((ClientData) comboPtr);
	    comboPtr->flags |= UPDATE_SCROLLBAR;
	    ComboComputeGeometry(comboPtr);
	    EventuallyRedraw(comboPtr);
	    Tcl_Release((ClientData) comboPtr);
	    break;
	case FocusIn:
	case FocusOut:
	    if (eventPtr->xfocus.detail != NotifyInferior) {
		ComboFocusProc(comboPtr, (eventPtr->type == FocusIn));
	    }
	    break;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComboCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
ComboCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to widget record for widget. */
{
    Combo *comboPtr = (Combo *) clientData;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (!(comboPtr->flags & COMBO_DELETED)) {
        Tk_DestroyWindow(comboPtr->tkwin);
    }
}

/*
 *---------------------------------------------------------------------------
 *
 * GetComboIndex --
 *
 *	Parse an index into an combo and return either its value
 *	or an error.
 *
 * Results:
 *	A standard Tcl result.  If all went well, then *indexPtr is
 *	filled in with the character index (into comboPtr) corresponding to
 *	string.  The index value is guaranteed to lie between 0 and
 *	the number of characters in the string, inclusive.  If an
 *	error occurs then an error message is left in the interp's result.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static int
GetComboIndex(interp, comboPtr, string, indexPtr)
    Tcl_Interp *interp;		/* For error messages. */
    Combo *comboPtr;		/* Combo for which the index is being
				 * specified. */
    char *string;		/* Specifies character in comboPtr. */
    int *indexPtr;		/* Where to store converted character
				 * index. */
{
    size_t length;

    length = strlen(string);

    if (string[0] == 'a') {
	if (strncmp(string, "anchor", length) == 0) {
	    *indexPtr = comboPtr->selectAnchor;
	} else {
	    badIndex:

	    /*
	     * Some of the paths here leave messages in the interp's result,
	     * so we have to clear it out before storing our own message.
	     */

	    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
	    Tcl_AppendResult(interp, "bad combo index \"", string,
		    "\"", (char *) NULL);
	    return TCL_ERROR;
	}
    } else if (string[0] == 'e') {
	if (strncmp(string, "end", length) == 0) {
	    *indexPtr = comboPtr->numChars;
	} else {
	    goto badIndex;
	}
    } else if (string[0] == 'i') {
	if (strncmp(string, "insert", length) == 0) {
	    *indexPtr = comboPtr->insertPos;
	} else {
	    goto badIndex;
	}
    } else if (string[0] == 's') {
	if (comboPtr->selectFirst < 0) {
	    Tcl_SetResult(interp, "selection isn't in combo", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (length < 5) {
	    goto badIndex;
	}
	if (strncmp(string, "sel.first", length) == 0) {
	    *indexPtr = comboPtr->selectFirst;
	} else if (strncmp(string, "sel.last", length) == 0) {
	    *indexPtr = comboPtr->selectLast;
	} else {
	    goto badIndex;
	}
    } else if (string[0] == '@') {
	int x, roundUp;

	if (Tcl_GetInt(interp, string + 1, &x) != TCL_OK) {
	    goto badIndex;
	}
	if (x < comboPtr->inset) {
	    x = comboPtr->inset;
	}
	roundUp = 0;
	if (x >= (Tk_Width(comboPtr->tkwin) - comboPtr->inset)) {
	    x = Tk_Width(comboPtr->tkwin) - comboPtr->inset - 1;
	    roundUp = 1;
	}
	*indexPtr = Tk_PointToChar(comboPtr->textLayout,
		x - comboPtr->layoutX, 0);

	/*
	 * Special trick:  if the x-position was off-screen to the right,
	 * round the index up to refer to the character just after the
	 * last visible one on the screen.  This is needed to enable the
	 * last character to be selected, for example.
	 */

	if (roundUp && (*indexPtr < comboPtr->numChars)) {
	    *indexPtr += 1;
	}
    } else {
	if (Tcl_GetInt(interp, string, indexPtr) != TCL_OK) {
	    goto badIndex;
	}
	if (*indexPtr < 0){
	    *indexPtr = 0;
	} else if (*indexPtr > comboPtr->numChars) {
	    *indexPtr = comboPtr->numChars;
	} 
    }
    return TCL_OK;
}

/*
 *---------------------------------------------------------------------------
 *
 * GetComboElement --
 *
 *	Return the element associated with an x,y coord.
 *
 * Results:
 *	Element type as enum selelement.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static int
GetComboElement(comboPtr, x, y)
    Combo *comboPtr;		/* Combo for which the index is being
				 * specified. */
    int x;			/* x coord */
    int y;			/* y coord */
{
    int inset, width;

    if ((x < 0) || (y < 0) || (y > Tk_Height(comboPtr->tkwin))
	    || (x > Tk_Width(comboPtr->tkwin))) {
	return SEL_NONE;
    }

    inset = comboPtr->inset - XPAD;
    width = Tk_Width(comboPtr->tkwin) - inset;
    switch (comboPtr->controls) {
	case CONTROLS_ALL:
	    if (x > (width - comboPtr->bWidth)) {
		if (y > (Tk_Height(comboPtr->tkwin) / 2)) {
		    return SEL_SPINDOWN;
		} else {
		    return SEL_SPINUP;
		}
	    } else if (x > (width - ((1+COMBO_MULT)*comboPtr->bWidth))) {
		return SEL_DROP;
	    } else {
		return SEL_ENTRY;
	    }
	    break;
	case CONTROLS_COMBO:
	    if (x > (width - COMBO_MULT*comboPtr->bWidth)) {
		return SEL_DROP;
	    } else {
		return SEL_ENTRY;
	    }
	    break;
	case CONTROLS_SPIN:
	    if (x > (width - comboPtr->bWidth)) {
		if (y > (Tk_Height(comboPtr->tkwin) / 2)) {
		    return SEL_SPINDOWN;
		} else {
		    return SEL_SPINUP;
		}
	    } else {
		return SEL_ENTRY;
	    }
	    break;
    }
    return SEL_ENTRY;
}

/*
 *----------------------------------------------------------------------
 *
 * ComboScanTo --
 *
 *	Given a y-coordinate (presumably of the curent mouse location)
 *	drag the view in the window to implement the scan operation.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The view in the window may change.
 *
 *----------------------------------------------------------------------
 */

static void
ComboScanTo(comboPtr, x)
    Combo *comboPtr;		/* Information about widget. */
    int x;			/* X-coordinate to use for scan operation. */
{
    int newLeftIndex;

    /*
     * Compute new leftIndex for combo by amplifying the difference
     * between the current position and the place where the scan
     * started (the "mark" position).  If we run off the left or right
     * side of the combo, then reset the mark point so that the current
     * position continues to correspond to the edge of the window.
     * This means that the picture will start dragging as soon as the
     * mouse reverses direction (without this reset, might have to slide
     * mouse a long ways back before the picture starts moving again).
     */

    newLeftIndex = comboPtr->scanMarkIndex
	    - (10 * (x - comboPtr->scanMarkX)) / comboPtr->avgWidth;
    if (newLeftIndex >= comboPtr->numChars) {
	newLeftIndex = comboPtr->scanMarkIndex = comboPtr->numChars - 1;
	comboPtr->scanMarkX = x;
    }
    if (newLeftIndex < 0) {
	newLeftIndex = comboPtr->scanMarkIndex = 0;
	comboPtr->scanMarkX = x;
    } 

    if (newLeftIndex != comboPtr->leftIndex) {
	comboPtr->leftIndex = newLeftIndex;
	comboPtr->flags |= UPDATE_SCROLLBAR;
	ComboComputeGeometry(comboPtr);
	if (newLeftIndex != comboPtr->leftIndex) {
	    comboPtr->scanMarkIndex = comboPtr->leftIndex;
	    comboPtr->scanMarkX = x;
	}
	EventuallyRedraw(comboPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComboSelectTo --
 *
 *	Modify the selection by moving its un-anchored end.  This could
 *	make the selection either larger or smaller.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection changes.
 *
 *----------------------------------------------------------------------
 */

static void
ComboSelectTo(comboPtr, index)
    Combo *comboPtr;		/* Information about widget. */
    int index;			/* Character index of element that is to
				 * become the "other" end of the selection. */
{
    int newFirst, newLast;

    /*
     * Grab the selection if we don't own it already.
     */

    if (!(comboPtr->flags & GOT_SELECTION) && (comboPtr->exportSelection)) {
	Tk_OwnSelection(comboPtr->tkwin, XA_PRIMARY, ComboLostSelection,
		(ClientData) comboPtr);
	comboPtr->flags |= GOT_SELECTION;
    }

    /*
     * Pick new starting and ending points for the selection.
     */

    if (comboPtr->selectAnchor > comboPtr->numChars) {
	comboPtr->selectAnchor = comboPtr->numChars;
    }
    if (comboPtr->selectAnchor <= index) {
	newFirst = comboPtr->selectAnchor;
	newLast = index;
    } else {
	newFirst = index;
	newLast = comboPtr->selectAnchor;
	if (newLast < 0) {
	    newFirst = newLast = -1;
	}
    }
    if ((comboPtr->selectFirst == newFirst)
	    && (comboPtr->selectLast == newLast)) {
	return;
    }
    comboPtr->selectFirst = newFirst;
    comboPtr->selectLast = newLast;
    EventuallyRedraw(comboPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ComboFetchSelection --
 *
 *	This procedure is called back by Tk when the selection is
 *	requested by someone.  It returns part or all of the selection
 *	in a buffer provided by the caller.
 *
 * Results:
 *	The return value is the number of non-NULL bytes stored
 *	at buffer.  Buffer is filled (or partially filled) with a
 *	NULL-terminated string containing part or all of the selection,
 *	as given by offset and maxBytes.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
ComboFetchSelection(clientData, offset, buffer, maxBytes)
    ClientData clientData;	/* Information about combo widget. */
    int offset;			/* Byte offset within selection of first
				 * character to be returned. */
    char *buffer;		/* Location in which to place selection. */
    int maxBytes;		/* Maximum number of bytes to place at
				 * buffer, not including terminating NULL
				 * character. */
{
    Combo *comboPtr = (Combo *) clientData;
    int byteCount;
    char *string, *selStart, *selEnd;

    if ((comboPtr->selectFirst < 0) || !(comboPtr->exportSelection)) {
	return -1;
    }
    string = comboPtr->displayString;
    selStart = Tcl_UtfAtIndex(string, comboPtr->selectFirst);
    selEnd = Tcl_UtfAtIndex(selStart,
	    comboPtr->selectLast - comboPtr->selectFirst);
    byteCount = selEnd - selStart - offset;
    if (byteCount > maxBytes) {
	byteCount = maxBytes;
    }
    if (byteCount <= 0) {
	return 0;
    }
    memcpy(buffer, selStart + offset, (size_t) byteCount);
    buffer[byteCount] = '\0';
    return byteCount;
}

/*
 *----------------------------------------------------------------------
 *
 * ComboLostSelection --
 *
 *	This procedure is called back by Tk when the selection is
 *	grabbed away from an combo widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The existing selection is unhighlighted, and the window is
 *	marked as not containing a selection.
 *
 *----------------------------------------------------------------------
 */

static void
ComboLostSelection(clientData)
    ClientData clientData;	/* Information about combo widget. */
{
    Combo *comboPtr = (Combo *) clientData;

    comboPtr->flags &= ~GOT_SELECTION;

    /*
     * On Windows and Mac systems, we want to remember the selection
     * for the next time the focus enters the window.  On Unix, we need
     * to clear the selection since it is always visible.
     */

#ifdef ALWAYS_SHOW_SELECTION
    if ((comboPtr->selectFirst >= 0) && comboPtr->exportSelection) {
	comboPtr->selectFirst = -1;
	comboPtr->selectLast = -1;
	EventuallyRedraw(comboPtr);
    }
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * EventuallyRedraw --
 *
 *	Ensure that an combo is eventually redrawn on the display.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets redisplayed.  Right now we don't do selective
 *	redisplays:  the whole window will be redrawn.  This doesn't
 *	seem to hurt performance noticeably, but if it does then this
 *	could be changed.
 *
 *----------------------------------------------------------------------
 */

static void
EventuallyRedraw(comboPtr)
    Combo *comboPtr;		/* Information about widget. */
{
    if ((comboPtr->tkwin == NULL) || !Tk_IsMapped(comboPtr->tkwin)) {
	return;
    }

    /*
     * Right now we don't do selective redisplays:  the whole window
     * will be redrawn.  This doesn't seem to hurt performance noticeably,
     * but if it does then this could be changed.
     */

    if (!(comboPtr->flags & REDRAW_PENDING)) {
	comboPtr->flags |= REDRAW_PENDING;
	Tcl_DoWhenIdle(DisplayCombo, (ClientData) comboPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComboVisibleRange --
 *
 *	Return information about the range of the combo that is
 *	currently visible.
 *
 * Results:
 *	*firstPtr and *lastPtr are modified to hold fractions between
 *	0 and 1 identifying the range of characters visible in the
 *	combo.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
ComboVisibleRange(comboPtr, firstPtr, lastPtr)
    Combo *comboPtr;		/* Information about widget. */
    double *firstPtr;		/* Return position of first visible
				 * character in widget. */
    double *lastPtr;		/* Return position of char just after last
				 * visible one. */
{
    int charsInWindow;

    if (comboPtr->numChars == 0) {
	*firstPtr = 0.0;
	*lastPtr = 1.0;
    } else {
	charsInWindow = Tk_PointToChar(comboPtr->textLayout,
		Tk_Width(comboPtr->tkwin) - comboPtr->inset
			- comboPtr->layoutX - 1, 0);
	if (charsInWindow < comboPtr->numChars) {
	    charsInWindow++;
	}
	charsInWindow -= comboPtr->leftIndex;
	if (charsInWindow == 0) {
	    charsInWindow = 1;
	}

	*firstPtr = (double) comboPtr->leftIndex / comboPtr->numChars;
	*lastPtr = (double) (comboPtr->leftIndex + charsInWindow)
		/ comboPtr->numChars;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComboUpdateScrollbar --
 *
 *	This procedure is invoked whenever information has changed in
 *	an combo in a way that would invalidate a scrollbar display.
 *	If there is an associated scrollbar, then this procedure updates
 *	it by invoking a Tcl command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional command may be
 *	invoked to process errors in the command.
 *
 *----------------------------------------------------------------------
 */

static void
ComboUpdateScrollbar(comboPtr)
    Combo *comboPtr;			/* Information about widget. */
{
    char args[TCL_DOUBLE_SPACE * 2];
    int code;
    double first, last;
    Tcl_Interp *interp;

    if (comboPtr->scrollCmd == NULL) {
	return;
    }

    interp = comboPtr->interp;
    Tcl_Preserve((ClientData) interp);
    ComboVisibleRange(comboPtr, &first, &last);
    sprintf(args, " %g %g", first, last);
    code = Tcl_VarEval(interp, comboPtr->scrollCmd, args, (char *) NULL);
    if (code != TCL_OK) {
	Tcl_AddErrorInfo(interp,
		"\n    (horizontal scrolling command executed by combo)");
	Tcl_BackgroundError(interp);
    }
    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
    Tcl_Release((ClientData) interp);
}

/*
 *----------------------------------------------------------------------
 *
 * ComboBlinkProc --
 *
 *	This procedure is called as a timer handler to blink the
 *	insertion cursor off and on.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor gets turned on or off, redisplay gets invoked,
 *	and this procedure reschedules itself.
 *
 *----------------------------------------------------------------------
 */

static void
ComboBlinkProc(clientData)
    ClientData clientData;	/* Pointer to record describing combo. */
{
    Combo *comboPtr = (Combo *) clientData;

    if ((comboPtr->state == STATE_DISABLED) ||
	    !(comboPtr->flags & GOT_FOCUS) ||
	    (comboPtr->insertOffTime == 0)) {
	return;
    }
    if (comboPtr->flags & CURSOR_ON) {
	comboPtr->flags &= ~CURSOR_ON;
	comboPtr->insertBlinkHandler = Tcl_CreateTimerHandler(
	    comboPtr->insertOffTime, ComboBlinkProc, (ClientData) comboPtr);
    } else {
	comboPtr->flags |= CURSOR_ON;
	comboPtr->insertBlinkHandler = Tcl_CreateTimerHandler(
	    comboPtr->insertOnTime, ComboBlinkProc, (ClientData) comboPtr);
    }
    EventuallyRedraw(comboPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ComboFocusProc --
 *
 *	This procedure is called whenever the combo gets or loses the
 *	input focus.  It's also called whenever the window is reconfigured
 *	while it has the focus.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor gets turned on or off.
 *
 *----------------------------------------------------------------------
 */

static void
ComboFocusProc(comboPtr, gotFocus)
    Combo *comboPtr;		/* Combo that got or lost focus. */
    int gotFocus;		/* 1 means window is getting focus, 0 means
				 * it's losing it. */
{
    Tcl_DeleteTimerHandler(comboPtr->insertBlinkHandler);
    if (gotFocus) {
	comboPtr->flags |= GOT_FOCUS | CURSOR_ON;
	if (comboPtr->insertOffTime != 0) {
	    comboPtr->insertBlinkHandler = Tcl_CreateTimerHandler(
		    comboPtr->insertOnTime, ComboBlinkProc,
		    (ClientData) comboPtr);
	}
	if (comboPtr->validate == VALIDATE_ALL ||
	    comboPtr->validate == VALIDATE_FOCUS ||
	    comboPtr->validate == VALIDATE_FOCUSIN) {
	    ComboValidateChange(comboPtr, (char *) NULL,
		    comboPtr->string, -1, VALIDATE_FOCUSIN);
	}
    } else {
	comboPtr->flags &= ~(GOT_FOCUS | CURSOR_ON);
	comboPtr->insertBlinkHandler = (Tcl_TimerToken) NULL;
	if (comboPtr->validate == VALIDATE_ALL ||
	    comboPtr->validate == VALIDATE_FOCUS ||
	    comboPtr->validate == VALIDATE_FOCUSOUT) {
	    ComboValidateChange(comboPtr, (char *) NULL,
		    comboPtr->string, -1, VALIDATE_FOCUSOUT);
	}
    }
    EventuallyRedraw(comboPtr);
}

/*
 *--------------------------------------------------------------
 *
 * ComboTextVarProc --
 *
 *	This procedure is invoked when someone changes the variable
 *	whose contents are to be displayed in an combo.
 *
 * Results:
 *	NULL is always returned.
 *
 * Side effects:
 *	The text displayed in the combo will change to match the
 *	variable.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static char *
ComboTextVarProc(clientData, interp, name1, name2, flags)
    ClientData clientData;	/* Information about combo. */
    Tcl_Interp *interp;		/* Interpreter containing variable. */
    char *name1;		/* Not used. */
    char *name2;		/* Not used. */
    int flags;			/* Information about what happened. */
{
    Combo *comboPtr = (Combo *) clientData;
    char *value;

    /*
     * If the variable is unset, then immediately recreate it unless
     * the whole interpreter is going away.
     */

    if (flags & TCL_TRACE_UNSETS) {
	if ((flags & TCL_TRACE_DESTROYED) && !(flags & TCL_INTERP_DESTROYED)) {
	    Tcl_SetVar(interp, comboPtr->textVarName, comboPtr->string,
		    TCL_GLOBAL_ONLY);
	    Tcl_TraceVar(interp, comboPtr->textVarName,
		    TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		    ComboTextVarProc, clientData);
	}
	return (char *) NULL;
    }

    /*
     * Update the combo's text with the value of the variable, unless
     * the combo already has that value (this happens when the variable
     * changes value because we changed it because someone typed in
     * the combo).
     */

    value = Tcl_GetVar(interp, comboPtr->textVarName, TCL_GLOBAL_ONLY);
    if (value == NULL) {
	value = "";
    }
    ComboSetValue(comboPtr, value);
    return (char *) NULL;
}

/*
 *--------------------------------------------------------------
 *
 * ComboListVarProc --
 *
 *	This procedure is invoked when someone changes the list var
 *	whose contents are used in an combo (dropdown).
 *
 * Results:
 *	NULL is always returned.
 *
 * Side effects:
 *	The list var in the combo will change to match the
 *	variable.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static char *
ComboListVarProc(clientData, interp, name1, name2, flags)
    ClientData clientData;	/* Information about combo. */
    Tcl_Interp *interp;		/* Interpreter containing variable. */
    char *name1;		/* Not used. */
    char *name2;		/* Not used. */
    int flags;			/* Information about what happened. */
{
    Combo *comboPtr = (Combo *) clientData;

    /*
     * If the variable is unset, then immediately recreate it unless
     * the whole interpreter is going away.
     */

    if (flags & TCL_TRACE_UNSETS) {
	if ((flags & TCL_TRACE_DESTROYED) && !(flags & TCL_INTERP_DESTROYED)) {
	    Tcl_SetVar2Ex(interp, comboPtr->listVarName,
		    (char *) NULL, comboPtr->listObj, TCL_GLOBAL_ONLY);
	    Tcl_TraceVar(interp, comboPtr->listVarName,
		    TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		    ComboListVarProc, clientData);
	}
	return (char *) NULL;
    }

    
    return (char *) NULL;
}

/*
 *--------------------------------------------------------------
 *
 * ComboInvoke --
 *
 *	This procedure is invoked when ...
 *
 * Results:
 *	TCL_OK if the command evals ok (or there is no cmd).
 *      TCL_ERROR if an error occurred while executing the cmd
 *
 * Side effects:
 *      An error condition may arise
 *
 *--------------------------------------------------------------
 */

static int
ComboInvoke(interp, comboPtr, element)
    register Tcl_Interp *interp;
    register Combo *comboPtr;
    int element;
{
    char *cmd;
    char *type;
    int val;

    switch (element) {
	case SEL_DROP:
	    cmd = comboPtr->dropCmd;
	    type = "drop";
	    val = 0;
	    break;
	case SEL_SPINUP:
	    //cmd = comboPtr->spinUpCmd;
	    cmd = comboPtr->spinDownCmd;
	    type = "up";
	    val = 1;
	    break;
	case SEL_SPINDOWN:
	    cmd = comboPtr->spinDownCmd;
	    type = "down";
	    val = -1;
	    break;
	default:
	    return TCL_OK;
    }

    if (cmd != NULL) {
	int code;
	Tcl_DString script;

	Tcl_DStringInit(&script);
	ExpandPercents(comboPtr, cmd, type, "", val,
		VALIDATE_BUTTON, &script);
	Tcl_DStringAppend(&script, "", 1);

	code = Tcl_EvalEx(interp, Tcl_DStringValue(&script), -1,
		TCL_EVAL_GLOBAL | TCL_EVAL_DIRECT);
	Tcl_DStringFree(&script);

	if (code != TCL_OK) {
	    Tcl_AddErrorInfo(interp,
		    "\n\t(in command executed by combo)");
	    Tcl_BackgroundError(interp);
	    return TCL_ERROR;
	}

	Tcl_SetResult(interp, NULL, 0);
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ComboValidate --
 *
 *	This procedure is invoked when any character is added or
 *	removed from the combo widget, or a focus has trigerred validation.
 *
 * Results:
 *	TCL_OK if the validatecommand passes the new string.
 *      TCL_BREAK if the vcmd executed OK, but rejects the string.
 *      TCL_ERROR if an error occurred while executing the vcmd
 *      or a valid Tcl_Bool is not returned.
 *
 * Side effects:
 *      An error condition may arise
 *
 *--------------------------------------------------------------
 */

static int
ComboValidate(comboPtr, cmd)
     register Combo *comboPtr;	/* Combo that needs validation. */
     register char *cmd;	/* Validation command (NULL-terminated
				 * string). */
{
    register Tcl_Interp *interp = comboPtr->interp;
    int code, bool;

    code = Tcl_GlobalEval(interp, cmd);

    if (code != TCL_OK && code != TCL_RETURN) {
	Tcl_AddErrorInfo(interp,
			 "\n\t(in validation command executed by combo)");
	Tcl_BackgroundError(interp);
	return TCL_ERROR;
    }

    if (Tcl_GetBooleanFromObj(interp, Tcl_GetObjResult(interp),
			      &bool) != TCL_OK) {
	Tcl_AddErrorInfo(interp,
		 "\nValid Tcl Boolean not returned by validation command");
	Tcl_BackgroundError(interp);
	Tcl_SetResult(interp, NULL, 0);
	return TCL_ERROR;
    }

    Tcl_SetResult(interp, NULL, 0);
    return (bool ? TCL_OK : TCL_BREAK);
}

/*
 *--------------------------------------------------------------
 *
 * ComboValidateChange --
 *
 *	This procedure is invoked when any character is added or
 *	removed from the combo widget, or a focus has trigerred validation.
 *
 * Results:
 *	TCL_OK if the validatecommand accepts the new string,
 *      TCL_ERROR if any problems occured with validatecommand.
 *
 * Side effects:
 *      The insertion/deletion may be aborted, and the
 *      validatecommand might turn itself off (if an error
 *      or loop condition arises).
 *
 *--------------------------------------------------------------
 */

static int
ComboValidateChange(comboPtr, change, new, index, type)
     register Combo *comboPtr;	/* Combo that needs validation. */
     char *change;		/* Characters to be added/deleted
				 * (NULL-terminated string). */
     char *new;                 /* Potential new value of combo string */
     int index;                 /* index of insert/delete, -1 otherwise */
     int type;                  /* forced, delete, insert,
				 * focusin or focusout */
{
    int code, varValidate = (comboPtr->flags & VALIDATE_VAR);
    char *p;
    Tcl_DString script;

    if (comboPtr->validateCmd == NULL ||
	comboPtr->validate == VALIDATE_NONE) {
	return (varValidate ? TCL_ERROR : TCL_OK);
    }

    /*
     * If we're already validating, then we're hitting a loop condition
     * Return and set validate to 0 to disallow further validations
     * and prevent current validation from finishing
     */
    if (comboPtr->flags & VALIDATING) {
	comboPtr->validate = VALIDATE_NONE;
	return (varValidate ? TCL_ERROR : TCL_OK);
    }

    comboPtr->flags |= VALIDATING;

    /*
     * Now form command string and run through the -validatecommand
     */

    Tcl_DStringInit(&script);
    ExpandPercents(comboPtr, comboPtr->validateCmd,
		   change, new, index, type, &script);
    Tcl_DStringAppend(&script, "", 1);

    p = Tcl_DStringValue(&script);
    code = ComboValidate(comboPtr, p);
    Tcl_DStringFree(&script);

    /*
     * If e->validate has become VALIDATE_NONE during the validation, or
     * we now have VALIDATE_VAR set (from ComboSetValue) and didn't before,
     * it means that a loop condition almost occured.  Do not allow
     * this validation result to finish.
     */
    if (comboPtr->validate == VALIDATE_NONE
	    || (!varValidate && (comboPtr->flags & VALIDATE_VAR))) {
	code = TCL_ERROR;
    }
    /*
     * If validate will return ERROR, then disallow further validations
     * Otherwise, if it didn't accept the new string (returned TCL_BREAK)
     * then eval the invalidCmd (if it's set)
     */
    if (code == TCL_ERROR) {
	comboPtr->validate = VALIDATE_NONE;
    } else if (code == TCL_BREAK) {
	/*
	 * If we were doing forced validation (like via a variable
	 * trace) and the command returned 0, the we turn off validation
	 * because we assume that textvariables have precedence in
	 * managing the value.  We also don't call the invcmd, as it
	 * may want to do combo manipulation which the setting of the
	 * var will later wipe anyway.
	 */
	if (varValidate) {
	    comboPtr->validate = VALIDATE_NONE;
	} else if (comboPtr->invalidCmd != NULL) {
	    Tcl_DStringInit(&script);
	    ExpandPercents(comboPtr, comboPtr->invalidCmd,
			   change, new, index, type, &script);
	    Tcl_DStringAppend(&script, "", 1);
	    p = Tcl_DStringValue(&script);
	    if (Tcl_GlobalEval(comboPtr->interp, p) != TCL_OK) {
		Tcl_AddErrorInfo(comboPtr->interp,
			"\n\t(in invalidcommand executed by combo)");
		Tcl_BackgroundError(comboPtr->interp);
		code = TCL_ERROR;
		comboPtr->validate = VALIDATE_NONE;
	    }
	    Tcl_DStringFree(&script);
	}
    }

    comboPtr->flags &= ~VALIDATING;

    return code;
}

/*
 *--------------------------------------------------------------
 *
 * ExpandPercents --
 *
 *	Given a command and an event, produce a new command
 *	by replacing % constructs in the original command
 *	with information from the X event.
 *
 * Results:
 *	The new expanded command is appended to the dynamic string
 *	given by dsPtr.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
ExpandPercents(comboPtr, before, change, new, index, type, dsPtr)
     register Combo *comboPtr;	/* Combo that needs validation. */
     register char *before;	/* Command containing percent
				 * expressions to be replaced. */
     char *change; 		/* Characters to added/deleted
				 * (NULL-terminated string). */
     char *new;                 /* Potential new value of combo string */
     int index;                 /* index of insert/delete */
     int type;                  /* INSERT or DELETE */
     Tcl_DString *dsPtr;        /* Dynamic string in which to append
				 * new command. */
{
    int spaceNeeded, cvtFlags;	/* Used to substitute string as proper Tcl
				 * list element. */
    int number, length;
    register char *string;
    Tcl_UniChar ch;
    char numStorage[2*TCL_INTEGER_SPACE];

    while (1) {
	if (*before == '\0') {
	    break;
	}
	/*
	 * Find everything up to the next % character and append it
	 * to the result string.
	 */

	string = before;
	/* No need to convert '%', as it is in ascii range */
	string = Tcl_UtfFindFirst(before, '%');
	if (string == (char *) NULL) {
	    Tcl_DStringAppend(dsPtr, before, -1);
	    break;
	} else if (string != before) {
	    Tcl_DStringAppend(dsPtr, before, string-before);
	    before = string;
	}

	/*
	 * There's a percent sequence here.  Process it.
	 */

	before++; /* skip over % */
	if (*before != '\0') {
	    before += Tcl_UtfToUniChar(before, &ch);
	} else {
	    ch = '%';
	}
	switch (ch) {
	case 'd': /* Type of call that caused validation */
	    switch (type) {
	    case VALIDATE_INSERT:
		number = 1;
		break;
	    case VALIDATE_DELETE:
		number = 0;
		break;
	    default:
		number = -1;
		break;
	    }
	    sprintf(numStorage, "%d", number);
	    string = numStorage;
	    break;
	case 'i': /* index of insert/delete */
	    sprintf(numStorage, "%d", index);
	    string = numStorage;
	    break;
	case 'P': /* 'Peeked' new value of the string */
	    string = new;
	    break;
	case 's': /* Current string value of combo */
	    string = comboPtr->string;
	    break;
	case 'S': /* string to be inserted/deleted, if any */
	    string = change;
	    break;
	case 'v': /* type of validation currently set */
	    string = validateStrings[comboPtr->validate];
	    break;
	case 'V': /* type of validation in effect */
	    switch (type) {
	    case VALIDATE_INSERT:
	    case VALIDATE_DELETE:
		string = validateStrings[VALIDATE_KEY];
		break;
	    case VALIDATE_FORCED:
		string = "forced";
		break;
	    default:
		string = validateStrings[type];
		break;
	    }
	    break;
	case 'W': /* widget name */
	    string = Tk_PathName(comboPtr->tkwin);
	    break;
	default:
	    length = Tcl_UniCharToUtf(ch, numStorage);
	    numStorage[length] = '\0';
	    string = numStorage;
	    break;
	}

	spaceNeeded = Tcl_ScanCountedElement(string, -1, &cvtFlags);
	length = Tcl_DStringLength(dsPtr);
	Tcl_DStringSetLength(dsPtr, length + spaceNeeded);
	spaceNeeded = Tcl_ConvertCountedElement(string, -1,
		Tcl_DStringValue(dsPtr) + length,
		cvtFlags | TCL_DONT_USE_BRACES);
	Tcl_DStringSetLength(dsPtr, length + spaceNeeded);
    }
}
