/*
 * tkCanvLabel.c is based on tkCanvText.c --
 *
 *	This file implements text items for canvas widgets.
 *
 * Copyright (c) 1991-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 * Text rotation code Copyright 1993-1998 Lucent Technologies, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: @(#) $Id: tkCanvLabel.c,v 1.2 2002/04/30 01:30:46 hobbs Exp $
 */

#include "pkg.h"

#include "tkInt.h"
#include "tkCanvas.h"

#define DEF_CANVTEXT_FONT	"Helvetica -12"

/*
 * A comparison function for double values
 */
#define MIN_DBL_VAL		1E-4
#define DOUBLES_EQ(d1, d2)	(fabs((d1) - (d2)) < MIN_DBL_VAL)

/*
 * The structure below defines the record for each text item.
 */

typedef struct TextItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    Tk_CanvasTextInfo *textInfoPtr;
				/* Pointer to a structure containing
				 * information about the selection and
				 * insertion cursor.  The structure is owned
				 * by (and shared with) the generic canvas
				 * code. */
    /*
     * Fields that are set by widget commands other than "configure".
     */

    double x, y;		/* Positioning point for text. */
    int insertPos;		/* Insertion cursor is displayed just to left
				 * of character with this index. */

    /*
     * Configuration settings that are updated by Tk_ConfigureWidget.
     */

    Tk_Anchor anchor;		/* Where to anchor text relative to (x,y). */
    Tk_TSOffset tsoffset;
    XColor *color;		/* Color for text. */
    XColor *activeColor;	/* Color for text. */
    XColor *disabledColor;	/* Color for text. */
    Tk_Font tkfont;		/* Font for drawing text. */
    Tk_Justify justify;		/* Justification mode for text. */
    Pixmap stipple;		/* Stipple bitmap for text, or None. */
    Pixmap activeStipple;	/* Stipple bitmap for text, or None. */
    Pixmap disabledStipple;	/* Stipple bitmap for text, or None. */
    char *text;			/* Text for item (malloc-ed). */
    int width;			/* Width of lines for word-wrap, pixels.
				 * Zero means no word-wrap. */
    double theta;		/* Rotation angle */
    Pixmap rotatedDrawable;	/* Pixmap containing the rotated text */
    Pixmap normalDrawable;	/* Pixmap containing the unrotated text */
    Pixmap rotatedMask;		/* Mask for drawing the rotated text */
    Pixmap normalMask;		/* Mask for drawing the unrotated text */
    int drawablesNeedUpdate;	/* Flag indicating if we should update *
    				 * the contents of the drawables */
    int normalWidth;		/* Width of the normalDrawable */
    int normalHeight;		/* Height of the normalDrawable */
    int rotatedWidth;		/* Width of the rotatedDrawable */
    int rotatedHeight;		/* Height of the rotatedDrawable */

    /*
     * Fields whose values are derived from the current values of the
     * configuration settings above.
     */

    int numChars;		/* Number of non-NULL characters in text. */
    int numBytes;               /* Length of text in bytes. */
    Tk_TextLayout textLayout;	/* Cached text layout information. */
    int leftEdge;		/* Pixel location of the left edge of the
				 * text item; where the left border of the
				 * text layout is drawn. */
    int rightEdge;		/* Pixel just to right of right edge of
				 * area of text item.  Used for selecting up
				 * to end of line. */
    GC gc;			/* Graphics context for drawing text. */
    GC maskGC;			/* Graphics context for manipulating 1-bit
    				 * bitmaps */
    GC drawableGC;		/* Graphics context for copying the rotated
    				 * bitmap onto the screen */
    GC selTextGC;		/* Graphics context for selected text. */
    GC cursorOffGC;		/* If not None, this gives a graphics context
				 * to use to draw the insertion cursor when
				 * it's off.  Used if the selection and
				 * insertion cursor colors are the same.  */
} TextItem;

/*
 * Information used for parsing configuration specs:
 */
static Tk_CustomOption stateOption = {
       (Tk_OptionParseProc *) TkStateParseProc,
       TkStatePrintProc, (ClientData) 2
};
static Tk_CustomOption offsetOption = {
       (Tk_OptionParseProc *) TkOffsetParseProc,
       TkOffsetPrintProc, (ClientData) (TK_OFFSET_RELATIVE)
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_COLOR, "-activefill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(TextItem, activeColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-activestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(TextItem, activeStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_ANCHOR, "-anchor", (char *) NULL, (char *) NULL,
	"center", Tk_Offset(TextItem, anchor),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-disabledfill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(TextItem, disabledColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-disabledstipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(TextItem, disabledStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(TextItem, color), TK_CONFIG_NULL_OK},
    {TK_CONFIG_FONT, "-font", (char *) NULL, (char *) NULL,
	DEF_CANVTEXT_FONT, Tk_Offset(TextItem, tkfont), 0},
    {TK_CONFIG_JUSTIFY, "-justify", (char *) NULL, (char *) NULL,
	"left", Tk_Offset(TextItem, justify),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-offset", (char *) NULL, (char *) NULL,
	"0,0", Tk_Offset(TextItem, tsoffset),
	TK_CONFIG_DONT_SET_DEFAULT, &offsetOption},
    {TK_CONFIG_CUSTOM, "-state", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, state), TK_CONFIG_NULL_OK,
	&stateOption},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(TextItem, stipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &customTagsOption},
    {TK_CONFIG_STRING, "-text", (char *) NULL, (char *) NULL,
	"", Tk_Offset(TextItem, text), 0},
    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(TextItem, width), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_DOUBLE, "-angle", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(TextItem, theta), TK_CONFIG_NULL_OK},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeTextBbox _ANSI_ARGS_((Tk_Canvas canvas,
			    TextItem *textPtr));
static int		ConfigureText _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    Tcl_Obj *CONST argv[], int flags));
static int		CreateText _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, struct Tk_Item *itemPtr,
			    int argc, Tcl_Obj *CONST argv[]));
static void		DeleteText _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display));
static void		DisplayCanvText _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int x, int y, int width, int height));
static void		DisplayCanvText2 _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int *, int *));
static int		GetSelText _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, int offset, char *buffer,
			    int maxBytes));
static int		GetTextIndex _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr,
			    Tcl_Obj *obj, int *indexPtr));
static void		ScaleText _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		SetTextCursor _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, int index));
static int		TextCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr,
			    int argc, Tcl_Obj *CONST argv[]));
static void		TextDeleteChars _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, int first, int last));
static void		TextInsert _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, int beforeThis, char *string));
static int		TextToArea _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *rectPtr));
static double		TextToPoint _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *pointPtr));
static int		TextToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int prepass));
static void		TranslateText _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double deltaX, double deltaY));
#ifdef ORIG /*CRRW visrotate ???*/
static void		RotateTextPixmap _ANSI_ARGS_((Pixmap src, Pixmap dest,
			    double theta));
#endif
static void		RotateAnchor _ANSI_ARGS_((int x, int y, int width,
			    int height, double theta, Tk_Anchor anchor,
			    int *newX, int *newY, int *newWidth,
			    int *newHeight));
static void		TranslateAnchor _ANSI_ARGS_((int x, int y, int width,
			    int height, Tk_Anchor anchor, int *newX,
			    int *newY));
static void		TextCreateDrawables _ANSI_ARGS_((Tk_Window canvas,
			    TextItem *textPtr));
static void		TextClearDrawables _ANSI_ARGS_((Tk_Window tkwin,
			    TextItem *textPtr));
static void		TextRotateBitmap _ANSI_ARGS_((Display *display,
			   Drawable draw, TextItem *textPtr, Tk_Canvas canvas,
			   Pixmap srcBitmap, int srcWidth, int srcHeight,
			   double theta, Pixmap destBitmap, int destWidth,
			   int destHeight));
static void		TextUpdateDrawableGC _ANSI_ARGS_((Tk_Window tkwin,
			    TextItem *textPtr));

/*
 * The structures below defines the label item types
 * by means of procedures that can be invoked by generic item code.
 */

Tk_ItemType TkCanvLabelType = {
    "label",				/* name */
    sizeof(TextItem),			/* itemSize */
    CreateText,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureText,			/* configureProc */
    TextCoords,				/* coordProc */
    DeleteText,				/* deleteProc */
    DisplayCanvText,			/* displayProc */
    TK_CONFIG_OBJS,             /* flags */
    TextToPoint,			/* pointProc */
    TextToArea,				/* areaProc */
    TextToPostscript,			/* postscriptProc */
    ScaleText,				/* scaleProc */
    TranslateText,			/* translateProc */
    (Tk_ItemIndexProc *) GetTextIndex,	/* indexProc */
    SetTextCursor,			/* icursorProc */
    GetSelText,				/* selectionProc */
    TextInsert,				/* insertProc */
    TextDeleteChars,			/* dTextProc */
    (Tk_ItemType *) NULL,		/* nextPtr */
};

/*
 *--------------------------------------------------------------
 *
 * CreateText --
 *
 *	This procedure is invoked to create a new text item
 *	in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new text item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateText(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas canvas;			/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    Tcl_Obj *CONST argv[]; 		/* Arguments describing rectangle. */
{
    TextItem *textPtr = (TextItem *) itemPtr;
    int i;

    if (argc==1) {
	i = 1;
    } else {
	char *arg = Tcl_GetStringFromObj( argv[1],NULL);
	if ((argc>1) && (arg[0] == '-')
		&& (arg[1] >= 'a') && (arg[1] <= 'z')) {
	    i = 1;
	} else {
	    i = 2;
	}
    }

    if (argc < i) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tk_PathName(Tk_CanvasTkwin(canvas)), " create ",
		itemPtr->typePtr->name, " x y ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    textPtr->textInfoPtr = Tk_CanvasGetTextInfo(canvas);

    textPtr->insertPos	= 0;

    textPtr->anchor		= TK_ANCHOR_CENTER;
    textPtr->tsoffset.flags	= 0;
    textPtr->tsoffset.xoffset	= 0;
    textPtr->tsoffset.yoffset	= 0;
    textPtr->color		= NULL;
    textPtr->activeColor	= NULL;
    textPtr->disabledColor	= NULL;
    textPtr->tkfont		= NULL;
    textPtr->justify		= TK_JUSTIFY_LEFT;
    textPtr->stipple		= None;
    textPtr->activeStipple	= None;
    textPtr->disabledStipple	= None;
    textPtr->text		= NULL;
    textPtr->width		= 0;

    textPtr->rotatedDrawable	= None;
    textPtr->normalDrawable	= None;
    textPtr->rotatedMask	= None;
    textPtr->normalMask		= None;
    textPtr->drawablesNeedUpdate = 1;
    textPtr->rotatedWidth	= 0;
    textPtr->rotatedHeight	= 0;
    textPtr->normalWidth	= 0;
    textPtr->normalHeight	= 0;

    textPtr->numChars		= 0;
    textPtr->numBytes		= 0;
    textPtr->textLayout		= NULL;
    textPtr->leftEdge		= 0;
    textPtr->rightEdge		= 0;
    textPtr->gc			= None;
    textPtr->maskGC		= None;
    textPtr->drawableGC		= None;
    textPtr->selTextGC		= None;
    textPtr->cursorOffGC	= None;

    /*
     * Process the arguments to fill in the item record.
     */

    if ((TextCoords(interp, canvas, itemPtr, i, argv) != TCL_OK)) {
	goto error;
    }
    if (ConfigureText(interp, canvas, itemPtr, argc-i, argv+i, 0) == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteText(canvas, itemPtr, Tk_Display(Tk_CanvasTkwin(canvas)));
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * TextCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on text items.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
TextCoords(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    Tcl_Obj *CONST argv[];		/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    TextItem *textPtr = (TextItem *) itemPtr;

    if (argc == 0) {
	Tcl_Obj *obj = Tcl_NewObj();
	Tcl_Obj *subobj = Tcl_NewDoubleObj(textPtr->x);
	Tcl_ListObjAppendElement(interp, obj, subobj);
	subobj = Tcl_NewDoubleObj(textPtr->y);
	Tcl_ListObjAppendElement(interp, obj, subobj);
	Tcl_SetObjResult(interp, obj);
    } else if (argc < 3) {
	if (argc==1) {
	    if (Tcl_ListObjGetElements(interp, argv[0], &argc,
		    (Tcl_Obj ***) &argv) != TCL_OK) {
		return TCL_ERROR;
	    } else if (argc != 2) {
		char buf[64 + TCL_INTEGER_SPACE];
		sprintf(buf, "wrong # coordinates: expected 2, got %d", argc);
		Tcl_SetResult(interp, buf, TCL_VOLATILE);
		return TCL_ERROR;
	    }
	}
	if ((Tk_CanvasGetCoordFromObj(interp, canvas, argv[0], &textPtr->x) != TCL_OK)
		|| (Tk_CanvasGetCoordFromObj(interp, canvas, argv[1],
  		    &textPtr->y) != TCL_OK)) {
	    return TCL_ERROR;
	}
	ComputeTextBbox(canvas, textPtr);
    } else {
	char buf[64 + TCL_INTEGER_SPACE];
	sprintf(buf, "wrong # coordinates: expected 0 or 2, got %d", argc);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureText --
 *
 *	This procedure is invoked to configure various aspects
 *	of a text item, such as its border and background colors.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureText(interp, canvas, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Interpreter for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Rectangle item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    Tcl_Obj *CONST argv[];	/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    TextItem *textPtr = (TextItem *) itemPtr;
    XGCValues gcValues;
    GC newGC, newSelGC;
    unsigned long mask;
    Tk_Window tkwin;
    Tk_CanvasTextInfo *textInfoPtr = textPtr->textInfoPtr;
    XColor *selBgColorPtr;
    XColor *color;
    Pixmap stipple;
    Tk_State state;

    tkwin = Tk_CanvasTkwin(canvas);
    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc,(char **) argv,
	    (char *) textPtr, flags|TK_CONFIG_OBJS) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    state = itemPtr->state;

    if (textPtr->activeColor != NULL ||
	    textPtr->activeStipple != None) {
	itemPtr->redraw_flags |= TK_ITEM_STATE_DEPENDANT;
    } else {
	itemPtr->redraw_flags &= ~TK_ITEM_STATE_DEPENDANT;
    }

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    color = textPtr->color;
    stipple = textPtr->stipple;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (textPtr->activeColor!=NULL) {
	    color = textPtr->activeColor;
	}
	if (textPtr->activeStipple!=None) {
	    stipple = textPtr->activeStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (textPtr->disabledColor!=NULL) {
	    color = textPtr->disabledColor;
	}
	if (textPtr->disabledStipple!=None) {
	    stipple = textPtr->disabledStipple;
	}
    }
    newGC = newSelGC = None;
    if (textPtr->tkfont != NULL) {
	gcValues.font = Tk_FontId(textPtr->tkfont);
	mask = GCFont;
        if (color != NULL) {
	    gcValues.foreground = color->pixel;
	    mask |= GCForeground;
	    if (stipple != None) {
		gcValues.stipple = stipple;
		gcValues.fill_style = FillStippled;
		mask |= GCStipple|GCFillStyle;
	    }
	    newGC = Tk_GetGC(tkwin, mask, &gcValues);
	}
	mask &= ~(GCTile|GCFillStyle|GCStipple);
	if (stipple != None) {
	    gcValues.stipple = stipple;
	    gcValues.fill_style = FillStippled;
	    mask |= GCStipple|GCFillStyle;
	}
	gcValues.foreground = textInfoPtr->selFgColorPtr->pixel;
	newSelGC = Tk_GetGC(tkwin, mask|GCForeground, &gcValues);
    }
    if (textPtr->gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), textPtr->gc);
    }
    textPtr->gc = newGC;
    if (textPtr->selTextGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), textPtr->selTextGC);
    }
    textPtr->selTextGC = newSelGC;

    selBgColorPtr = Tk_3DBorderColor(textInfoPtr->selBorder);
    if (Tk_3DBorderColor(textInfoPtr->insertBorder)->pixel
	    == selBgColorPtr->pixel) {
	if (selBgColorPtr->pixel == BlackPixelOfScreen(Tk_Screen(tkwin))) {
	    gcValues.foreground = WhitePixelOfScreen(Tk_Screen(tkwin));
	} else {
	    gcValues.foreground = BlackPixelOfScreen(Tk_Screen(tkwin));
	}
	newGC = Tk_GetGC(tkwin, GCForeground, &gcValues);
    } else {
	newGC = None;
    }
    if (textPtr->cursorOffGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), textPtr->cursorOffGC);
    }
    textPtr->cursorOffGC = newGC;

    /*
     * If the text was changed, move the selection and insertion indices
     * to keep them inside the item.
     */

    textPtr->numBytes = strlen(textPtr->text);
    textPtr->numChars = Tcl_NumUtfChars(textPtr->text, textPtr->numBytes);
    if (textInfoPtr->selItemPtr == itemPtr) {
	if (textInfoPtr->selectFirst >= textPtr->numChars) {
	    textInfoPtr->selItemPtr = NULL;
	} else {
	    if (textInfoPtr->selectLast >= textPtr->numChars) {
		textInfoPtr->selectLast = textPtr->numChars-1;
	    }
	    if ((textInfoPtr->anchorItemPtr == itemPtr)
		    && (textInfoPtr->selectAnchor >= textPtr->numChars)) {
		textInfoPtr->selectAnchor = textPtr->numChars-1;
	    }
	}
    }
    if (textPtr->insertPos >= textPtr->numChars) {
	textPtr->insertPos = textPtr->numChars;
    }

    ComputeTextBbox(canvas, textPtr);

    /*
     * This Creates and clears the text drawables
     */

    TextCreateDrawables(Tk_CanvasTkwin(canvas), textPtr);
    textPtr->drawablesNeedUpdate = 1;

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteText --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a text item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteText(canvas, itemPtr, display)
    Tk_Canvas canvas;			/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
    Display *display;			/* Display containing window for
					 * canvas. */
{
    TextItem *textPtr = (TextItem *) itemPtr;

    if (textPtr->color != NULL) {
	Tk_FreeColor(textPtr->color);
    }
    if (textPtr->activeColor != NULL) {
	Tk_FreeColor(textPtr->activeColor);
    }
    if (textPtr->disabledColor != NULL) {
	Tk_FreeColor(textPtr->disabledColor);
    }
    Tk_FreeFont(textPtr->tkfont);
    if (textPtr->stipple != None) {
	Tk_FreeBitmap(display, textPtr->stipple);
    }
    if (textPtr->activeStipple != None) {
	Tk_FreeBitmap(display, textPtr->activeStipple);
    }
    if (textPtr->disabledStipple != None) {
	Tk_FreeBitmap(display, textPtr->disabledStipple);
    }
    if (textPtr->text != NULL) {
	ckfree(textPtr->text);
    }

    Tk_FreeTextLayout(textPtr->textLayout);
    if (textPtr->gc != None) {
	Tk_FreeGC(display, textPtr->gc);
    }
    if (textPtr->maskGC != None) {
	Tk_FreeGC(display, textPtr->maskGC);
    }
    if (textPtr->drawableGC != None) {
	Tk_FreeGC(display, textPtr->drawableGC);
    }
    if (textPtr->selTextGC != None) {
	Tk_FreeGC(display, textPtr->selTextGC);
    }
    if (textPtr->cursorOffGC != None) {
	Tk_FreeGC(display, textPtr->cursorOffGC);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeTextBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a text item.
 *	In addition, it recomputes all of the geometry information
 *	used to display a text item or check for mouse hits.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr, and the linePtr structure is regenerated
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeTextBbox(canvas, textPtr)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    TextItem *textPtr;			/* Item whose bbos is to be
					 * recomputed. */
{
    Tk_CanvasTextInfo *textInfoPtr;
    int leftX, topY, width, height, fudge;
    Tk_State state = textPtr->header.state;

    if (state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    Tk_FreeTextLayout(textPtr->textLayout);
    textPtr->textLayout = Tk_ComputeTextLayout(textPtr->tkfont,
	    textPtr->text, textPtr->numChars, textPtr->width,
	    textPtr->justify, 0, &width, &height);

    if (state == TK_STATE_HIDDEN || textPtr->color == NULL) {
	width = height = 0;
    }

    RotateAnchor(textPtr->x, textPtr->y, width, height,
	    textPtr->theta, textPtr->anchor,
	    &leftX, &topY, &width, &height);

    textPtr->header.x1 = leftX;
    textPtr->header.x2 = leftX + width;
    textPtr->header.y1 = topY;
    textPtr->header.y2 = topY + height;

    textPtr->leftEdge  = leftX;
    textPtr->rightEdge = leftX + width;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayCanvText --
 *
 *	This procedure is invoked to display the text pixmap in a
 *	given drawable, usually ending up on the screen.
 *
 * Results:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
DisplayCanvText(canvas, itemPtr, display, drawable, x, y, width, height)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Display *display;			/* Display on which to draw item. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
    int x, y, width, height;		/* Describes region of canvas that
					 * must be redisplayed (not used). */
{
    TextItem *textPtr = (TextItem *)itemPtr;
    Tk_State state = itemPtr->state;
    int w, h;
    short x1, y1;
    XGCValues gcValues;
    unsigned long mask = 0;
    GC newGC;

    if (state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    gcValues.font = Tk_FontId(textPtr->tkfont);
    XChangeGC(display, textPtr->maskGC, GCFont, &gcValues);

    if (textPtr->drawablesNeedUpdate) {
	DisplayCanvText2(canvas, itemPtr, display, textPtr->normalDrawable,
		&w, &h);

	TextRotateBitmap(display, drawable, textPtr, canvas,
	    textPtr->normalDrawable,
	    textPtr->normalWidth, textPtr->normalHeight,
	    textPtr->theta,
	    textPtr->rotatedDrawable,
	    textPtr->rotatedWidth,
	    textPtr->rotatedHeight);

	textPtr->drawablesNeedUpdate = 0;
    }

    Tk_CanvasDrawableCoords(canvas, textPtr->header.x1, textPtr->header.y1,
	    &x1, &y1);

    gcValues.background = 0;
    gcValues.function = GXcopy;
    gcValues.clip_mask = textPtr->rotatedMask;
    gcValues.clip_x_origin = (int)x1;
    gcValues.clip_y_origin = (int)y1;
    mask = GCFunction|GCBackground|GCClipMask|GCClipXOrigin|GCClipYOrigin;
    newGC = XCreateGC(display, drawable, mask, &gcValues);
    if (textPtr->gc)
	XCopyGC(display, textPtr->gc, GCForeground|GCFont, newGC);

    /*
    XCopyPlane(display, textPtr->normalMask, drawable, newGC, 0, 0, w, h, (int)x1, (int)y1+h, 1);
    */

    /*
     * Now copy the rotated drawable to the screen.
     */

    /*
    XSetClipMask(display, textPtr->drawableGC, textPtr->rotatedMask);
    XSetClipOrigin(display, textPtr->drawableGC, (int)x1, (int)y1);
    */

    if (textPtr->gc) {
	XCopyPlane(display, textPtr->rotatedMask, drawable, newGC,
	    0, 0, textPtr->rotatedWidth, textPtr->rotatedHeight,
	    (int)x1, (int)y1, 1);
    }

    XFreeGC(display, newGC);

    /*
    XCopyArea(display, textPtr->rotatedDrawable, drawable, textPtr->drawableGC,
	0, 0, textPtr->rotatedWidth, textPtr->rotatedHeight,
	(int)x1, (int)y1);
    */
}

/*
 *--------------------------------------------------------------
 *
 * DisplayCanvText2 --
 *
 *	This procedure is invoked to draw a text item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas.
 *
 *--------------------------------------------------------------
 */

static void
DisplayCanvText2(canvas, itemPtr, display, drawable, width, height)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Display *display;			/* Display on which to draw item. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
    int *width, *height;		/* Describes size of normal bitmap */
{
    TextItem *textPtr = (TextItem *)itemPtr;
    Tk_CanvasTextInfo *textInfoPtr = textPtr->textInfoPtr;
    int selFirst, selLast;
    short drawableX, drawableY;
    Pixmap stipple;
    Tk_State state = itemPtr->state;
    int x, y;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    stipple = textPtr->stipple;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (textPtr->activeStipple!=None) {
	    stipple = textPtr->activeStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (textPtr->disabledStipple!=None) {
	    stipple = textPtr->disabledStipple;
	}
    }

    if (textPtr->gc == None) {
	return;
    }

    /*
     * If we're stippling, then modify the stipple offset in the GC.  Be
     * sure to reset the offset when done, since the GC is supposed to be
     * read-only.
     */
    selFirst = -1;
    selLast = 0;		/* lint. */
    /*
     * Don't bother with selection for now
     */
    if (textInfoPtr->selItemPtr == itemPtr && 0) {
	selFirst = textInfoPtr->selectFirst;
	selLast = textInfoPtr->selectLast;
	if (selLast >= textPtr->numChars) {
	    selLast = textPtr->numChars - 1;
	}
	if ((selFirst >= 0) && (selFirst <= selLast)) {
	    /*
	     * Draw a special background under the selection.
	     */

	    int xFirst, yFirst, hFirst;
	    int xLast, yLast, wLast;

	    Tk_CharBbox(textPtr->textLayout, selFirst,
		    &xFirst, &yFirst, NULL, &hFirst);
	    Tk_CharBbox(textPtr->textLayout, selLast,
		    &xLast, &yLast, &wLast, NULL);

	    /*
	     * If the selection spans the end of this line, then display
	     * selection background all the way to the end of the line.
	     * However, for the last line we only want to display up to the
	     * last character, not the end of the line.
	     */

	    x = xFirst;
	    *height = hFirst;
	    for (y = yFirst ; y <= yLast; y += *height) {
		if (y == yLast) {
		    *width = (xLast + wLast) - x;
		} else {
		    *width = textPtr->rightEdge - textPtr->leftEdge - x;
		}
		Tk_CanvasDrawableCoords(canvas,
			(double) (textPtr->leftEdge + x
				- textInfoPtr->selBorderWidth),
			(double) (textPtr->header.y1 + y),
			&drawableX, &drawableY);
		Tk_Fill3DRectangle(Tk_CanvasTkwin(canvas), drawable,
			textInfoPtr->selBorder, drawableX, drawableY,
			*width + 2 * textInfoPtr->selBorderWidth,
			*height, textInfoPtr->selBorderWidth, TK_RELIEF_RAISED);
		/*
		 * Once more for the mask
		 */
		Tk_Fill3DRectangle(Tk_CanvasTkwin(canvas), textPtr->normalMask,
			textInfoPtr->selBorder, drawableX, drawableY,
			*width + 2 * textInfoPtr->selBorderWidth,
			*height, textInfoPtr->selBorderWidth, TK_RELIEF_RAISED);
		x = 0;
	    }
	}
    }

    /*
     * If the insertion point should be displayed, then draw a special
     * background for the cursor before drawing the text.  Note:  if
     * we're the cursor item but the cursor is turned off, then redraw
     * background over the area of the cursor.  This guarantees that
     * the selection won't make the cursor invisible on mono displays,
     * where both are drawn in the same color.
     */

    /*
     * Don't draw the insertion cursor
     */

    if ((textInfoPtr->focusItemPtr == itemPtr) && (textInfoPtr->gotFocus) && 0) {
	if (Tk_CharBbox(textPtr->textLayout, textPtr->insertPos,
		&x, &y, NULL, height)) {
	    Tk_CanvasDrawableCoords(canvas,
		    (double) (textPtr->leftEdge + x
			    - (textInfoPtr->insertWidth / 2)),
		    (double) (textPtr->header.y1 + y),
		    &drawableX, &drawableY);
	    if (textInfoPtr->cursorOn) {
		/*
		 * The cursor is useless with rotated text
		Tk_Fill3DRectangle(Tk_CanvasTkwin(canvas), drawable,
			textInfoPtr->insertBorder,
			drawableX, drawableY,
			textInfoPtr->insertWidth, *height,
			textInfoPtr->insertBorderWidth, TK_RELIEF_RAISED);
		*/
		/*
		 * Once more for the mask
		 */
		/*
		Tk_Fill3DRectangle(Tk_CanvasTkwin(canvas), textPtr->normalMask,
			textInfoPtr->insertBorder,
			drawableX, drawableY,
			textInfoPtr->insertWidth, *height,
			textInfoPtr->insertBorderWidth, TK_RELIEF_RAISED);
		*/
	    } else if (textPtr->cursorOffGC != None) {
		/*
		 * Redraw the background over the area of the cursor,
		 * even though the cursor is turned off.  This
		 * guarantees that the selection won't make the cursor
		 * invisible on mono displays, where both may be drawn
		 * in the same color.
		 */

		XFillRectangle(display, drawable, textPtr->cursorOffGC,
			drawableX, drawableY,
			(unsigned) textInfoPtr->insertWidth,
			(unsigned) (*height));
		/*
		 * Once more for the mask
		 */
		XFillRectangle(display, textPtr->normalMask, textPtr->maskGC,
			drawableX, drawableY,
			(unsigned) textInfoPtr->insertWidth,
			(unsigned) (*height));
	    }
	}
    }


    /*
     * Display the text in two pieces: draw the entire text item, then
     * draw the selected text on top of it.  The selected text then
     * will only need to be drawn if it has different attributes (such
     * as foreground color) than regular text.
     */

    Tk_CanvasDrawableCoords(canvas, (double) textPtr->leftEdge,
	    (double) textPtr->header.y1, &drawableX, &drawableY);
    /*
    Tk_DrawTextLayout(display, drawable, textPtr->gc, textPtr->textLayout,
	    drawableX, drawableY, 0, -1);
    */
    Tk_DrawTextLayout(display, drawable, textPtr->gc, textPtr->textLayout,
	    0, 0, 0, -1);
    /*
     * Once more for the mask
     */
    Tk_DrawTextLayout(display, textPtr->normalMask, textPtr->maskGC,
	    textPtr->textLayout, 0, 0, 0, -1);

    /*
     * Don't draw the selection for now.
     */
    if ((selFirst >= 0) && (textPtr->selTextGC != textPtr->gc) && 0) {
	Tk_DrawTextLayout(display, drawable, textPtr->selTextGC,
	    textPtr->textLayout, drawableX, drawableY, selFirst,
	    selLast + 1);
	/*
	 * Once more for the mask
	 */
	Tk_DrawTextLayout(display, textPtr->normalMask, textPtr->maskGC,
	    textPtr->textLayout, drawableX, drawableY, selFirst,
	    selLast + 1);
    }
}

/*
 *--------------------------------------------------------------
 *
 * TextInsert --
 *
 *	Insert characters into a text item at a given position.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The text in the given item is modified.  The cursor and
 *	selection positions are also modified to reflect the
 *	insertion.
 *
 *--------------------------------------------------------------
 */

static void
TextInsert(canvas, itemPtr, index, string)
    Tk_Canvas canvas;		/* Canvas containing text item. */
    Tk_Item *itemPtr;		/* Text item to be modified. */
    int index;			/* Index of character before which text is
				 * to be inserted. */
    char *string;		/* New characters to be inserted. */
{
    TextItem *textPtr = (TextItem *) itemPtr;
    int byteIndex, byteCount, charsAdded;
    char *new, *text;
    Tk_CanvasTextInfo *textInfoPtr = textPtr->textInfoPtr;

    string = Tcl_GetStringFromObj((Tcl_Obj *) string, &byteCount);

    text = textPtr->text;

    if (index < 0) {
	index = 0;
    }
    if (index > textPtr->numChars) {
	index = textPtr->numChars;
    }
    byteIndex = Tcl_UtfAtIndex(text, index) - text;
    byteCount = strlen(string);
    if (byteCount == 0) {
	return;
    }

    new = (char *) ckalloc((unsigned) textPtr->numBytes + byteCount + 1);
    memcpy(new, text, (size_t) byteIndex);
    strcpy(new + byteIndex, string);
    strcpy(new + byteIndex + byteCount, text + byteIndex);

    ckfree(text);
    textPtr->text = new;
    charsAdded = Tcl_NumUtfChars(string, byteCount);
    textPtr->numChars += charsAdded;
    textPtr->numBytes += byteCount;

    /*
     * Inserting characters invalidates indices such as those for the
     * selection and cursor.  Update the indices appropriately.
     */

    if (textInfoPtr->selItemPtr == itemPtr) {
	if (textInfoPtr->selectFirst >= index) {
	    textInfoPtr->selectFirst += charsAdded;
	}
	if (textInfoPtr->selectLast >= index) {
	    textInfoPtr->selectLast += charsAdded;
	}
	if ((textInfoPtr->anchorItemPtr == itemPtr)
		&& (textInfoPtr->selectAnchor >= index)) {
	    textInfoPtr->selectAnchor += charsAdded;
	}
    }
    if (textPtr->insertPos >= index) {
	textPtr->insertPos += charsAdded;
    }
    ComputeTextBbox(canvas, textPtr);
}

/*
 *--------------------------------------------------------------
 *
 * TextDeleteChars --
 *
 *	Delete one or more characters from a text item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Characters between "first" and "last", inclusive, get
 *	deleted from itemPtr, and things like the selection
 *	position get updated.
 *
 *--------------------------------------------------------------
 */

static void
TextDeleteChars(canvas, itemPtr, first, last)
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item in which to delete characters. */
    int first;			/* Index of first character to delete. */
    int last;			/* Index of last character to delete. */
{
    TextItem *textPtr = (TextItem *) itemPtr;
    int byteIndex, byteCount, charsRemoved;
    char *new, *text;
    Tk_CanvasTextInfo *textInfoPtr = textPtr->textInfoPtr;

    text = textPtr->text;
    if (first < 0) {
	first = 0;
    }
    if (last >= textPtr->numChars) {
	last = textPtr->numChars - 1;
    }
    if (first > last) {
	return;
    }
    charsRemoved = last + 1 - first;

    byteIndex = Tcl_UtfAtIndex(text, first) - text;
    byteCount = Tcl_UtfAtIndex(text + byteIndex, charsRemoved)
	- (text + byteIndex);
    
    new = (char *) ckalloc((unsigned) (textPtr->numBytes + 1 - byteCount));
    memcpy(new, text, (size_t) byteIndex);
    strcpy(new + byteIndex, text + byteIndex + byteCount);

    ckfree(text);
    textPtr->text = new;
    textPtr->numChars -= charsRemoved;
    textPtr->numBytes -= byteCount;

    /*
     * Update indexes for the selection and cursor to reflect the
     * renumbering of the remaining characters.
     */

    if (textInfoPtr->selItemPtr == itemPtr) {
	if (textInfoPtr->selectFirst > first) {
	    textInfoPtr->selectFirst -= charsRemoved;
	    if (textInfoPtr->selectFirst < first) {
		textInfoPtr->selectFirst = first;
	    }
	}
	if (textInfoPtr->selectLast >= first) {
	    textInfoPtr->selectLast -= charsRemoved;
	    if (textInfoPtr->selectLast < first - 1) {
		textInfoPtr->selectLast = first - 1;
	    }
	}
	if (textInfoPtr->selectFirst > textInfoPtr->selectLast) {
	    textInfoPtr->selItemPtr = NULL;
	}
	if ((textInfoPtr->anchorItemPtr == itemPtr)
		&& (textInfoPtr->selectAnchor > first)) {
	    textInfoPtr->selectAnchor -= charsRemoved;
	    if (textInfoPtr->selectAnchor < first) {
		textInfoPtr->selectAnchor = first;
	    }
	}
    }
    if (textPtr->insertPos > first) {
	textPtr->insertPos -= charsRemoved;
	if (textPtr->insertPos < first) {
	    textPtr->insertPos = first;
	}
    }
    ComputeTextBbox(canvas, textPtr);
    return;
}

/*
 *--------------------------------------------------------------
 *
 * TextToPoint --
 *
 *	Computes the distance from a given point to a given
 *	text item, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the text item.  If
 *	the point isn't inside the text item then the return value
 *	is the distance from the point to the text item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static double
TextToPoint(canvas, itemPtr, pointPtr)
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    TextItem *textPtr;
    Tk_State state = itemPtr->state;
    double value;
    double xDiff, yDiff;

    if (state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    textPtr = (TextItem *) itemPtr;

    value =  (double) Tk_DistanceToTextLayout(textPtr->textLayout,
	    (int) pointPtr[0] - textPtr->leftEdge,
	    (int) pointPtr[1] - textPtr->header.y1);

    /*
     * Just use the bounding box for the text.  It's quite complicated
     * to calculate the distance using Tk_DistanceToTextLayout
     */

    if ((state == TK_STATE_HIDDEN) || (textPtr->color == NULL) ||
	    (textPtr->text == NULL) || (*textPtr->text == 0)) {
	return 1.0e36;
    }

    if ((pointPtr[0] >= textPtr->header.x1) &&
	    (pointPtr[0] < textPtr->header.x2) &&
	    (pointPtr[1] >= textPtr->header.y1) &&
	    (pointPtr[1] < textPtr->header.y2)) {
	return 0.0;
    }

    if (pointPtr[0] < textPtr->header.x1) {
	xDiff = textPtr->header.x1 - pointPtr[0];
    } else if (pointPtr[0] > textPtr->header.x2)  {
	xDiff = pointPtr[0] - textPtr->header.x2;
    } else {
	xDiff = 0;
    }

    if (pointPtr[1] < textPtr->header.y1) {
	yDiff = textPtr->header.y1 - pointPtr[1];
    } else if (pointPtr[1] > textPtr->header.y2)  {
	yDiff = pointPtr[1] - textPtr->header.y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * TextToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
TextToArea(canvas, itemPtr, rectPtr)
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item to check against rectangle. */
    double *rectPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    TextItem *textPtr;
    Tk_State state = itemPtr->state;

    if (state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    textPtr = (TextItem *) itemPtr;
    return Tk_IntersectTextLayout(textPtr->textLayout,
	    (int) (rectPtr[0] + 0.5) - textPtr->leftEdge,
	    (int) (rectPtr[1] + 0.5) - textPtr->header.y1,
	    (int) (rectPtr[2] - rectPtr[0] + 0.5),
	    (int) (rectPtr[3] - rectPtr[1] + 0.5));
}

/*
 *--------------------------------------------------------------
 *
 * ScaleText --
 *
 *	This procedure is invoked to rescale a text item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Scales the position of the text, but not the size
 *	of the font for the text.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ScaleText(canvas, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas canvas;			/* Canvas containing rectangle. */
    Tk_Item *itemPtr;			/* Rectangle to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    TextItem *textPtr = (TextItem *) itemPtr;

    textPtr->x = originX + scaleX*(textPtr->x - originX);
    textPtr->y = originY + scaleY*(textPtr->y - originY);
    ComputeTextBbox(canvas, textPtr);
    return;
}

/*
 *--------------------------------------------------------------
 *
 * TranslateText --
 *
 *	This procedure is called to move a text item by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the text item is offset by (xDelta, yDelta),
 *	and the bounding box is updated in the generic part of the
 *	item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateText(canvas, itemPtr, deltaX, deltaY)
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    TextItem *textPtr = (TextItem *) itemPtr;

    textPtr->x += deltaX;
    textPtr->y += deltaY;
    ComputeTextBbox(canvas, textPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GetTextIndex --
 *
 *	Parse an index into a text item and return either its value
 *	or an error.
 *
 * Results:
 *	A standard Tcl result.  If all went well, then *indexPtr is
 *	filled in with the index (into itemPtr) corresponding to
 *	string.  Otherwise an error message is left in
 *	interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
GetTextIndex(interp, canvas, itemPtr, obj, indexPtr)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item for which the index is being
				 * specified. */
    Tcl_Obj *obj;		/* Specification of a particular character
				 * in itemPtr's text. */
    int *indexPtr;		/* Where to store converted index. */
{
    TextItem *textPtr = (TextItem *) itemPtr;
    int length;
    int c;
    TkCanvas *canvasPtr = (TkCanvas *) canvas;
    Tk_CanvasTextInfo *textInfoPtr = textPtr->textInfoPtr;
    char *string = Tcl_GetStringFromObj(obj, &length);

    c = string[0];

    if ((c == 'e') && (strncmp(string, "end", length) == 0)) {
	*indexPtr = textPtr->numChars;
    } else if ((c == 'i') && (strncmp(string, "insert", length) == 0)) {
	*indexPtr = textPtr->insertPos;
    } else if ((c == 's') && (strncmp(string, "sel.first", length) == 0)
	    && (length >= 5)) {
	if (textInfoPtr->selItemPtr != itemPtr) {
	    Tcl_AppendResult(interp, "selection isn't in item", (char *) NULL);
	    return TCL_ERROR;
	}
	*indexPtr = textInfoPtr->selectFirst;
    } else if ((c == 's') && (strncmp(string, "sel.last", length) == 0)
	    && (length >= 5)) {
	if (textInfoPtr->selItemPtr != itemPtr) {
	    Tcl_AppendResult(interp, "selection isn't in item", (char *) NULL);
	    return TCL_ERROR;
	}
	*indexPtr = textInfoPtr->selectLast;
    } else if (c == '@') {
	int x, y;
	double tmp;
	char *end, *p;

	p = string+1;
	tmp = strtod(p, &end);
	if ((end == p) || (*end != ',')) {
	    goto badIndex;
	}
	x = (int) ((tmp < 0) ? tmp - 0.5 : tmp + 0.5);
	p = end+1;
	tmp = strtod(p, &end);
	if ((end == p) || (*end != 0)) {
	    goto badIndex;
	}
	y = (int) ((tmp < 0) ? tmp - 0.5 : tmp + 0.5);
	*indexPtr = Tk_PointToChar(textPtr->textLayout,
		x + canvasPtr->scrollX1 - textPtr->leftEdge,
		y + canvasPtr->scrollY1 - textPtr->header.y1);
    } else if (Tcl_GetIntFromObj((Tcl_Interp *)NULL, obj, indexPtr) == TCL_OK) {
	if (*indexPtr < 0){
	    *indexPtr = 0;
	} else if (*indexPtr > textPtr->numChars) {
	    *indexPtr = textPtr->numChars;
	}
    } else {
	badIndex:
	Tcl_AppendResult(interp, "bad index \"", string, "\"",
		(char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SetTextCursor --
 *
 *	Set the position of the insertion cursor in this item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor position will change.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
SetTextCursor(canvas, itemPtr, index)
    Tk_Canvas canvas;			/* Record describing canvas widget. */
    Tk_Item *itemPtr;			/* Text item in which cursor position
					 * is to be set. */
    int index;				/* Index of character just before which
					 * cursor is to be positioned. */
{
    TextItem *textPtr = (TextItem *) itemPtr;

    if (index < 0) {
	textPtr->insertPos = 0;
    } else  if (index > textPtr->numChars) {
	textPtr->insertPos = textPtr->numChars;
    } else {
	textPtr->insertPos = index;
    }
}

/*
 *--------------------------------------------------------------
 *
 * GetSelText --
 *
 *	This procedure is invoked to return the selected portion
 *	of a text item.  It is only called when this item has
 *	the selection.
 *
 * Results:
 *	The return value is the number of non-NULL bytes stored
 *	at buffer.  Buffer is filled (or partially filled) with a
 *	NULL-terminated string containing part or all of the selection,
 *	as given by offset and maxBytes.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
GetSelText(canvas, itemPtr, offset, buffer, maxBytes)
    Tk_Canvas canvas;			/* Canvas containing selection. */
    Tk_Item *itemPtr;			/* Text item containing selection. */
    int offset;				/* Offset within selection of first
					 * character to be returned. */
    char *buffer;			/* Location in which to place
					 * selection. */
    int maxBytes;			/* Maximum number of bytes to place
					 * at buffer, not including terminating
					 * NULL character. */
{
    TextItem *textPtr = (TextItem *) itemPtr;
    int count;
    Tk_CanvasTextInfo *textInfoPtr = textPtr->textInfoPtr;

    count = textInfoPtr->selectLast + 1 - textInfoPtr->selectFirst - offset;
    if (textInfoPtr->selectLast == textPtr->numChars) {
	count -= 1;
    }
    if (count > maxBytes) {
	count = maxBytes;
    }
    if (count <= 0) {
	return 0;
    }
    strncpy(buffer, textPtr->text + textInfoPtr->selectFirst + offset,
	    (size_t) count);
    buffer[count] = '\0';
    return count;
}

/*
 *--------------------------------------------------------------
 *
 * TextToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	text items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
TextToPostscript(interp, canvas, itemPtr, prepass)
    Tcl_Interp *interp;			/* Leave Postscript or error message
					 * here. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    int prepass;			/* 1 means this is a prepass to
					 * collect font information;  0 means
					 * final Postscript is being created. */
{
    TextItem *textPtr = (TextItem *) itemPtr;
    int x, y;
    Tk_FontMetrics fm;
    char *justify;
    char buffer[500];
    XColor *color;
    Pixmap stipple;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    color = textPtr->color;
    stipple = textPtr->stipple;
    if (state == TK_STATE_HIDDEN || textPtr->color == NULL ||
	    textPtr->text == NULL || *textPtr->text == 0) {
	return TCL_OK;
    } else if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (textPtr->activeColor!=NULL) {
	    color = textPtr->activeColor;
	}
	if (textPtr->activeStipple!=None) {
	    stipple = textPtr->activeStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (textPtr->disabledColor!=NULL) {
	    color = textPtr->disabledColor;
	}
	if (textPtr->disabledStipple!=None) {
	    stipple = textPtr->disabledStipple;
	}
    }

    if (Tk_CanvasPsFont(interp, canvas, textPtr->tkfont) != TCL_OK) {
	return TCL_ERROR;
    }
    if (prepass != 0) {
	return TCL_OK;
    }
    if (Tk_CanvasPsColor(interp, canvas, color) != TCL_OK) {
	return TCL_ERROR;
    }
    if (stipple != None) {
	Tcl_AppendResult(interp, "/StippleText {\n    ",
		(char *) NULL);
	Tk_CanvasPsStipple(interp, canvas, stipple);
	Tcl_AppendResult(interp, "} bind def\n", (char *) NULL);
    }

    /*
    sprintf(buffer, "%.15g %.15g [\n", textPtr->x,
	    Tk_CanvasPsY(canvas, textPtr->y));
    Tcl_AppendResult(interp, buffer, (char *) NULL);
    */
    /*
     * Rotate the damned stuff
     */
    sprintf(buffer, "%.15g %.15g translate\n%.15g rotate\n0 0\n[", textPtr->x,
	    Tk_CanvasPsY(canvas, textPtr->y), textPtr->theta);
    Tcl_AppendResult(interp, buffer, (char *) NULL);


    Tk_TextLayoutToPostscript(interp, textPtr->textLayout);

    x = 0;  y = 0;  justify = NULL;	/* lint. */
    switch (textPtr->anchor) {
	case TK_ANCHOR_NW:	x = 0; y = 0;	break;
	case TK_ANCHOR_N:	x = 1; y = 0;	break;
	case TK_ANCHOR_NE:	x = 2; y = 0;	break;
	case TK_ANCHOR_E:	x = 2; y = 1;	break;
	case TK_ANCHOR_SE:	x = 2; y = 2;	break;
	case TK_ANCHOR_S:	x = 1; y = 2;	break;
	case TK_ANCHOR_SW:	x = 0; y = 2;	break;
	case TK_ANCHOR_W:	x = 0; y = 1;	break;
	case TK_ANCHOR_CENTER:	x = 1; y = 1;	break;
    }
    switch (textPtr->justify) {
        case TK_JUSTIFY_LEFT:	justify = "0";	break;
	case TK_JUSTIFY_CENTER: justify = "0.5";break;
	case TK_JUSTIFY_RIGHT:  justify = "1";	break;
    }

    Tk_GetFontMetrics(textPtr->tkfont, &fm);
    sprintf(buffer, "] %d %g %g %s %s DrawText\n",
	    fm.linespace, x / -2.0, y / 2.0, justify,
	    ((stipple == None) ? "false" : "true"));
    Tcl_AppendResult(interp, buffer, (char *) NULL);

    return TCL_OK;
}

static void
RotateAnchor(int x, int y, int width, int height, double theta, Tk_Anchor anchor, int *newX, int *newY, int *newWidth, int *newHeight)
{
    double cosTheta, sinTheta;
    double maxX, maxY, minX, minY;
    double rx, ry;
    double x0=0, x1=0, x2=0, x3=0;
    double y0=0, y1=0, y2=0, y3=0;

    /*
     * fooX, fooY is our new top-left corner.
     * Rotate the upper-left corner about this point.
     *
     * width, height are the dimensions of the rotated box.
     */

    switch(anchor) {
	case TK_ANCHOR_NW:
		x2 = x3 = width;
		y1 = y3 = -height;
		break;
	case TK_ANCHOR_W:
		x2 = x3 = width;
		y0 = y2 = height/2;
		y1 = y3 = -height/2;
		break;
	case TK_ANCHOR_SW:
		x2 = x3 = width;
		y0 = y2 = height;
		break;
	case TK_ANCHOR_N:
		x0 = x1 = -width/2;
		x2 = x3 = width/2;
		y1 = y3 = -height;
		break;
	case TK_ANCHOR_CENTER:
		x0 = x1 = -width/2;
		x2 = x3 = width/2;
		y0 = y2 = height/2;
		y1 = y3 = -height/2;
		break;
	case TK_ANCHOR_S:
		x0 = x1 = -width/2;
		x2 = x3 = width/2;
		y0 = y2 = height;
		break;
	case TK_ANCHOR_NE:
		x0 = x1 = -width;
		y1 = y3 = -height;
		break;
	case TK_ANCHOR_E:
		x0 = x1 = -width;
		y0 = y2 = height/2;
		y1 = y3 = -height/2;
		break;
	case TK_ANCHOR_SE:
		x0 = x1 = -width;
		y0 = y2 = height;
		break;
    }

    cosTheta = cos(theta / 180.0 * M_PI);
    sinTheta = sin(theta / 180.0 * M_PI);

    /*
     * top-left
     */
    rx = (x0 * cosTheta) - (y0 * sinTheta);
    ry = (x0 * sinTheta) + (y0 * cosTheta);

    maxX = rx, maxY = ry;
    minX = rx, minY = ry;

    /*
     * top-right
     */
    rx = (x2) * cosTheta - (y2) * sinTheta;
    ry = (x2) * sinTheta + (y2) * cosTheta;

    if (rx < minX) minX = rx;
    if (rx > maxX) maxX = rx;
    if (ry < minY) minY = ry;
    if (ry > maxY) maxY = ry;

    /*
     * bottom-left
     */
    rx = (x1) * cosTheta - (y1) * sinTheta;
    ry = (x1) * sinTheta + (y1) * cosTheta;

    if (rx < minX) minX = rx;
    if (rx > maxX) maxX = rx;
    if (ry < minY) minY = ry;
    if (ry > maxY) maxY = ry;

    /*
     * bottom-right
     */
    rx = (x3) * cosTheta - (y3) * sinTheta;
    ry = (x3) * sinTheta + (y3) * cosTheta;

    if (rx < minX) minX = rx;
    if (rx > maxX) maxX = rx;
    if (ry < minY) minY = ry;
    if (ry > maxY) maxY = ry;

    *newX = x+minX;
    *newY = y-maxY;

    if (newWidth)
	*newWidth = maxX - minX;
    if (newHeight)
	*newHeight = maxY - minY;
}

/*
 * -----------------------------------------------------------------
 *
 * TranslateAnchor -- (this code taken from BLT)
 *
 * 	Translate the coordinates of a given bounding box based
 *	upon the anchor specified.  The anchor indicates where
 *	the given xy position is in relation to the bounding box.
 *
 *  		nw --- n --- ne
 *  		|            |
 *  		w   center   e
 *  		|            |
 *  		sw --- s --- se
 *
 * 	The coordinates returned are translated to the origin of the
 * 	bounding box (suitable for giving to XCopyArea, etc.)
 *
 * Results:
 *	The translated coordinates of the bounding box are returned.
 *
 * -----------------------------------------------------------------
 */
void
TranslateAnchor(x, y, width, height, anchor, transXPtr, transYPtr)
    int x, y;			/* Window coordinates of anchor */
    int width, height;		/* Extents of the bounding box */
    Tk_Anchor anchor;		/* Direction of the anchor */
    int *transXPtr, *transYPtr;
{
    switch (anchor) {
    case TK_ANCHOR_NW:		/* Upper left corner */
	break;
    case TK_ANCHOR_W:		/* Left center */
	y -= (height / 2);
	break;
    case TK_ANCHOR_SW:		/* Lower left corner */
	y -= height;
	break;
    case TK_ANCHOR_N:		/* Top center */
	x -= (width / 2);
	break;
    case TK_ANCHOR_CENTER:	/* Centered */
	x -= (width / 2);
	y -= (height / 2);
	break;
    case TK_ANCHOR_S:		/* Bottom center */
	x -= (width / 2);
	y -= height;
	break;
    case TK_ANCHOR_NE:		/* Upper right corner */
	x -= width;
	break;
    case TK_ANCHOR_E:		/* Right center */
	x -= width;
	y -= (height / 2);
	break;
    case TK_ANCHOR_SE:		/* Lower right corner */
	x -= width;
	y -= height;
	break;
    }
    *transXPtr = x;
    *transYPtr = y;
}

static void
TextCreateDrawables(Tk_Window tkwin, TextItem *textPtr)
{
    Display *display;
    Drawable d;
    int width, height, depth;

    display = Tk_Display(tkwin);
    d = RootWindowOfScreen(Tk_Screen(tkwin));
    width = (int) (textPtr->header.x2 - textPtr->header.x1);
    height = (int) (textPtr->header.y2 - textPtr->header.y1);
    depth = Tk_Depth(tkwin);

    if (textPtr->rotatedDrawable != None) {
	Tk_FreePixmap(display, textPtr->rotatedDrawable);
    }
    if (textPtr->normalDrawable != None) {
	Tk_FreePixmap(display, textPtr->normalDrawable);
    }
    if (textPtr->rotatedMask != None) {
	Tk_FreePixmap(display, textPtr->rotatedMask);
    }
    if (textPtr->normalMask != None) {
	Tk_FreePixmap(display, textPtr->normalMask);
    }
    textPtr->normalDrawable = None;
    textPtr->rotatedDrawable = None;
    textPtr->normalMask = None;
    textPtr->rotatedMask = None;

    if (d != (Drawable) NULL) {
	int oldWidth, oldHeight;

	if (width <= 0) width = 1;
	if (height <= 0) height = 1;

	textPtr->rotatedWidth = width;
	textPtr->rotatedHeight = height;

	textPtr->rotatedDrawable = Tk_GetPixmap(display, d,
		textPtr->rotatedWidth, textPtr->rotatedHeight,
		depth);
	textPtr->rotatedMask = Tk_GetPixmap(display, d,
		textPtr->rotatedWidth, textPtr->rotatedHeight, 1);

	Tk_ComputeTextLayout(textPtr->tkfont, textPtr->text, textPtr->numChars,
		textPtr->width, textPtr->justify, 0, &oldWidth, &oldHeight);

	if (oldWidth <= 0) oldWidth = 1;
	if (oldHeight <= 0) oldHeight = 1;

	textPtr->normalWidth = oldWidth;
	textPtr->normalHeight = oldHeight;

	textPtr->normalDrawable = Tk_GetPixmap(display, d,
		textPtr->normalWidth, textPtr->normalHeight,
		depth);
	textPtr->normalMask = Tk_GetPixmap(display, d,
		textPtr->normalWidth, textPtr->normalHeight, 1);

	/*
	 * Clear the drawables
	 */

	TextUpdateDrawableGC(tkwin, textPtr);

	TextClearDrawables(tkwin, textPtr);
    }
}

static void
TextUpdateDrawableGC(Tk_Window tkwin, TextItem *textPtr)
{
    Display *display;
    Drawable d;
    int depth;
    XGCValues gcValues;
    GC newGC;

    display = Tk_Display(tkwin);
    d = RootWindowOfScreen(Tk_Screen(tkwin));
    depth = Tk_Depth(tkwin);

    gcValues.function = GXcopy;
    if (textPtr->color)
	gcValues.foreground = textPtr->color->pixel;
    else
	gcValues.foreground = 1;
    gcValues.background = 0;
    gcValues.fill_style = FillSolid;
    gcValues.font = Tk_FontId(textPtr->tkfont);
    gcValues.plane_mask = AllPlanes;
    gcValues.tile = None;
    gcValues.stipple = None;
    gcValues.clip_mask = None;
    gcValues.clip_x_origin = 0;
    gcValues.clip_y_origin = 0;

    /*
     * Create a GC for copying the rotated pixmap to the screen
     */

    newGC = Tk_GetGC(tkwin,
	    (unsigned long) GCFunction|GCPlaneMask|GCFillStyle|GCForeground|GCBackground, &gcValues);

    if (textPtr->drawableGC != None) {
	Tk_FreeGC(display, textPtr->drawableGC);
    }
    textPtr->drawableGC = newGC;

    /*
     * Create a GC for drawing on the masks.  Don't use Tk_GetGC
     * because we won't have control over the GC depth.
     */

    gcValues.foreground = 1;
    newGC = XCreateGC(display, textPtr->normalMask,
	    (unsigned long) GCFunction|GCPlaneMask|GCFillStyle|GCForeground|GCBackground, &gcValues);

    if (textPtr->maskGC != None) {
	XFreeGC(display, textPtr->maskGC);
    }
    textPtr->maskGC = newGC;
}


static void
TextClearDrawables(Tk_Window tkwin, TextItem *textPtr)
{
    Display *display;
    Drawable d;
    XGCValues gcValues;
    GC newGC;
    int depth;

    display = Tk_Display(tkwin);
    d = RootWindowOfScreen(Tk_Screen(tkwin));
    depth = Tk_Depth(tkwin);

    gcValues.function = GXclear;
    if (textPtr->rotatedDrawable != None) {
	newGC = XCreateGC(display, textPtr->rotatedDrawable, GCFunction,
	    &gcValues);
	XFillRectangle(display, textPtr->rotatedDrawable, newGC, 0, 0,
	    (unsigned int) textPtr->rotatedWidth,
	    (unsigned int) textPtr->rotatedHeight);
	if (textPtr->normalDrawable != None) {
	    XFillRectangle(display, textPtr->normalDrawable, newGC, 0, 0,
		(unsigned int) textPtr->normalWidth,
		(unsigned int) textPtr->normalWidth);
	}
	XFreeGC(display, newGC);
    }

    if (textPtr->rotatedMask != None) {
	newGC = XCreateGC(display, textPtr->rotatedMask, GCFunction,
	    &gcValues);
	XFillRectangle(display, textPtr->rotatedMask, newGC, 0, 0,
	    (unsigned int) textPtr->rotatedWidth,
	    (unsigned int) textPtr->rotatedHeight);
	if (textPtr->normalMask != None) {
	    XFillRectangle(display, textPtr->normalMask, newGC, 0, 0,
		(unsigned int) textPtr->normalWidth,
		(unsigned int) textPtr->normalHeight);
	}
	XFreeGC(display, newGC);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * TextRotateBitmap --
 *
 *	Creates a new bitmap containing the rotated image of the given
 *	bitmap.  We also need a special GC of depth 1, so that we do
 *	not need to rotate more than one plane of the bitmap.
 *
 * Results:
 *	Returns a new bitmap containing the rotated image.
 *
 * -----------------------------------------------------------------
 */
static void
TextRotateBitmap(display, draw, textPtr, canvas, srcBitmap,
	srcWidth, srcHeight, theta, destBitmap, destWidth, destHeight)
    Display *display;		/* X display */
    Drawable draw;		/* Root window drawable */
    TextItem *textPtr;		/* Text item containing the GC to use */
    Tk_Canvas canvas;		/* Canvas onto which the text will be drawn */
    Pixmap srcBitmap;		/* Source bitmap to be rotated */
    int srcWidth, srcHeight;	/* Width and height of the source bitmap */
    double theta;		/* Right angle rotation to perform */
    Pixmap destBitmap;		/* Destination drawable */
    int destWidth, destHeight;	/* Size of destination */
{
    XImage *src, *dest, *destMask, *srcMask;
    int dx, dy;	/* Destination bitmap coordinates */
    int sx, sy;	/* Source bitmap coordinates */

    if (srcWidth < 1 || srcHeight < 1 || destWidth < 1 || destHeight < 1)
	return;

    if (textPtr->gc == None)
	return;

    /*
    RotateAnchor((int)(textPtr->x), (int)(textPtr->y),
	    textPtr->header.x2-textPtr->header.x1,
	    textPtr->header.y2-textPtr->header.y1,
	    textPtr->theta,
	    textPtr->anchor,
	    &newdx, &newdy,
	    (int *)NULL, (int *)NULL);
    Tk_CanvasDrawableCoords(canvas, (double)textPtr->header.x1, (double)textPtr->header.y1, &destCanvasX, &destCanvasY);
    */

    src = XGetImage(display, srcBitmap, 0, 0, srcWidth, srcHeight,
	    AllPlanes, XYPixmap);
    dest = XGetImage(display, draw, 0, 0, destWidth, destHeight,
	    AllPlanes, XYPixmap);
    srcMask = XGetImage(display, textPtr->normalMask, 0, 0,
	    srcWidth, srcHeight, AllPlanes, XYPixmap);
    destMask = XGetImage(display, textPtr->rotatedMask, 0, 0,
	    destWidth, destHeight, AllPlanes, XYPixmap);

    while (theta >= 360.0) { theta -= 360.0; }
    while (theta < 0.0)    { theta += 360.0; }

    /*
     * Handle right-angle rotations specifically
     */
    if (DOUBLES_EQ(theta, 0.0)) {
	/*
	 * Special case.  Don't copy pixel by pixel.  Just
	 * change a couple of pointers.  It's much faster.
	 */

	XDestroyImage(dest);
	dest = src;
	src = (XImage *)NULL;

	XDestroyImage(destMask);
	destMask = srcMask;
	srcMask = (XImage *)NULL;
    } else if (DOUBLES_EQ(theta, 90.0)) {
	for (dx = 0; dx < destWidth; dx++) {
	    for (dy = 0; dy < destHeight; dy++) {
		sx = destHeight - dy - 1;
		sy = dx;
		XPutPixel(dest, dx, dy, XGetPixel(src, sx, sy));
		XPutPixel(destMask, dx, dy, XGetPixel(srcMask, sx, sy));
	    }
	}
    } else if (DOUBLES_EQ(theta, 180.0)) {
	for (dx = 0; dx < destWidth; dx++) {
	    for (dy = 0; dy < destHeight; dy++) {
		sx = destWidth - dx - 1;
		sy = destHeight - dy - 1;
		XPutPixel(dest, dx, dy, XGetPixel(src, sx, sy));
		XPutPixel(destMask, dx, dy, XGetPixel(srcMask, sx, sy));
	    }
	}
    } else if (DOUBLES_EQ(theta, 270.0)) {
	for (dx = 0; dx < destWidth; dx++) {
	    for (dy = 0; dy < destHeight; dy++) {
		sx = dy;
		sy = destWidth - dx - 1;
		XPutPixel(dest, dx, dy, XGetPixel(src, sx, sy));
		XPutPixel(destMask, dx, dy, XGetPixel(srcMask, sx, sy));
	    }
	}
    } else {
	double radians, sinTheta, cosTheta;
	double srcX, srcY;	/* Center of source rectangle */
	double destX, destY;	/* Center of destination rectangle */
	double transX, transY;
	double rx, ry;		/* Angle of rotation for x and y coordinates */

	radians = (theta / 180.0) * M_PI;
	sinTheta = sin(radians), cosTheta = cos(radians);

	/*
	 * Coordinates of the centers of the source and destination rectangles
	 */
	srcX = srcWidth * 0.5;
	srcY = srcHeight * 0.5;
	destX = destWidth * 0.5;
	destY = destHeight * 0.5;

	/* Rotate each pixel of dest image, placing results in source image */

	for (dx = 0; dx < destWidth; dx++) {
	    for (dy = 0; dy < destHeight; dy++) {

		/* Translate origin to center of destination image */
		transX = dx - destX;
		transY = dy - destY;

		/* Rotate the coordinates about the origin */
		rx = (transX * cosTheta) - (transY * sinTheta);
		ry = (transX * sinTheta) + (transY * cosTheta);

		/* Translate back to the center of the source image */
		rx += srcX;
		ry += srcY;

		sx = ROUND(rx);
		sy = ROUND(ry);

		/*
		 * Verify the coordinates, since the destination image can be
		 * bigger than the source
		 */

		if ((sx >= srcWidth) || (sx < 0)
			|| (sy >= srcHeight) || (sy < 0)) {
		    continue;
		}

		XPutPixel(dest, dx, dy, XGetPixel(src, sx, sy));
		XPutPixel(destMask, dx, dy, XGetPixel(srcMask, sx, sy));
	    }
	}
    }
    /* Write the rotated image into the destination bitmap */

    XPutImage(display, textPtr->rotatedMask, textPtr->maskGC, destMask,
	    0, 0, 0, 0, destWidth, destHeight);
    XPutImage(display, textPtr->rotatedDrawable, textPtr->gc, dest,
	    0, 0, 0, 0, destWidth, destHeight);

    /* Clean up temporary resources used */
    if (src) XDestroyImage(src);
    if (dest) XDestroyImage(dest);
    if (destMask) XDestroyImage(destMask);
    if (srcMask) XDestroyImage(srcMask);
}
