/* 
 * tkgsTk3d.c --
 *
 *	This is a modified tk3D.c that uses TkGS instead of X. All original
 *	names have been prefixed by "TkGS".
 *	Original header follows.
 *
 *	This module provides procedures to draw borders in
 *	the three-dimensional Motif style.
 *
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1997 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: @(#) $Id: tkgsTk3d.c,v 1.2 2001/03/08 12:30:44 fbonnet Exp $
 */

#include <math.h> /* For cos/sin et al */
#include <tkgs.h>

#include "tkgsTk.h"
#include "tkgsTk3d.h"

/* TODO: get rid of this. Needed for the current color parsing method, in order
   to get system colors properly */
#ifdef WIN32
#include <tkWinInt.h>
#endif


/*
 * The following table defines the string values for reliefs, which are
 * used by TkGSTk_GetReliefFromObj.
 */

static char *reliefStrings[] = {"flat", "groove", "raised", "ridge", "solid",
	"sunken", (char *) NULL};

/*
 * Unlike Tk borders, TkGS borders are device-independent. Thus, we don't need
 * to use per-display hash tables, but only global ones
 */

static int borderInit = 0;
static Tcl_HashTable borderTable;


/*
 * Forward declarations for procedures defined in this file:
 */

static void		BorderInit _ANSI_ARGS_(());
static void		DupBorderObjProc _ANSI_ARGS_((Tcl_Obj *srcObjPtr,
			    Tcl_Obj *dupObjPtr));
static void		FreeBorderObjProc _ANSI_ARGS_((Tcl_Obj *objPtr));
static int		Intersect _ANSI_ARGS_((TkGS_Point *a1Ptr, TkGS_Point *a2Ptr,
			    TkGS_Point *b1Ptr, TkGS_Point *b2Ptr, TkGS_Point *iPtr));
static void		InitBorderObj _ANSI_ARGS_((Tcl_Obj *objPtr));
static void		ShiftLine _ANSI_ARGS_((TkGS_Point *p1Ptr, TkGS_Point *p2Ptr,
			    int distance, TkGS_Point *p3Ptr));

/*
 * The following structure defines the implementation of the "border" Tcl
 * object, used for drawing. The border object remembers the hash table entry
 * associated with a border. The actual allocation and deallocation of the
 * border should be done by the configuration package when the border option
 * is set.
 */

static Tcl_ObjType borderObjType = {
    "tkgsBorder",		/* name */
    FreeBorderObjProc,		/* freeIntRepProc */
    DupBorderObjProc,		/* dupIntRepProc */
    NULL,			/* updateStringProc */
    NULL			/* setFromAnyProc */
};

/*
 *----------------------------------------------------------------------
 *
 * TkGSTk_Alloc3DBorderFromObj --
 *
 *	Given a Tcl_Obj *, map the value to a corresponding
 *	TkGSTk_3DBorder structure.
 *
 * Results:
 *	The return value is a token for a data structure describing a
 *	3-D border.  This token may be passed to procedures such as
 *	TkGSTk_Draw3DRectangle and TkGSTk_Free3DBorder.  If an error prevented
 *	the border from being created then NULL is returned and an error
 *	message will be left in the interp's result.
 *
 * Side effects:
 *	The border is added to an internal database with a reference
 *	count. For each call to this procedure, there should eventually
 *	be a call to FreeBorderObjProc so that the database is
 *	cleaned up when borders aren't in use anymore.
 *
 *----------------------------------------------------------------------
 */

TkGSTk_3DBorder
TkGSTk_Alloc3DBorderFromObj(interp, objPtr)
    Tcl_Interp *interp;		/* Interp for error results. */
    Tcl_Obj *objPtr;		/* Object giving name of color for window
				 * background. */
{
    TkGSTkBorder *borderPtr;

    if (objPtr->typePtr != &borderObjType) {
	InitBorderObj(objPtr);
    }
    borderPtr = (TkGSTkBorder *) objPtr->internalRep.twoPtrValue.ptr1;

    /*
     * If the object currently points to a TkGSTkBorder, increment 
     * its reference count and return.
     */

    if (borderPtr != NULL) {
	if (borderPtr->resourceRefCount == 0) {
	    /*
	     * This is a stale reference: it refers to a border that's
	     * no longer in use.  Clear the reference.
	     */

	    FreeBorderObjProc(objPtr);
	    borderPtr = NULL;
	} else {
	    borderPtr->resourceRefCount++;
	    return (TkGSTk_3DBorder) borderPtr;
	}
    }

    /*
     * Call TkGSTk_Get3DBorder to allocate a new border.
     */

    borderPtr = (TkGSTkBorder *) TkGSTk_Get3DBorder(interp,
	    Tcl_GetString(objPtr));
    objPtr->internalRep.twoPtrValue.ptr1 = (VOID *) borderPtr;
    if (borderPtr != NULL) {
	borderPtr->objRefCount++;
    }
    return (TkGSTk_3DBorder) borderPtr;
}

/*
 *--------------------------------------------------------------
 *
 * TkGSTk_Get3DBorder --
 *
 *	Create a data structure for displaying a 3-D border.
 *
 * Results:
 *	The return value is a token for a data structure describing a
 *	3-D border.  This token may be passed to procedures such as
 *	Tk_Draw3DRectangle and Tk_Free3DBorder.  If an error prevented
 *	the border from being created then NULL is returned and an error
 *	message will be left in the interp's result.
 *
 * Side effects:
 *	Data structures, graphics contexts, etc. are allocated.
 *	It is the caller's responsibility to eventually call
 *	Tk_Free3DBorder to release the resources.
 *
 *--------------------------------------------------------------
 */

TkGSTk_3DBorder
TkGSTk_Get3DBorder(interp, colorName)
    Tcl_Interp *interp;		/* Place to store an error message. */
    char *colorName;		/* String giving name of color
				 * for window background. */
{
    Tcl_HashEntry *hashPtr;
    TkGSTkBorder *borderPtr;
    int new;
    TkGS_Color bgColor;

    if (!borderInit) {
	BorderInit();
    }

    hashPtr = Tcl_CreateHashEntry(&borderTable, colorName, &new);
    if (!new) {
	borderPtr = (TkGSTkBorder *) Tcl_GetHashValue(hashPtr);
	borderPtr->resourceRefCount++;
	return (TkGSTk_3DBorder) borderPtr;
    }

    /*
     * No satisfactory border exists yet.  Initialize a new one.
     */

    /*****/


    /* For now, do a simple call to Tk_GetColor */
    /* TODO: write a real color parsing proc, with X format (eg #rrggbb)
     * as well as extended format such as "rgb(r,g,b)", "cmyk(c,m,y,k)"... */

    /******/
    {
    XColor *xcolor = Tk_GetColor(interp, Tk_MainWindow(interp), colorName);
    if (xcolor == NULL) {
	if (new) {
	    Tcl_DeleteHashEntry(hashPtr);
	}
	return NULL;
    }
#ifdef WIN32
    if (TkWinIndexOfColor(xcolor) != -1) {
	bgColor = TkGS_GetSystemColor(TkWinIndexOfColor(xcolor));
    } else {
	bgColor = TkGS_GetRGBColor(xcolor->red, xcolor->green, xcolor->blue);
    }
#else
    bgColor = TkGS_GetRGBColor(xcolor->red, xcolor->green, xcolor->blue);
#endif
    Tk_FreeColor(xcolor);
    }
    /*****/

    borderPtr = TkGSTkpGetBorder();
    borderPtr->resourceRefCount = 1;
    borderPtr->objRefCount = 0;
    borderPtr->bgColor = bgColor;
    borderPtr->darkColor = NULL;
    borderPtr->lightColor = NULL;
    borderPtr->hashPtr = hashPtr;
    Tcl_SetHashValue(hashPtr, borderPtr);

    return (TkGSTk_3DBorder) borderPtr;
}

/*
 *--------------------------------------------------------------
 *
 * TkGSTk_Draw3DRectangle --
 *
 *	Draw a 3-D border at a given place in a given window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A 3-D border will be drawn in the indicated drawable.
 *	The outside edges of the border will be determined by x,
 *	y, width, and height.  The inside edges of the border
 *	will be determined by the borderWidth argument.
 *
 *--------------------------------------------------------------
 */

void
TkGSTk_Draw3DRectangle(drawable, border, x, y, width, height,
	borderWidth, relief)
    TkGS_Drawable drawable;	/* TkGS drawable in which to draw. */
    TkGSTk_3DBorder border;	/* Token for border to draw. */
    int x, y, width, height;	/* Outside area of region in
				 * which border will be drawn. */
    int borderWidth;		/* Desired width for border, in
				 * pixels. */
    int relief;			/* Type of relief: TK_RELIEF_RAISED,
				 * TK_RELIEF_SUNKEN, TK_RELIEF_GROOVE, etc. */
{
    if (width < 2*borderWidth) {
	borderWidth = width/2;
    }
    if (height < 2*borderWidth) {
	borderWidth = height/2;
    }
    TkGSTk_3DVerticalBevel(drawable, border, x, y, borderWidth, height,
	    1, relief);
    TkGSTk_3DVerticalBevel(drawable, border, x+width-borderWidth, y,
	    borderWidth, height, 0, relief);
    TkGSTk_3DHorizontalBevel(drawable, border, x, y, width, borderWidth,
	    1, 1, 1, relief);
    TkGSTk_3DHorizontalBevel(drawable, border, x, y+height-borderWidth,
	    width, borderWidth, 0, 0, 0, relief);
}

/*
 *--------------------------------------------------------------
 *
 * TkGSTk_NameOf3DBorder --
 *
 *	Given a border, return a textual string identifying the
 *	border's color.
 *
 * Results:
 *	The return value is the string that was used to create
 *	the border.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

char *
TkGSTk_NameOf3DBorder(border)
    TkGSTk_3DBorder border;	/* Token for border. */
{
    TkGSTkBorder *borderPtr = (TkGSTkBorder *) border;

    return borderPtr->hashPtr->key.string;
}

/*
 *--------------------------------------------------------------------
 *
 * TkGSTk_3DBorderColor --
 *
 *	Given a 3D border, return the TkGS color used for the "flat"
 *	surfaces.
 *
 * Results:
 *	Returns the color used drawing flat surfaces with the border.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------
 */
TkGS_Color
TkGSTk_3DBorderColor(border)
    Tk_3DBorder border;		/* Border whose color is wanted. */
{
    return(((TkGSTkBorder *) border)->bgColor);
}

/*
 *--------------------------------------------------------------
 *
 * TkGSTk_Free3DBorder --
 *
 *	This procedure is called when a 3D border is no longer
 *	needed.  It frees the resources associated with the
 *	border.  After this call, the caller should never again
 *	use the "border" token.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources are freed.
 *
 *--------------------------------------------------------------
 */

void
TkGSTk_Free3DBorder(border)
    TkGSTk_3DBorder border;		/* Token for border to be released. */
{
    TkGSTkBorder *borderPtr = (TkGSTkBorder *) border;
    TkGSTkBorder *prevPtr;

    borderPtr->resourceRefCount--;
    if (borderPtr->resourceRefCount > 0) {
	return;
    }

    prevPtr = (TkGSTkBorder *) Tcl_GetHashValue(borderPtr->hashPtr);
    TkGSTkpFreeBorder(borderPtr);
    if (borderPtr->bgColor != NULL) {
	TkGS_FreeColor(borderPtr->bgColor);
    }
    if (borderPtr->darkColor != NULL) {
	TkGS_FreeColor(borderPtr->darkColor);
    }
    if (borderPtr->lightColor != NULL) {
	TkGS_FreeColor(borderPtr->lightColor);
    }
    Tcl_DeleteHashEntry(borderPtr->hashPtr);
    if (borderPtr->objRefCount == 0) {
	ckfree((char *) borderPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTk_Free3DBorderFromObj --
 *
 *	This procedure is called to release a border allocated by
 *	TkGSTk_Alloc3DBorderFromObj. It does not throw away the Tcl_Obj *;
 *	it only gets rid of the hash table entry for this border
 *	and clears the cached value that is normally stored in the object.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The reference count associated with the border represented by
 *	objPtr is decremented, and the border's resources are released 
 *	if there are no remaining uses for it.
 *
 *----------------------------------------------------------------------
 */

void
TkGSTk_Free3DBorderFromObj(objPtr)
    Tcl_Obj *objPtr;		/* The Tcl_Obj * to be freed. */
{
    TkGSTk_Free3DBorder(TkGSTk_Get3DBorderFromObj(objPtr));
    FreeBorderObjProc(objPtr);
}

/*
 *---------------------------------------------------------------------------
 *
 * FreeBorderObjProc -- 
 *
 *	This proc is called to release an object reference to a border.
 *	Called when the object's internal rep is released or when
 *	the cached borderPtr needs to be changed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The object reference count is decremented. When both it
 *	and the hash ref count go to zero, the border's resources
 *	are released.
 *
 *---------------------------------------------------------------------------
 */

static void
FreeBorderObjProc(objPtr)
    Tcl_Obj *objPtr;		/* The object we are releasing. */
{
    TkGSTkBorder *borderPtr = (TkGSTkBorder *) objPtr->internalRep.twoPtrValue.ptr1;

    if (borderPtr != NULL) {
	borderPtr->objRefCount--;
	if ((borderPtr->objRefCount == 0) 
		&& (borderPtr->resourceRefCount == 0)) {
	    ckfree((char *) borderPtr);
	}
	objPtr->internalRep.twoPtrValue.ptr1 = (VOID *) NULL;
    }
}

/*
 *---------------------------------------------------------------------------
 *
 * DupBorderObjProc -- 
 *
 *	When a cached border object is duplicated, this is called to
 *	update the internal reps.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The border's objRefCount is incremented and the internal rep
 *	of the copy is set to point to it.
 *
 *---------------------------------------------------------------------------
 */

static void
DupBorderObjProc(srcObjPtr, dupObjPtr)
    Tcl_Obj *srcObjPtr;		/* The object we are copying from. */
    Tcl_Obj *dupObjPtr;		/* The object we are copying to. */
{
    TkGSTkBorder *borderPtr = (TkGSTkBorder *) srcObjPtr->internalRep.twoPtrValue.ptr1;
    
    dupObjPtr->typePtr = srcObjPtr->typePtr;
    dupObjPtr->internalRep.twoPtrValue.ptr1 = (VOID *) borderPtr;

    if (borderPtr != NULL) {
	borderPtr->objRefCount++;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTk_SetBackgroundFromBorder --
 *
 *	Change the background of a window to one appropriate for a given
 *	3-D border.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Tkwin's background gets modified.
 *
 *----------------------------------------------------------------------
 */

void
TkGSTk_SetBackgroundFromBorder(tkwin, border)
    Tk_Window tkwin;		/* Window whose background is to be set. */
    Tk_3DBorder border;		/* Token for border. */
{
    register TkGSTkBorder *borderPtr = (TkGSTkBorder *) border;

    /* TODO: implement TkGSTk_SetWindowBackground */
    /*TkGSTk_SetWindowBackground(tkwin, borderPtr->bgColor);*/
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTk_GetReliefFromObj --
 *
 *	Return an integer value based on the value of the objPtr.
 *
 * Results:
 *	The return value is a standard Tcl result. If an error occurs during
 *	conversion, an error message is left in the interpreter's result
 *	unless "interp" is NULL.
 *
 * Side effects:
 *	The object gets converted by Tcl_GetIndexFromObj.
 *
 *----------------------------------------------------------------------
 */

int
TkGSTk_GetReliefFromObj(interp, objPtr, resultPtr)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tcl_Obj *objPtr;		/* The object we are trying to get the 
				 * value from. */
    int *resultPtr;		/* Where to place the answer. */
{
    return Tcl_GetIndexFromObj(interp, objPtr, reliefStrings, "relief", 0, 
	    resultPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTk_GetRelief --
 *
 *	Parse a relief description and return the corresponding
 *	relief value, or an error.
 *
 * Results:
 *	A standard Tcl return value.  If all goes well then
 *	*reliefPtr is filled in with one of the values
 *	TK_RELIEF_RAISED, TK_RELIEF_FLAT, or TK_RELIEF_SUNKEN.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TkGSTk_GetRelief(interp, name, reliefPtr)
    Tcl_Interp *interp;		/* For error messages. */
    char *name;			/* Name of a relief type. */
    int *reliefPtr;		/* Where to store converted relief. */
{
    char c;
    size_t length;

    c = name[0];
    length = strlen(name);
    if ((c == 'f') && (strncmp(name, "flat", length) == 0)) {
	*reliefPtr = TK_RELIEF_FLAT;
    } else if ((c == 'g') && (strncmp(name, "groove", length) == 0)
	    && (length >= 2)) {
        *reliefPtr = TK_RELIEF_GROOVE;
    } else if ((c == 'r') && (strncmp(name, "raised", length) == 0)
	    && (length >= 2)) {
	*reliefPtr = TK_RELIEF_RAISED;
    } else if ((c == 'r') && (strncmp(name, "ridge", length) == 0)) {
        *reliefPtr = TK_RELIEF_RIDGE;
    } else if ((c == 's') && (strncmp(name, "solid", length) == 0)) {
	*reliefPtr = TK_RELIEF_SOLID;
    } else if ((c == 's') && (strncmp(name, "sunken", length) == 0)) {
	*reliefPtr = TK_RELIEF_SUNKEN;
    } else {
	char buf[200];

	sprintf(buf, "bad relief type \"%.50s\": must be %s",
		name, "flat, groove, raised, ridge, solid, or sunken");
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * TkGSTk_NameOfRelief --
 *
 *	Given a relief value, produce a string describing that
 *	relief value.
 *
 * Results:
 *	The return value is a static string that is equivalent
 *	to relief.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

char *
TkGSTk_NameOfRelief(relief)
    int relief;		/* One of TK_RELIEF_FLAT, TK_RELIEF_RAISED,
			 * or TK_RELIEF_SUNKEN. */
{
    if (relief == TK_RELIEF_FLAT) {
	return "flat";
    } else if (relief == TK_RELIEF_SUNKEN) {
	return "sunken";
    } else if (relief == TK_RELIEF_RAISED) {
	return "raised";
    } else if (relief == TK_RELIEF_GROOVE) {
	return "groove";
    } else if (relief == TK_RELIEF_RIDGE) {
	return "ridge";
    } else if (relief == TK_RELIEF_SOLID) {
	return "solid";
    } else {
	return "unknown relief";
    }
}

/*
 *--------------------------------------------------------------
 *
 * TkGSTk_Draw3DPolygon --
 *
 *	Draw a border with 3-D appearance around the edge of a
 *	given polygon.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information is drawn in "drawable" in the form of a
 *	3-D border borderWidth units width wide on the left
 *	of the trajectory given by pointPtr and numPoints (or
 *	-borderWidth units wide on the right side, if borderWidth
 *	is negative).
 *
 *--------------------------------------------------------------
 */

void
TkGSTk_Draw3DPolygon(drawable, border, pointPtr, numPoints,
	borderWidth, leftRelief)
    TkGS_Drawable drawable;	/* TkGS drawable in which to draw. */
    TkGSTk_3DBorder border;	/* Token for border to draw. */
    TkGS_Point *pointPtr;	/* Array of points describing
				 * polygon.  All points must be
				 * absolute (CoordModeOrigin). */
    int numPoints;		/* Number of points at *pointPtr. */
    int borderWidth;		/* Width of border, measured in
				 * pixels to the left of the polygon's
				 * trajectory.   May be negative. */
    int leftRelief;		/* TK_RELIEF_RAISED or
				 * TK_RELIEF_SUNKEN: indicates how
				 * stuff to left of trajectory looks
				 * relative to stuff on right. */
{
    TkGS_Point poly[4], b1, b2, newB1, newB2;
    TkGS_Point perp, c, shift1, shift2;	/* Used for handling parallel lines. */
    register TkGS_Point *p1Ptr, *p2Ptr;
    TkGS_Color color;
    TkGSTkBorder *borderPtr = (TkGSTkBorder *) border;
    int i, lightOnLeft, dx, dy, parallel, pointsSeen;

    if (borderPtr->lightColor == NULL) {
	TkGSTkpGetShadows(borderPtr);
    }

    /*
     * Handle grooves and ridges with recursive calls.
     */

    if ((leftRelief == TK_RELIEF_GROOVE) || (leftRelief == TK_RELIEF_RIDGE)) {
	int halfWidth;

	halfWidth = borderWidth/2;
	TkGSTk_Draw3DPolygon(drawable, border, pointPtr, numPoints,
		halfWidth, (leftRelief == TK_RELIEF_GROOVE) ? TK_RELIEF_RAISED
		: TK_RELIEF_SUNKEN);
	TkGSTk_Draw3DPolygon(drawable, border, pointPtr, numPoints,
		-halfWidth, (leftRelief == TK_RELIEF_GROOVE) ? TK_RELIEF_SUNKEN
		: TK_RELIEF_RAISED);
	return;
    }

    /*
     * If the polygon is already closed, drop the last point from it
     * (we'll close it automatically).
     */

    p1Ptr = &pointPtr[numPoints-1];
    p2Ptr = &pointPtr[0];
    if ((p1Ptr->x == p2Ptr->x) && (p1Ptr->y == p2Ptr->y)) {
	numPoints--;
    }

    /*
     * The loop below is executed once for each vertex in the polgon.
     * At the beginning of each iteration things look like this:
     *
     *          poly[1]       /
     *             *        /
     *             |      /
     *             b1   * poly[0] (pointPtr[i-1])
     *             |    |
     *             |    |
     *             |    |
     *             |    |
     *             |    |
     *             |    | *p1Ptr            *p2Ptr
     *             b2   *--------------------*
     *             |
     *             |
     *             x-------------------------
     *
     * The job of this iteration is to do the following:
     * (a) Compute x (the border corner corresponding to
     *     pointPtr[i]) and put it in poly[2].  As part of
     *	   this, compute a new b1 and b2 value for the next
     *	   side of the polygon.
     * (b) Put pointPtr[i] into poly[3].
     * (c) Draw the polygon given by poly[0..3].
     * (d) Advance poly[0], poly[1], b1, and b2 for the
     *     next side of the polygon.
     */

    /*
     * The above situation doesn't first come into existence until
     * two points have been processed;  the first two points are
     * used to "prime the pump", so some parts of the processing
     * are ommitted for these points.  The variable "pointsSeen"
     * keeps track of the priming process;  it has to be separate
     * from i in order to be able to ignore duplicate points in the
     * polygon.
     */

    pointsSeen = 0;
    for (i = -2, p1Ptr = &pointPtr[numPoints-2], p2Ptr = p1Ptr+1;
	    i < numPoints; i++, p1Ptr = p2Ptr, p2Ptr++) {
	if ((i == -1) || (i == numPoints-1)) {
	    p2Ptr = pointPtr;
	}
	if ((p2Ptr->x == p1Ptr->x) && (p2Ptr->y == p1Ptr->y)) {
	    /*
	     * Ignore duplicate points (they'd cause core dumps in
	     * ShiftLine calls below).
	     */
	    continue;
	}
	ShiftLine(p1Ptr, p2Ptr, borderWidth, &newB1);
	newB2.x = newB1.x + (p2Ptr->x - p1Ptr->x);
	newB2.y = newB1.y + (p2Ptr->y - p1Ptr->y);
	poly[3] = *p1Ptr;
	parallel = 0;
	if (pointsSeen >= 1) {
	    parallel = Intersect(&newB1, &newB2, &b1, &b2, &poly[2]);

	    /*
	     * If two consecutive segments of the polygon are parallel,
	     * then things get more complex.  Consider the following
	     * diagram:
	     *
	     * poly[1]
	     *    *----b1-----------b2------a
	     *                                \
	     *                                  \
	     *         *---------*----------*    b
	     *        poly[0]  *p2Ptr   *p1Ptr  /
	     *                                /
	     *              --*--------*----c
	     *              newB1    newB2
	     *
	     * Instead of using x and *p1Ptr for poly[2] and poly[3], as
	     * in the original diagram, use a and b as above.  Then instead
	     * of using x and *p1Ptr for the new poly[0] and poly[1], use
	     * b and c as above.
	     *
	     * Do the computation in three stages:
	     * 1. Compute a point "perp" such that the line p1Ptr-perp
	     *    is perpendicular to p1Ptr-p2Ptr.
	     * 2. Compute the points a and c by intersecting the lines
	     *    b1-b2 and newB1-newB2 with p1Ptr-perp.
	     * 3. Compute b by shifting p1Ptr-perp to the right and
	     *    intersecting it with p1Ptr-p2Ptr.
	     */

	    if (parallel) {
		perp.x = p1Ptr->x + (p2Ptr->y - p1Ptr->y);
		perp.y = p1Ptr->y - (p2Ptr->x - p1Ptr->x);
		(void) Intersect(p1Ptr, &perp, &b1, &b2, &poly[2]);
		(void) Intersect(p1Ptr, &perp, &newB1, &newB2, &c);
		ShiftLine(p1Ptr, &perp, borderWidth, &shift1);
		shift2.x = shift1.x + (perp.x - p1Ptr->x);
		shift2.y = shift1.y + (perp.y - p1Ptr->y);
		(void) Intersect(p1Ptr, p2Ptr, &shift1, &shift2, &poly[3]);
	    }
	}
	if (pointsSeen >= 2) {
	    dx = poly[3].x - poly[0].x;
	    dy = poly[3].y - poly[0].y;
	    if (dx > 0) {
		lightOnLeft = (dy <= dx);
	    } else {
		lightOnLeft = (dy < dx);
	    }
	    if (lightOnLeft ^ (leftRelief == TK_RELIEF_RAISED)) {
		color = borderPtr->lightColor;
	    } else {
		color = borderPtr->darkColor;
	    }
	    TkGS_SetDrawableForeground(drawable, color);
	    /* TODO: implement TkGS_DrawPolygon */
	    /*TkGS_DrawPolygon(drawable, 1, poly, 4);*/
	}
	b1.x = newB1.x;
	b1.y = newB1.y;
	b2.x = newB2.x;
	b2.y = newB2.y;
	poly[0].x = poly[3].x;
	poly[0].y = poly[3].y;
	if (parallel) {
	    poly[1].x = c.x;
	    poly[1].y = c.y;
	} else if (pointsSeen >= 1) {
	    poly[1].x = poly[2].x;
	    poly[1].y = poly[2].y;
	}
	pointsSeen++;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTk_Fill3DRectangle --
 *
 *	Fill a rectangular area, supplying a 3D border if desired.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets drawn on the screen.
 *
 *----------------------------------------------------------------------
 */

void
TkGSTk_Fill3DRectangle(drawable, border, x, y, width,
	height, borderWidth, relief)
    TkGS_Drawable drawable;	/* TkGS drawable in which to draw. */
    TkGSTk_3DBorder border;	/* Token for border to draw. */
    int x, y, width, height;	/* Outside area of rectangular region. */
    int borderWidth;		/* Desired width for border, in
				 * pixels. Border will be *inside* region. */
    int relief;			/* Indicates 3D effect: TK_RELIEF_FLAT,
				 * TK_RELIEF_RAISED, or TK_RELIEF_SUNKEN. */
{
    register TkGSTkBorder *borderPtr = (TkGSTkBorder *) border;
    int doubleBorder;

    /*
     * This code is slightly tricky because it only draws the background
     * in areas not covered by the 3D border. This avoids flashing
     * effects on the screen for the border region.
     */
  
    if (relief == TK_RELIEF_FLAT) {
	borderWidth = 0;
    } else {
	/*
	 * We need to make this extra check, otherwise we will leave
	 * garbage in thin frames [Bug: 3596]
	 */
	if (width < 2*borderWidth) {
	    borderWidth = width/2;
	}
	if (height < 2*borderWidth) {
	    borderWidth = height/2;
	}
    }
    doubleBorder = 2*borderWidth;

    if ((width > doubleBorder) && (height > doubleBorder)) {
	TkGS_SetDrawableForeground(drawable, borderPtr->bgColor);
	TkGS_DrawRectangle(drawable, 1,
		x + borderWidth, y + borderWidth,
		(unsigned int) (width - doubleBorder),
		(unsigned int) (height - doubleBorder));
    }
    if (borderWidth) {
	TkGSTk_Draw3DRectangle(drawable, border, x, y, width,
		height, borderWidth, relief);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTk_Fill3DPolygon --
 *
 *	Fill a polygonal area, supplying a 3D border if desired.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets drawn on the screen.
 *
 *----------------------------------------------------------------------
 */

void
TkGSTk_Fill3DPolygon(drawable, border, pointPtr, numPoints,
	borderWidth, leftRelief)
    TkGS_Drawable drawable;	/* TkGS drawable in which to draw. */
    TkGSTk_3DBorder border;	/* Token for border to draw. */
    TkGS_Point *pointPtr;	/* Array of points describing
				 * polygon.  All points must be
				 * absolute (CoordModeOrigin). */
    int numPoints;		/* Number of points at *pointPtr. */
    int borderWidth;		/* Width of border, measured in
				 * pixels to the left of the polygon's
				 * trajectory.   May be negative. */
    int leftRelief;			/* Indicates 3D effect of left side of
				 * trajectory relative to right:
				 * TK_RELIEF_FLAT, TK_RELIEF_RAISED,
				 * or TK_RELIEF_SUNKEN. */
{
    register TkGSTkBorder *borderPtr = (TkGSTkBorder *) border;

    TkGS_SetDrawableForeground(drawable, borderPtr->bgColor);
    /* TODO: implement TkGS_DrawPolygon */
    /*TkGS_DrawPolygon(drawable, 1, pointPtr, numPoints);*/
    if (leftRelief != TK_RELIEF_FLAT) {
	TkGSTk_Draw3DPolygon(drawable, border, pointPtr, numPoints,
		borderWidth, leftRelief);
    }
}

/*
 *--------------------------------------------------------------
 *
 * BorderInit --
 *
 *	Initialize the structures used for border management.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Read the code.
 *
 *-------------------------------------------------------------
 */

static void
BorderInit()
{
    borderInit = 1;
    Tcl_InitHashTable(&borderTable, TCL_STRING_KEYS);
}

/*
 *--------------------------------------------------------------
 *
 * ShiftLine --
 *
 *	Given two points on a line, compute a point on a
 *	new line that is parallel to the given line and
 *	a given distance away from it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
ShiftLine(p1Ptr, p2Ptr, distance, p3Ptr)
    TkGS_Point *p1Ptr;		/* First point on line. */
    TkGS_Point *p2Ptr;		/* Second point on line. */
    int distance;		/* New line is to be this many
				 * units to the left of original
				 * line, when looking from p1 to
				 * p2.  May be negative. */
    TkGS_Point *p3Ptr;		/* Store coords of point on new
				 * line here. */
{
    int dx, dy, dxNeg, dyNeg;

    /*
     * The table below is used for a quick approximation in
     * computing the new point.  An index into the table
     * is 128 times the slope of the original line (the slope
     * must always be between 0 and 1).  The value of the table
     * entry is 128 times the amount to displace the new line
     * in y for each unit of perpendicular distance.  In other
     * words, the table maps from the tangent of an angle to
     * the inverse of its cosine.  If the slope of the original
     * line is greater than 1, then the displacement is done in
     * x rather than in y.
     */

    static int shiftTable[129];

    /*
     * Initialize the table if this is the first time it is
     * used.
     */

    if (shiftTable[0] == 0) {
	int i;
	double tangent, cosine;

	for (i = 0; i <= 128; i++) {
	    tangent = i/128.0;
	    cosine = 128/cos(atan(tangent)) + .5;
	    shiftTable[i] = (int) cosine;
	}
    }

    *p3Ptr = *p1Ptr;
    dx = p2Ptr->x - p1Ptr->x;
    dy = p2Ptr->y - p1Ptr->y;
    if (dy < 0) {
	dyNeg = 1;
	dy = -dy;
    } else {
	dyNeg = 0;
    }
    if (dx < 0) {
	dxNeg = 1;
	dx = -dx;
    } else {
	dxNeg = 0;
    }
    if (dy <= dx) {
	dy = ((distance * shiftTable[(dy<<7)/dx]) + 64) >> 7;
	if (!dxNeg) {
	    dy = -dy;
	}
	p3Ptr->y += dy;
    } else {
	dx = ((distance * shiftTable[(dx<<7)/dy]) + 64) >> 7;
	if (dyNeg) {
	    dx = -dx;
	}
	p3Ptr->x += dx;
    }
}

/*
 *--------------------------------------------------------------
 *
 * Intersect --
 *
 *	Find the intersection point between two lines.
 *
 * Results:
 *	Under normal conditions 0 is returned and the point
 *	at *iPtr is filled in with the intersection between
 *	the two lines.  If the two lines are parallel, then
 *	-1 is returned and *iPtr isn't modified.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Intersect(a1Ptr, a2Ptr, b1Ptr, b2Ptr, iPtr)
    TkGS_Point *a1Ptr;		/* First point of first line. */
    TkGS_Point *a2Ptr;		/* Second point of first line. */
    TkGS_Point *b1Ptr;		/* First point of second line. */
    TkGS_Point *b2Ptr;		/* Second point of second line. */
    TkGS_Point *iPtr;		/* Filled in with intersection point. */
{
    int dxadyb, dxbdya, dxadxb, dyadyb, p, q;

    /*
     * The code below is just a straightforward manipulation of two
     * equations of the form y = (x-x1)*(y2-y1)/(x2-x1) + y1 to solve
     * for the x-coordinate of intersection, then the y-coordinate.
     */

    dxadyb = (a2Ptr->x - a1Ptr->x)*(b2Ptr->y - b1Ptr->y);
    dxbdya = (b2Ptr->x - b1Ptr->x)*(a2Ptr->y - a1Ptr->y);
    dxadxb = (a2Ptr->x - a1Ptr->x)*(b2Ptr->x - b1Ptr->x);
    dyadyb = (a2Ptr->y - a1Ptr->y)*(b2Ptr->y - b1Ptr->y);

    if (dxadyb == dxbdya) {
	return -1;
    }
    p = (a1Ptr->x*dxbdya - b1Ptr->x*dxadyb + (b1Ptr->y - a1Ptr->y)*dxadxb);
    q = dxbdya - dxadyb;
    if (q < 0) {
	p = -p;
	q = -q;
    }
    if (p < 0) {
	iPtr->x = - ((-p + q/2)/q);
    } else {
	iPtr->x = (p + q/2)/q;
    }
    p = (a1Ptr->y*dxadyb - b1Ptr->y*dxbdya + (b1Ptr->x - a1Ptr->x)*dyadyb);
    q = dxadyb - dxbdya;
    if (q < 0) {
	p = -p;
	q = -q;
    }
    if (p < 0) {
	iPtr->y = - ((-p + q/2)/q);
    } else {
	iPtr->y = (p + q/2)/q;
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTk_Get3DBorderFromObj --
 *
 *	Returns the border referred to by a Tcl object.  The border must
 *	already have been allocated via a call to TkGSTk_Alloc3DBorderFromObj 
 *	or TkGSTk_Get3DBorder.
 *
 * Results:
 *	Returns the TkGSTk_3DBorder that matches the string rep
 *	of the name of the border given in objPtr.
 *
 * Side effects:
 *	If the object is not already a border, the conversion will free
 *	any old internal representation. 
 *
 *----------------------------------------------------------------------
 */

TkGSTk_3DBorder
TkGSTk_Get3DBorderFromObj(objPtr)
    Tcl_Obj *objPtr;		/* The object whose string value selects
				 * a border. */
{
    TkGSTkBorder *borderPtr = NULL;
    Tcl_HashEntry *hashPtr;

    if (objPtr->typePtr != &borderObjType) {
	InitBorderObj(objPtr);
    }

    /*
     * Check if the TkGSTkBorder structure is cached in the internal
     * representation of the Tcl_Obj.
     */

    borderPtr = (TkGSTkBorder *) objPtr->internalRep.twoPtrValue.ptr1;
    if ((borderPtr != NULL)
	    && (borderPtr->resourceRefCount > 0)) {
	/*
	 * The object already points to the right border structure.
	 * Just return it.
	 */
	return (TkGSTk_3DBorder) borderPtr;
    }

    /*
     * There was no cached TkGSTkBorder in the Tcl_Obj
     */

    hashPtr = Tcl_FindHashEntry(&borderTable, Tcl_GetString(objPtr));
    if (hashPtr == NULL) {
	goto error;
    }
    borderPtr = (TkGSTkBorder *) Tcl_GetHashValue(hashPtr);
    if (borderPtr != NULL) {
	FreeBorderObjProc(objPtr);
	objPtr->internalRep.twoPtrValue.ptr1 = (VOID *) borderPtr;
	borderPtr->objRefCount++;
	return (TkGSTk_3DBorder) borderPtr;
    }

    error:
    panic("Tk_Get3DBorderFromObj called with non-existent border!");
    /*
     * The following code isn't reached; it's just there to please compilers.
     */
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * InitBorderObj --
 *
 *	Attempt to generate a border internal form for the Tcl object
 *	"objPtr".
 *
 * Results:
 *	The return value is a standard Tcl result. If an error occurs during
 *	conversion, an error message is left in the interpreter's result
 *	unless "interp" is NULL.
 *
 * Side effects:
 *	If no error occurs, a blank internal format for a border value
 *	is intialized.
 *
 *----------------------------------------------------------------------
 */

static void
InitBorderObj(objPtr)
    Tcl_Obj *objPtr;		/* The object to convert. */
{
    Tcl_ObjType *typePtr;

    /*
     * Free the old internalRep before setting the new one. 
     */

    Tcl_GetString(objPtr);
    typePtr = objPtr->typePtr;
    if ((typePtr != NULL) && (typePtr->freeIntRepProc != NULL)) {
	(*typePtr->freeIntRepProc)(objPtr);
    }
    objPtr->typePtr = &borderObjType;
    objPtr->internalRep.twoPtrValue.ptr1 = (VOID *) NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTkDebugBorder --
 *
 *	This procedure returns debugging information about a border.
 *
 * Results:
 *	The return value is a list with one sublist for each TkBorder
 *	corresponding to "name".  Each sublist has two elements that
 *	contain the resourceRefCount and objRefCount fields from the
 *	TkBorder structure.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj *
TkGSTkDebugBorder(name)
    char *name;			/* Name of the desired color. */
{
    TkGSTkBorder *borderPtr;
    Tcl_HashEntry *hashPtr;
    Tcl_Obj *resultPtr, *objPtr;
    
    resultPtr = Tcl_NewObj();
    hashPtr = Tcl_FindHashEntry(&borderTable, name);
    if (hashPtr != NULL) {
	borderPtr = (TkGSTkBorder *) Tcl_GetHashValue(hashPtr);
	if (borderPtr == NULL) {
	    panic("TkGSTkDebugBorder found empty hash table entry");
	}
	objPtr = Tcl_NewObj();
	Tcl_ListObjAppendElement(NULL, objPtr,
		Tcl_NewIntObj(borderPtr->resourceRefCount));
	Tcl_ListObjAppendElement(NULL, objPtr,
		Tcl_NewIntObj(borderPtr->objRefCount)); 
	Tcl_ListObjAppendElement(NULL, resultPtr, objPtr);
    }
    return resultPtr;
}
