# HList.tcl --
#
#	This file defines the default bindings for Tix Hierarchical Listbox
#	widgets.
#
# Copyright (c) 1993-1999 Ioi Kim Lam.
# Copyright (c) 2000-2001 Tix Project Group.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# $Id: HList.tcl,v 1.5 2001/04/04 04:39:30 ioilam Exp $


#--------------------------------------------------------------------------
# tkPriv elements used in this file:
#
# afterId -		Token returned by "after" for autoscanning.
# fakeRelease -		Cancel the ButtonRelease-1 after the user double click
#--------------------------------------------------------------------------
#
proc tixHListBind {} {
    tixBind TixHList <ButtonPress-1> {
	tixHList:Button-1 %W %x %y ""
    }
    tixBind TixHList <Shift-ButtonPress-1> {
	tixHList:Button-1 %W %x %y s
    }
    tixBind TixHList <Control-ButtonPress-1> {
	tixHList:Button-1 %W %x %y c
    }
    tixBind TixHList <ButtonRelease-1> {
	tixHList:ButtonRelease-1 %W %x %y
    }
    tixBind TixHList <Double-ButtonPress-1> {
	tixHList:Double-1 %W  %x %y
    }
    tixBind TixHList <B1-Motion> {
	set tkPriv(x) %x 
	set tkPriv(y) %y
	set tkPriv(X) %X
	set tkPriv(Y) %Y

	tixHList:B1-Motion %W %x %y
    }
    tixBind TixHList <B1-Leave> {
	set tkPriv(x) %x 
	set tkPriv(y) %y
	set tkPriv(X) %X
	set tkPriv(Y) %Y

	tixHList:B1-Leave %W
    }
    tixBind TixHList <B1-Enter> {
	tixHList:B1-Enter %W %x %y
    }

    # Keyboard bindings
    #
    tixBind TixHList <Up> {
	tixHList:UpDown %W prev ""
    }
    tixBind TixHList <Down> {
	tixHList:UpDown %W next ""
    }
    tixBind TixHList <Shift-Up> {
	tixHList:UpDown %W prev s
    }
    tixBind TixHList <Shift-Down> {
	tixHList:UpDown %W next s
    }
    tixBind TixHList <Left> {
	tixHList:LeftRight %W left
    }
    tixBind TixHList <Right> {
	tixHList:LeftRight %W right
    }
    tixBind TixHList <Prior> {
	%W yview scroll -1 pages
    }
    tixBind TixHList <Next> {
	%W yview scroll 1 pages
    }
    tixBind TixHList <Return> {
	tixHList:Keyboard-Activate %W 
    }
    tixBind TixHList <space> {
	tixHList:Keyboard-Browse %W 
    }
    #
    # Don't use tixBind because %A causes Tk 8.3.2 to crash
    #
    bind TixHList <MouseWheel> {
        %W yview scroll [expr {- (%D / 120) * 2}] units
    }
}

#----------------------------------------------------------------------
#
#
#			 Key bindings
#
#
#----------------------------------------------------------------------
proc tixHList:Keyboard-Activate {w} {
    if {[tixHList:GetState $w] != 0} {
	return
    }
    set ent [$w info anchor]

    if {$ent == ""} {
	return
    }

    if {[$w cget -selectmode] == "single"} {
	$w select clear
	$w select set $ent
    }

    set command [$w cget -command]
    if {$command != ""} {
	set bind(specs) {%V}
	set bind(%V)    $ent

	tixEvalCmdBinding $w $command bind $ent
    }
}

proc tixHList:Keyboard-Browse {w} {
    if {[tixHList:GetState $w] != 0} {
	return
    }
    set ent [$w info anchor]

    if {$ent == ""} {
	return
    }

    if {[$w cget -selectmode] == "single"} {
	$w select clear
	$w select set $ent
    }

    tixHList:Browse $w $ent
}

proc tixHList:LeftRight {w spec} {
    catch {
	uplevel #0 unset $w:priv:shiftanchor
    }
    if {[tixHList:GetState $w] != 0} {
	return
    }

    set anchor [$w info anchor]
    if {$anchor == ""} {
	set anchor [lindex [$w info children] 0]
    }
    if {$anchor == ""} {
	return
    }

    set ent $anchor
    while {1} {
	set e $ent
	if {$spec == "left"} {
	    set ent [$w info parent $e]

	    if {$ent == "" || [$w entrycget $ent -state] == "disabled"} {
		set ent [$w info prev $e]
	    }
	} else {
	    set ent [lindex [$w info children $e] 0]

	    if {$ent == "" || [$w entrycget $ent -state] == "disabled"} {
		set ent [$w info next $e]
	    }
	}

	if {$ent == ""} {
	    break
	}
 	if {[$w entrycget $ent -state] == "disabled"} {
	    continue
	}
 	if {[$w info hidden $ent]} {
	    continue
	}
	break
    }

    if {$ent == ""} {
       return
    }

    $w anchor set $ent
    $w see $ent

    if {[$w cget -selectmode] != "single"} {
	$w select clear
	$w selection set $ent

	tixHList:Browse $w $ent
    }
}

proc tixHList:UpDown {w spec mod} {
    if {[tixHList:GetState $w] != 0} {
	return
    }
    set anchor [$w info anchor]
    set done 0

    if {$anchor == ""} {
	set anchor [lindex [$w info children] 0]

	if {$anchor == ""} {
	    return
	}

	if {[$w entrycget $anchor -state] != "disabled"} {
	    # That's a good anchor
	    set done 1
	} else {
	    # We search for the first non-disabled entry (downward)
	    set spec next
	}
    }

    set ent $anchor
    # Find the prev/next non-disabled entry
    #
    while {!$done} {
	set ent [$w info $spec $ent]
	if {$ent == ""} {
	    break
	}
	if {[$w entrycget $ent -state] == "disabled"} {
	    continue
	}
	if {[$w info hidden $ent]} {
	    continue
	}
	break
    }

    if {$ent == ""} {
	return
    } else {
	$w see $ent
	$w anchor set $ent

	set selMode [$w cget -selectmode]
       if {$mod == "s" && ($selMode == "extended" || $selMode == "multiple")} {
	    global $w:priv:shiftanchor

	    if {![info exists $w:priv:shiftanchor]} {
		set $w:priv:shiftanchor $anchor
	    }

	    $w selection clear
	    $w selection set $ent [set $w:priv:shiftanchor]
    
	    tixHList:Browse $w $ent
	} else {
	    catch {
		uplevel #0 unset $w:priv:shiftanchor
	    }

	    if {[$w cget -selectmode] != "single"} {
		$w select clear
		$w selection set $ent

		tixHList:Browse $w $ent
	    }
	}
    }
}

#----------------------------------------------------------------------
#
#
#			 Mouse bindings
#
#
#----------------------------------------------------------------------

proc tixHList:Button-1 {w x y mod} {
#    if {[$w cget -state] == "disabled"} {
#	return
#    }

    if {[$w cget -takefocus]} {
	focus $w
    }

    set selMode [$w cget -selectmode]

    case [tixHList:GetState $w] {
	{0} {
	    if {$mod == "s" && $selMode == "multiple"} {
		tixHList:GoState 28 $w $x $y
		return
	    }
	    if {$mod == "s" && $selMode == "extended"} {
		tixHList:GoState 28 $w $x $y
		return
	    }
	    if {$mod == "c" && $selMode == "extended"} {
		tixHList:GoState 33 $w $x $y
		return
	    }

	    tixHList:GoState 1 $w $x $y
	}
    }
}

proc tixHList:ButtonRelease-1 {w x y} {
    case [tixHList:GetState $w] {
	{5 16} {
	    tixHList:GoState 6 $w $x $y
	}
	{15} {
	    tixHList:GoState 17 $w $x $y
	}
	{10 11} {
	    tixHList:GoState 18 $w
	}
	{13 20} {
	    tixHList:GoState 14 $w $x $y
	}
	{21} {
	    tixHList:GoState 22 $w
	}
	{24} {
	    tixHList:GoState 25 $w
	}
	{26 28 33} {
	    tixHList:GoState 27 $w
	}
	{30} {
	    tixHList:GoState 32 $w
	}
    }
}

proc tixHList:Double-1 {w x y} {
    case [tixHList:GetState $w] {
	{0} {
	    tixHList:GoState 23 $w $x $y
	}
    }
}

proc tixHList:B1-Motion {w x y} {
    case [tixHList:GetState $w] {
	{1} {
	    tixHList:GoState 5 $w $x $y 
	}
	{5 16} {
	    tixHList:GoState 5 $w $x $y 
	}
	{13 20 21} {
	    tixHList:GoState 20 $w $x $y 
	}
	{24 26 28} {
	    tixHList:GoState 26 $w $x $y
	}
    }
}

proc tixHList:B1-Leave {w} {
    case [tixHList:GetState $w] {
	{5} {
	    tixHList:GoState 10 $w
	}
	{26} {
	    tixHList:GoState 29 $w
	}
    }
}

proc tixHList:B1-Enter {w x y} {
    case [tixHList:GetState $w] {
	{10 11} {
	    tixHList:GoState 12 $w $x $y
	}
	{29 30} {
	    tixHList:GoState 31 $w $x $y
	}
    }
}

proc tixHList:AutoScan {w} {
    case [tixHList:GetState $w] {
	{10 11} {
	    tixHList:GoState 11 $w
	}
	{29 30} {
	    tixHList:GoState 30 $w
	}
    }
}

#----------------------------------------------------------------------
#
#			STATE MANIPULATION
#
#
#----------------------------------------------------------------------
proc tixHList:GetState {w} {
    global $w:priv:state

    if {![info exists $w:priv:state]} {
	set $w:priv:state 0
    }
    return [set $w:priv:state]
}

proc tixHList:SetState {w n} {
    global $w:priv:state

    set $w:priv:state $n
}

proc tixHList:GoState {n w args} {

#   puts "going from [tixHList:GetState $w] --> $n"

    tixHList:SetState $w $n
    eval tixHList:GoState-$n $w $args
}

#----------------------------------------------------------------------
#			States
#----------------------------------------------------------------------
proc tixHList:GoState-0 {w} {

}
proc tixHList:GoState-1 {w x y} {
    set oldEnt [$w info anchor]
    set ent [tixHList:SetAnchor $w $x $y 1]

    if {$ent == ""} {
	tixHList:GoState 0 $w
	return
    }

    set info [$w info item $x $y]
    if {[lindex $info 1] == "indicator"} {
	tixHList:GoState 13 $w $ent $oldEnt
    } else {
	if {[$w entrycget $ent -state] == "disabled"} {
	    tixHList:GoState 0 $w
	} else {
	    case [$w cget -selectmode] {
		{single browse} {
		    tixHList:GoState 16 $w $ent
		}
		default {
		    tixHList:GoState 24 $w $ent
		}
	    }
	}
    }
}

proc tixHList:GoState-5 {w x y} {
    set oldEnt [$w info anchor]

    set ent [tixHList:SetAnchor $w $x $y]

    if {$ent == "" || $oldEnt == $ent} {
	return
    }

    if {[$w cget -selectmode] != "single"} {
	tixHList:Select $w $ent
	tixHList:Browse $w $ent
    }
}

proc tixHList:GoState-6 {w x y} {
    set ent [tixHList:SetAnchor $w $x $y]

    if {$ent == ""} {
	tixHList:GoState 0 $w
	return
    }
    tixHList:Select $w $ent
    tixHList:Browse $w $ent

    tixHList:GoState 0 $w
}

# start auto-scanning inside a single/browse widget
proc tixHList:GoState-10 {w} {
    tixHList:StartScan $w
}

# repeat auto-scanning inside a single/browse widget
proc tixHList:GoState-11 {w} {
    global tkPriv

    tixHList:DoScan $w

    set oldEnt [$w info anchor]
    set ent [tixHList:SetAnchor $w $tkPriv(x) $tkPriv(y) 0 1]

    if {$ent == "" || $oldEnt == $ent} {
	return
    }

    if {[$w cget -selectmode] != "single"} {
	tixHList:Select $w $ent
	tixHList:Browse $w $ent
    }
}

proc tixHList:GoState-12 {w x y} {
    tkCancelRepeat
    tixHList:GoState 5 $w $x $y 
}

proc tixHList:GoState-13 {w ent oldEnt} {
    global tkPriv
    set tkPriv(tix,indicator) $ent
    set tkPriv(tix,oldEnt)    $oldEnt
    tixHList:CallIndicatorCmd $w <Arm> $ent
}

proc tixHList:GoState-14 {w x y} {
    global tkPriv

    if {[tixHList:InsideArmedIndicator $w $x $y]} {
	$w anchor set $tkPriv(tix,indicator)
	$w select clear
	$w select set $tkPriv(tix,indicator)
	tixHList:CallIndicatorCmd $w <Activate> $tkPriv(tix,indicator)
    } else {
	tixHList:CallIndicatorCmd $w <Disarm>   $tkPriv(tix,indicator)
    }

    unset tkPriv(tix,indicator)
    tixHList:GoState 0 $w
}

proc tixHList:GoState-16 {w ent} {
    if {$ent != "" && [$w cget -selectmode] != "single"} {
	tixHList:Select $w $ent
	tixHList:Browse $w $ent
    }
}

proc tixHList:GoState-18 {w} {
    global tkPriv
    tkCancelRepeat
    tixHList:GoState 6 $w $tkPriv(x) $tkPriv(y) 
}

proc tixHList:GoState-20 {w x y} {
    global tkPriv

    if {![tixHList:InsideArmedIndicator $w $x $y]} {
	tixHList:GoState 21 $w $x $y
    } else {
	tixHList:CallIndicatorCmd $w <Arm> $tkPriv(tix,indicator)
    }
}

proc tixHList:GoState-21 {w x y} {
    global tkPriv

    if {[tixHList:InsideArmedIndicator $w $x $y]} {
	tixHList:GoState 20 $w $x $y
    } else {
	tixHList:CallIndicatorCmd $w <Disarm> $tkPriv(tix,indicator)
    }
}

proc tixHList:GoState-22 {w} {
    global tkPriv

    if {$tkPriv(tix,oldEnt) != ""} {
	$w anchor set $tkPriv(tix,oldEnt)
    } else {
	$w anchor clear
    }
    tixHList:GoState 0 $w
}

proc tixHList:GoState-23 {w x y} {
    set ent [tixHList:GetNearest $w $y]

    if {$ent != ""} {
	set info [$w info item $x $y]

	if {[lindex $info 1] == "indicator"} {
	    tixHList:CallIndicatorCmd $w <Activate> $ent
	} else {
	    $w select set $ent
	    set command [$w cget -command]
	    if {$command != ""} {
		set bind(specs) {%V}
		set bind(%V)    $ent

		tixEvalCmdBinding $w $command bind $ent
	    }
	}
    }
    tixHList:GoState 0 $w
}

proc tixHList:GoState-24 {w ent} {
    if {$ent != ""} {
	tixHList:Select $w $ent
	tixHList:Browse $w $ent
    }
}

proc tixHList:GoState-25 {w} {
    set ent [$w info anchor]

    if {$ent != ""} {
	tixHList:Select $w $ent
	tixHList:Browse $w $ent
    }

    tixHList:GoState 0 $w
}


proc tixHList:GoState-26 {w x y} {
    set anchor [$w info anchor]
    if {$anchor == ""} {
	set first [lindex [$w info children ""] 0]
	if {$first != ""} {
	    $w anchor set $first
	    set anchor $first
	} else {
	    return
	}
    }

    set ent [tixHList:GetNearest $w $y 1]

    if {$ent != ""} {
	$w selection clear
	$w selection set $anchor $ent

	tixHList:Browse $w $ent
    }
}

proc tixHList:GoState-27 {w} {
    set ent [$w info anchor]

    if {$ent != ""} {
	tixHList:Browse $w $ent
    }

    tixHList:GoState 0 $w
}

proc tixHList:GoState-28 {w x y} {
    set anchor [$w info anchor]
    if {$anchor == ""} {
	set first [lindex [$w info children ""] 0]
	if {$first != ""} {
	    $w anchor set $first
	    set anchor $first
	} else {
	    return
	}
    }

    set ent [tixHList:GetNearest $w $y 1]
    if {$ent != ""} {
	$w selection clear
	$w selection set $anchor $ent

	tixHList:Browse $w $ent
    }
}

proc tixHList:GoState-29 {w} {
    tixHList:StartScan $w
}

proc tixHList:GoState-30 {w} {
    global tkPriv

    tixHList:DoScan $w

    set anchor [$w info anchor]
    if {$anchor == ""} {
	set first [lindex [$w info children ""] 0]
	if {$first != ""} {
	    $w anchor set $first
	    set anchor $first
	} else {
	    return
	}
    }

    set ent [tixHList:GetNearest $w $tkPriv(y) 1]
    if {$ent != ""} {
	$w selection clear
	$w selection set $anchor $ent

	tixHList:Browse $w $ent
    }
}

proc tixHList:GoState-31 {w x y} {
    tkCancelRepeat
    tixHList:GoState 26 $w $x $y 
}

proc tixHList:GoState-32 {w} {
    tkCancelRepeat
    tixHList:GoState 0 $w
}

# ctrl-button-1 has been pressed in extended selection mode
proc tixHList:GoState-33 {w x y} {
    set ent [tixHList:GetNearest $w $y]
    if {$ent != ""} {
	$w anchor set $ent
        if { [lsearch [$w info selection] $ent] >= 0 } {
            # toggle
            $w selection clear $ent
        } else {
            $w selection set $ent
            tixHList:Browse $w $ent
        }
    }
}


#----------------------------------------------------------------------
#
#		Common actions
#
#----------------------------------------------------------------------
proc tixHList:GetNearest {w y {disableOK 0}} {
    set ent [$w nearest $y]

    if {$ent != ""} {
	if {!$disableOK && [$w entrycget $ent -state] == "disabled"} {
	    return ""
	}
    }

    return $ent
}

proc tixHList:SetAnchor {w x y {disableOK 0} {dontsee 0}} {
    set ent [tixHList:GetNearest $w $y $disableOK]

    if {$ent != ""} {
	if {[$w entrycget $ent -state] != "disabled"} {
	    $w anchor set $ent
            if {!$dontsee} {
                $w see $ent
            }
	    return $ent
	} elseif $disableOK {
	    return $ent
	}
    }

    return ""
}

proc tixHList:Select {w ent} {
    if {"x[$w info selection]" != "x$ent"} {
	$w selection clear
	$w select set $ent
    }
}

#----------------------------------------------------------------------
#
#		       Auto scan
#
#----------------------------------------------------------------------
proc tixHList:StartScan {w} {
    global tkPriv
    set tkPriv(afterId) [after 50 tixHList:AutoScan $w]
}

proc tixHList:DoScan {w} {
    global tkPriv
    set x $tkPriv(x)
    set y $tkPriv(y)
    set X $tkPriv(X)
    set Y $tkPriv(Y)

    if {$y >= [winfo height $w]} {
	$w yview scroll 1 units
    } elseif {$y < 0} {
	$w yview scroll -1 units
    } elseif {$x >= [winfo width $w]} {
	$w xview scroll 2 units
    } elseif {$x < 0} {
	$w xview scroll -2 units
    } else {
	return
    }

    set tkPriv(afterId) [after 50 tixHList:AutoScan $w]
}


#----------------------------------------------------------------------
#
#		Indicator handling
#
#----------------------------------------------------------------------

proc tixHList:CallIndicatorCmd {w event ent} {
    set cmd [$w cget -indicatorcmd]

    if {$cmd != ""} {
	set bind(type)  $event
	set bind(specs) {%V}
	set bind(%V)    $ent

	tixEvalCmdBinding $w $cmd bind $ent
    }
}

proc tixHList:InsideArmedIndicator {w x y} {
    global tkPriv

    set ent [tixHList:GetNearest $w $y 1]
    if {$ent == "" || $ent != $tkPriv(tix,indicator)} {
	return 0
    }

    set info [$w info item $x $y]
    if {[lindex $info 1] == "indicator"} {
	return 1
    } else {
	return 0
    }
}

proc tixHList:Browse {w ent} {
    set browsecmd [$w cget -browsecmd]
    if {$browsecmd != ""} {
	set bind(specs) {%V}
	set bind(%V)    $ent

	tixEvalCmdBinding $w $browsecmd bind $ent
    }
}


#----------------------------------------------------------------------
#
# Header resizing. Based on Art Morel's work.
#
#----------------------------------------------------------------------

bind TixHListHeader <Motion> {
    tixHList:header:setHeaderCursor [winfo parent %W] %W %x %y
}
bind TixHListHeader <Enter> {
    tixHList:header:setHeaderCursor [winfo parent %W] %W %x %y
}
bind TixHListHeader <Leave> {
    tixHList:header:setHeaderCursor [winfo parent %W] %W %x %y
}
bind TixHListHeader <1> {
    tixHList:header:mark [winfo parent %W] %x
}
bind TixHListHeader <Button1-Enter> {
    tixHList:header:drag [winfo parent %W] %x
}
bind TixHListHeader <Button1-Leave> {
    tixHList:header:drag [winfo parent %W] %x
}
bind TixHListHeader <Button1-Motion> {
    tixHList:header:drag [winfo parent %W] %x
}
bind TixHListHeader <ButtonRelease-1> {
    tixHList:header:endDrag [winfo parent %W] %x
}

# tixHList:header:setHeaderCursor --
#     Change the header window's cursor to <-> when it's placed on top
#     of the border between two header columns.
# Arguments:
#     hlist     The HList whose column is being resized.
#     header    The private header window of the HList.
#     x         Mouse X location (relative to header window)
#     y         Mouse Y location (relative to header window)
# Results:
#     Returns nothing
#
proc tixHList:header:setHeaderCursor {hlist header x y} {
    global tixPriv

    if {$y >= 0 && $y < [winfo height $header] &&
            [tixHList:header:getColumn $hlist $x] != -1} {
        if {![info exists tixPriv(hlist:oldcursor)]} {
            set tixPriv(hlist:oldcursor) [$hlist cget -cursor]
            $hlist config -cursor sb_h_double_arrow
        }
    } else {
        if {[info exists tixPriv(hlist:oldcursor)]} {
            $hlist config -cursor $tixPriv(hlist:oldcursor)
            unset tixPriv(hlist:oldcursor)
        }
    }
}

# tixHList:header:mark --
#     Initialize the header dragging indicator line.
# Arguments:
#     hlist     The Hlist whose column is being resized.
#     x         Mouse X location (relative to header window)
# Results:
#     Returns nothing
#
proc tixHList:header:mark {hlist x} {
    global tixPriv

    if {![info exists tixPriv(hlist:oldcursor)]} {
        return
    }

    set col [tixHList:header:getColumn $hlist $x]
    set colx [tixHList:header:getColumnX $hlist $col]
    tixHList:header:drawDragLines set $hlist $colx

    set tixPriv(hlist:startx) $x
    set tixPriv(hlist:col)  $col
}

# tixHList:header:drag --
#     Move the header dragging indicator line.
# Arguments:
#     hlist     The Hlist whose column is being resized.
#     x         Mouse X location (relative to header window)
# Results:
#     Returns nothing
#
proc tixHList:header:drag {hlist x} {
    global tixPriv

    if {![info exists tixPriv(hlist:oldcursor)]} {
        return
    }

    set x [tixHList:header:restrictDrag $hlist $tixPriv(hlist:col) $x]
    set delta [expr $x - $tixPriv(hlist:startx)]
    set colx [tixHList:header:getColumnX $hlist $tixPriv(hlist:col)]

    tixHList:header:drawDragLines update $hlist [expr $colx + $delta]
}

# tixHList:header:endDrag --
#     End header dragging and resize the affected column
# Arguments:
#     hlist     The Hlist whose column is being resized.
#     x         Mouse X location (relative to header window)
# Results:
#     Returns nothing
#
proc tixHList:header:endDrag {hlist x} {
    global tixPriv

    if {![info exists tixPriv(hlist:oldcursor)]} {
        return
    }
    tixHList:header:drawDragLines cancel $hlist

    set x [tixHList:header:restrictDrag $hlist $tixPriv(hlist:col) $x]
    set width [$hlist column width $tixPriv(hlist:col)]
    set delta [expr $x - $tixPriv(hlist:startx)]
    set nwidth [expr $delta + $width]
    if {$nwidth < 8} {
        set nwidth 8
    }
    $hlist column width $tixPriv(hlist:col) $nwidth
    $hlist config -cursor $tixPriv(hlist:oldcursor)

    unset tixPriv(hlist:startx)
    unset tixPriv(hlist:oldcursor)
    unset tixPriv(hlist:col)
    $hlist config -cursor {}
}

# Returns the vertical line that divides col and col+1
proc tixHList:header:getColumnX {hlist col} {
    set colx 0
    for {set i 0} {$i <= $col} {incr i} {
        incr colx [$hlist column width $i]
    }
    incr colx -[$hlist xview]

    return $colx
}

proc tixHList:header:restrictDrag {hlist col x} {
    set min [expr [tixHList:header:getColumnX $hlist [expr $col - 1]] + 8]
    set max [expr [winfo width $hlist] - [$hlist cget -bd] - 2]

    if {$x < $min} {
        set x $min
    }
    if {$x > $max} {
        set x $max
    }

    if {$x < 0} {
        set x 0
    }
    if {$x > [winfo width $hlist]} {
        set x [expr [winfo width $hlist] - 1]
    }
    return $x
}


proc tixHList:header:drawDragLines {mode {hlist {}} {x {}}} {
    global tixPriv

    if {$mode != "set"} {
        if {[info exists tixPriv(hlist:drX)]} {
            set oldX  $tixPriv(hlist:drX)
            set oldY1 $tixPriv(hlist:drY1)
            set oldY2 $tixPriv(hlist:drY2)

            tixTmpLine $oldX $oldY1 $oldX $oldY2 $hlist
        }
    }
    
    if {$mode == "cancel"} {
        catch {
            unset tixPriv(hlist:drX)
            unset tixPriv(hlist:drY1)
            unset tixPriv(hlist:drY2)
        }
        return
    }

    set y1 [$hlist cget -bd]
    set y2 [expr [winfo height $hlist] - 2 * $y1]

    incr x  [winfo rootx $hlist]
    incr y1 [winfo rooty $hlist]
    incr y2 [winfo rooty $hlist]

    tixTmpLine $x $y1 $x $y2 $hlist

    set tixPriv(hlist:drX)  $x
    set tixPriv(hlist:drY1) $y1
    set tixPriv(hlist:drY2) $y2
}

# Returns the column which is just to the left of the mouse cursor
proc tixHList:header:getColumn {hlist x} {
    set fuzzy 3
    set sum 0
    set max [expr [$hlist cget -columns] - 1]
    incr x [$hlist xview] 

    for {set col 0} {$col < $max} {incr col} {
        set width [$hlist column width $col]
        incr sum $width
        if {abs($x - $sum) <= $fuzzy} {
            return $col
            break
        }
    }

    return -1
}

