
/*
 * bltPicture.h --
 *
 *	Copyright 2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef _BLT_PICTURE_H
#define _BLT_PICTURE_H

#include "bltPs.h"

/*
 *----------------------------------------------------------------------
 *
 * Pix32 --
 *
 *      A union representing either a pixel as a RGB triplet or a
 *	single word value.
 *
 *----------------------------------------------------------------------
 */
typedef union {
    unsigned int color;		
    struct {
#ifdef WORDS_BIGENDIAN
	unsigned char a, r, g, b;
#else 
	unsigned char b, g, r, a;
#endif    
    } u8;
} Pix32;

#define Red	u8.r
#define Blue	u8.b
#define Green	u8.g
#define Alpha	u8.a

#define ALPHA_OPAQUE		(0xFF)
#define ALPHA_TRANSPARENT	(0)
#define MAXINTENSITY		(0xFF)

#define GAMMA	(1.0)

/*
 *----------------------------------------------------------------------
 *
 * Blt_Picture --
 *
 *      The structure below represents a picture.  Each pixel
 *	occupies a 32-bit word of memory: one byte for each of the
 *	red, green, and blue color intensities, and another for
 *	alpha-channel image compositing (e.g. transparency).
 *
 *----------------------------------------------------------------------
 */
struct Blt_PictureStruct {
    int width, height;		/* Dimensions of the image in pixels. */
    int pixelsPerRow;		/* Stride of the image. */
    Pix32 *bits;		/* Array of pixels (malloc'ed)
				 * containing the RGBA values. */
    int align;			/* Offset of start of picture in bits
				 * array. */
    unsigned int flags;		/* Flags describing the picture. */
    void *buffer;

};
#define BLT_PICTURE_COLOR (1<<0) /* Indicates if color or greyscale. */
#define BLT_PICTURE_BLEND (1<<1) /* Picture has partial opaque pixels. */
#define BLT_PICTURE_MASK  (1<<2) /* Pixels are either 100% opaque or
				  * transparent. The separate BLEND
				  * and MASK flags are so that don't
				  * premultiply alphas for masks. */

#define BLT_PICTURE_PREMULTIPLIED_ALPHAS (1<<3)
				/* Indicates if RGB components have
				 * been premultiplied by their
				 * alphas. */

#define BLT_PICTURE_DIRTY (1<<4) /* Indicates that the picture
				  * contents have changed. Cached
				  * items may need to be
				  * refreshed. For example, may need
				  * to premultiply alphas again. */ 

#define BLT_PAINTER_DITHER		(1<<0)

#define BLT_PAINTER_BLEND_MASK		(0x0F)
#define BLT_PAINTER_BLEND_NONE		(0)
#define BLT_PAINTER_BLEND_MIN_ALPHAS	(1<<1)
#define BLT_PAINTER_BLEND_MAX_ALPHAS	(1<<2)
#define BLT_PAINTER_BLEND_DIFF		(1<<3)
#define BLT_PAINTER_BLEND_MULTIPLY	(1<<4)
#define BLT_PAINTER_BLEND_UNDER		(1<<6)

typedef enum GradientShapes {
    BLT_GRADIENT_SHAPE_LINEAR,
    BLT_GRADIENT_SHAPE_BILINEAR,
    BLT_GRADIENT_SHAPE_RADIAL,	
    BLT_GRADIENT_SHAPE_RECTANGULAR
} Blt_GradientShape;

typedef enum GradientPaths {
    BLT_GRADIENT_PATH_X,
    BLT_GRADIENT_PATH_Y,
    BLT_GRADIENT_PATH_XY,
    BLT_GRADIENT_PATH_YX,
} Blt_GradientPath;

typedef struct {
    Blt_GradientShape shape;	/* Determines the shape of gradient. */

    Blt_GradientPath path;	/* Determines the direction of a
				 * linear or bilinear gradient. */

    int logScale;		/* If non-zero, use a logarithmic
				 * scale to ramp colors. */

    int jitter;			/* If non-zero, add jitter error to
				 * gradient. */

} Blt_Gradient;

typedef struct Blt_PictureImageStruct *Blt_PictureImage;
typedef struct Blt_PictureFilterStruct *Blt_PictureFilter;
typedef struct Blt_PictureStruct *Blt_Picture;

/*
 * Blt_Picture is supposed to be an opaque type.  Use the macros below
 * to access its members.
 */

#define Blt_PictureBits(p)	((p)->bits)
#define Blt_PictureFlags(p)	((p)->flags)
#define Blt_PictureHeight(p)	((p)->height)
#define Blt_PicturePixel(p,x,y) ((p)->bits + ((p)->pixelsPerRow * (y)) + (x))
#define Blt_PictureWidth(p)	((p)->width)
#define Blt_PictureStride(p)	((p)->pixelsPerRow)

#define Blt_PictureIsDirty(p)	((p)->flags & BLT_PICTURE_DIRTY)
#define Blt_PictureIsOpaque(p)	\
	(((p)->flags & (BLT_PICTURE_BLEND | BLT_PICTURE_MASK)) == 0)
#define Blt_PictureIsMasked(p)  ((p)->flags &  BLT_PICTURE_MASK)
#define Blt_PictureIsBlended(p) ((p)->flags &  BLT_PICTURE_BLEND)
#define Blt_PictureIsColor(p)   ((p)->flags &  BLT_PICTURE_COLOR)
#define Blt_PictureIsGreyscale(p)   (!Blt_IsColorPicture(p))

typedef enum PictureArithOps {
    PICTURE_ARITH_ADD,
    PICTURE_ARITH_AND,
    PICTURE_ARITH_NAND,
    PICTURE_ARITH_NOR,
    PICTURE_ARITH_OR,
    PICTURE_ARITH_RSUB,
    PICTURE_ARITH_SUB,
    PICTURE_ARITH_XOR,
    PICTURE_ARITH_MIN,
    PICTURE_ARITH_MAX,
} Blt_PictureArithOps;

typedef struct {
    unsigned int x, y;
} PictureCoordinate;

typedef enum BlendingModes {
    BLT_BLEND_NORMAL,		/* C = F */
    BLT_BLEND_MULTIPLY,		/* C = F * B */
    BLT_BLEND_SCREEN,		/* C = 1 - (1 - F * B */
    BLT_BLEND_DARKEN,		/* C = min(F,B) */
    BLT_BLEND_LIGHTEN,		/* C = max(F,B) */
    BLT_BLEND_DIFFERENCE,	/* C = |F - B| */
    BLT_BLEND_HARDLIGHT,
    BLT_BLEND_SOFTLIGHT,
    BLT_BLEND_COLORDODGE,	/* C = B / (1 - F) */ 
    BLT_BLEND_COLORBURN,	/* C = (1 - B) / F */
    BLT_BLEND_OVERLAY,		/* C = B * (F + (2 * F) * (1 - B)) */
} Blt_BlendingMode;

extern Blt_PictureFilter bltBoxFilter; /* The ubiquitous box filter */
extern Blt_PictureFilter bltMitchellFilter; 
extern Blt_PictureFilter bltTableFilter;

typedef struct {
    int x, y, width, height;
} BBox;

#define Blt_AddPictures(dest, src) \
	Blt_ApplyPictureToPicture(dest, src, PICTURE_ARITH_ADD)
#define Blt_SubtractPictures(dest, src) \
	Blt_ApplyPictureToPicture(dest, src, PICTURE_ARITH_SUB)
#define Blt_AndPictures(dest, src) \
	Blt_ApplyPictureToPicture(dest, src, PICTURE_ARITH_AND)
#define Blt_OrPictures(dest, src) \
	Blt_ApplyPictureToPicture(dest, src, PICTURE_ARITH_OR)
#define Blt_XorPictures(dest, src) \
	Blt_ApplyPictureToPicture(dest, src, PICTURE_ARITH_XOR)

/* Prototypes of picture routines */

extern void Blt_ApplyPictureToPicture(Blt_Picture dest, Blt_Picture src,
	Blt_PictureArithOps op);

extern void Blt_ApplyScalarToPicture(Blt_Picture dest, Pix32 *colorPtr, 
	Blt_PictureArithOps op);

extern void Blt_ApplyPictureToPictureWithMatte(Blt_Picture dest, 
	Blt_Picture src, Blt_Picture matte, int invert, Blt_PictureArithOps op);

extern void Blt_ApplyScalarToPictureWithMatte(Blt_Picture dest, 
	Pix32 *colorPtr, Blt_Picture matte, int invert, Blt_PictureArithOps op);

extern void Blt_BlankPicture(Blt_Picture picture, Pix32 *colorPtr);

extern void Blt_BlendPictures(Blt_Picture dest, Blt_Picture src);

extern void Blt_BlendPictureArea(Blt_Picture bg, Blt_Picture fg, int fgX, 
	int fgY, int fgWidth, int fgHeight, int bgX, int bgY);

extern int Blt_ReallocatePicture(Blt_Picture picture, int width, int height,
	int save);

extern Blt_Picture Blt_ClonePicture(Blt_Picture picture);

extern void Blt_ConvolvePicture(Blt_Picture dest, Blt_Picture src, 
	Blt_PictureFilter vFilter, Blt_PictureFilter hFilter);

extern void Blt_CopyPicture(Blt_Picture dest, Blt_Picture src);

extern void Blt_CopyPictureArea(Blt_Picture dest, Blt_Picture src, int srcX, 
	int srcY, int width, int height, int destX, int destY);

extern Blt_Picture Blt_CreatePicture(int width,int height);

extern Blt_Picture Blt_DitherPicture(Blt_Picture picture, Pix32 *palette);

extern void Blt_FadePicture(Blt_Picture dest, Blt_Picture src, double fade);

extern void Blt_FreePicture(Blt_Picture picture);

extern void Blt_GradientPicture(Blt_Picture picture, Pix32 *highPtr, 
	Pix32 *lowPtr, Blt_Gradient *gradientPtr);

extern Blt_Picture Blt_GreyscalePicture(Blt_Picture picture);

extern Blt_Picture Blt_QuantizePicture (Blt_Picture picture, int nColors);

extern void Blt_ResamplePicture (Blt_Picture dest, Blt_Picture src, 
	Blt_PictureFilter hFilter, Blt_PictureFilter vFilter);

extern Blt_Picture Blt_ScalePicture(Blt_Picture picture, int x, int y, 
	int width, int height, int destWidth, int destHeight);

extern Blt_Picture Blt_ScalePictureArea(Blt_Picture picture, int x, int y,
	int width, int height, int destWidth, int destHeight);

extern Blt_Picture Blt_RotatePicture (Blt_Picture picture, double angle);

extern void Blt_TilePicture(Blt_Picture dest, Blt_Picture src, int xOrigin, 
	int yOrigin, int x, int y, int width, int height);

extern Blt_Picture Blt_DrawableToPicture(Tk_Window tkwin, Drawable drawable, 
	int x, int y, int width, int height, double inputGamma);

extern Blt_Picture Blt_PhotoToPicture (Tk_PhotoHandle photo);

extern Blt_Picture Blt_PhotoAreaToPicture (Tk_PhotoHandle photo, int x,
	int y, int width, int height);

extern Blt_Picture Blt_WindowToPicture(Display *display, Drawable drawable, 
	int x, int y, int width, int height, double inputGamma);

extern int Blt_PictureToPsData(Blt_Picture picture, int nComponents, 
	Tcl_DString *resultPtr, char *prefix);

extern void Blt_PictureToPostScript(Blt_PostScript ps, Blt_Picture picture,
	double x, double y);

extern void Blt_PictureToPhoto(Blt_Picture picture, Tk_PhotoHandle photo);

extern int Blt_PictureToPsData(Blt_Picture picture, int nComponents, 
	Tcl_DString *resultPtr, char *prefix);

extern void Blt_SelectPixels(Blt_Picture dest, Blt_Picture src, 
	Pix32 *lowerPtr, Pix32 *upperPtr);

extern int Blt_GetPicture(Tcl_Interp *interp, char *string, 
	Blt_Picture *picturePtr);

extern int Blt_GetPictureFromObj(Tcl_Interp *interp, Tcl_Obj *objPtr, 
	Blt_Picture *picturePtr);

extern int Blt_GetPictureFilterFromObj(Tcl_Interp *interp, Tcl_Obj *objPtr,
	Blt_PictureFilter *filterPtr);

extern char *Blt_NameOfPictureFilter(Blt_PictureFilter filter);

#if HAVE_JPEG
extern Blt_Picture Blt_JPEGToPicture(Tcl_Interp *interp, char *fileName);
#endif

extern void Blt_UnmultiplyAlphas(Blt_Picture picture);
extern void Blt_PremultiplyAlphas(Blt_Picture picture);
extern void Blt_MultiplyPixels(Blt_Picture picture, float value);

extern int Blt_GetBBoxFromObjv(Tcl_Interp *interp, int objc, 
	Tcl_Obj *CONST *objv, BBox *bboxPtr);

extern int Blt_AdjustBBoxToPicture(Blt_Picture picture, BBox *bboxPtr);

extern int Blt_GetPix32FromObj(Tcl_Interp *interp, Tcl_Obj *objPtr, 
	Pix32 *pixelPtr);

extern int Blt_GetPix32(Tcl_Interp *interp, char *string, Pix32 *pixelPtr);

extern Blt_Picture Blt_PictureFromImage(Blt_PictureImage image);

extern void Blt_NotifyImageChanged(Blt_PictureImage image);

extern int Blt_QueryColors(Blt_Picture picture, Blt_HashTable *tablePtr);

extern int Blt_SnapPhoto(Tcl_Interp *interp, Tk_Window tkwin, Drawable drawable,
	 int x, int y, int width, int height, int destWidth, int destHeight, 
	char *photoName, double inputGamma);

extern int Blt_SnapPicture(Tcl_Interp *interp, Tk_Window tkwin, 
	Drawable drawable, int x, int y, int width, int height,
	int destWidth, int destHeight, char *imageName, double inputGamma);

#endif /*_BLT_PICTURE_H*/
