#include <tcl.h>
#include <ldap.h>
#include "ldap_tcl.h"

int LDAPSetNamePasswordCmd(ClientData clientData, Tcl_Interp *interp,
          int argc, char *argv[])
{
	// Does the TCL interpreter support version 8.3 of TCL?
	if (Tcl_InitStubs(interp,"8.3",0) == NULL)
		return TCL_ERROR;

	// Verify the user has sent all the needed arguments.
	if (argc != 3)
	{
		interp->result = "FAILED {Usage : LDAP_SetNamePassword \"username\" \"password\"}";
		return TCL_ERROR;
	}

	ldapUserName = argv[1];
	ldapPassword = argv[2];
	interp->result = "SUCCEEDED {LDAP username and password set for future searches.}";
	return TCL_OK;

}

int LDAPDisconnectCmd(ClientData clientData, Tcl_Interp *interp,
          int argc, char *argv[])
{
	// Does the TCL interpreter support version 8.3 of TCL?
	if (Tcl_InitStubs(interp,"8.3",0) == NULL)
		return TCL_ERROR;

	if (ld !=NULL)
	{
		ldap_unbind(ld);
		ld = NULL;
		interp->result = "SUCCEEDED {LDAP directory unbinded.}";
	}
	else
	{
		interp->result = "FAILED {LDAP directory already unbinded.}";
	}
	return TCL_OK;
}

int LDAPConnectCmd(ClientData clientData, Tcl_Interp *interp,
          int argc, char *argv[])
{

	int				intPort;

	// Does the TCL interpreter support version 8.3 of TCL?
	if (Tcl_InitStubs(interp,"8.3",0) == NULL)
		return TCL_ERROR;

	// Is a connection already open?
	if (ld != NULL)
	{
		interp->result = "FAILED {LDAP connection already open. Please disconnect first using LDAP_Disconnect.}";
		return TCL_ERROR;
	}

	// Verify the user has sent all the needed arguments.
	if (argc != 3)
	{
		interp->result = "FAILED {Usage : LDAP_Connect \"Server\" \"Port\" }";
		return TCL_ERROR;
	}

	// Parse the LDAP Port.
	if( Tcl_GetInt(interp, argv[2], &intPort))
	{
		interp->result = "FAILED {Parsing LDAP port:Should be second parameter.}";
		return TCL_ERROR;
	}

	/*get a handle to an LDAP Connection */
	if ( ( ld = ldap_init(argv[1], intPort)) == NULL )
	{
		interp->result = "FAILED {Connecting to LDAP Server}";
		return TCL_ERROR;
	}

	/*Authenticate to the directory*/
	if ( ldapUserName != NULL && ldapPassword != NULL)
	{
		if ( ldap_simple_bind_s(ld,ldapUserName,ldapPassword) != LDAP_SUCCESS )
		{
			ldap_perror (ld, "ldap_simple_bind_s");
			interp->result = "FAILED {During ldap_simple_bind_s}";
			return TCL_ERROR;
		}

	}
	else
	{
		if ( ldap_simple_bind_s(ld,NULL,NULL) != LDAP_SUCCESS )
		{
			ldap_perror (ld, "ldap_simple_bind_s");
			interp->result = "FAILED {During ldap_simple_bind_s}";
			return TCL_ERROR;
		}
	}

 	interp->result = "SUCCEEDED {Successfully simple binded synchronously to the LDAP database.}";
	return TCL_OK;

}

int LDAPAllInOneSearchCmd(ClientData clientData, Tcl_Interp *interp,
          int argc, char *argv[])
{

	LDAPMessage	*result, *e;
	BerElement		   *ber;
	char			*a, *dn;
	char			 **vals;
	int					  i;
	Tcl_Obj		*obj_result;
	Tcl_Obj		   *obj_row;
	Tcl_Obj 	   *obj_col;
	#if UTF_ENCODING
		Tcl_DString ds;
	#endif

	int				intPort;

	// Does the TCL interpreter support version 8.3 of TCL?
	if (Tcl_InitStubs(interp,"8.3",0) == NULL)
		return TCL_ERROR;

	// Verify the user has sent all the needed arguments.
	if (argc < 5)
	{
		interp->result = "FAILED {Usage : LDAP_AllInOneSearch \"Server\" \"Port\" \"Directory\" \"SearchFilter\" \"Attributes[Optional]\"}";
		return TCL_ERROR;
	}

	// Parse the LDAP Port.
	if( Tcl_GetInt(interp, argv[2], &intPort))
	{
		interp->result = "FAILED {Parsing LDAP port:Should be second parameter.}";
		return TCL_ERROR;
	}
	/*get a handle to an LDAP Connection */
	/* Example: "158.137.200.174" */
	/* Example Port: 25389 */
	if ( ( ld = ldap_init(argv[1], intPort)) == NULL )
	{
		interp->result = "FAILED {Connecting to LDAP Server}";
		return TCL_ERROR;
	}

	/*Authenticate to the directory*/
	if ( ldapUserName != NULL && ldapPassword != NULL)
	{
		// The user has specified a username and password for the LDAP directory.
		if ( ldap_simple_bind_s(ld,ldapUserName,ldapPassword) != LDAP_SUCCESS )
		{
			ldap_perror (ld, "ldap_simple_bind_s");
			interp->result = "FAILED {During ldap_simple_bind_s}";
			return TCL_ERROR;
		}

	}
	else
	{
		// The user has not specified a username or password.
		if ( ldap_simple_bind_s(ld,NULL,NULL) != LDAP_SUCCESS )
		{
			ldap_perror (ld, "ldap_simple_bind_s");
			interp->result = "FAILED {During ldap_simple_bind_s}";
			return TCL_ERROR;
		}
	}

	/*Search for all entries*/
	if (ldap_search_s(ld, argv[3],
		LDAP_SCOPE_SUBTREE, argv[4],NULL,0,&result)
		!=LDAP_SUCCESS)
	{
		ldap_perror(ld, "ldap_search_s");
		interp->result = "FAILED {Possible bad LDAP directory tree or search filter}";
		return TCL_ERROR;
	}

	obj_result = Tcl_GetObjResult(interp);
	if (argc == 5)
	{
		// The user did not specify which attributes they want returned
		// We will retrieve all attributes.
		//Tcl_SetResult(interp,"{SUCCEEDED}",TCL_DYNAMIC);
		for ( e=ldap_first_entry(ld,result); e !=NULL;e = ldap_next_entry(ld,e))
		{
			obj_row = Tcl_NewListObj(0,NULL);
			if ((dn=ldap_get_dn(ld,e)) !=NULL)
			{
				ldap_memfree(dn);
			}

			for ( a=ldap_first_attribute(ld,e,&ber);a !=NULL; a=ldap_next_attribute(ld,e,ber))
			{
				if ((vals=ldap_get_values(ld,e,a)) != NULL)
				{
					for (i=0;vals[i] !=NULL; i++)
					{

						obj_col = Tcl_NewStringObj(a,strlen(a));
						Tcl_ListObjAppendElement(interp,obj_row,obj_col);

						obj_col = Tcl_NewStringObj(vals[i],strlen(vals[i]));
						Tcl_ListObjAppendElement(interp,obj_row,obj_col);

					}
					ldap_value_free(vals);
				}
					ldap_memfree(a);
			}
			if (ber !=NULL)
			{
				ber_free(ber,0);
			}
			Tcl_ListObjAppendElement(interp,obj_result,obj_row);
		}

	}
	else
	{
		// The user has specified which attributes they want returned.
		// We will only retrieve attributes the user wants.
		/* Build the response message. */
		for ( e=ldap_first_entry(ld,result); e !=NULL;e = ldap_next_entry(ld,e))
		{
			obj_row = Tcl_NewListObj(0,NULL);
			if ((dn=ldap_get_dn(ld,e)) !=NULL)
			{
				ldap_memfree(dn);
			}

			for(int x=5;x < argc;x++)
			{
				if ((vals = ldap_get_values(ld,e,argv[x])) != NULL)
				{
					for (i=0;vals[i] !=NULL; i++)
					{

						obj_col = Tcl_NewStringObj(argv[x],strlen(argv[x]));
						Tcl_ListObjAppendElement(interp,obj_row,obj_col);

						obj_col = Tcl_NewStringObj(vals[i],strlen(vals[i]));
						Tcl_ListObjAppendElement(interp,obj_row,obj_col);

					}
					ldap_value_free(vals);
				}
				ldap_memfree(argv[x-1]);
			}

			Tcl_ListObjAppendElement(interp,obj_result,obj_row);
		}
	}

	ldap_msgfree (result);
	ldap_unbind(ld);
	ld = NULL;
	return TCL_OK;
}


int LDAPSearchCmd(ClientData clientData, Tcl_Interp *interp,
          int argc, char *argv[])
{

	LDAPMessage	*result, *e;
	BerElement		   *ber;
	char			*a, *dn;
	char			 **vals;
	int					  i;
	Tcl_Obj		*obj_result;
	Tcl_Obj		   *obj_row;
	Tcl_Obj 	   *obj_col;
	#if UTF_ENCODING
		Tcl_DString ds;
	#endif

	// Is the user currently connected to the LDAP Directory.
	if (ld == NULL)
	{
		interp->result = "FAILED {Not binded to an LDAP directory}";
		return TCL_ERROR;
	}

	// Verify the user has sent all the needed arguments.
	if (argc < 3)
	{
		interp->result = "FAILED {Usage : LDAP_Search \"Directory\" \"SearchFilter\" \"Attributes[Optional]}";
		return TCL_ERROR;
	}

	// Does the TCL interpreter support version 8.3 of TCL?
	if (Tcl_InitStubs(interp,"8.3",0) == NULL)
		return TCL_ERROR;

	/*Search for all entries*/
	if (ldap_search_s(ld, argv[1],
		LDAP_SCOPE_SUBTREE, argv[2],NULL,0,&result)
		!=LDAP_SUCCESS)
	{
		ldap_perror(ld, "ldap_search_s");
		interp->result = "FAILED {Possible bad LDAP directory tree or search filter}";
		return TCL_ERROR;
	}

	obj_result = Tcl_GetObjResult(interp);
	if (argc == 3)
	{
		// The user did not specify which attributes they want returned
		// We will retrieve all attributes.
		//Tcl_SetResult(interp,"{SUCCEEDED}",TCL_DYNAMIC);
		for ( e=ldap_first_entry(ld,result); e !=NULL;e = ldap_next_entry(ld,e))
		{
			obj_row = Tcl_NewListObj(0,NULL);
			if ((dn=ldap_get_dn(ld,e)) !=NULL)
			{
				ldap_memfree(dn);
			}

			for ( a=ldap_first_attribute(ld,e,&ber);a !=NULL; a=ldap_next_attribute(ld,e,ber))
			{
				if ((vals=ldap_get_values(ld,e,a)) != NULL)
				{
					for (i=0;vals[i] !=NULL; i++)
					{

						obj_col = Tcl_NewStringObj(a,strlen(a));
						Tcl_ListObjAppendElement(interp,obj_row,obj_col);

						obj_col = Tcl_NewStringObj(vals[i],strlen(vals[i]));
						Tcl_ListObjAppendElement(interp,obj_row,obj_col);

					}
					ldap_value_free(vals);
				}
					ldap_memfree(a);
			}
			if (ber !=NULL)
			{
				ber_free(ber,0);
			}
			Tcl_ListObjAppendElement(interp,obj_result,obj_row);
		}

	}
	else
	{
		// The user has specified which attributes they want returned.
		// We will only retrieve attributes the user wants.
		/* Build the response message. */
		for ( e=ldap_first_entry(ld,result); e !=NULL;e = ldap_next_entry(ld,e))
		{
			obj_row = Tcl_NewListObj(0,NULL);
			if ((dn=ldap_get_dn(ld,e)) !=NULL)
			{
				ldap_memfree(dn);
			}

			for(int x=3;x < argc;x++)
			{
				if ((vals = ldap_get_values(ld,e,argv[x])) != NULL)
				{
					for (i=0;vals[i] !=NULL; i++)
					{

						obj_col = Tcl_NewStringObj(argv[x],strlen(argv[x]));
						Tcl_ListObjAppendElement(interp,obj_row,obj_col);

						obj_col = Tcl_NewStringObj(vals[i],strlen(vals[i]));
						Tcl_ListObjAppendElement(interp,obj_row,obj_col);

					}
					ldap_value_free(vals);
				}
				ldap_memfree(argv[x-1]);
			}

			Tcl_ListObjAppendElement(interp,obj_result,obj_row);
		}
	}

	ldap_msgfree (result);
	return TCL_OK;
}

int Ldap_tcl_Init(Tcl_Interp *interp)
{
		Tcl_CreateCommand(interp, "LDAP_Search", LDAPSearchCmd,(ClientData)NULL, (Tcl_CmdDeleteProc *) NULL);
		Tcl_CreateCommand(interp, "LDAP_AllInOneSearch", LDAPAllInOneSearchCmd,(ClientData)NULL, (Tcl_CmdDeleteProc *) NULL);
		Tcl_CreateCommand(interp, "LDAP_SetNamePassword", LDAPSetNamePasswordCmd,(ClientData)NULL, (Tcl_CmdDeleteProc *) NULL);
		Tcl_CreateCommand(interp, "LDAP_Connect", LDAPConnectCmd,(ClientData)NULL, (Tcl_CmdDeleteProc *) NULL);
		Tcl_CreateCommand(interp, "LDAP_Disconnect", LDAPDisconnectCmd,(ClientData)NULL, (Tcl_CmdDeleteProc *) NULL);
		return (Tcl_PkgProvide(interp,"LDAP","1.0") == TCL_ERROR ? TCL_ERROR : TCL_OK);
}