/* forward declarations */
#include "config.h"

#include <stdio.h>
#include <math.h>
#include <string.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif

#include <signal.h>
#include <unistd.h>

#include <slang.h>

#include <stdarg.h>

#include "slplplot.h"
#include "sldxe.h"

#define DXE_MAX_PATH_LEN 1024

static int TTY_Inited;
static void init_tty (void);
static void reset_tty (void);

static int open_readline (void);
static void close_readline (void);

void sldxe_exit_error (int status, char *fmt, ...)
{
   va_list ap;
   
   if (fmt != NULL)
     {
	unsigned int len = strlen (fmt);

	va_start (ap, fmt);
	vfprintf (stderr, fmt, ap);
	va_end (ap);
	if (len && (fmt [len - 1] != '\n'))
	  fputs ("\n", stderr);
     }
   close_readline ();
   SLang_reset_tty ();
   exit (status);
}

void sldxe_verror (char *fmt, ...)
{
   va_list ap;
   
   if (fmt != NULL)
     {
	unsigned int len = strlen (fmt);

	va_start (ap, fmt);
	vfprintf (stderr, fmt, ap);
	va_end (ap);
	if (len && (fmt [len - 1] != '\n'))
	  fputs ("\n", stderr);
     }
   if (SLang_Error == 0)
     SLang_Error = INTRINSIC_ERROR;
}

void sldxe_vmessage (char *fmt, ...)
{
   va_list ap;
   
   if (fmt != NULL)
     {
	unsigned int len = strlen (fmt);

	va_start (ap, fmt);
	vfprintf (stdout, fmt, ap);
	va_end (ap);

	if (len && (fmt [len - 1] != '\n'))
	  fputs ("\n", stdout);
	
	fflush (stdout);
     }
}

#if USE_PLPLOT
extern int plD_input_pending_xw (int, unsigned int);
#endif

static int my_input_pending (int tsecs)
{
#if USE_PLPLOT
   return plD_input_pending_xw (SLang_TT_Read_FD, tsecs * 100);
#else
   return SLang_input_pending (tsecs);
#endif
}

unsigned int sldxe_getkey (void)
{
   int reset;
   unsigned int ch;

   reset = (TTY_Inited == 0);

   if (SLang_Input_Buffer_Len == 0)
     {
#if USE_PLPLOT
	while (0 == my_input_pending (300))
	  ;
#endif
     }
   ch = SLang_getkey ();
   if (reset)
     reset_tty ();
   return ch;
}

#ifdef SIGFPE
static int Float_Exception;
static void floating_point_execption (int sig)
{
   (void) sig;
   Float_Exception = 1;
   SLsignal (SIGFPE, SIG_IGN);
}
#endif


static SLang_RLine_Info_Type Dxe_RLI;
static unsigned char Dxe_RLI_Buf[256];

static void (*last_sig_sigtstp) (int);

static void sig_sigtstp (int sig)
{
   (void) sig;
   SLsig_block_signals ();
   reset_tty ();
   kill(getpid(),SIGSTOP);
   init_tty ();
   SLrline_redraw (&Dxe_RLI);
   SLsig_unblock_signals ();
}


static void init_tty (void)
{
   if (TTY_Inited)
     return;
   TTY_Inited++;

   SLsig_block_signals ();

   SLang_TT_Read_FD = fileno (stdin);
   
   last_sig_sigtstp = SLsignal (SIGTSTP, sig_sigtstp);
   
   if (-1 == SLang_init_tty (-1, 1, 0))
     {
	SLsignal (SIGTSTP, last_sig_sigtstp);
	SLsig_unblock_signals ();
	sldxe_exit_error (1, "Error initializing terminal.");
     }
   SLtty_set_suspend_state (1);
   SLsig_unblock_signals ();
}

static void reset_tty (void)
{
   if (TTY_Inited == 0)
     return;
   TTY_Inited = 0;

   SLsig_block_signals ();
   SLsignal (SIGTSTP, last_sig_sigtstp);
   SLang_reset_tty ();
   SLsig_unblock_signals ();
   fputc ('\n', stdout);
   fflush (stdout);
}

static int Popen_Reopen_TTY;
FILE *sldxe_popen (char *s, char *mode)
{
   FILE *fp;

   Popen_Reopen_TTY = TTY_Inited;
   reset_tty ();
   fp = popen (s, mode);
   if ((fp == NULL) && (Popen_Reopen_TTY))
     init_tty ();
   return fp;
}

int sldxe_pclose (FILE *fp)
{
   int ret = -1;
   if (fp != NULL) ret = pclose (fp);
   if (Popen_Reopen_TTY) init_tty ();
   return ret;
}

static SLang_Load_Type *Readline_Load_Object;

static char *read_using_readline (SLang_Load_Type *x)
{
   int n;
   
   Dxe_RLI_Buf[0] = 0;

   if (0 == x->parse_level) 
     {
	Dxe_RLI.prompt = "sldxe> ";
	if (-1 == SLang_run_hooks ("sldxe_take_input_hook", 0))
	  return NULL;
     }
   else Dxe_RLI.prompt = "       ";

   SLtt_get_screen_size ();
   Dxe_RLI.edit_width = SLtt_Screen_Cols - 1;
   
   init_tty ();
   n = SLang_read_line (&Dxe_RLI);
   reset_tty ();

   if (n < 0) return NULL;
   if ((n == 0) && (SLang_Last_Key_Char == SLang_RL_EOF_Char))
     sldxe_exit_error (0, NULL);

   SLang_rline_save_line (&Dxe_RLI);
   return (char *) Dxe_RLI_Buf;
}

static void close_readline (void)
{
   if (Readline_Load_Object == NULL)
     return;
     
   SLdeallocate_load_type (Readline_Load_Object);
   Readline_Load_Object = NULL;
}



static int open_readline (void)
{
   Dxe_RLI.buf = Dxe_RLI_Buf;
   Dxe_RLI.buf_len = 255;
   Dxe_RLI.tab = 8;
   Dxe_RLI.edit_width = 79;
   Dxe_RLI.dhscroll = 20;
   Dxe_RLI.prompt = "sldxe> ";
   Dxe_RLI.getkey = sldxe_getkey;
   Dxe_RLI.flags = SL_RLINE_BLINK_MATCH;
   Dxe_RLI.input_pending = my_input_pending;
   
#ifdef USE_ANSI_COLOR
   Dxe_RLI.flags |= SL_RLINE_USE_ANSI;
#endif

   if (-1 == SLang_init_readline (&Dxe_RLI))
     {
	close_readline ();
	return -1;
     }

   if (NULL == (Readline_Load_Object = SLallocate_load_type ("<stdin>")))
     {
	close_readline ();
	return -1;
     }

   Readline_Load_Object->read = read_using_readline;
   Readline_Load_Object->auto_declare_globals = 1;
   return 0;
}

static char *file_type (char *file)
{
   char *f;
   
   f = file + strlen (file);
   do
     {
	if (*f == '.')
	  return f + 1;
	f--;
     }
   while (f >= file);
   return NULL;
}


static int dircat (char *buf, char *dir, char *name, unsigned int blen)
{
   unsigned int dirlen, namelen;
   
   dirlen = strlen (dir);

   if (name == NULL) 
     name = "";
   
   namelen = strlen (name);
   
   if (namelen + dirlen + 1 >= blen)
     return -1;
   
   if (dir != buf) strcpy (buf, dir);
   if (dirlen && (buf [dirlen - 1] != '/'))
     {
	buf [dirlen] = '/';
	dirlen++;
     }
   
   strcpy (buf + dirlen, name);
   
   return 0;
}
   
static int fixup_dir (char *dir, unsigned int len)
{
   return dircat (dir, dir, NULL, len);
}

#if 0
static int safe_strcpy (char *a, char *b, unsigned int len)
{
   unsigned int blen;
   
   blen = strlen (b);
   if (blen >= len)
     return -1;
   
   strcpy (a, b);
   return 0;
}
#endif

static int safe_strncpy (char *a, char *b, unsigned int maxcpy, unsigned int alen)
{
   char *amax, *bmax, ch;
   
   amax = a + alen;
   bmax = b + maxcpy;
   
   while (a < amax)
     {
	if (b == bmax)
	  {
	     *a = 0;
	     return 0;
	  }

	ch = *b++;
	*a++ = ch;
	if (ch == 0)
	  return 0;
     }

   return -1;
}

static int safe_strcat (char *a, char *b, unsigned int len)
{
   unsigned int blen, alen;
   
   blen = strlen (b);
   alen = strlen (a);
   
   if (alen + blen >= len)
     return -1;
   
   strcpy (a + alen, b);
   return 0;
}

   
static char *Sldxe_Library;
char *Sldxe_Root_Dir;

static int set_sldxe_root (char *r)
{
   char buf [DXE_MAX_PATH_LEN];
   
   if (r == NULL)
     r = getenv ("SLDXE_ROOT");

   if (r == NULL)
     {
	sldxe_verror ("SLDXE_ROOT environment variable not set");
	return -1;
     }
   
   r = SLang_create_slstring (r);
   if (r == NULL)
     return -1;
   
   if (Sldxe_Root_Dir == NULL)
     SLang_free_slstring (Sldxe_Root_Dir);

   Sldxe_Root_Dir = r;
   
   if (-1 == dircat (buf, r, "lib", sizeof (buf)))
     return -1;
   
   return sldxe_set_lib_path (buf);
}

int sldxe_set_lib_path (char *path)
{
   path = SLang_create_slstring (path);
   if (path == NULL)
     return -1;
   
   if (Sldxe_Library != NULL)
     SLang_free_slstring (Sldxe_Library);
   
   Sldxe_Library = path;
   return 0;
}

char *sldxe_get_lib_path (void)
{
   return Sldxe_Library;
}


static FILE *open_lib_file (char *file, char *libfsl, unsigned int len)
{
   char *lib, *type;
   FILE *fp;

   type = file_type(file);

   lib = Sldxe_Library;

   while (lib != NULL)
     {
	unsigned int n;
	char *next;

	if (*file == '/')
	  {
	     next = NULL;
	     *libfsl = 0;
	  }
	else
	  {	     
	     next = strchr (lib, ':');
	     if (next == NULL)
	       n = strlen (lib);
	     else
	       {
		  n = next - lib;
		  next++;
	       }

	     if (-1 == safe_strncpy (libfsl, lib, n, len))
	       continue;

	     if (-1 == fixup_dir(libfsl, len))
	       continue;
	  }

	if (-1 == safe_strcat(libfsl, file, len))
	  continue;

	if ((type == NULL) || (*type == 0))
	  if (-1 == safe_strcat (libfsl, ".sl", len))
	    continue;
	
	fp = fopen (libfsl, "r");
	if (fp != NULL)
	  {
	     sldxe_vmessage ("loading %s", libfsl);
	     return fp;
	  }
	
	lib = next;
     }
   
   SLang_verror (SL_OBJ_NOPEN, "Unable to open %s", file);
   return NULL;
}

#define DXE_LOAD_BUF_SIZE 512

typedef struct
{
   FILE *fp;
   char buf [DXE_LOAD_BUF_SIZE];
}
File_Load_Type;

static char *read_from_file (SLang_Load_Type *x)
{
   File_Load_Type *c;
   
   c = (File_Load_Type *) x->client_data;

   if (NULL == fgets (c->buf, DXE_LOAD_BUF_SIZE, c->fp))
     return NULL;

   return c->buf;
}

static int load_file (char *file)
{
   SLang_Load_Type *x;
   File_Load_Type client_data;
   char filename [DXE_MAX_PATH_LEN];
   int ret;
   FILE *fp;

   if (NULL == (fp = open_lib_file (file, filename, sizeof(filename))))
     return -1;

   ret = -1;
   if (NULL != (x = SLallocate_load_type (filename)))
     {
	client_data.fp = fp;
	x->read = read_from_file;
	x->client_data = (VOID_STAR) &client_data;
	ret = SLang_load_object (x);
     }
   
   SLdeallocate_load_type (x);
   fclose (fp);
   return ret;
}

int main (int argc, char **argv)
{
#if 0
   (void) plParseInternalOpts(&argc, argv, PL_PARSE_FULL);
#endif
   
   /* Initialize the library.  This is always needed. */
   
   if ((-1 == SLang_init_slang ())     /* basic interpreter functions */
       || (-1 == SLang_init_slmath ()) /* sin, cos, etc... */
#ifdef __unix__
       || (-1 == SLang_init_slunix ()) /* common unix system calls (stat, etc) */
#endif
       || (-1 == SLang_init_slfile ()) /* file I/O */
#if USE_PLPLOT
       || (-1 == sldxe_init_plplot ())
#endif
       || (-1 == sldxe_init_intrinsics ())
       )
     {
	fprintf(stderr, "Unable to initialize S-Lang.\n");
	exit (1);
     }

   if (-1 == open_readline ())
     sldxe_exit_error (1, "Unable to initialized readline.\n");
	   
   if (-1 == set_sldxe_root (NULL))
     sldxe_exit_error (1, "Error setting DXE_ROOT.\n");

   /* Turn on debugging */
   SLang_Traceback = 1;

   SLsignal (SIGPIPE, SIG_IGN);

   SLang_set_abort_signal (NULL);
#ifdef __linux__
   /* __setfpucw (0x1372); */
#endif

   SLang_Load_File_Hook = load_file;

   SLang_Traceback = 1;
   SLang_load_file ("sldxe");

   
   while (1)
     {
#ifdef SIGFPE
	if (Float_Exception)
	  {
	     SLang_doerror ("Floating point exception");
	     Float_Exception = 0;
	  }
	SLsignal (SIGFPE, floating_point_execption);
#endif
	if (SLang_Error) 
	  {
	     SLang_doerror (NULL);
	     SLang_restart(1);
	  }
	
	SLKeyBoard_Quit = SLang_Error = 0;
	SLang_load_object (Readline_Load_Object);
     }
   sldxe_exit_error (0, NULL);
   return 1;
}

