/* The usual include files */
#include "config.h"

#include <stdio.h>
#include <math.h>
#include <string.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif

#include <signal.h>
#include <unistd.h>

#include <slang.h>

#include <jdmath.h>
#include <_slang.h>

#include "sldxe.h"
#include "util.h"

#define SLDXE_OP_ISNAN		0x1000
#define SLDXE_OP_ISFINITE	0x1001
#define SLDXE_OP_ISINF		0x1002
#define SLDXE_OP_ERF		0x1003

#define SLDXE_OP_MIN		0x2000
#define SLDXE_OP_MAX		0x2001
#define SLDXE_OP_SUM		0x2002
#define SLDXE_OP_LEN		0x2003


#define NOT_READY 0

typedef int ATS_Fun_Type (VOID_STAR, unsigned int, VOID_STAR);

typedef struct
{
   unsigned char arg_type;
   unsigned char ret_type;
   ATS_Fun_Type *f;
}
Array_To_Scalar_Type;

static Array_To_Scalar_Type *
find_function (Array_To_Scalar_Type *funs, unsigned char arg_type)
{
   while (funs->arg_type != 0)
     {
	if (funs->arg_type == arg_type)
	  return funs;
	funs++;
     }

   SLang_verror (SL_TYPE_MISMATCH, "%s not supported by function",
		 SLclass_get_datatype_name (arg_type));
   return NULL;
}

/* If the top of the stack contains a scalar, it will apply the appropriate
 * function to the scalar and push it. The function returns 0 upon
 * sucess and -1 upon error. The arguments that are function
 * pointers, have the names: X_fun_Y where X and Y are either 'i'
 * (int) or d (double).
 */
static int 
array_to_scalar (Array_To_Scalar_Type *funs, unsigned int num_return_vals)
{
   SLang_Array_Type *at;
   int type;
   double arg_buf[8];		       /* should be sufficient for all other types */
   double ret_buf[8];
   unsigned int num;
   VOID_STAR arg_data;
   ATS_Fun_Type *f;
   int ret;

   at = NULL;
   num = 1;
   type = SLang_peek_at_stack ();
   arg_data = (VOID_STAR) arg_buf;

   if (num_return_vals > 4)
     {
	SLang_verror (SL_APPLICATION_ERROR, "%u values not implemented", num_return_vals);
	return -1;
     }

   switch (type)
     {
      case -1:		       /* Stack underflow */
	return -1;
	
      case SLANG_COMPLEX_TYPE:
	if (-1 == SLang_pop_complex (arg_buf, arg_buf + 1))
	  return -1;
	break;
	
      case SLANG_INT_TYPE:
	if (-1 == SLang_pop_integer ((int *) arg_data))
	  return -1;
	num = 1;
	break;
	
      case SLANG_DOUBLE_TYPE:
	if (-1 == SLang_pop_double (arg_buf, NULL, NULL))
	  return -1;
	break;
	
      case SLANG_ARRAY_TYPE:
      default:
	if (-1 == SLang_pop_array (&at, 0))
	  return -1;
	type = at->data_type;
	arg_data = at->data;
	num = at->num_elements;
	break;
     }

   ret = -1;
   if (NULL == (funs = find_function (funs, type)))
     goto return_error;
   
   f = funs->f;

   if (-1 == (*f)(arg_data, num, (VOID_STAR) ret_buf))
     goto return_error;
   
   switch (funs->ret_type)
     {
	unsigned int i;
	int *iret_buf;

      case SLANG_INT_TYPE:
	iret_buf = (int *) ret_buf;
	for (i = 0; i < num_return_vals; i++)
	  ret = SLang_push_integer (iret_buf[i]);
	break;
	
      case SLANG_DOUBLE_TYPE:
	for (i = 0; i < num_return_vals; i++)
	  ret = SLang_push_double (ret_buf[i]);
	break;
	
      case SLANG_COMPLEX_TYPE:
	for (i = 0; i < num_return_vals; i += 2)
	  ret = SLang_push_complex (ret_buf[i], ret_buf[i + 1]);
	break;
     }
   /* drop */
   return_error:

   if (at != NULL)
     SLang_free_array (at);

   return ret;
}

static int 
pop_matrix (SLang_Array_Type **at_ptr, unsigned int *nr, unsigned int *nc)
{
   SLang_Array_Type *at;

   if (-1 == SLang_pop_array (&at, 0))
     return -1;

   switch (at->num_dims)
     {
      case 0:
	*nr = *nc = 0;
	break; 
      case 1:
	*nr = (unsigned int)at->dims[0];
	*nc = 1;
	break;
      case 2:
	*nr = (unsigned int)at->dims[0];
	*nc = (unsigned int)at->dims[1];
	break;

      default:
	SLang_verror (SL_TYPE_MISMATCH, "operation limited to 2-d arrays");
	SLang_free_array (at);
	*at_ptr = NULL;
	return -1;
     }
   *at_ptr = at;
   return 0;
}


static int 
matrix_rows_to_vector (Array_To_Scalar_Type *funs)
{
   SLang_Array_Type *at, *bt;
   unsigned char type;
   unsigned int num_rows, num_cols;
   int d;
   char *a_data, *b_data;
   unsigned int sizeof_a, sizeof_b;
   unsigned int i;
   ATS_Fun_Type *f;

   switch (SLang_peek_at_stack ())
     {
      default:
	return array_to_scalar (funs, 1);

      case -1:		       /* Stack underflow */
	return -1;

      case SLANG_ARRAY_TYPE:
	if (-1 == pop_matrix (&at, &num_rows, &num_cols))
	  return -1;

	if (num_rows == 1)
	  {
	     if (-1 == SLang_push_array (at, 1))
	       return -1;
	     SLang_free_array (at);
	     return array_to_scalar (funs, 1);
	  }
	break;
     }

   type = at->data_type;

   if (NULL == (funs = find_function (funs, type)))
     {
	SLang_free_array (at);
	return -1;
     }
   
   d = (int) num_rows;
   bt = SLang_create_array (funs->ret_type, 0, NULL, &d, 1);
   if (bt == NULL)
     {
	SLang_free_array (at);
	return -1;
     }

   f = funs->f;
   
   a_data = (char *)at->data;
   b_data = (char *)bt->data;
   sizeof_a = at->sizeof_type * num_cols;
   sizeof_b = bt->sizeof_type;

   for (i = 0; i < num_rows; i++)
     {
	if (-1 == (*f) ((VOID_STAR) a_data, num_cols, (VOID_STAR) b_data))
	  goto return_error;
	
	a_data += sizeof_a;
	b_data += sizeof_b;
     }
   SLang_free_array (at);
   return SLang_push_array (bt, 1);

   return_error:

   SLang_free_array (at);
   SLang_free_array (bt);
   return -1;
}

static int 
matrix_cols_to_vector (Array_To_Scalar_Type *funs)
{
   SLang_Array_Type *at, *bt;
   unsigned char type;
   unsigned int num_rows, num_cols;
   unsigned int i, j;
   char *a_data, *b_data, *c_data;
   unsigned int sizeof_a, sizeof_b;
   int ret;
   int d;
   ATS_Fun_Type *f;

   switch (SLang_peek_at_stack ())
     {
      case -1:		       /* Stack underflow */
	return -1;
	
      default:
	return array_to_scalar (funs, 1);

      case SLANG_ARRAY_TYPE:
	if (-1 == pop_matrix (&at, &num_rows, &num_cols))
	  return -1;

	if (num_cols == 1)
	  {
	     if (-1 == SLang_push_array (at, 1))
	       return -1;
	     SLang_free_array (at);
	     return array_to_scalar (funs, 1);
	  }
	break;
     }
   
   type = at->data_type;
   if (NULL == (funs = find_function (funs, type)))
     {
	SLang_free_array (at);
	return -1;
     }
   
   sizeof_a = at->sizeof_type;

   c_data = SLmalloc (sizeof_a * num_rows + 1);   /* add 1 incase num_rows is 0 */
   if (c_data == NULL)
     {
	SLang_free_array (at);
	return -1;
     }

   d = (int) num_cols;
   bt = SLang_create_array (funs->ret_type, 0, NULL, &d, 1);
   if (bt == NULL)
     {
	SLfree (c_data);
	SLang_free_array (at);
	return -1;
     }

   ret = -1;

   a_data = (char *) at->data;
   b_data = (char *) bt->data;
   sizeof_b = bt->sizeof_type;
   
   f = funs->f;

   for (i = 0; i < num_cols; i++)
     {
	for (j = 0; j < num_rows; j++)
	  {
	     unsigned int j_sizeof_a = (unsigned int)j * sizeof_a;
	     memcpy (c_data + j_sizeof_a, 
		     a_data + num_cols * j_sizeof_a,
		     sizeof_a);
	  }
	if (-1 == (*f)((VOID_STAR) c_data, num_rows, (VOID_STAR) b_data))
	  goto return_error;
	
	a_data += sizeof_a;
	b_data += sizeof_b;
     }

   ret = SLang_push_array (bt, 0);
   /* drop */

   return_error:
   if (ret == -1) SLang_free_array (bt);   
   /* bt is not to be freed since it was created and not popped.
    * If the push failed, then free it.
    */

   SLang_free_array (at);
   SLfree (c_data);
   return ret;
}

static int 
array_length_function (int what_dim)   /* -1 ==> total_length */
{
   SLang_Array_Type *at;
   unsigned int len;
   char *s;
   
   len = 1;			       /* default value */
   switch (SLang_peek_at_stack ())
     {
      default:
	SLdo_pop ();
	break;
	
      case SLANG_STRING_TYPE:

	if (-1 == SLang_pop_slstring (&s))
	  return -1;

	if (what_dim <= 0)
	  len = strlen (s);
	SLang_free_slstring (s);
	break;
	
      case SLANG_ARRAY_TYPE:
	if (-1 == SLang_pop_array (&at, 0))
	  return -1;

	if (what_dim < 0)
	  len = at->num_elements;
	else if ((unsigned int) what_dim < at->num_dims)
	  len = at->dims [len];
	SLang_free_array (at);
	break;
     }

   return (int) len;
}

static int 
array_length (void)
{
   return array_length_function (-1);
}

static int 
array_length_rows (void)
{
   return array_length_function (0);
}

static int 
array_length_cols (void)
{
   return array_length_function (1);
}


static int 
sum_integers (int *i, unsigned int num, double *s)
{
   double sum;
   unsigned int n;

   sum = 0;
   for (n = 0; n < num; n++)
     sum += i[n];
   *s = sum;
   return 0;
}

static int 
isum_integers (int *i, unsigned int num, int *s)
{
   int sum;
   unsigned int n;

   sum = 0;
   for (n = 0; n < num; n++)
     sum += i[n];
   *s = sum;
   return 0;
}

static int 
mean_stddev_doubles (double *x, unsigned int num, double *s)
{
   unsigned int i;
   double mean_i, variance_i;
   
   mean_i = variance_i = 0.0;
   i = 0;
   while (i < num)
     {
	double diff, x_i;
	
	x_i = x[i];
	diff = x_i - mean_i;
	i++;
	mean_i += diff / i;
	variance_i += diff * (x_i - mean_i);
     }
   
   s[0] = mean_i;
   if (num > 1)
     s[1] = sqrt (variance_i / (num - 1));
   else
     s[1] = 0;

   return 0;
}


static int 
sum_doubles (double *d, unsigned int num, double *s)
{
   double sum;
   unsigned int n;

   sum = 0;
   for (n = 0; n < num; n++)
     sum += d[n];
   *s = sum;
   return 0;
}

static int 
sum_complex (double *z, unsigned int num, double *s)
{
   double *zmax;
   double sr, si;
   
   zmax = z + 2 * num;
   sr = si = 0.0;
   while (z < zmax)
     {
	sr += *z++;
	si += *z++;
     }
   *s++ = sr;
   *s = si;
   return 0;
}

static int 
min_integers (int *i, unsigned int num, int *s)
{
   unsigned int n;
   int m;

   if (num == 0)
     {
	SLang_verror (SL_INVALID_PARM, "min: array is empty");
	return -1;
     }
   
   m = i[0];

   for (n = 1; n < num; n++)
     if (m > i[n]) m = i[n];
   
   *s = m;
   return 0;
}

static int 
max_integers (int *i, unsigned int num, int *s)
{
   unsigned int n;
   int m;
   
   if (num == 0)
     {
	SLang_verror (SL_INVALID_PARM, "max: array is empty");
	return -1;
     }

   m = i[0];

   for (n = 1; n < num; n++)
     if (m < i[n]) m = i[n];
   
   *s = m;
   return 0;
}

static int 
min_doubles (double *i, unsigned int num, double *s)
{
   unsigned int n;
   double m;
   
   if (num == 0)
     {
	SLang_verror (SL_INVALID_PARM, "min: array is empty");
	return -1;
     }

   m = i[0];

   for (n = 1; n < num; n++)
     if (m > i[n]) m = i[n];
   
   *s = m;
   return 0;
}

static int 
max_doubles (double *i, unsigned int num, double *s)
{
   unsigned int n;
   double m;
   
   if (num == 0)
     {
	SLang_verror (SL_INVALID_PARM, "max: array is empty");
	return -1;
     }

   m = i[0];

   for (n = 1; n < num; n++)
     if (m < i[n]) m = i[n];
   
   *s = m;
   return 0;
}

static int 
sumsq_integers (int *a, unsigned int num, double *s)
{
   unsigned int i;
   double sum = 0;
   
   for (i = 0; i < num; i++)
     sum += a[i] * a[i];
   
   *s = sum;
   return 0;
}

static int 
sumsq_doubles (double *a, unsigned int num, double *s)
{
   unsigned int i;
   double sum = 0;
   
   for (i = 0; i < num; i++)
     sum += a[i] * a[i];
   
   *s = sum;
   return 0;
}

static int 
sumsq_complex (double *a, unsigned int num, double *s)
{
   double sr, si, z[2];
   double *amax;

   sr = si = 0;
   
   amax = a + 2 * num;

   while (a < amax)
     {
	SLcomplex_times (z, a, a);
	sr += z[0];
	si += z[1];
	a += 2;
     }
   s[0] = sr;
   s[1] = si;
   return 0;
}


static Array_To_Scalar_Type Array_Sum_Funs [] = 
{
     {SLANG_INT_TYPE, SLANG_DOUBLE_TYPE, (ATS_Fun_Type *) sum_integers},
     {SLANG_DOUBLE_TYPE, SLANG_DOUBLE_TYPE, (ATS_Fun_Type *) sum_doubles},
     {SLANG_COMPLEX_TYPE, SLANG_COMPLEX_TYPE, (ATS_Fun_Type *) sum_complex},
     {0, 0, NULL}
};


static void 
array_sum_rows (void)
{
   (void) matrix_rows_to_vector (Array_Sum_Funs);
}


static void 
array_sum_cols (void)
{
   (void) matrix_cols_to_vector (Array_Sum_Funs);
}


static void 
array_sum (void)
{
   (void) array_to_scalar (Array_Sum_Funs, 1);
}

static Array_To_Scalar_Type Array_Sumsq_Funs [] =
{
     {SLANG_INT_TYPE, SLANG_DOUBLE_TYPE, (ATS_Fun_Type *) sumsq_integers},
     {SLANG_DOUBLE_TYPE, SLANG_DOUBLE_TYPE, (ATS_Fun_Type *) sumsq_doubles},
     {SLANG_COMPLEX_TYPE, SLANG_COMPLEX_TYPE, (ATS_Fun_Type *) sumsq_complex},
     {0, 0, NULL}
};
   

static void 
array_sumsq_cols (void)
{
   (void) matrix_cols_to_vector (Array_Sumsq_Funs);
}

static void 
array_sumsq_rows (void)
{
   (void) matrix_rows_to_vector (Array_Sumsq_Funs);
}

static void 
array_sumsq (void)
{
   (void) array_to_scalar (Array_Sumsq_Funs, 1);
}


static Array_To_Scalar_Type Array_Min_Funs [] = 
{
     {SLANG_INT_TYPE, SLANG_INT_TYPE, (ATS_Fun_Type *) min_integers},
     {SLANG_DOUBLE_TYPE, SLANG_DOUBLE_TYPE, (ATS_Fun_Type *) min_doubles},
     {0, 0, NULL}
};
   
static void 
array_min (void)
{
   (void) array_to_scalar (Array_Min_Funs, 1);
}

static void 
array_min_rows (void)
{
   (void) matrix_rows_to_vector (Array_Min_Funs);
}

static void 
array_min_cols (void)
{
   (void) matrix_cols_to_vector (Array_Min_Funs);
}

static Array_To_Scalar_Type Array_Max_Funs [] =
{
     {SLANG_INT_TYPE, SLANG_INT_TYPE, (ATS_Fun_Type *) max_integers},
     {SLANG_DOUBLE_TYPE, SLANG_DOUBLE_TYPE, (ATS_Fun_Type *) max_doubles},
     {0, 0, NULL}
};

static void 
array_max (void)
{
   (void) array_to_scalar (Array_Max_Funs, 1);
}

static void 
array_max_rows (void)
{
   (void) matrix_rows_to_vector (Array_Max_Funs);
}

static void 
array_max_cols (void)
{
   (void) matrix_cols_to_vector (Array_Max_Funs);
}

static Array_To_Scalar_Type Array_Mean_Stddev_Funs [] =
{
     {SLANG_DOUBLE_TYPE, SLANG_DOUBLE_TYPE, (ATS_Fun_Type *) mean_stddev_doubles},
     {0, 0, NULL}
};

static void mean_stddev (void)
{
   (void) array_to_scalar (Array_Mean_Stddev_Funs, 2);
}

static void create_and_init_double_array (double (*fun)(void))
{
   unsigned int i, num;
   double *d;
   SLang_Array_Type *at;

   if (SLang_Num_Function_Args == 0)
     {
	SLang_push_double ((*fun)());
	return;
     }
   
   /* FIXME!!: This hack should not be required */
   SLang_Num_Function_Args += 1;
   if (-1 == _SLang_push_datatype (SLANG_DOUBLE_TYPE))
     return;
   
   if (-1 == _SLarray_aget ())
     return;
   
   at = sldxe_pop_double_array (1);
   if (at == NULL)
     return;
   
   d = (double *) at->data;
   num = at->num_elements;
   
   for (i = 0; i < num; i++)
     d[i] = (*fun)();
   
   SLang_push_array (at, 0);
   SLang_free_array (at);
}

static void array_urand (void)
{
   create_and_init_double_array (JDMrandom);
}

static void array_grand (void)
{
   create_and_init_double_array (JDMgaussian_random);
}

static void array_hypot (void)
{
   SLang_Array_Type *at, *bt, *ct;
   double *a, *b, *c;
   double x;
   unsigned int i, imax;

   at = bt = ct = NULL;

   if (SLANG_ARRAY_TYPE == SLang_peek_at_stack ())
     {
	if (NULL == (bt = sldxe_pop_double_array (0)))
	  return;
	
	ct = SLang_create_array (bt->data_type, 0, NULL, bt->dims, bt->num_dims);
	if (bt == NULL)
	  goto return_error;

	imax = bt->num_elements;
	b = (double *) bt->data;
	c = (double *) ct->data;

	if (SLANG_ARRAY_TYPE != SLang_peek_at_stack ())
	  {
	     if (-1 == SLang_pop_double (&x, NULL, NULL))
	       goto return_error;
	     
	     for (i = 0; i < imax; i++)
	       c[i] = SLmath_hypot (x, b[i]);
	  }
	else
	  {
	     if (NULL == (at = sldxe_pop_double_array (0)))
	       goto return_error;
	     
	     if (at->num_elements != imax)
	       {
		  SLang_verror (SL_TYPE_MISMATCH, "hypot: arrays do not match in size");
		  goto return_error;
	       }
	     a = (double *) at->data;
	     for (i = 0; i < imax; i++)
	       c[i] = SLmath_hypot (a[i], b[i]);
	  }
     }
   else
     {
	if (-1 == SLang_pop_double (&x, NULL, NULL))
	  return;
	
	if (SLANG_ARRAY_TYPE != SLang_peek_at_stack ())
	  {
	     double y;
	     if (0 == SLang_pop_double (&y, NULL, NULL))
	       (void) SLang_push_double (SLmath_hypot (y, x));
	     return;
	  }
	
	if (NULL == (bt = sldxe_pop_double_array (0)))
	  return;
	
	ct = SLang_create_array (bt->data_type, 0, NULL, bt->dims, bt->num_dims);
	if (ct == NULL)
	  goto return_error;
	
	b = (double *) bt->data;
	c = (double *) ct->data;
	imax = bt->num_elements;
	for (i = 0; i < imax; i++) c[i] = SLmath_hypot (b[i], x);
     }

   /* We get here only if we have an array to push */
   (void) SLang_push_array (ct, 1);  
   ct = NULL;
	     
   /* drop */
   return_error:
   if (ct != NULL) SLang_free_array (ct);
   if (bt != NULL) SLang_free_array (bt);
   if (at != NULL) SLang_free_array (at);
}

   
static void array_histogram (void)
{
   SLang_Array_Type *a, *b;
   double *a_data;
   double min, max;
   double binsize;
   unsigned int num_pts;
   int requires_min_max;
   int num_bin_pts;
   
   /* Usage: array_histogram (array, binsize, min, max);
    *        array_histogram (array, binsize);
    *        array_histogram (array);
    */
   
   binsize = 1.0;
   requires_min_max = 1;

   switch (SLang_Num_Function_Args)
     {
      default:
	SLang_verror (SL_INVALID_PARM, "Usage: histogram(a, binsize [,min, max]);");
	return;

      case 4:
	if (SLang_pop_double (&max, NULL, NULL)
	    || (SLang_pop_double (&min, NULL, NULL)))
	  return;
	if (min > max)
	  {
	     double tmp = max;
	     max = min;
	     min = tmp;
	  }
	requires_min_max = 0;
      case 2:
	if (SLang_pop_double (&binsize, NULL, NULL))
	  return;
      case 1:
	if (NULL == (a = sldxe_pop_double_array (0)))
	  return;
     }
   
   if (binsize <= 0.0)
     {
	SLang_verror (SL_INVALID_PARM, "Binsize must be > 0");
	SLang_free_array (a);
	return;
     }
   
   num_pts = a->num_elements;
   a_data = (double *) a->data;

   (void) min_doubles (a_data, num_pts, &min);
   (void) max_doubles (a_data, num_pts, &max);
   
   num_bin_pts = (int) (1.0 + ((max - min) / binsize));
   if (NULL == (b = SLang_create_array (SLANG_INT_TYPE, 0, NULL, &num_bin_pts, 1)))
     {
	SLang_free_array (a);
	return;
     }

   (void) JDMhistogram_d (a_data, num_pts,
			  (unsigned int *)b->data, num_bin_pts,
			  min, max);

   (void) SLang_push_array (b, 1);
   SLang_free_array (a);
}


static void array_sort (void)
{
   SLang_Array_Type *a;
   double *dat;
   int *indices;
   int len;

   a = sldxe_pop_double_array (0);
   if (a == NULL)
     return;
   
   len = (int)a->num_elements;
   dat = (double *) a->data;
   
   indices = (int *) JDMsort_doubles (dat, (unsigned int)len);
   SLang_free_array (a);
   
   if (indices == 0)
     {
	SLang_Error = SL_MALLOC_ERROR;
	return;
     }
   
   a = SLang_create_array (SLANG_INT_TYPE, 0, (VOID_STAR) indices, &len, 1);
   if (a == NULL)
     return;
   
   SLang_push_array (a, 1);
}

/* Three arrays are expected with usage like:
 *    new_y = interpol (new_x, old_x, old_y);
 */
static void 
array_interp (void)
{	     
   SLang_Array_Type *x, *y, *new_x, *new_y;
   unsigned int new_npts, npts;
   
   x = y = new_x = new_y = NULL;
   
   if (-1 == sldxe_pop_2_double_arrays (&x, &y, 0))
     return;

   if (NULL == (new_x = sldxe_pop_double_array (0)))
     goto return_error;

   npts = x->num_elements;

   if (npts < 2)
     {
	SLang_doerror ("Need at least two points for interpolation.");
	goto return_error;
     }
   
   if (NULL == (new_y = SLang_create_array (SLANG_DOUBLE_TYPE, 0, NULL,
					    new_x->dims, new_x->num_dims)))
     goto return_error;
   
   new_npts = new_x->num_elements;
   
   (void) JDMinterpolate_dvector ((double *)new_x->data, (double *)new_y->data, new_npts,
				  (double *)x->data, (double *)y->data, npts);
   
   SLang_push_array (new_y, 1);
   new_y = NULL;
   /* drop */

   return_error:
   
   if (x != NULL) SLang_free_array (x);
   if (y != NULL) SLang_free_array (y);
   if (new_x != NULL) SLang_free_array (new_x);
   if (new_y != NULL) SLang_free_array (new_y);
}
   

static void array_readcol (void)
{  
   int col_index [20];
   double *data[20];
   int nargs;
   int i, j;
   char *file;
   int max_col, nread;

   if ((nargs = SLang_Num_Function_Args) < 2)
     {
	SLang_doerror ("Usage: readcol(file, col, ...);");
	return;
     }
   
   memset ((char *) data, 0, sizeof (data));
   memset ((char *) col_index, 0, sizeof (col_index));
   max_col = 0;
   
   for (i = 1; i < nargs; i++)
     {
	int col;
	
	if (SLang_pop_integer (&col))
	  return;
	
	if ((col <= 0) || (col > 20))
	  {
	     SLang_doerror ("Column must be <= 20");
	     return;
	  }
	/* Set it to nargs - i so that we can push the arrays in the
	 * same order
	 */
	col_index [col - 1] = (nargs - i);
	if (col > max_col) max_col = col;
     }
   
   if (SLpop_string (&file))
     return;

   nread = JDMread_column_ddata (file, data, col_index, max_col);
   SLfree (file);
   
   if (nread == -1)
     {
	SLang_doerror ("Error reading file");
	return;
     }

   if (nread == 0)
     {
	SLang_doerror ("File contains no rows.");
	return;
     }
   

   j = 1;
   while (j < nargs)
     {
	for (i = 0; i < max_col; i++)
	  {
	     SLang_Array_Type *at;


	     if (col_index[i] != j)
	       continue;
	     
	     at = SLang_create_array (SLANG_DOUBLE_TYPE, 0, (VOID_STAR) data[i],
				      &nread, 1);
	     if (at != NULL)
	       {
		  SLang_push_array (at, 1);
		  data [i] = NULL;
		  j++;
		  break;
	       }
	     
	     for (i = 0; i < max_col; i++)
	       {
		  if (data[i] != NULL)
		    SLfree ((char *)data[i]);
	       }
	     return;
	  }
     }
}
   

/* File routines */
 
#define FILE_BYTE_TYPE		1
#define FILE_SHORT_TYPE		2
#define FILE_INT_TYPE		3
#define FILE_LONG_TYPE		4
#define FILE_INT16_TYPE		5
#define FILE_INT32_TYPE		6
#define FILE_FLOAT_TYPE		7
#define FILE_FLOAT32_TYPE	8
#define FILE_FLOAT64_TYPE	9
#define FILE_DOUBLE_TYPE	10
#define FILE_STRING_TYPE	11

static int read_from_file (int type, unsigned int sizeof_type)
{
   int number, number_requested;
   char *buf;
   unsigned int buf_len;
   unsigned int nread;
   int i_nread;
   SLang_Array_Type *at;
   SLang_MMT_Type *mmt;
   FILE *fp;

   if (SLang_pop_integer (&number_requested)
       || (-1 == SLang_pop_fileptr (&mmt, &fp)))
     return -1;

   buf = NULL;

   number = number_requested;

   /* number <= 0 means that read to EOF */
   if (number <= 0)
     {
	long here, end;
	long num_bytes;
	char *err1 = "ftell failed.";
	char *err2 = "fseek failed.";
	
	here = ftell (fp);
	
	if (here == -1)
	  {
	     SLang_doerror (err1);
	     goto return_error;
	  }
	if (-1 == fseek (fp, 0L, SEEK_END))
	  {
	     SLang_doerror (err2);
	     goto return_error;
	  }
	if (-1 == (end = ftell (fp)))
	  {
	     SLang_doerror (err1);
	     goto return_error;
	  }
	if (-1 == fseek (fp, here, SEEK_SET))
	  {
	     SLang_doerror (err2);
	     goto return_error;
	  }
	num_bytes = end - here;
	number = num_bytes / sizeof_type;
     }
   
   buf_len = sizeof_type * number;
   if (NULL == (buf = SLmalloc (buf_len + 1)))
     goto return_error;

   buf[buf_len] = 0;
   
   nread = fread (buf, sizeof_type, number, fp);
   if ((nread == 0) && (number_requested == 1))
     {
	SLang_verror  (INTRINSIC_ERROR, "No data to read from file");
	goto return_error;
     }

   SLang_free_mmt (mmt); mmt = NULL;
   
   i_nread = (int) nread;
   at = NULL;
   switch (type)
     {
	int *ibuf;
	double *dbuf;
	float *fbuf;
	short *sbuf;
	unsigned int i;

      case FILE_STRING_TYPE:
	SLang_push_malloced_string (buf);
	return 0;

      case FILE_INT32_TYPE:
	JDMstr_read_int32 ((int32 *)buf, nread, (unsigned char *)buf);
      case FILE_INT_TYPE:
	if (number_requested == 1)
	  {
	     SLang_push_integer (*(int *) buf);
	     SLfree (buf);
	     return 0;
	  }
	at = SLang_create_array (SLANG_INT_TYPE, 0, (VOID_STAR)buf, &i_nread, 1);
	break;

      case FILE_FLOAT64_TYPE:
	JDMstr_read_float64 ((float64 *)buf, nread, (unsigned char *)buf);
      case FILE_DOUBLE_TYPE:
	if (number_requested == 1)
	  {
	     SLang_push_double (*(float *) buf);
	     SLfree (buf);
	     return 0;
	  }
	at = SLang_create_array (SLANG_DOUBLE_TYPE, 0, (VOID_STAR)buf, &i_nread, 1);
	break;

      case FILE_BYTE_TYPE:
	if (number_requested == 1)
	  {
	     SLang_push_integer (*buf);
	     SLfree (buf);
	     return 0;
	  }
	at = SLang_create_array (SLANG_INT_TYPE, 0, NULL, &i_nread, 1);
	if (at == NULL) break;
	ibuf = (int *) at->data;
	for (i = 0; i < nread; i++)
	  ibuf[i] = (signed char) buf[i];
	SLfree (buf);
	buf = NULL;
	break;

      case FILE_INT16_TYPE:
	JDMstr_read_int16 ((int16 *)buf, nread, (unsigned char *)buf);
      case FILE_SHORT_TYPE:
	if (number_requested == 1)
	  {
	     SLang_push_integer (*(short *) buf);
	     SLfree (buf);
	     return 0;
	  }
	
	at = SLang_create_array (SLANG_INT_TYPE, 0, NULL, &i_nread, 1);
	if (at == NULL)
	  break;
	ibuf = (int *) at->data;
	sbuf = (short *) buf;
	for (i = 0; i < nread; i++)
	  ibuf[i] = sbuf[i];
	SLfree (buf);
	buf = NULL;
	break;

      case FILE_FLOAT32_TYPE:
	JDMstr_read_float32 ((float32 *)buf, nread, (unsigned char *)buf);
      case FILE_FLOAT_TYPE:
	if (number_requested == 1)
	  {
	     SLang_push_double (*(float *) buf);
	     SLfree (buf);
	     return 0;
	  }
	at = SLang_create_array (SLANG_DOUBLE_TYPE, 0, NULL, &i_nread, 1);
	if (at == NULL) break;
	dbuf = (double *) at->data;
	fbuf = (float *) buf;
	for (i = 0; i < nread; i++)
	  dbuf[i] = fbuf[i];
	SLfree (buf);
	buf = NULL;
	break;

      case FILE_LONG_TYPE:
      default:
	at = NULL;
	SLang_doerror ("file read not implemnted for LONG type");
     }
   
   if (at != NULL)
     return SLang_push_array (at, 1);
   
   return_error:
   
   if (buf != NULL) SLfree (buf);
   if (mmt != NULL) SLang_free_mmt (mmt);
   return -1;
}

static void fread_float (void)
{	
   read_from_file (FILE_FLOAT_TYPE, sizeof (float));
}

static void fread_double (void)
{	
   read_from_file (FILE_DOUBLE_TYPE, sizeof (double));
}

static void fread_float32 (void)
{	
   read_from_file (FILE_FLOAT32_TYPE, sizeof (float32));
}

static void fread_float64 (void)
{
   read_from_file (FILE_FLOAT64_TYPE, sizeof (float64));
}

static void fread_int (void)
{
   read_from_file (FILE_INT_TYPE, sizeof (int));
}

static void fread_int16 (void)
{
   read_from_file (FILE_INT16_TYPE, sizeof (int16));
 }

static void fread_int32 (void)
{
   read_from_file (FILE_INT32_TYPE, sizeof (int32));
}

static void fread_bytes (void)
{
   read_from_file (FILE_BYTE_TYPE, 1);
}

static void fread_string (void)
{
   read_from_file (FILE_STRING_TYPE, 1);
}

static void fwrite_float_types (int type)
{
   FILE *fp;
   SLang_MMT_Type *mmt;
   SLang_Array_Type *a;
   double *xdat;
   unsigned int len;
   unsigned int nwrite;

   if (-1 == SLang_pop_fileptr (&mmt, &fp))
     return;

   if (NULL == (a = sldxe_pop_double_array (0)))
     goto return_error;
   
   xdat = (double *) a->data;
   len = a->num_elements;
   
   nwrite = 0;
   switch (type)
     {
      case FILE_FLOAT_TYPE:
	for (nwrite = 0; nwrite < len; nwrite++)
	  {
	     float f;
	     f = xdat[nwrite];
	     if (1 != fwrite ((char *)&f, sizeof (float), 1, fp))
	       break;
	  }
	break;
	
      case FILE_DOUBLE_TYPE:
	nwrite = fwrite ((char *)xdat, sizeof(double), len, fp);
	break;
	
      case FILE_FLOAT32_TYPE:
	nwrite = JDMwrite_d_float32 (xdat, len, fp);
	break;
	
      case FILE_FLOAT64_TYPE:
	nwrite = JDMwrite_d_float64 (xdat, len, fp);
	break;
     }
   
   if (nwrite != len)
     SLang_doerror ("Write error.  Not all data written");
   
   /* drop */

   return_error:
   if (a != NULL) SLang_free_array (a);
   if (mmt != NULL) SLang_free_mmt (mmt);
}

static void fwrite_float (void)
{   
   fwrite_float_types (FILE_FLOAT_TYPE);
}
static void fwrite_float32 (void)
{   
   fwrite_float_types (FILE_FLOAT32_TYPE);
}
static void fwrite_float64 (void)
{   
   fwrite_float_types (FILE_FLOAT64_TYPE);
}
static void fwrite_double (void)
{   
   fwrite_float_types (FILE_DOUBLE_TYPE);
}


/* Everthing below here is standard stuff.  It consists of the intrinsic
 * table and main.
 */

/* Function to quit */
static void c_quit (void)
{
   sldxe_exit_error (SLang_Error, NULL);
}


static void print_array (void)
{
   SLang_Array_Type *at;
   unsigned int i, num_rows, num_cols;
   int *idat;
   double *ddat;
   unsigned char type;
   FILE *fp;
   int just_one_line;

   if (-1 == pop_matrix (&at, &num_rows, &num_cols))
     return;
   
   type = at->data_type;
   switch (type)
     {
      case SLANG_INT_TYPE:
      case SLANG_DOUBLE_TYPE:
      case SLANG_COMPLEX_TYPE:
	break;

      default:
	SLang_verror (SL_TYPE_MISMATCH, "print_array: %s is not supported",
		      SLclass_get_datatype_name (type));
	SLang_free_array (at);
	return;
     }
	
   fp = NULL;

   if (num_rows > 24)
     {
	char *pager;
	
	pager = getenv ("PAGER");
	if (pager == NULL) pager = "more";
	fp = sldxe_popen (pager, "w");
     }

   if (fp == NULL) fp = stdout;
   
   idat = (int *)at->data;
   ddat = (double *)at->data;

   just_one_line = 0;
   for (i = 0; i < num_rows; i++)
     {
	unsigned int j;

	for (j = 0; j < num_cols; j++)
	  {
	     switch (type)
	       {
		case SLANG_INT_TYPE:
		  if (0 >= fprintf (fp, "%d\t", *idat))
		    goto done;
		  idat++;
		  break;
		  
		case SLANG_DOUBLE_TYPE:
		  if (0 >= fprintf (fp, "%e  ", *ddat))
		    goto done;
		  ddat++;
		  break;
	     
		case SLANG_COMPLEX_TYPE:
		  if (0 >= fprintf (fp, "(%e, %e)  ", ddat[0], ddat[1]))
		    goto done;
		  ddat += 2;
		  break;
	       }
	  }
	
	if (0 >= fputs ("\n", fp))
	  break;

	if ((fp == stdout) 
	    && (((0 == (num_rows % 24)) && (num_rows != 0))
		|| just_one_line))
	  {
	     unsigned int key;
	     
	     if (just_one_line == 0)
	       fprintf (stdout, "Press SPACE to continue");
	     fflush (stdout);
	     
	     key = sldxe_getkey ();
	     if (key == ' ')
	       just_one_line = 0;
	     else if (key == '\r')
	       just_one_line = 1;
	     else break;
	  }
     }
   
   done:

   if (fp != stdout)
     sldxe_pclose (fp);
   
   fputs ("\n", stdout);
   SLang_free_array (at);
}

#if 0
/* This is very dirty and should be thrown out in place of something more
 * sophisticated.
 */
static void smear (void)
{
   SLuser_Object_Type *ux, *uy;
   SLArray_Type *x, *y;
   unsigned int npts;
   double dx;
   unsigned int new_npts;
   double min, max;
   double *xdat, *ydat;

   if (SLang_pop_float (&dx, NULL, NULL))
     return;
   
   if (dx <= 0.0)
     {
	SLang_doerror ("smear requires a positive dx");
	return;
     }
   
   if (sldxe_pop_2_float_arrays (&ux, &uy))
     return;
   
   x = (SLArray_Type *)ux->obj;
   y = (SLArray_Type *)uy->obj;
   
   npts = x->x * x->y * x->z;
   if (npts < 2)
     {
	SLang_doerror ("smear requires at least 2 points.");
	goto return_error;
     }
   
   xdat = x->buf.f_ptr;
   ydat = x->buf.f_ptr;
   
   max = farray_max_fun (xdat->npts);
   min = farray_min_fun (xdat->npts);
   
   npts =
   return_error:

   SLang_free_user_object (ux);
   SLang_free_user_object (uy);
}

#endif

static char *get_lib_load_path (void)
{
   char *path;

   if (NULL == (path = sldxe_get_lib_path ()))
     path = "";
   
   return path;
}

static void set_lib_load_path (void)
{
   char *path;

   if (SLpop_string (&path))
     return;
   
   sldxe_set_lib_path (path);
   SLfree (path);
}

static char *get_sldxe_root (void)
{
   if (Sldxe_Root_Dir == NULL)
     return "";
   return Sldxe_Root_Dir;
}


static int array_unary_op_result (int op, unsigned char a, unsigned char *b)
{
   switch (op)
     {
      default:
	return 0;
	
      case SLDXE_OP_ISNAN:
      case SLDXE_OP_ISINF:
      case SLDXE_OP_ISFINITE:
	switch (a)
	  {
	   case SLANG_COMPLEX_TYPE:
	   case SLANG_INT_TYPE:
	   case SLANG_DOUBLE_TYPE:
	     *b = SLANG_INT_TYPE;
	     break;
	     
	   default:
	     return 0;
	  }
     }
   return 1;
}

static void 
array_complex_isxxx (double *z, unsigned int na, int *i, int (*xxx)(double))
{
   unsigned int n;
   
   for (n = 0; n < na; n++)
     {
	i[n] = ((*xxx)(z[0]) || (*xxx)(z[1]));
	z += 2;
     }
}

static void 
array_double_isxxx (double *z, unsigned int na, int *i, int (*xxx)(double))
{
   unsigned int n;
   
   for (n = 0; n < na; n++)
     i[n] = (*xxx)(z[n]);
}

static void 
array_int_isxxx (int *i, unsigned int na, int val)
{
   unsigned int n;
   
   for (n = 0; n < na; n++) i[n] = val;
}

static int array_unary_op (int op, 
			   unsigned char a, VOID_STAR ap, unsigned int na,
			   VOID_STAR bp)
{
   switch (a)
     {
      default:
	return 0;

      case SLANG_COMPLEX_TYPE:
	switch (op)
	  {
	   case SLDXE_OP_ISNAN:
	     array_complex_isxxx ((double *) ap, na, (int *) bp, JDMisnan);
	     break;
	   case SLDXE_OP_ISINF:
	     array_complex_isxxx ((double *) ap, na, (int *) bp, JDMisinf);
	     break;
	   case SLDXE_OP_ISFINITE:
	     array_complex_isxxx ((double *) ap, na, (int *) bp, JDMfinite);
	     break;
	   default:
	     return 0;
	  }
	break;
	
      case SLANG_DOUBLE_TYPE:
	switch (op)
	  {
	   case SLDXE_OP_ISNAN:
	     array_double_isxxx ((double *) ap, na, (int *) bp, JDMisnan);
	     break;
	   case SLDXE_OP_ISINF:
	     array_double_isxxx ((double *) ap, na, (int *) bp, JDMisinf);
	     break;
	   case SLDXE_OP_ISFINITE:
	     array_double_isxxx ((double *) ap, na, (int *) bp, JDMfinite);
	     break;
	   default:
	     return 0;
	  }
	break;
	
      case SLANG_INT_TYPE:
	switch (op)
	  {
	   case SLDXE_OP_ISNAN:
	   case SLDXE_OP_ISINF:
	     array_int_isxxx ((int *) bp, na, 0);
	     break;

	   case SLDXE_OP_ISFINITE:
	     array_int_isxxx ((int *) bp, na, 1);
	     break;

	   default:
	     return 0;
	  }
	break;
     }
   
   return 1;
}

static SLang_Intrin_Fun_Type DXE_Intrinsics[] =
{
   MAKE_INTRINSIC("print_array", print_array, VOID_TYPE, 0),
   MAKE_INTRINSIC("quit", c_quit, VOID_TYPE, 0),

   MAKE_INTRINSIC("mean_stddev", mean_stddev, VOID_TYPE, 0),

   MAKE_INTRINSIC("length", array_length, INT_TYPE, 0),
   MAKE_INTRINSIC("length_rows", array_length_rows, INT_TYPE, 0),
   MAKE_INTRINSIC("length_cols", array_length_cols, INT_TYPE, 0),
   MAKE_INTRINSIC("sumsq", array_sumsq, VOID_TYPE, 0),
   MAKE_INTRINSIC("sumsq_rows", array_sumsq_rows, VOID_TYPE, 0),
   MAKE_INTRINSIC("sumsq_cols", array_sumsq_cols, VOID_TYPE, 0),
   MAKE_INTRINSIC("sum", array_sum, VOID_TYPE, 0),
   MAKE_INTRINSIC("sum_rows", array_sum_rows, VOID_TYPE, 0),
   MAKE_INTRINSIC("sum_cols", array_sum_cols, VOID_TYPE, 0),
   MAKE_INTRINSIC("min", array_min, VOID_TYPE, 0),
   MAKE_INTRINSIC("min_rows", array_min_rows, VOID_TYPE, 0),
   MAKE_INTRINSIC("min_cols", array_min_cols, VOID_TYPE, 0),
   MAKE_INTRINSIC("max", array_max, VOID_TYPE, 0),
   MAKE_INTRINSIC("max_rows", array_max_rows, VOID_TYPE, 0),
   MAKE_INTRINSIC("max_cols", array_max_cols, VOID_TYPE, 0),

   MAKE_INTRINSIC("histogram", array_histogram, VOID_TYPE, 0),
   MAKE_INTRINSIC("urand", array_urand, VOID_TYPE, 0),
   MAKE_INTRINSIC("grand", array_grand, VOID_TYPE, 0),

   MAKE_INTRINSIC("hypot", array_hypot, VOID_TYPE, 0),
   MAKE_INTRINSIC("sort", array_sort, VOID_TYPE, 0),
   MAKE_INTRINSIC("readcol", array_readcol, VOID_TYPE, 0),
   MAKE_INTRINSIC("interpol", array_interp, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_bytes", fread_bytes, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_string", fread_string, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_int32", fread_int32, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_int16", fread_int16, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_int", fread_int, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_float", fread_float, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_double", fread_double, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_float32", fread_float32, VOID_TYPE, 0),
   MAKE_INTRINSIC("fread_float64", fread_float64, VOID_TYPE, 0),
   MAKE_INTRINSIC("fwrite_float64", fwrite_float64, VOID_TYPE, 0),
   MAKE_INTRINSIC("fwrite_float32", fwrite_float32, VOID_TYPE, 0),
   MAKE_INTRINSIC("fwrite_float", fwrite_float, VOID_TYPE, 0),
   MAKE_INTRINSIC("fwrite_double", fwrite_double, VOID_TYPE, 0),

   MAKE_INTRINSIC("set_library_path", set_lib_load_path, VOID_TYPE, 0),
   MAKE_INTRINSIC("get_library_path", get_lib_load_path, STRING_TYPE, 0),
   MAKE_INTRINSIC("get_sldxe_root", get_sldxe_root, STRING_TYPE, 0),
   SLANG_END_TABLE
};

static SLang_App_Unary_Type App_Unary_Table [] =
{
   MAKE_APP_UNARY("isnan", SLDXE_OP_ISNAN),
   MAKE_APP_UNARY("isfinite", SLDXE_OP_ISFINITE),
   MAKE_APP_UNARY("isinf", SLDXE_OP_ISINF),
   MAKE_APP_UNARY("erf", SLDXE_OP_ERF),
   SLANG_END_TABLE
};



int sldxe_init_intrinsics (void)
{
   if ((-1 == SLclass_add_app_unary_op (SLANG_INT_TYPE,
				     array_unary_op,
				     array_unary_op_result))
       || (-1 == SLclass_add_app_unary_op (SLANG_DOUBLE_TYPE,
					 array_unary_op,
					 array_unary_op_result))
       || (-1 == SLclass_add_app_unary_op (SLANG_COMPLEX_TYPE,
					 array_unary_op,
					 array_unary_op_result)))
     return -1;

   /* Now add intrinsics for this application */
   if ((-1 == SLadd_intrin_fun_table (DXE_Intrinsics, NULL))
       || (-1 == SLadd_app_unary_table (App_Unary_Table, NULL)))
     return -1;

   return 0;
}

