/*
 * Copyright 1990 the Regents of the University of California.  All
 * rights reserved.  Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without fee is
 * hereby granted, provided that this copyright notice appear in all
 * copies.  See the file copyright.h for more information.
 *
 */

/* debug.c -- debugging routines.  Levels are:
   
   0 -- no debug
   1 -- Debug messages only
   2 -- Entering and debug messages
   3 -- Entering, debug messages, and exiting messages
   4 -- all of 3 and malloc_verify called at every procedure call and
        malloc_debug set to 1
   5 -- all of 4 and malloc_debug set to 2
*/

#include <stdio.h>
#include "global.h"

int debuglevel=0;
char debugmsg[MAXMAXLINE];  /* need to be big because some routines 
			       send a lot of data at a time */
char errmsg[MAXMAXLINE];

static indent_level=0;

static void PrintToXcbiError(msg)
     char *msg;
     /* note: I can't call any of my functions in this function because
	of the debugging calls to this function by my function. */
{
	printf(msg);
}

void myerror(msg)
     char *msg;
     /* prints run-time error message to stdout. */
{
	char temp[MAXMAXLINE];
	sprintf(temp,"Error: %s.\n",msg);
	PrintToXcbiError(temp);
}

static void PrintTabs()
     /* prints tabs according to the indent_level */
{
	int i;
	for (i=0; i<indent_level; i++)
	  PrintToXcbiError("    ");
}

void exit_proc()
{
	char temp[MAXLINE];
	PrintTabs();
	indent_level--;
	if (indent_level<0)
	  indent_level=0;
	if (debuglevel>2) {
		sprintf(temp,"Out.)\n");
		PrintToXcbiError(temp);
#ifdef MALLOCDEBUG
	if (debuglevel>3)
	  malloc_verify();
#endif
	}
}

void entering_proc(routine)
     char *routine;
     /* for debug purposes, prints what routine it's in */
{
	char temp[MAXLINE];
#ifdef MALLOCDEBUG
	if (debuglevel>3)
	  malloc_verify();
#endif
	if (debuglevel>1) {
		indent_level++;
		PrintTabs();
		sprintf(temp,"(In %s. \n",routine);
		PrintToXcbiError(temp);
	}
}

int get_debuglevel()
{
	return debuglevel;
}

void set_debuglevel(level)
     int level;
{
	char temp[MAXLINE];
	if (debuglevel!=0 || level!=0) {
		sprintf(temp,"Setting debuglevel to %d\n",level);
		PrintToXcbiError(temp);
	}
	debuglevel = level;
#ifdef MALLOCDEBUG
	if (debuglevel<4)
	  malloc_debug(0);
	if (debuglevel>3)
	  malloc_debug(debuglevel-3);
#endif
	indent_level = 0;
}

void dprint(msg)
     char *msg;
     /* prints debug messages to stdout. */
{
	char *temp;
	if (debuglevel>0) {
		if (msg&&*msg) {
			temp = XtMalloc(strlen(msg)+11);
			sprintf(temp,"Debug: %s.\n",msg);
		} else {
			temp = XtNewString("Debug: NULL.   ");
			sprintf(temp,"Debug: NULL.\n");
		}
		PrintToXcbiError(temp);
		free(temp);
	}
}
