#include "world.h"


/**************************************************************************/
/* MACRO  **************        WritePragmas       ************************/
/**************************************************************************/
/* PURPOSE: WRITE NAME, MARK, REFERENCE COUNT, FILE, FUNCTION, AND SOURCE */
/*          LINE PRAGMAS, FOLLOWED BY EOLN, TO output.                    */
/**************************************************************************/

#define WritePragmas(x)  if ( (x)->name != NULL )                        \
			     fprintf( output, " %%na=%s", (x)->name );   \
                         if ( (x)->file != NULL )                        \
			     fprintf( output, " %%sf=%s", (x)->file );   \
                         if ( (x)->funct )                               \
			     fprintf( output, " %%fn=%s", (x)->funct );  \
                         if ( (x)->line )                                \
			     fprintf( output, " %%sl=%d", (x)->line );   \
                         if ( (x)->sr > 0 )                              \
			     fprintf( output, " %%sr=%d", (x)->sr );     \
                         if ( (x)->pm > 0 )                              \
			     fprintf( output, " %%pm=%d", (x)->pm );     \
                         if ( (x)->pl > 0 )                              \
			     fprintf( output, " %%pl=%d", (x)->pl );     \
                         if ( (x)->cm != 0 )                             \
			     fprintf( output, " %%cm=%d", (x)->cm );     \
                         if ( (x)->pmark )                               \
			     fprintf( output, " %%mk=P" );               \
                         if ( (x)->rmark1 == RMARK )                     \
			     fprintf( output, " %%mk=R" );               \
                         if ( (x)->rmark1 == rMARK )                     \
			     fprintf( output, " %%mk=r" );               \
                         if ( (x)->omark1 )                              \
			     fprintf( output, " %%mk=O" );               \
                         if ( (x)->dmark )                               \
			     fprintf( output, " %%mk=D" );               \
                         if ( (x)->lmark )                               \
			     fprintf( output, " %%mk=L" );               \
                         if ( (x)->mmark )                               \
			     fprintf( output, " %%mk=M" );               \
                         if ( (x)->cmark )                               \
			     fprintf( output, " %%mk=C" );               \
                         if ( (x)->nmark )                               \
			     fprintf( output, " %%mk=N" );               \
                         if ( (x)->bmark )                               \
			     fprintf( output, " %%mk=B" );               \
                         if ( (x)->wmark )                               \
			     fprintf( output, " %%mk=W" );               \
			 if ( (x)->smark )                               \
			     fprintf( output, " %%mk=S" );               \
			 if ( (x)->ccost > 0.0 )                         \
			     fprintf( output, " %%cc=%4.3e", (x)->ccost ); \
                         fprintf( output, "\n" )


/**************************************************************************/
/* LOCAL  **************  FixSignedConstantImports ************************/
/**************************************************************************/
/* PURPOSE: IF sgnok IS TRUE, REMOVE THE SIGN OF ALL  NEGATIVE ARITHMETIC */
/*          IMPORTS TO NODE n AND INSERT THE APPROPRIATE NEG NODES; LLNL  */
/*          SOFTWARE DOES NOT ACCEPT SIGNED CONSTANTS. A NULL CONSTANT    */
/*          DEFINES AN ERROR VALUE.                                       */
/**************************************************************************/

static int FixSignedConstantImports( n, lab )
PNODE n;
int   lab;
{
    register PEDGE  i;
    register PEDGE  ii;
    register PNODE  neg;

    for ( i = n->imp; i != NULL; i = i->isucc ) {
	if ( !IsConst( i ) || (i->constant == NULL) )
	    continue;

	if ( IsArithmetic( i->info ) && (!sgnok) )
	    if ( i->constant[0] == '-' ) {
		neg = NodeAlloc( ++lab, IFNeg );
		ii  = EdgeAlloc( NULL, CONST_LABEL, neg, 1 );

		ii->info = i->info;
		ii->constant = &(i->constant[1]);

		LinkImport( neg, ii );

		i->constant = NULL;
		i->eport = 1;

		LinkExport( neg, i );
		LinkNode( (IsGraph(n))? n : n->npred, neg );
		}
	}

    return( lab );
}


/**************************************************************************/
/* STATIC **************      AssignNewLabels      ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN NEW LABELS TO THE NODES OF GRAPH g AND FIX SIGNED      */
/*          ARITHMETIC CONSTANTS (GIVEN sgnok IS TRUE).                   */
/**************************************************************************/

static void  AssignNewLabels( g )
register PNODE g;
{
    register int   lab;
    register PNODE n;

    FixSignedConstantImports( g, 0 );

    for ( lab = 0, n = g->G_NODES; n != NULL; n = n->nsucc ) {
	lab = FixSignedConstantImports( n, lab );

	n->label = ++lab;
        }
}


/**************************************************************************/
/* LOCAL  **************        WriteStamps        ************************/
/**************************************************************************/
/* PURPOSE: WRITE THE IF1 STAMPS IN THE STAMP TABLE TO output.            */
/**************************************************************************/

static void WriteStamps()
{
    register int i;

    for ( i = 0; i < 127; i++ )
	if ( stamps[i] != NULL )
	    fprintf( output, "C$  %c %s\n", i, stamps[i] );
}


/**************************************************************************/
/* LOCAL  **************         WriteInfo         ************************/
/**************************************************************************/
/* PURPOSE: WRITE THE INFO NODE LIST HEADED BY ihead TO OUTPUT.           */
/**************************************************************************/

static void WriteInfo()
{
    register PINFO i;
    register int   t;

    for ( i = ihead; i != NULL; i = i->next ) {
        if ( IsBasic(i) )
	    t = IF_BASIC;
        else
            t = i->type;

        fprintf( output, "T %2d %2d", i->label, t );

        switch ( i->type ) {
            case IF_FUNCTION:
	        fprintf( output, " %2d", (i->F_IN == NULL)? 0 : i->F_IN->label);
	        fprintf( output, " %2d", i->F_OUT->label );
	        break;

            case IF_STREAM:
            case IF_MULTPLE:
            case IF_ARRAY:
	    case IF_BUFFER:
	        fprintf( output, " %2d   ", i->A_ELEM->label );
                break;

            case IF_UNION:
            case IF_RECORD:
	        fprintf( output, " %2d   ", i->R_FIRST->label );
                break;

            case IF_TUPLE: 
            case IF_FIELD:
            case IF_TAG:
	        fprintf( output, " %2d", i->L_SUB->label );
	        fprintf( output, " %2d", 
				 (i->L_NEXT == NULL)? 0 : i->L_NEXT->label );
					 
                break;

            case IF_UNKNOWN:
	        fprintf( output, "      " );
	        break;

	    default:
	        fprintf( output, " %2d   ", i->type - BASIC_TYPE_START );
	        break;
            }

        WritePragmas( i );
	}
}


/**************************************************************************/
/* LOCAL  **************         WriteConst        ************************/
/**************************************************************************/
/* PURPOSE: WRITE A CONSTANT TO output.  IF THE CONSTANT IS REPRESENTED   */
/*          BY A NULL POINTER, THEN IT IS AN ERROR CONSTANT.  ELSE THE    */
/*          APPROPRIATE DELIMITERS ARE DETERMINED BASED ON THE CONSTANTS  */
/*          TYPE. THE PORT NUMBERS MAY BE NEGATIVE.                       */
/**************************************************************************/

void WriteConst( c )
PEDGE c;
{
    fprintf( output, "L         %2d %2d  %2d", c->dst->label,
                     abs( c->iport ), c->info->label         );

    if ( c->constant == NULL ) {
        fprintf( output, " \"%s\"", ERROR_CONSTANT );
	return;
	}

    if ( IsDefArrayBuf( c->dst ) && (c->isucc == NULL) ) {
	fprintf( output, " \"%s\"", c->constant );         /* BUFFER LITERAL */
	return;
	}

    switch ( c->info->type ) {
        case IF_CHAR:
            fprintf( output, " \"\'%s\'\"", c->constant );
            break;

        case IF_ARRAY:
        case IF_STREAM:
            fprintf( output, " \"\"%s\"\"", c->constant );
            break;

        default:
            fprintf( output, " \"%s\"", c->constant );
            break;
        }
}


/**************************************************************************/
/* MACRO  **************          WriteAde         ************************/
/**************************************************************************/
/* PURPOSE: WRITE ARITFICIAL DEPENDENCE EDGE x TO OUTPUT.                 */
/**************************************************************************/

#define WriteAde(x)    fprintf( output, "D %2d      %2d\n",                \
	                                (x)->src->label, (x)->dst->label )


/**************************************************************************/
/* MACRO  **************         WriteEdge         ************************/
/**************************************************************************/
/* PURPOSE: WRITE EDGE e TO output. THE PORT NUMBERS MAY BE NEGATIVE.     */
/**************************************************************************/

#define WriteEdge(e)   fprintf( output, "E %2d %2d   %2d %2d  %2d",        \
	                                e->src->label, abs( e->eport ),    \
                                        e->dst->label, abs( e->iport ),    \
					e->info->label               )


/**************************************************************************/
/* LOCAL  **************        WriteImports       ************************/
/**************************************************************************/
/* PURPOSE: WRITE NODE n's IMPORTS, BOTH DATA AND CONTROL, TO output.     */
/**************************************************************************/

static void WriteImports( n )
PNODE n;
{
    register PEDGE  i;
    register PADE   a;

    for ( i = n->imp; i != NULL; i = i->isucc ) {
	if ( IsConst( i ) )
	    WriteConst( i );
	else
            WriteEdge( i );

	WritePragmas( i );
        }

    for ( a = n->aimp; a != NULL; a = a->isucc )
	WriteAde( a );
}


/**************************************************************************/
/* LOCAL  **************        WriteNodes         ************************/
/**************************************************************************/
/* PURPOSE: PRINT THE NODES OF GRAPH g AND THEIR IMPORT EDGES TO output.  */
/*          NODE RELABELING IS DONE BEFORE PRINTING.                      */
/**************************************************************************/

static void WriteNodes( g )
PNODE g;
{
    register PNODE  n;
    register PALIST l;

    switch ( g->type ) {
        case IFLGraph:
	    fprintf( output, "G %2d", g->G_INFO->label );
	    fprintf( output, " \"%s\"", g->G_NAME );
	    break;

	case IFIGraph:
	    fprintf( output, "I %2d", g->G_INFO->label );
	    fprintf( output, " \"%s\"", g->G_NAME );
	    break;

	case IFXGraph:
	    fprintf( output, "X %2d", g->G_INFO->label );
	    fprintf( output, " \"%s\"", g->G_NAME );
	    break;

	case IFSGraph:
	    fprintf( output, "G %2d", g->label );
            break;
	}

    WritePragmas( g );

    AssignNewLabels( g );
    WriteImports( g );

    for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
        switch ( n->type ) {
	    case IFLoopA:
	    case IFLoopB:
	    case IFTagCase:
	    case IFForall:
	    case IFSelect:
	    case IFIfThenElse:
	    case IFIterate:
	        fprintf( output, "{ Compound %2d %2d\n", n->label, n->type );

	        for ( g = n->C_SUBS; g != NULL; g = g->gsucc )
	            WriteNodes( g );

                fprintf( output, "} %2d %2d %2d   ", n->label, 
				  n->type, n->C_SCNT        );

	        for ( l = n->C_ALST; l != NULL; l = l->next )
                    fprintf( output, " %d", l->datum );

	        break;

            default:
	        fprintf( output, "N %2d %2d", n->label, n->type );
            }

        WritePragmas( n );
	WriteImports( n );
	}
}


/**************************************************************************/
/* GLOBAL **************         If2Write          ************************/
/**************************************************************************/
/* PURPOSE: PRINT ALL IF2 TYPES, STAMPS, AND FUNCTION GRAPHS TO output.   */
/*          THE TYPE INFORMATION IS FOUND TRAVERSING THE BINARY TREE      */
/*          itree; THE STAMPS ARE FOUND IN ARRAY stamps; AND THE FUNCTION */
/*          GRAPHS ARE FOUND TRAVERSING glstop.  NODES ARE RELABELED.     */
/*          (SIGNED CONSTANTS ARE ALLOWED IF sgnok IS TRUE).              */
/**************************************************************************/

void If2Write()
{
    register PNODE f;

    WriteInfo();
    WriteStamps();

    for ( f = glstop->gsucc; f != NULL; f = f->gsucc )
	WriteNodes( cfunct = f );
}
