#include "SC.h"
#include "SCLib.h"


static	int	CurrentChar = EOF;
static	long	CharacterNumber = 0;
static	int	LineNumber = 1;
static	int	ColumnNumber = 1;

/* ------------------------------------------------------------ */
static char	*InputStrings[MaxStringIO] ;
static int	InputBufferLevel = 0 ;

/* Reset the Fibre input */
void FlushFibre()
{
    CurrentChar = EOF;
    CharacterNumber = 0;
    LineNumber = 1;
    ColumnNumber = 1;
}

/* ------------------------------------------------------------ */
void
OpenStringInput(string)
     char	*string;
{
  if ( InputBufferLevel >= MaxStringIO ) Oops("Too many strings open");

  InputStrings[InputBufferLevel++] = string;
  CharacterNumber = 0;
  LineNumber = 1;
}

/* ------------------------------------------------------------ */
void
CloseStringInput()
{
  /* Remove the string from the buffer location and flush any */
  /* lookaheads in the output. */
  if ( InputBufferLevel > 0 ) InputBufferLevel--;
  CurrentChar = EOF;
}

/* ------------------------------------------------------------ */
static int
GetAChar()
{
  int	c;

  if ( InputBufferLevel ) {
    if ( InputStrings[InputBufferLevel-1] &&
	*InputStrings[InputBufferLevel-1] ) {
      c = *InputStrings[InputBufferLevel-1]++;
    } else {
      c = EOF;
    }
  } else {
    c = FileReadChar();
  }
  CharacterNumber++;

  if (c == EndOfLine) {
    LineNumber++;
    ColumnNumber = 1;
  }

  return c;
}

/* ------------------------------------------------------------ */
static int
Peek()
{
  if (CurrentChar == EOF) {
    CurrentChar = GetAChar();
  }

  return CurrentChar;
}

/* ------------------------------------------------------------ */
static int
Read()
{
  int		LastChar;

  LastChar = (CurrentChar==EOF)?GetAChar():CurrentChar;
  CurrentChar = GetAChar();

  return LastChar;
}

/* ------------------------------------------------------------ */
static int
AtEOF()
{
  if (CurrentChar == EOF) {
    CurrentChar = GetAChar();
  }

  return (CurrentChar == EOF);
}

/* ------------------------------------------------------------ */
static void
SkipWhite()
{
  int	c;
  do {
    c = Peek();

    /* If the char is a blank, just skip it */
    if ( isspace(c) ) {
      (void)Read();
      continue;
    }

    /* If the character doesn't start a comment, then the current */
    /* Peek() value is the first non-blank.  We're done. */
    if ( c != (int)('#') ) break;

    /* For comments, skip characters upto a newline */
    while(Peek() != (int)('\n')) (void)Read();
  } while (TRUE);
}

/* ------------------------------------------------------------ */
static int
MatchToken(s)
     char	*s;
{
  int	c;
  int	i;
  int	length;

  length = strlen(s);
  for(i=0;i<length;i++) {
    c = Peek();
    if (islower(c)) c = toupper(c);
    if (c != s[i]) return 0;
    (void)Read();		/* Consume the character */
  }

  return 1;
}


/* ------------------------------------------------------------ */
static void
ReportError(s,Error)
     char	*s;
     char	*Error;
{
  FibreError(s,Peek(),CharacterNumber,LineNumber,ColumnNumber,Error);
  CheckWarning(;);
}

/* ------------------------------------------------------------ */
static void
ReadError(s,Error)
     char	*s;
     char	*Error;
{
  char		errorstring[1024];

  if ( !MatchToken("ERROR") ) {
    /* Isn't an error value */
    (void)sprintf(errorstring,"Cannot read %s",s);
    ReportError(errorstring,Error);
    CheckWarning(;);
  }
}

/* ------------------------------------------------------------ */
int
isodigit(x)
     int	x;
{
  int	IsOctal;

  switch(x) {
    case '0': case '1': case '2': case '3':
    case '4': case '5': case '6': case '7':
      IsOctal = 1;
      break;
    default:
      IsOctal = 0;
  }
  return IsOctal;
}

/* ------------------------------------------------------------ */
static int
ReadOctal()
{
  int	val;

  val = 0;
  while(isodigit(Peek())) {
    val = val*8 + DigitValueOf(Read());
  }

  return val;
}

/* ------------------------------------------------------------ */
static SisalBoolean
ReadInteger(val,Error)
     SisalInteger	*val;
     char		*Error;
{
  char		s[1024];
  int		length;
  SisalBoolean	IsError;

  /* Collect an integer */
  SkipWhite();
  length = 0;
  if (Peek() == '-') s[length++] = Read();
  while (isdigit(Peek())) {
    s[length++] = Read();
  }

  /* If there is no integer, try to read an integer, else decode it */
  if (length == 0) {
    ReadError("Integer",Error);
    CheckWarningV(;,FALSE);
    *val = IntegerZero;
    IsError = TRUE;
  } else {
    s[length] = '\0';
    StringToInteger(s,val);
    IsError = FALSE;
  }

  return IsError;
}

/* ------------------------------------------------------------ */
static SisalBoolean
ReadReal(val,Error)
     SisalReal	*val;
     char	*Error;
{
  char		s[1024];
  int		length;
  SisalBoolean	IsError,HaveFraction,HavePower;

  /* Collect an real */
  length = 0;
  SkipWhite();

  /* Optional Sign */
  if (Peek()=='-') {
    s[length++] = Read();
  }

  /* Mantissa */
  while(isdigit(Peek())) {
    s[length++] = Read();
  }

  /* Optional Fraction */
  if (Peek() == '.') {
    HaveFraction = TRUE;
    s[length++] = Read();
    while(isdigit(Peek())) {
      s[length++] = Read();
    }
  } else {
    HaveFraction = FALSE;
  }

  if (length != 0) {		/* Have a valid Real */
    /* Optional Power */
    if (Peek() == 'e' || Peek() == 'E') {
      HavePower = TRUE;
      s[length++] = Read();
      if (Peek()=='-') {
	s[length++] = Read();
      }
      while(isdigit(Peek())) {
	s[length++] = Read();
      }
    } else {
      HavePower = FALSE;
    }

    if (! ( HaveFraction || HavePower ) ) {
      ReportError("Fibre Real without fraction or exponent",Error);
      CheckWarningV(;,FALSE);
    }

    s[length] = '\0';
    *val = atof(s);
    IsError = FALSE;
  } else {
    /* Try to read in an error value */
    ReadError("Real",Error);
    CheckWarningV(;,FALSE);
    *val = 0;
    IsError = TRUE;
  }
  
  return IsError;
}

/* ------------------------------------------------------------ */
static SisalBoolean
ReadDouble(val,Error)
     SisalDouble	*val;
     char		*Error;
{
  char		s[1024];
  int		length;
  SisalBoolean	IsError;

  /* Collect a double */
  length = 0;
  SkipWhite();

  /* Optional Sign */
  if (Peek()=='-') {
    s[length++] = Read();
  }

  /* Mantissa */
  while(isdigit(Peek())) {
    s[length++] = Read();
  }

  /* Optional Fraction */
  if (Peek() == '.') {
    s[length++] = Read();
    while(isdigit(Peek())) {
      s[length++] = Read();
    }
  }

  if ( length != 0 ) {
    /* Required Power */
    if (Peek() != 'd' && Peek() != 'D') {
      FibreError("Invalid Double -- No double power (d or D)",
		 Peek(),CharacterNumber,LineNumber,ColumnNumber,Error
		 );
    }

    (void)Read();		/* Consume the power */
    s[length++] = 'e';		/* Convert to C form */

    if (Peek()=='-') {
      s[length++] = Read();
    }
    while(isdigit(Peek())) {
      s[length++] = Read();
    }

    s[length] = '\0';
    *val = atof(s);
    IsError = FALSE;
  } else {
    /* Try to read in an error value */
    ReadError("Double",Error);
    CheckWarningV(;,FALSE);
    *val = 0;
    IsError = TRUE;
  }
  
  return IsError;
}

/* ------------------------------------------------------------ */
static SisalBoolean
ReadBoolean(val,Error)
     SisalBoolean	*val;
     char		*Error;
{
  SisalBoolean	IsError;

  /* Collect a boolean */
  SkipWhite();
  if ( MatchToken("T") ) {
    *val = TRUE;
    IsError = FALSE;
  } else if ( MatchToken("F") ) {
    *val = FALSE;
    IsError = FALSE;
  } else {
    ReadError("Boolean",Error);
    CheckWarningV(;,FALSE);
    *val = FALSE;
    IsError = TRUE;
  }

  return IsError;
}

/* ------------------------------------------------------------ */
static SisalBoolean
ReadNull(val,Error)
     SisalNull	*val;
     char	*Error;
{
  SisalBoolean	IsError;

  /* Collect a null */
  SkipWhite();
  if ( MatchToken("NIL") ) {
    *val = NIL;
    IsError = FALSE;
  } else {
    ReadError("Null",Error);
    CheckWarningV(;,FALSE);
    *val = NIL;
    IsError = TRUE;
  }

  return IsError;
}

/* ------------------------------------------------------------ */
static SisalBoolean
ReadCharacter(val,Error)
     SisalCharacter	*val;
     char		*Error;
{
  SisalBoolean	IsError;

  /* Collect a character */
  SkipWhite();

  if ( MatchToken("'") ) {
    IsError = FALSE;		/* No errors in quotes */

    if ( MatchToken("\\") ) {	/* Look for Escapes */
      switch( Peek() ) {
	case 'b':
	  (void)Read();		/* Consume Character */
	  *val = '\010';
	  break;
	  
	case 'f':
	  (void)Read();		/* Consume Character */
	  *val = '\014';
	  break;

	case 'n':
	  (void)Read();		/* Consume Character */
	  *val = '\012';
	  break;

	case 'r':
	  (void)Read();		/* Consume Character */
	  *val = '\015';
	  break;

	case 't':
	  (void)Read();		/* Consume Character */
	  *val = '\011';
	  break;

	case '0': case '1': case '2': case '3':
	case '4': case '5': case '6': case '7':
	  *val = ReadOctal();
	  break;

	default:
	  *val = Read();
	}
    } else {
      *val = Read();
    }

    if ( !MatchToken("'") ) {
      FibreError("No close quote for character",
		 Peek(),CharacterNumber,LineNumber,ColumnNumber,
		 Error
		 );
    }

  } else {
    ReadError("Character",Error);
    CheckWarningV(;,FALSE);
    *val = '\0';
    IsError = TRUE;
  }

  return IsError;
}

/* ------------------------------------------------------------ */
static void
ReadArray(T,TheArray,Error)
     TypeD	T;
     IF1OBJECT	*TheArray;
     char	*Error;
{
  TypeD		ElementType;
  SisalBoolean	IsOK;
  IF1OBJECT	Lower,Element,*El;
  BagPtr	B;
  unsigned	Count;

  ElementType = ElementTypeOfArray(T);

  SkipWhite();

  /* Get opening bracket */
  if (Peek() == '[') {
    IsOK = TRUE;
    (void)Read();
  } else if (MatchToken("ERROR[")) {
    IsOK = FALSE;
  } else {
    ReportError("Cannot open Fibre Array",Error);
    CheckWarning(;);
  }

  /* Read Lower Bound and consume the colon */
  FibreRead(IntegerTypeD,&Lower,Error);
  CheckWarning(;);
  SkipWhite();
  if ( Peek() == ':' ) {
    (void)Read();
  } else {
    ReportError("Missing colon in Fibre Array",Error);
    CheckWarning(;);
  }

  /* While there are values, get elements */
  SkipWhite();
  B = EmptyBag(ArrayInitialSize);
  Count = 0;
  while(!AtEOF() && Peek() != ']') {
    FibreRead(ElementType, &Element,Error);
    CheckWarning(;);
    El = GetPointerIntoBag(B,Count++,TRUE);
    Copy(El,&Element);

    SkipWhite();
  }

  /* Consume the final bracket */
  if (Peek() == ']') {
    (void)Read();
  } else {
    ReportError("Cannot close Fibre Array",Error);
    CheckWarning(;);
  }

  /* Put it all together */
  CreateArray(TheArray,T,IsOK,BasErr(&Lower),IVal(&Lower),Count,Count,B,0);
}
/* ------------------------------------------------------------ */
static void
ReadRecord(T,TheRecord,Error)
     TypeD	T;
     IF1OBJECT	*TheRecord;
     char	*Error;
{
  TypeD		ElementType;
  BagPtr	Collection;
  SisalBoolean	IsErr;
  unsigned	i;
  IF1OBJECT	*Element;

  SkipWhite();

  /* Get opening bracket */
  if (Peek() == '<') {
    IsErr = FALSE;
    (void)Read();

    Collection = EmptyBag(FieldCountOfRecord(T));

    for(i=0;i<FieldCountOfRecord(T);i++) {
      ElementType = FieldTypesOfRecord(T)[i];
      Element = PointerIntoBag(Collection,i);
      FibreRead(ElementType,Element,Error);
      CheckWarning(;);
    }
    SkipWhite();

  } else if (MatchToken("ERROR<")) {
    IsErr = TRUE;
    Collection = NULL;
  } else {
    ReportError("Cannot open Fibre Record",Error);
    CheckWarning(;);
  }

  TypeOf(TheRecord) = T;
  RecErr(TheRecord) = IsErr;
  RecCol(TheRecord) = Collection;

  /* Consume the final bracket */
  if (Peek() == '>') {
    (void)Read();
  } else {
    ReportError("Cannot close Fibre Record",Error);
    CheckWarning(;);
  }
}
/* ------------------------------------------------------------ */
static void
ReadStream(T,TheStream,Error)
     TypeD	T;
     IF1OBJECT	*TheStream;
     char	*Error;
{
  TypeD		ElementType;
  SisalBoolean	IsOK;
  IF1OBJECT	Lower,Element,*El;
  BagPtr	B;
  unsigned	Count;

  ElementType = ElementTypeOfStream(T);

  if (StrictStreams) {
    /* Prepare an empty bag to receive the stream elements */
    B = EmptyBag(StreamInitialSize);
    Count = 0;

    if ( UseStdio &&
	TypeEntryOf(ElementType) == IF1BASIC &&
	KindOfBasic(ElementType) == IF1Character
	) {
      /* Set Error and Lowerbound  */
	IsOK = TRUE;

	SetDes(&Lower,IntegerTypeD);
	BasErr(&Lower)	= FALSE;
	IVal(&Lower)	= IntegerOne;

      /* Just read in characters */
      while(!AtEOF()) {
	SetDes(&Element,CharacterTypeD);
	BasErr(&Element) = FALSE;
	CVal(&Element)	 = LocalToCharacter(Read());
	El = GetPointerIntoBag(B,Count++,TRUE);
	Copy(El,&Element);
      }
    } else {
      /* Get opening bracket */
      SkipWhite();
      if (Peek() == '{') {
	IsOK = TRUE;
	(void)Read();
      } else if (MatchToken("ERROR{")) {
	IsOK = FALSE;
      } else {
	ReportError("Cannot open Fibre Stream",Error);
	CheckWarning(;);
      }

      /* Read Lower Bound and consume the colon */
      FibreRead(IntegerTypeD,&Lower,Error);
      CheckWarning(;);
      SkipWhite();
      if ( Peek() == ':' ) {
	(void)Read();
      } else {
	ReportError("Missing colon in Fibre Stream",Error);
	CheckWarning(;);
      }

      /* While there are values, get elements */
      SkipWhite();
      while(!AtEOF() && Peek() != '}') {
	FibreRead(ElementType, &Element,Error);
	CheckWarning(;);
	El = GetPointerIntoBag(B,Count++,TRUE);
	Copy(El,&Element);

	SkipWhite();
      }

      /* Consume the final bracket */
      if (Peek() == '}') {
	(void)Read();
      } else {
	ReportError("Cannot close Fibre Stream",Error);
	CheckWarning(;);
      }
    }
    /* Put it all together */
    CreateArray(TheStream,T,IsOK,BasErr(&Lower),IVal(&Lower),Count,Count,B,0);
  } else {
    Oops("Cannot create a non-strict stream");
  }
}

/* ------------------------------------------------------------ */
static void
ReadBasic(Description,OBJ,Error)
     TypeD	Description;
     IF1OBJECT	*OBJ;
     char	*Error;
{
  SisalBoolean		BooleanValue;
  SisalCharacter	CharacterValue;
  SisalDouble		DoubleValue;
  SisalInteger		IntegerValue;
  SisalNull		NullValue;
  SisalReal		RealValue;

  /* Read in the value and flags */
  switch(KindOfBasic(Description)) {
    case IF1Boolean:
    if ( ReadBoolean(&BooleanValue,Error) ) {
      BasErr(OBJ)	= TRUE;
    } else {
      BasErr(OBJ)	= FALSE;
      BVal(OBJ)   = BooleanValue;
    }
    break;

    case IF1Character:
    if ( ReadCharacter(&CharacterValue,Error) ) {
      BasErr(OBJ) = TRUE;
    } else {
      BasErr(OBJ) = FALSE;
      CVal(OBJ)   = CharacterValue;
    }
    break;

    case IF1Double:
    if ( ReadDouble(&DoubleValue,Error) ) {
      BasErr(OBJ) = TRUE;
    } else {
      BasErr(OBJ) = FALSE;
      DVal(OBJ)   = DoubleValue;
    }
    break;

    case IF1Integer:
    if ( ReadInteger(&IntegerValue,Error) ) {
      BasErr(OBJ) = TRUE;
    } else {
      BasErr(OBJ) = FALSE;
      IVal(OBJ)   = IntegerValue;
    }
    break;

    case IF1Null:
    if ( ReadNull(&NullValue,Error) ) {
      BasErr(OBJ) = TRUE;
    } else {
      BasErr(OBJ) = FALSE;
    }
    break;

    case IF1Real:
    if ( ReadReal(&RealValue,Error) ) {
      BasErr(OBJ) = TRUE;
    } else {
      BasErr(OBJ) = FALSE;
      RVal(OBJ)   = RealValue;
    }
    break;

    default:
    Oops("Unknown Basic Type");
    break;

  }

  /* Set the appropriate type */
  SetDes(OBJ,Description);
}
/* ------------------------------------------------------------ */
void
ReadUnion(Description,TheUnion,Error)
     TypeD	Description;
     IF1OBJECT	*TheUnion;
     char	*Error;
{
  SisalBoolean	IsErr;
  IF1OBJECT	TagVal,*Element;
  SisalInteger	Tag;
  BagPtr	Collection;

  SkipWhite();

  /* Get opening bracket */
  if (Peek() == '(') {
    IsErr = FALSE;
    (void)Read();

    /* Read the tag and colon*/
    FibreRead(IntegerTypeD,&TagVal,Error);
    CheckWarning(;);
    SkipWhite();
    if ( Peek() == ':' ) {
      (void)Read();
    } else {
      ReportError("Missing colon in Fibre Union",Error);
      CheckWarning(;);
    }

    /* Make sure the tag is in range */
    if (BasErr(&TagVal) 
	|| IntegerToLocal(IVal(&TagVal)) < 0 || IntegerToLocal(IVal(&TagVal)) >= TagCountOfUnion(Description)
	) {
      ReportError("Bad tag in Fibre Union",Error);
      CheckWarning(;);
    }

    IsErr	= FALSE;
    Tag		= IVal(&TagVal);
    Collection	= EmptyBag(1);

    Element	= PointerIntoBag(Collection,0);
    FibreRead(TagTypesOfUnion(Description)[IntegerToLocal(Tag)],Element,Error);
    CheckWarning(;);
    SkipWhite();

  } else if (MatchToken("ERROR(")) {
    IsErr	= TRUE;
    Tag		= IntegerZero;
    Collection	= NULL;
  } else {
    ReportError("Cannot open Fibre Union",Error);
    CheckWarning(;);
  }

  SetDes(TheUnion,	Description);
  UniErr(TheUnion)	= IsErr;
  UniTag(TheUnion)	= Tag;
  UniVal(TheUnion)	= Collection;

  /* Consume the final bracket */
  if (Peek() == ')') {
    (void)Read();
  } else {
    ReportError("Cannot close Fibre Union",Error);
    CheckWarning(;);
  }
}
/* ------------------------------------------------------------ */
void
FibreRead(Description,OBJ,Error)
     TypeD	Description;
     IF1OBJECT	*OBJ;
     char	*Error;
{
  if (!Description) {
    SkipWhite();
    SkipFibre(Read(),Error);
    FCopy(OBJ,EmptyObject,aEMPTY);
  } else {
    switch( TypeEntryOf(Description) ) {
      case IF1ARRAY:
      ReadArray(Description,OBJ,Error);
      CheckWarning(;);
      break;

      case IF1BASIC:
      ReadBasic(Description,OBJ,Error);
      CheckWarning(;);
      break;

      case IF1FIELD:
      Oops("");
      break;

      case IF1FUNCTION:
      Oops("");
      break;

      case IF1MULTIPLE:
      Oops("");
      break;

      case IF1RECORD:
      ReadRecord(Description,OBJ,Error);
      CheckWarning(;);
      break;

      case IF1STREAM:
      ReadStream(Description,OBJ,Error);
      CheckWarning(;);
      break;

      case IF1TAG:
      Oops("");
      break;

      case IF1TUPLE:
      Oops("");
      break;

      case IF1UNION:
      ReadUnion(Description,OBJ,Error);
      CheckWarning(;);
      break;

      default:
      Oops("");
      break;

    }
  }
}
/* ------------------------------------------------------------ */
void
SkipFibre(Start,Error)
     int	Start;
     char	*Error;
{
  int		c;

  switch(Start) {
   case '[':
    SkipWhite();
    while(Peek() !=  ']') SkipFibre(Read(),Error);
    (void)Read();		/* Consume the closing bracket */
    break;
   case '{':
    SkipWhite();
    while(Peek() !=  '}') SkipFibre(Read(),Error);
    (void)Read();		/* Consume the closing bracket */
    break;
   case '<':
    SkipWhite();
    while(Peek() !=  '>') SkipFibre(Read(),Error);
    (void)Read();		/* Consume the closing bracket */
    break;
   case '(':
    SkipWhite();
    while(Peek() !=  ')') SkipFibre(Read(),Error);
    (void)Read();		/* Consume the closing bracket */
    break;

   case 'e': case 'E':
    /* It could be we're skipping something like error[ ... ]! */
    if ((c=Read(), c == (int)('r') || c == (int)('R')) &&
	(c=Read(), c == (int)('r') || c == (int)('R')) &&
	(c=Read(), c == (int)('o') || c == (int)('O')) &&
	(c=Read(), c == (int)('r') || c == (int)('R')) ) {
      /* If the next character is an aggregate start char, then */
      /* continue the skip.  Otherwise we're all done */
      c = Peek();
      if ( c == '[' || c == '{' || c=='<' || c=='(' ) SkipFibre(Read(),Error);
    }
    break;
   default:
    /* Its a number or boolean or something.  Just read up to an */
    /* space or aggregate delimiter of some sort. */
    while(c=Peek(),
	  c != EOF && c != ']' && c != '}' && c!='>' && c!=')' && !isspace(c)
	  ) (void)Read();
  }

  SkipWhite();		/* Point at next object */
}
