/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* TWINE is the frontend to the TWINE SISAL compiler/debugger system. */
/* It provides a functionality somewhat like CC or OSC or other UNIX */
/* compilers. */
/* ------------------------------------------------------------ */
/* Release	Comment				Date	Author */
/* 0.0		Initial release			8/12/91	Miller */
/* ------------------------------------------------------------ */
/* Modification History:
/*   Walter Cedeno, 12/91: Corrected bugs with options and default */
/*     paths.
/*   Walter Cedeno, 3/92: Added -dbx optio to compile the program */
/*     using -g option                                             */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */

#include <stdio.h>

/* ------------------------------------------------------------ */
/* If these parameters (size and default values) are not set by the */
/* creator of TWINE, set them to something meaningful in the */
/* development environment.  The production Makefile should override */
/* all of these parameters. */

#ifndef MAXFILES
#define MAXFILES		(100)
#endif

#ifndef NAMELEN
#define NAMELEN			(100)
#endif

#ifndef COMMANDSIZE
#define COMMANDSIZE		(1000)
#endif

#ifndef  DEFAULT_TWINEDIR
#define DEFAULT_TWINEDIR	"/usr/new/sisal/TWINE";
#endif

#ifndef  DEFAULT_SISAL
#define DEFAULT_SISAL		"/usr/new/sisal/Osc/bin/osc -IF1";
#endif

#ifndef  DEFAULT_SISALFIX
#define DEFAULT_SISALFIX	"/usr/new/sisal/TWINE/bin/fixloop";
#endif

#ifndef  DEFAULT_IF1ORDER
#define DEFAULT_IF1ORDER        "/usr/new/sisal/TWINE/bin/lineorder"
#endif

#ifndef  DEFAULT_YIFT
#define DEFAULT_YIFT		"/usr/new/sisal/TWINE/bin/yift";
#endif

#ifndef  DEFAULT_CC
#define DEFAULT_CC		"/bin/cc";
#endif

#ifndef  DEFAULT_LD
#define DEFAULT_LD		"/bin/cc";
#endif

#ifndef  DEFAULT_EXE
#define DEFAULT_EXE		"t.out";
#endif

#ifndef  DEFAULT_ENTRY
#define DEFAULT_ENTRY		"main";
#endif

#ifndef  DEFAULT_MACHINE
#define DEFAULT_MACHINE		"SUN3";
#endif

#ifndef  DEFAULT_OS
#define DEFAULT_OS		"BSD";
#endif

/* ------------------------------------------------------------ */
/* Assume that all steps of the compile will be done.  Options on the */
/* execution line may change what the terminal step will be. */
int	DoYift		= 1;
int	DoCC		= 1;
int	DoLink		= 1;

/* ------------------------------------------------------------ */
/* Set optimization off unless requested as well as the debug (don't */
/* execute compiler commands) and the verbose mode (echo commands as */
/* they are executed).  The Debugger option will turn on the debugging */
/* facility of TWINE.  */
int	Optimize	= 0;
int	Debug		= 0;
int	Verbose		= 0;
int	Debugger	= 0;
int     DebugTwine      = 0;

/* ------------------------------------------------------------ */
/* Set the default values for the compilers to use, the default name */
/* for an executable, and the default entry point. */
char	*TwineDir	= DEFAULT_TWINEDIR;
char	*TwineSisal	= DEFAULT_SISAL;
char	*TwineSisalFix	= DEFAULT_SISALFIX;
char    *TwineIf1Order  = DEFAULT_IF1ORDER;
char	*TwineYift	= DEFAULT_YIFT;
char	*TwineCC	= DEFAULT_CC;
char	*TwineLD	= DEFAULT_LD;
char	*ExeName	= DEFAULT_EXE;
char	*Entry		= DEFAULT_ENTRY;
char	*Machine	= DEFAULT_MACHINE;
char	*OS		= DEFAULT_OS;       

/* ------------------------------------------------------------ */
char	LoadFiles[MAXFILES*NAMELEN];	/* Load string */
char	LoadOptions[MAXFILES*NAMELEN];	/* Options for the loader step */
char	TempFile[NAMELEN] = "";		/* The /tmp/#twine<pid>.c file */
char	TempObj[NAMELEN]  = "";		/* The /tmp/#twine<pid>.o file */

/* ------------------------------------------------------------ */
/* Standard string stuff */
extern char		*strcat();
extern char		*strcpy();
#ifndef SYSV
extern char		*sprintf();
#endif
extern char		*rindex();

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* CheckEnvironment looks up a string in the environment.  If it */
/* exists there, then the value is set to the environment's value. */
void
CheckEnvironment(var,value)
     char	*var;
     char	**value;
{
  char	*getenv();
  char	*env;

  env = getenv(var);
  if (env) *value = env;
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ProcessOptions first looks for environment overrides to internal */
/* variables and then looks for overrides to the default flags. */
/* Anything that isn't recognized as an option is assumed to be a file */
/* and is appended to the file list.  The file list is always */
/* terminated with a NULL name. */
void
ProcessOptions(argc,argv,files)
     int	argc;
     char	*argv[];
     char	*files[];
{
  int		i;

  CheckEnvironment("TwineDir",&TwineDir);
  CheckEnvironment("TwineSisal",&TwineSisal);
  CheckEnvironment("TwineSisalFix",&TwineSisalFix);
  CheckEnvironment("TwineIf1Order",&TwineIf1Order);
  CheckEnvironment("TwineYift",&TwineYift);
  CheckEnvironment("TwineCC",&TwineCC);
  CheckEnvironment("TwineLD",&TwineLD);

  for(i=1;i<argc;i++) {
    if ( strcmp(argv[i],"-o") == 0 && i+1 < argc ) {
      ExeName = argv[++i];
    } else if ( strcmp(argv[i],"-D") == 0 && i+1 < argc ) {
      TwineDir = argv[++i];
    } else if ( strcmp(argv[i],"-entry") == 0 && i+1 < argc ) {
      Entry = argv[++i];
    } else if ( strcmp(argv[i],"-d") == 0 ) {
      Debug = 1;
    } else if ( strcmp(argv[i],"-g") == 0 ) {
      Debugger = 1;
      Optimize = 0;
    } else if ( strcmp(argv[i],"-O") == 0 ) {
      Debugger = 0;
      Optimize = 1;
    } else if ( strcmp(argv[i],"-v") == 0 ) {
      Verbose = 1;
    } else if ( strcmp(argv[i],"-i") == 0 ) {
      DoLink = 0;
      DoYift = 0;
    } else if ( strcmp(argv[i],"-C") == 0 ) {
      DoLink = 0;
      DoCC = 0;
    } else if ( strcmp(argv[i],"-c") == 0 ) {
      DoLink = 0;
    } else if ( strcmp(argv[i],"-dbx") == 0 ) {
      DebugTwine = 1;
    } else {
      *(files++) = argv[i];
    }
  }

  *(files++) = NULL;
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* Remove the files from /tmp (if created). */
void
Cleanup()
{
  if ( TempFile[0] ) unlink(TempFile);
  if ( TempObj[0] ) unlink(TempObj);
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* The Command routine will fork a process off to run the specified */
/* command (com) with the specified options (option).  The optformat */
/* argument is used to specify how the call is to be made.  In verbose */
/* mode, the command is echoed to the stderr output.  In debug mode, */
/* the command is not really executed.  If the command fails, cleanup */
/* is performed, and TWINE takes an error exit. */
void
Command(com,optformat,file,option)
     char	*com,*optformat,*file,*option;
{
  char	command[COMMANDSIZE];

  (void)sprintf(command,optformat,com,file,option);

  if ( Verbose ) {
    (void)fputs(command,stderr);
    (void)putc('\n',stderr);
  }

  if ( !Debug ) {
    if (system(command)) { Cleanup(); exit(1);}
  }
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* Return the first character of the extension in a filename.  If */
/* there is no extension, return NULL */
char
Extension(s)
     char	*s;
{
  char	*dotp;			/* Pointer to the . in .ext */

  dotp = rindex(s,'.');

  if (dotp && *(dotp+1)) {
    return *(dotp+1);
  } else {
    return '\0';
  }
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* Change the extension on a file. */
void
ChangeExtension(s,new)
     char	*s;
     char	*new;
{
  char	*dotp;

  dotp = rindex(s,'.');

  if (dotp) {
    (void)strcpy(dotp+1,new);
  }
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* This routine builds a TWINE linkage module in /tmp.  This file */
/* creates a table called ``Linkage'' which is used by TWINE as a */
/* dynamic linkage mechanism */
void
LinkModule(modules,count)
     char	*modules[];
{
  int		Pid = getpid();	/* Process ID for unique names */
  int		i;		/* Loop var */
  char		*p;		/* Points to .ext on file */
  FILE		*fp;		/* FP to the /tmp/#twine<pid>.c file */
  char		Options[100];	/* Options to the cc compile */

  /* ------------------------------------------------------------ */
  /* Open up a temp file to fullfil the linkage requiresments */
  sprintf(TempFile,"/tmp/#twine%d.c",Pid);
  strcpy(TempObj,TempFile);
  *(rindex(TempObj,'.')+1) = 'o'; /* Make it a .o file */
  fp = fopen(TempFile,"w");
  if ( !fp ) {perror("twine"); exit(1);}

  /* ------------------------------------------------------------ */
  /* Build the header to the link file */
  fprintf(fp,"#include \"SC.h\"\n");
  fprintf(fp,"#include \"SCLib.h\"\n");
  fprintf(fp,"char *DefaultEntry = \"%s\";\n",Entry);

  /* ------------------------------------------------------------ */
  /* Create a forward declaration for each module */
  for(i=0;i<count;i++) {
    p = rindex(modules[i],'.');
    if ( p ) *p = '\0';	 /* Remove the extension and replace others with '_' */
    for( p = rindex(modules[i],'.'); p; p = rindex(modules[i],'.') )
	*p = '_';
    fprintf(fp,"extern NodeStuff %s_NT[];\n",modules[i]);
  }

  /* ------------------------------------------------------------ */
  /* Create a forward declaration for the system list */
  fprintf(fp,"extern NodeStuff SystemNodeTable[];\n");

  /* ------------------------------------------------------------ */
  /* Create a table of modules.  The table ends with the system module */
  /* followed by a NULL to indicate the end of the table */
  fprintf(fp,"NodeInfo Linkage[] = {\n");
  for(i=0;i<count;i++) {
    fprintf(fp,"  %s_NT,\n",modules[i]);
  }
  fprintf(fp,"  SystemNodeTable,\n");
  fprintf(fp,"  NULL\n");
  fprintf(fp,"};\n");
  fclose(fp);

  /* ------------------------------------------------------------ */
  /* Build an option list that will compile the linkage */
  (void)sprintf(Options,"%%s -c %s %%s -D%s -D%s -I%%s/include -o %s",
		(DebugTwine)?"-g":"",Machine,OS,TempObj);
  Command(TwineCC,Options,TempFile,TwineDir);
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* The TWINE compiler will compile and link .sis, .if1, .c, and .o */
/* files.  The compilers for each stage are fixed at compile time */
/* except as overridden by environment variables.  First, flags are */
/* set and environment overrides are implemented.  Then, each file in */
/* the TWINE compile line is processed in turn, starting at the */
/* appropriate stage and ending with the compile to a .o file.  After */
/* all the .o files are produced, a linkage module is created to tie */
/* together all the component pieces (in /tmp).  The final executable */
/* is then produced */
void
main(argc,argv)
     int	argc;
     char	*argv[];
{
  char		*files[MAXFILES]; 	/* The list of files to process */
  char		**file;			/* A file in the above list */
  char		WorkFile[NAMELEN];	/* The file as its .ext changes */
  char		TFile[NAMELEN];		/* The .c file output from YIFT */
  char		*LinkFiles[MAXFILES];	/* The list of files to be */
					/* linked.  This list will */
					/* form the modules list for */
					/* the TWINE Linkage table */
  int		LinkCount = 0;		/* Number of files to link */
  char		Options[100];		/* Options to the cc compile */

  /* ------------------------------------------------------------ */
  /* Start the strings as empty */
  LoadFiles[0] = '\0';
  LoadOptions[0] = '\0';

  /* ------------------------------------------------------------ */
  /* Set (or reset) compile flags and get the list of files */
  ProcessOptions(argc,argv,files);

  /* ------------------------------------------------------------ */
  /* Process each file in turn */
  for(file=files;*file;file++) {
    (void)strcpy(WorkFile,*file);

    switch( Extension(WorkFile) ) {
      /* ------------------------------------------------------------ */
      /* .sis --> .if1 */
     case 's':
      sprintf( Options, "%%s %s %s %%s", (Debugger)?"-noopt":"", 
	      (Verbose)?"-v":"" );
      Command(TwineSisal,Options,WorkFile,(char*)NULL);
      ChangeExtension(WorkFile,"if1");
      if ( Optimize )
	Command(TwineSisalFix,"%s %s",WorkFile,(char*)NULL);
      if (!DoYift) break;

      /* ------------------------------------------------------------ */
      /* .if1 --> .c */
     case 'i':
      (void)strcpy(TFile,WorkFile);

      /* When -g is specified, re-order if1 file by source line */
      if ( Debugger ) {
          ChangeExtension(TFile,"IF1");
          Command(TwineIf1Order,"%s %s %s",WorkFile,TFile);
          ChangeExtension(WorkFile,"IF1");
      }
      
      ChangeExtension(TFile,"c");
      Command(TwineYift,"%s %s %s",WorkFile,TFile);
      ChangeExtension(WorkFile,"c");
      if (!DoCC) break;

      /* ------------------------------------------------------------ */
      /* .c --> .o */
     case 'c':
      (void)sprintf(Options,"%%s -c %s -D%s -D%s %%s -I%%s/include",
		    (Optimize)?"-O":((Debugger)?"-g -DDebugger":""),
		    Machine,OS
		    );
      Command(TwineCC,Options,WorkFile,TwineDir);
      ChangeExtension(WorkFile,"o");

      /* ------------------------------------------------------------ */
      /* .o  */
     case 'o':
      LinkFiles[LinkCount++] = strcpy(malloc(strlen(WorkFile)+1),WorkFile);

      /* ------------------------------------------------------------ */
      /* .a files or whatever */
     default:
      (void)strcat(strcat(LoadFiles," "),WorkFile);
    }
  }

  /* ------------------------------------------------------------ */
  /* ------------------------------------------------------------ */
  /* Link up all the specified and created objects */
  if (DoLink) {
    /* ------------------------------------------------------------ */
    /* Build and compile a link object to combine all the modules */
    LinkModule(LinkFiles,LinkCount);

    /* ------------------------------------------------------------ */
    /* Finish up the Load options string */
    (void)sprintf(LoadOptions+strlen(LoadOptions),
	    " %s -o %s %s -L%s/lib -l%stwine -ltbug -ltwinedef -lm",
	    (DebugTwine)?"-g":"",
	    ExeName,
	    TempObj,
	    TwineDir,
	    Optimize?"O":Debugger?"G":""
	    );

    /* ------------------------------------------------------------ */
    /* Perform the link */
    Command(TwineLD,"%s %s %s",LoadFiles,LoadOptions);
  }

  /* ------------------------------------------------------------ */
  /* ------------------------------------------------------------ */
  /* Remove the temp files (if they exist) */
  Cleanup();
}
