#include "world.h"


/**************************************************************************/
/* GLOBAL **************          MyAlloc          ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE AND RETURN A POINTER TO size BYTES OF MEMORY. IF THE */
/*          ALLOCATION FAILS, AN ERROR MESSAGE IS PRINTED AND EXECUTION   */
/*          TERMINATES.                                                   */
/**************************************************************************/

char *MyAlloc( size )
int size;
{
    char *p;

    if ( (p = malloc( size )) == NULL )
        Error1( "MALLOC FAILED" );

    return( p );
}


/**************************************************************************/
/* LOCAL  **************      MyBBlockAlloc        ************************/
/**************************************************************************/
/* PURPOSE: RETURN A POINTER TO A BBLOCK WHICH CAN BECOME EITHER A NODE  */
/*          OR EDGE OR INFO.  IT MAY NEVER BE FREED!!!!                   */
/**************************************************************************/

#define MAX_BBLOCKS 5000

union bblock {
  NODE n;
  EDGE e;
  INFO i;
  SET  s;
  ADE  a;
  CALL c;
  };

typedef union bblock BBLOCK, *PBBLOCK;

static PBBLOCK pool;
static int     pidx = MAX_BBLOCKS+100;

static PBBLOCK MyBBlockAlloc()
{
  if ( pidx >= MAX_BBLOCKS ) {
    pool = (PBBLOCK) MyAlloc( sizeof(BBLOCK)*MAX_BBLOCKS );
    pidx = 0;
    }

  return( &(pool[pidx++]) );
}


/**************************************************************************/
/* GLOBAL **************      GetNewCallNode       ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE AND RETURN A NEW CALL STRUCTURE.                     */
/**************************************************************************/

PCALL GetNewCallNode()
{
  PBBLOCK b;

  b = MyBBlockAlloc();
  return( &(b->c) );
}


/**************************************************************************/
/* GLOBAL **************      HasWriteExport       ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF NODE n HAS AN EXPORT CLASSIFIED AS A WRITE.    */
/*          SUCH AN EXPORT IS GUARANTEED TO BE AN AGGREGATE.              */
/**************************************************************************/

int HasWriteExport( n )
PNODE n;
{
    register PEDGE e;

    for ( e = n->exp; e != NULL; e = e->esucc )
	if ( e->wmark )
	    return( TRUE );

    return( FALSE );
}


/**************************************************************************/
/* LOCAL  **************       AreConstsEqual      ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF CONSTANT c1 IS THE SAME AS CONSTANT c2, ELSE   */
/*          RETURN FALSE.  TWO ERROR CONSTANTS ARE NOT CONSIDERED EQUAL.  */
/**************************************************************************/

static int AreConstsEqual( c1, c2 )
PEDGE c1;
PEDGE c2;
{
    if ( c1->CoNsT == NULL || c2->CoNsT == NULL )
	return( FALSE );

    if ( c1->info->type == c2->info->type )
	if ( strcmp( c1->CoNsT, c2->CoNsT ) == 0 )
	    return( TRUE );

    return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************       AreEdgesEqual       ************************/
/**************************************************************************/
/* PURPOSE: RETURNS TRUE IF THE EDGES e1 AND e2 ARE EQUAL.  TO BE EQUAL,  */
/*          THEY MUST EITHER BOTH DEFINE THE SAME CONSTANT OR ADDRESS     */
/*          THE SAME SOURCE NODE AND HAVE THE SAME SOURCE PORT NUMBERS.   */
/*          IF THEY ARE NOT EQUAL, FALSE IS RETURNED.                     */
/**************************************************************************/

int AreEdgesEqual( e1, e2 )
PEDGE e1;
PEDGE e2;
{
    if ( IsConst( e1 ) ^ IsConst( e2 ) ) /* XOR */
	return( FALSE );

    if ( IsConst( e1 ) )
	return( AreConstsEqual( e1, e2 ) );

    if ( (e1->src == e2->src) && (e1->eport == e2->eport) )
        return( TRUE );

    return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************     AreNodesDependent     ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF NODE n1 EXPORTS A VALUE USED BY NODE n2, ELSE  */
/*          RETURN FALSE.                                                 */
/**************************************************************************/

int AreNodesDependent( n1, n2 )
PNODE n1;
PNODE n2;
{
    register PEDGE e;

    for ( e = n1->exp; e != NULL; e = e->esucc ) {
	if ( e->dst == n2 )
	    return( TRUE );

	if ( IsGraph( e->dst ) )
	    continue;

        if ( AreNodesDependent( e->dst, n2 ) )
	    return( TRUE );
	}

    return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************       FindFunction        ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE FUNCTION GRAPH WITH NAME nm.  IF THE FUNCTION IS   */
/*          NOT FOUND, AN ERROR MESSAGE IS PRINTED.  NOTE: THIS ROUTINE   */
/*          WILL NOT WORK CORRECTLY UNTIL AFTER THE CALL GRAPH HAS BEEN   */
/*          BUILT AND THE FUNCTION GRAPHS HAVE BEEN SORTED.               */
/**************************************************************************/

PNODE FindFunction( nm )
char *nm;
{
    register PNODE f;

    for ( f = fhead; f != NULL; f = f->gsucc )
	if ( strcmp( f->G_NAME, nm ) == 0 )
	    return( f );

    fprintf( stderr, "HELP: %s\n", nm );
    Error1( "FindFunction: FUNCTION NOT FOUND" );
}


/**************************************************************************/
/* GLOBAL **************        IsAggregate        ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF SYMBOL TABLE ENTRY i DEFINES AN AGGREGATE:     */
/*          UNION, RECORD, ARRAY, STREAM, OR MULTIPLE THERE OF.           */
/**************************************************************************/

int IsAggregate( i )
PINFO i;
{
    if ( IsMultiple( i ) )
	i = i->A_ELEM;

    switch( i->type ) {
	case IF_ARRAY:
	case IF_UNION:
	case IF_STREAM:
	case IF_RECORD:
	    return( TRUE );

        default:
	    break;
        }

    return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************        UsageCount         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE NUMBER OF EXPORT EDGES ATTACHED TO NODE n WITH     */
/*          EXPORT PORT eport.                                            */
/**************************************************************************/

int UsageCount( n, eport )
PNODE n;
int   eport;
{
    register PEDGE e;
    register int   cnt = 0;

    for ( e = n->exp; e != NULL; e = e->esucc )
	if ( e->eport == eport )
	    cnt++;

    return( cnt );
}


/**************************************************************************/
/* GLOBAL **************        FindLastNode       ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE LAST NODE IN THE NODE LIST BEGINNING WITH NODE n;  */
/*          NOTE n ITSELF MIGHT BE THE LAST NODE.                         */
/**************************************************************************/

PNODE FindLastNode( n )
register PNODE n;
{
    while ( n->nsucc != NULL )
	n = n->nsucc;

    return( n );
}


/**************************************************************************/
/* GLOBAL **************        FindExport         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE FIRST EXPORT OF NODE n WITH EXPORT PORT NUMBER     */ 
/*          eport.  IF NOT FOUND, RETURN NULL.                            */
/**************************************************************************/

PEDGE FindExport( n, eport )
PNODE n;
int   eport;
{
    register PEDGE e;

    for ( e = n->exp; e != NULL; e = e->esucc )
	if ( e->eport == eport )
	    return( e );

    return( NULL );
}


/**************************************************************************/
/* GLOBAL **************        FindImport         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE IMPORT OF NODE n WITH IMPORT PORT NUMBER iport. IF */
/*          NOT FOUND, RETURN NULL.                                       */
/**************************************************************************/

PEDGE FindImport( n, iport )
PNODE n;
int   iport;
{
    register PEDGE i;

    for ( i = n->imp; i != NULL; i = i->isucc )
	if ( i->iport == iport )
	    return( i );

    return( NULL );
}


/**************************************************************************/
/* GLOBAL **************        LinkGraph          ************************/
/**************************************************************************/
/* PURPOSE: LINK GRAPH NODE g TO THE DOUBLE LINK LIST CONTAINING GRAPH    */
/*          NODE pred SO TO DIRECTLY FOLLOW pred.  IF pred IS NULL, THE   */
/*          INSERTION WILL NOT TAKE PLACE; REGUARDLESS, NODE g IS ALWAYS  */
/*          RETURNED.                                                     */
/**************************************************************************/

PNODE LinkGraph( pred, g )
PNODE pred;
PNODE g;
{
    if ( pred == NULL )
	return( g );

    g->gsucc = pred->gsucc;
    g->gpred = pred;

    if ( pred->gsucc != NULL )
	pred->gsucc->gpred = g;

    pred->gsucc = g;

    return( g );
}


/**************************************************************************/
/* GLOBAL **************        UnlinkGraph        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK GRAPH NODE g FROM ITS DOUBLE LINK LIST. IF IT EXISTS,  */
/*          THE PREDECESSOR OF g IS RETURNED, ELSE g'S SUCCESSOR IS       */
/*          RETURNED.                                                     */
/**************************************************************************/

PNODE UnlinkGraph( g )
PNODE g;
{
    register PNODE pred = g->gpred;

    if ( pred != NULL )
        pred->gsucc = g->gsucc;

    if ( g->gsucc != NULL )
	g->gsucc->gpred = pred;

    pred     = (pred != NULL)? pred : g->gsucc;
    g->gsucc = NULL;
    g->gpred = NULL;

    return( pred );
}


/**************************************************************************/
/* GLOBAL **************         LinkNode          ************************/
/**************************************************************************/
/* PURPOSE: LINK NODE n TO THE DOUBLE LINK LIST CONTAINING NODE pred SO   */
/*          TO FOLLOW pred. IF pred IS NULL, THE INSERTION IS NOT DONE;   */
/*          REGUARDLESS, NODE n IS RETURNED.                              */
/**************************************************************************/

PNODE LinkNode( pred, n )
PNODE pred;
PNODE n;
{
    if ( pred == NULL )
	return( n );

    n->nsucc = pred->nsucc;
    n->npred = pred;

    if ( pred->nsucc != NULL )
	pred->nsucc->npred = n;

    pred->nsucc = n;

    return( n );
}


/**************************************************************************/
/* GLOBAL **************         UnlinkNode        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK NODE n FROM ITS DOUBLE LINK LIST. IF IT EXISTS, THE    */
/*          PREDECESSOR OF n IS RETURNED, ELSE n'S SUCCESSOR IS RETURNED. */
/**************************************************************************/

PNODE UnlinkNode( n )
PNODE n;
{
    register PNODE pred = n->npred;

    if ( pred != NULL )
        pred->nsucc = n->nsucc;

    if ( n->nsucc != NULL )
	n->nsucc->npred = pred;

    pred     = (pred != NULL)? pred : n->nsucc;
    n->nsucc = NULL;
    n->npred = NULL;

    return( pred );
}


/**************************************************************************/
/* GLOBAL **************        LinkImport         ************************/
/**************************************************************************/
/* PURPOSE: ADD IMPORT e TO THE IMPORT LIST of NODE dst IN iport ORDER.   */
/*          THE PREDECESSOR OF THE FIRST IMPORT IS ALWAYS NULL.           */
/**************************************************************************/

void LinkImport( dst, e )
register PNODE dst;
register PEDGE e;
{
    register PEDGE i;

    e->ipred = NULL;
    e->isucc = NULL;
    e->dst   = dst;

    if ( dst->imp == NULL ) {                       /* IMPORT LIST EMPTY */
        dst->imp = e;
        return;
        }

    if ( dst->imp->iport > e->iport ) {          /* BEFORE FIRST IN LIST */
        e->isucc        = dst->imp;
        dst->imp->ipred = e;
	dst->imp        = e;
        return;
        }

    for( i = dst->imp; i->isucc != NULL; i = i->isucc )       /* WHERE? */
	if ( i->isucc->iport > e->iport )
	    break;

    e->isucc  = i->isucc;                               /* LINK AFTER i */
    e->ipred  = i;

    if ( i->isucc != NULL )
        i->isucc->ipred = e;

    i->isucc = e;
}


/**************************************************************************/
/* GLOBAL **************       UnlinkImport        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK IMPORT i FROM ITS DESTINATION NODE's IMPORT LIST.      */
/**************************************************************************/

void UnlinkImport( i )
PEDGE i;
{
    if ( i->ipred == NULL ) {                         /* FIRST ONE IN LIST */
	i->dst->imp = i->isucc;

	if ( i->isucc != NULL )
	    i->isucc->ipred = NULL;
    } else {
	i->ipred->isucc = i->isucc;

	if ( i->isucc != NULL )
	    i->isucc->ipred = i->ipred;
    }
}


/**************************************************************************/
/* GLOBAL **************        LinkExport         ************************/
/**************************************************************************/
/* PURPOSE: ADD EDGE e TO THE HEAD OF src'S EXPORT LIST. THE PEDECESSOR   */  
/*          OF THE FIRST EXPORT IS ALWAYS NULL.                           */
/**************************************************************************/


void LinkExport( src, e )
PNODE src;
PEDGE e;
{
    e->src = src;

    e->epred = NULL;
    e->esucc = src->exp;

    if ( src->exp != NULL )
	src->exp->epred = e;

    src->exp = e;
}


/**************************************************************************/
/* GLOBAL **************       UnlinkExport        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK EDGE e FROM ITS SOURCE NODE'S EXPORT LIST.  IF e IS A  */
/*          CONSTANT, NOTHING IS DONE.                                    */
/**************************************************************************/

void UnlinkExport( e )
PEDGE e;
{
    if ( IsConst( e ) )
	return;

    if ( e->epred == NULL ) {                         /* FIRST ONE IN LIST */
	e->src->exp = e->esucc;

	if ( e->esucc != NULL )
	    e->esucc->epred = NULL;
    } else {
	e->epred->esucc = e->esucc;

	if ( e->esucc != NULL )
	    e->esucc->epred = e->epred;
    }
}


/**************************************************************************/
/* GLOBAL **************       LinkAdeImport       ************************/
/**************************************************************************/
/* PURPOSE: ADD ADE IMPORT a TO THE HEAD OF dst'S ADE IMPORT LIST.  THE   */
/*          PREDECESSOR OF THE FIRST IMPORT IS ALWAYS NULL.               */
/**************************************************************************/

void LinkAdeImport( dst, a )
register PNODE dst;
register PADE  a;
{
    a->dst = dst;

    a->ipred = NULL;
    a->isucc = dst->aimp;

    if ( dst->aimp != NULL )
	dst->aimp->ipred = a;

    dst->aimp = a;
}


/**************************************************************************/
/* GLOBAL **************      UnlinkAdeImport      ************************/
/**************************************************************************/
/* PURPOSE: UNLINK ADE IMPORT a FROM ITS DESTINATION NODE's ADE IMPORT    */
/*          LIST.                                                         */
/**************************************************************************/

void UnlinkAdeImport( a )
PADE a;
{
    if ( a->ipred == NULL ) {                         /* FIRST ONE IN LIST */
	a->dst->aimp = a->isucc;

	if ( a->isucc != NULL )
	    a->isucc->ipred = NULL;
    } else {
	a->ipred->isucc = a->isucc;

	if ( a->isucc != NULL )
	    a->isucc->ipred = a->ipred;
    }
}


/**************************************************************************/
/* GLOBAL **************       LinkAdeExport       ************************/
/**************************************************************************/
/* PURPOSE: ADD ADE EXPORT a TO THE HEAD OF src'S ADE EXPORT LIST. THE    */
/*          PEDECESSOR OF THE FIRST EXPORT IS ALWAYS NULL.                */
/**************************************************************************/


void LinkAdeExport( src, a )
PNODE src;
PADE  a;
{
    a->src = src;

    a->epred = NULL;
    a->esucc = src->aexp;

    if ( src->aexp != NULL )
	src->aexp->epred = a;

    src->aexp = a;
}


/**************************************************************************/
/* GLOBAL **************       UnlinkAdeExport     ************************/
/**************************************************************************/
/* PURPOSE: UNLINK ADE a FROM ITS SOURCE NODE'S ADE EXPORT LIST.          */
/**************************************************************************/

void UnlinkAdeExport( a )
PADE a;
{
    if ( a->epred == NULL ) {                         /* FIRST ONE IN LIST */
	a->src->aexp = a->esucc;

	if ( a->esucc != NULL )
	    a->esucc->epred = NULL;
    } else {
	a->epred->esucc = a->esucc;

	if ( a->esucc != NULL )
	    a->esucc->epred = a->epred;
    }
}

static void IsPathInit( nn )
PNODE nn;
{
    register PNODE g, n;

    for ( g = nn; !IsGraph(g); g = g->npred );
    for ( n = g->nsucc; n != NULL; n = n->nsucc )
	n->checked = FALSE;
}


/**************************************************************************/
/* LOCAL  **************          IsPath           ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF A PATH EXISTS BETWEEN NODE src AND dst.  THIS  */
/*          ROUTINE ASSUMES THE GRAPH IS ACYCLIC.                         */
/**************************************************************************/

static int IsPath( src, dst )
PNODE src;
PNODE dst; /* FIXED */
{
    register PEDGE e;
    register PADE  a;

    if ( src == dst ) 
	return( TRUE );

    if ( src->checked ) 
	return( FALSE );

    src->checked = TRUE;

    for ( e = src->exp; e != NULL; e = e->esucc )
	if ( !IsGraph( e->dst ) )
	    if ( IsPath( e->dst, dst ) )
		return( TRUE );

    for ( a = src->aexp; a != NULL; a = a->esucc )
	if ( !IsGraph( a->dst ) )
	    if ( IsPath( a->dst, dst ) )
	        return( TRUE );

    return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************     CreateAndInsertAde    ************************/
/**************************************************************************/
/* PURPOSE: CREATE AN ARTIFICIAL DEPENDENCE EDGE WITH SOURCE NODE src,    */
/*          DESTINATION NODE dst, AND PRIORITY p; MAKING THE APPROPRIATE  */
/*          LINK LIST INSERTIONS. IF THE ADE WOULD INTRODUCE A CYCLE, IT  */
/*          IS NOT INSERTED. NOTE: BOUND EDGES NEVER INTRODUCE CYCLES.    */
/**************************************************************************/

void CreateAndInsertAde( src, dst, p )
PNODE src;
PNODE dst;
int   p;
{
    register PADE a;

    if ( p != BOUND ) {
	IsPathInit( src );
	if ( IsPath( dst, src ) ) {
	    cycle++;
	    return;
	    }
        }

    a = AdeAlloc( src, dst, p );

    LinkAdeExport( src, a );
    LinkAdeImport( dst, a );
}


/**************************************************************************/
/* GLOBAL **************       IsAdePresent        ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF AN ADE EXIST BETWEEN NODe src (THE ORIGIN) AND */
/*          NODE dst (THE DESTINATION), ELSE RETURN FALSE.                */
/**************************************************************************/

int IsAdePresent( src, dst )
PNODE src;
PNODE dst;
{
    register PADE a;

    for ( a = src->aexp; a != NULL; a = a->esucc )
	if ( a->dst == dst )
	    return( TRUE );

    return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************      LinkAssocLists       ************************/
/**************************************************************************/
/* PURPOSE: LINK THE ASSOCIATION LIST lst2 TO lst1 AND RETURN THE RESULT. */
/*          IF THE FIRST LIST IS EMPTY, THE SECOND IS RETURNED.           */
/**************************************************************************/

PALIST LinkAssocLists( lst1, lst2 )
PALIST lst1;
PALIST lst2;
{
    register PALIST l;
    register PALIST prev = NULL;

    for( l = lst1; l != NULL; l = l->next )
	prev = l;

    if ( prev == NULL )
	return( lst2 );

    prev->next = lst2;

    return( lst1 );
}


/**************************************************************************/
/* GLOBAL **************        CopyString         ************************/
/**************************************************************************/
/* PURPOSE: RETURN A COPY OF THE INPUT STRING s.                          */
/**************************************************************************/

char *CopyString( s ) 
char *s;
{
    return( strcpy( MyAlloc( strlen( s ) + 1 ), s ) );
}


/**************************************************************************/
/* GLOBAL **************         Warning1          ************************/
/**************************************************************************/
/* PURPOSE: PRINT A WARNING MESSAGE TO stderr, THEN CONTINUE AS NORMAL.   */
/**************************************************************************/

void Warning1( msg1 )
char *msg1;
{
    fprintf( stderr, "%s: W - %s\n", program, msg1 );
}


/**************************************************************************/
/* GLOBAL **************          Error1           ************************/
/**************************************************************************/
/* PURPOSE: PRINT AN ERROR MEASAGE TO stderr AND ABORT EXECUTION.         */
/**************************************************************************/

void Error1( msg1 )
char *msg1;
{
    fprintf( stderr, "%s: E - %s\n", program, msg1 );
    Stop( ERROR );
}


/**************************************************************************/
/* GLOBAL **************          Error2           ************************/
/**************************************************************************/
/* PURPOSE: PRINT TWO ERROR MEASAGES TO stderr AND ABORT EXECUTION.       */
/**************************************************************************/

void Error2( msg1, msg2 )
char *msg1;
char *msg2;
{
    fprintf( stderr, "%s: E - %s %s\n", program, msg1, msg2 );
    Stop( ERROR );
}


/**************************************************************************/
/* GLOBAL **************          AdeAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN ARTIFICIAL DEPENDENCE     */
/*          EDGE.                                                         */
/**************************************************************************/

PADE AdeAlloc( src, dst, p )
PNODE src;
PNODE dst;
int   p;
{
    PADE a;
    PBBLOCK b;

    /* a = (PADE) MyAlloc( sizeof(ADE) ); */
    b = MyBBlockAlloc();
    a = &(b->a);

    a->dst = dst;
    a->src = src;

    a->epred = a->esucc = NULL;
    a->ipred = a->isucc = NULL;

    a->priority = p;

    return( a );
}


/**************************************************************************/
/* GLOBAL **************         EdgeAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN EDGE.                     */
/**************************************************************************/

PEDGE EdgeAlloc( src, eport, dst, iport )
PNODE src;
int   eport;
PNODE dst;
int   iport;
{
    register PEDGE e;
    PBBLOCK b;

    /* e = (PEDGE) MyAlloc( sizeof(EDGE) ); */
    b = MyBBlockAlloc();
    e = &(b->e);

    e->eport = eport;
    e->src   = src;
    e->iport = iport;
    e->dst   = dst;

    e->usucc = NULL;
    e->epred = e->esucc = NULL;
    e->ipred = e->isucc = NULL;
    e->grset = e->gwset = NULL;
    e->lrset = e->lwset = NULL;

    InitPragmas( e );

    return( e );
}


/**************************************************************************/
/* GLOBAL **************         NodeAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN A NODE.                      */
/**************************************************************************/

PNODE NodeAlloc( label, type )
int   label;
int   type;
{
    register PNODE n;
    PBBLOCK b;

    /* n = (PNODE) MyAlloc( sizeof(NODE) ); */
    b = MyBBlockAlloc();
    n = &(b->n);

    n->label = label;
    n->type  = type;

    n->imp   = NULL;
    n->exp   = NULL;
    n->aimp  = NULL;
    n->aexp  = NULL;
    n->usucc = NULL;
    n->color = WHITE;

    n->gpred = n->gsucc = NULL;
    n->npred = n->nsucc = NULL;

    n->visited  = FALSE;
    n->sorted   = FALSE;
    n->executed = FALSE;
    n->checked  = FALSE;

    InitPragmas( n );

    return( n );
}


/**************************************************************************/
/* GLOBAL **************         InfoAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN INFO NODE.                */
/**************************************************************************/


PINFO InfoAlloc( label, type )
int   label;
int   type;
{
    register PINFO i;
    PBBLOCK b;

    /* i = (PINFO) MyAlloc( sizeof(INFO) ); */
    b = MyBBlockAlloc();
    i = &(b->i);

    i->info1 = i->info2 = NULL;

    i->label = label;
    i->type  = type;

    InitPragmas( i );

    i->next  = NULL;

    return( i );
}


/**************************************************************************/
/* GLOBAL **************      AssocListAlloc       ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN ASSOCIATION LIST NODE.    */
/**************************************************************************/

PALIST AssocListAlloc( datum )
int datum;
{
    register PALIST l;

    l = (PALIST) MyAlloc( sizeof(ALIST) );

    l->next  = NULL;
    l->datum = datum;

    return( l );
}


/**************************************************************************/
/* GLOBAL **************          SetAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN A READ/WRITE SET.            */
/**************************************************************************/

PSET SetAlloc( ssucc, scope )
PSET  ssucc;
PNODE scope;
{
    PSET s;
    PBBLOCK b;

    /* s = (PSET) MyAlloc( sizeof(SET) ); */
    b = MyBBlockAlloc();
    s = &(b->s);

    s->ssucc = ssucc;
    s->graph = scope;

    s->set   = NULL;
    s->last  = -1;
    s->ssize = -1;

    s->gen   = NULL;

    return( s );
}


/**************************************************************************/
/* GLOBAL **************        EnterInSet         ************************/
/**************************************************************************/
/* PURPOSE: ENTER EDGE e IN READ/WRITE SET s.  OVERFLOW RESULTS IN AN     */
/*          ERROR MESSAGE.                                                */
/**************************************************************************/

void EnterInSet( s, e )
PSET  s;
PEDGE e;
{
    register PEDGE *p;
    register int    i;

    if ( s == NULL )
	return;


    if ( s->last == s->ssize ) {
      p = s->set;
      s->set = (PEDGE*) MyAlloc( sizeof(PEDGE) * (s->ssize + SET_INC + 1) );

      for ( i = 0; i <= s->last; i++ )
	s->set[i] = p[i];

      s->ssize += SET_INC;

      /* if ( p != NULL ) */
        /* free( p ); */
      }

    s->last++;
    s->set[s->last] = e;
}
