#include "world.h"

#include <sys/time.h>

#ifdef USE_TIMES
#include <sys/types.h>
#include <sys/times.h>
#ifdef SGI
#include <sys/param.h>
#endif
#else
#ifndef CRAY
#include <sys/resource.h>
static struct rusage StartUsage;
static struct rusage StopUsage;
#endif
#endif

static double StartTime = 0.0;
static double StopTime  = 0.0;


/**************************************************************************/
/* LOCAL  **************      ElapsedTime          ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE ELAPSED TIME BETWEEN Start AND Stop.               */
/**************************************************************************/

static double ElapsedTime( Start, Stop )
struct timeval *Start;
struct timeval *Stop;
{
  double Seconds;
  double USeconds; /* MICROSECONDS */

  Seconds  = (double) (Stop->tv_sec  - Start->tv_sec);
  USeconds = (double) (Stop->tv_usec - Start->tv_usec); 

  if ( USeconds < 0.0 ) {
    USeconds += 1000000.0;
    Seconds  -= 1.0;
    }

  return( Seconds + (USeconds / 1000000.0) );
}


#ifndef CRAY
/**************************************************************************/
/* GLOBAL **************          TSECND           ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE ELAPSED CPU TIME SINCE THE START OF THE PROGRAM.   */
/*          NOTE THAT TSECND IS A CRAY INTRINSIC!                         */
/**************************************************************************/

double TSECND()
{
  register double CurrentCpuTime;

#ifdef USE_TIMES
  struct tms         StopTime;
#else
  struct rusage StartTimerInfo;
  struct rusage StopTimerInfo;
#endif

#ifdef USE_TIMES
  times(&StopTime);
  CurrentCpuTime  = ((double) (StopTime.tms_utime + StopTime.tms_stime));
#ifdef SGI
  CurrentCpuTime /= ((double) HZ);
#else
  CurrentCpuTime /= 60.0; /* SYSTEM DEPENDENT!!! */
#endif

#else
  getrusage( RUSAGE_SELF, &StopTimerInfo );

  StartTimerInfo.ru_utime.tv_sec  = 0;
  StartTimerInfo.ru_utime.tv_usec = 0;
  StartTimerInfo.ru_stime.tv_sec  = 0;
  StartTimerInfo.ru_stime.tv_usec = 0;

  CurrentCpuTime = 
	 ElapsedTime( &StartTimerInfo.ru_utime, &StopTimerInfo.ru_utime ) +
         ElapsedTime( &StartTimerInfo.ru_stime, &StopTimerInfo.ru_stime );
#endif

  return( CurrentCpuTime );
}
#endif


/**************************************************************************/
/* GLOBAL **************       StartProfiler       ************************/
/**************************************************************************/
/* PURPOSE: RECORD THE CURRENT TSECND VALUE IN StartTime                  */
/**************************************************************************/

void StartProfiler()
{
  if ( prof )
    StartTime = TSECND();
}

/**************************************************************************/
/* GLOBAL **************        StopProfiler       ************************/
/**************************************************************************/
/* PURPOSE: PRINT THE CPU TIME SINCE THE LAST CALL TO StartProfiler ALONG */
/*          WITH msg TO stderr.                                           */
/**************************************************************************/

void StopProfiler( msg )
char *msg;
{
  if ( prof )
    fprintf( stderr, "PROFILE [%s,%s] = %lf seconds\n",
	     program, msg, TSECND() - StartTime );
}
