#include "world.h"


#define MAX_NHASH 11

static PNODE nhash[MAX_NHASH];
static PNODE nprd;


PINFO  itail       = NULL;                  /* SYMBOL TABLE TAIL POINTER  */
PINFO  ihead       = NULL;                  /* SYMBOL TABLE HEAD POINTER  */

PNODE  glstop      = NULL;                  /* GRAPH LIST STACK TOP       */

static PNODE nlstop = NULL;                 /* NODE  LIST STACK TOP       */

char  *stamps[128] =                        /* STAMPS FOR ANY CHARACTER   */
    { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };

char  *sfile  = NULL;                            /* SISAL FILE BEING READ */
PNODE  cfunct = NULL;                   /* FUNCTION BEING BUILT OR EXAMED */

PINFO ptr_real    = NULL;                 /* VARIOUS SYMBOL TABLE ENTRIES */
PINFO ptr_double  = NULL;
PINFO ptr_integer = NULL;
PINFO ptr         = NULL;
PINFO integer     = NULL;


/**************************************************************************/
/* GLOBAL **************       MakePtrTypes        ************************/
/**************************************************************************/
/* PURPOSE: MAKE POINTER SYMBOL TABLE ENTRIES.                            */
/**************************************************************************/

void MakePtrTypes()
{
  ptr_real = FindInfo( ++maxint, IF_PTR_REAL );
  ptr_real->tname = "float*";

  ptr_double = FindInfo( ++maxint, IF_PTR_DOUBLE );
  ptr_double->tname = "double*";

  ptr_integer = FindInfo( ++maxint, IF_PTR_INTEGER );
  ptr_integer->tname = "int*";

  ptr = FindInfo( ++maxint, IF_PTR );
  ptr->tname = "POINTER";
}


/**************************************************************************/
/* GLOBAL **************    AssignSourceFileName   ************************/
/**************************************************************************/
/* PURPOSE: IF PRESENT, REPLACE FILE NAME n's ".if1" SUFFIX WITH ".sis".  */
/*          N is UPDATED IN SITU AND ASSIGNED TO sfile.                   */
/**************************************************************************/

void AssignSourceFileName( n )
char *n;
{
    register char *p;

    for ( p = n + (strlen( n ) - 1); p != n; p-- )
        if ( *p == '.' )
	    if ( strcmp( p, ".if1" ) == 0 ) {
	        *(++p) = 's'; *(++p) = 'i'; *(++p) = 's';
	        break;
	        }

    sfile = n;
}


/**************************************************************************/
/* LOCAL  **************       PushNodeList        ************************/
/**************************************************************************/
/* PURPOSE: ADD A NEW NODE LIST HEADER TO THE CURRENT NODE LIST HEADER    */
/*          STACK. WHEN DONE, nlstop WILL ADDRESS THIS NEW HEADER.        */
/**************************************************************************/

static void PushNodeList()
{
    register int idx;

    /* CLEAR THE FAST NODE LOOKUP HASH TABLE */
    for ( idx = 0; idx < MAX_NHASH; idx++ )
	nhash[idx] = NULL;

    if ( nlstop != NULL )
      nlstop->usucc = nprd;
    nprd = NULL;

    nlstop = LinkGraph( nlstop, NodeAlloc( 0, 0 ) );
}


/**************************************************************************/
/* LOCAL  **************       PushGraphList       ************************/
/**************************************************************************/
/* PURPOSE: ADD A NEW GRAPH LIST HEADER TO THE CURRENT GRAPH LIST HEADER  */
/*          STACK. WHEN DONE, glstop WILL ADDRESS THIS NEW HEADER.        */
/**************************************************************************/

static void PushGraphList() 
{ 
    glstop = LinkNode( glstop, NodeAlloc( 0, 0 ) );
}


/**************************************************************************/
/* LOCAL  **************       PopGraphList        ************************/
/**************************************************************************/
/* PURPOSE: REMOVE THE TOP GRAPH LIST HEADER FROM GRAPH LIST HEADER STACK */
/*          IF IT IS NOT ALREADY EMPTY. WHEN DONE, glstop WILL ADDRESS THE*/
/*          PREVIOUS GRAPH LIST HEADER.  THE REMOVED HEADER NODE IS FREED.*/
/**************************************************************************/

static void PopGraphList() 
{ 
    register PNODE r;

    if ( (r = glstop) == NULL )
	return;

    glstop = UnlinkNode( glstop );

    /* free( r ); */
}


/**************************************************************************/
/* LOCAL  **************        PopNodeList        ************************/
/**************************************************************************/
/* PURPOSE: REMOVE THE TOP NODE LIST HEADER FROM NODE LIST HEADER STACK   */
/*          IF IT IS NOT ALREADY EMPTY. WHEN DONE, nlstop WILL ADDRESS    */
/*          THE PREVIOUS NODE LIST HEADER ON THE STACK. THE NODE LIST OF  */
/*          THE HEADER IS ASSIGNED TO THE GRAPH NODE OF THE GRAPH         */
/*          CURRENTLY BEING COMPLETED (ADDRESSED BY glstop->gpred ). THEN */
/*          THE REMOVED HEADER IS FREED.                                  */
/**************************************************************************/

static void  PopNodeList() 
{ 
    register PNODE r;
    register int   idx;

    /* CLEAR THE FAST NODE LOOKUP HASH TABLE */
    for ( idx = 0; idx < MAX_NHASH; idx++ )
	nhash[idx] = NULL;
    nprd = NULL;

    if ( nlstop == NULL )
	return;

    if ( glstop->gpred != NULL ) {
        glstop->gpred->G_NODES = nlstop->nsucc;

        if ( nlstop->nsucc != NULL )
	    nlstop->nsucc->npred = glstop->gpred;
        }

    nlstop = UnlinkGraph( r = nlstop );

    /* RESTORE OLD NODE LIST TAIL */
    if ( nlstop != NULL )
      nprd = nlstop->usucc;

    /* free( r ); */
}


/**************************************************************************/
/* GLOBAL **************        EnterScope         ************************/
/**************************************************************************/
/* PURPOSE: PUSH NEW GRAPH LIST AND NODE LIST HEADERS ON THE ASSOCIATED   */
/*          HEADER STACKS. THIS ROUTINE IS CALLED BEFORE A NEW COMPOUND   */
/*          NODE IS TO BE BUILT (WHEN } SEEN IN IF1 INPUT).               */
/**************************************************************************/

void EnterScope() { PushGraphList(); PushNodeList(); }


/**************************************************************************/
/* GLOBAL **************         ExitScope         ************************/
/**************************************************************************/
/* PURPOSE: PERFORM A PARTIAL SCOPE EXIT BY POPING THE TOP NODE LIST      */
/*          HEADER OFF THE NODE LIST HEADER STACK.  THIS ROUTINE IS ONLY  */
/*          CALLED FROM If1Read AFTER EOF IS ENCOUNTERED SO TO PRESERVE   */
/*          gsltop AS A POINTER TO ALL OUTER MOST GRAPHS THAT WERE BUILT  */
/*          (THE FUNCTIONS).                                              */
/**************************************************************************/

void ExitScope() { PopNodeList(); }


/**************************************************************************/
/* LOCAL  **************       LookupInfo          ************************/
/**************************************************************************/
/* PURPOSE: UTILITY USED BY FindInfo TO LOCATE AN INFO NODE WITH LABEL    */
/*          label in THE INFO LIST HEADED BY ihead. IF IT IS NOT IN THE   */
/*          LIST, THEN A NEW INFO NODE IS ALLOCATED, APPENDED TO THE END, */
/*          AND RETURNED.                                                 */
/**************************************************************************/

static PINFO LookupInfo( label, type )
register int label;
         int type;
{
    register PINFO i;

    for ( i = ihead; i != NULL; i = i->next )
        if ( label == i->label ) {
            if ( i->type == IF_NONTYPE )
                i->type = type;

            return( i );
            }

    i = InfoAlloc( label, type );

    itail->next = i;
    itail       = i;

    return( i );
}


/**************************************************************************/
/* GLOBAL **************        FindInfo           ************************/
/**************************************************************************/
/* PURPOSE: LOCATE AND RETURN THE INFO NODE WITH LABEL label IN THE INFO  */
/*          LIST HEADED BY ihead.  IF THE NODE DOES NOT EXIST, IT IS      */
/*          ALLOCATED AND INSERTED INFO THE LIST. NULL IS RETURNED IF THE */
/*          REQUESTED LABEL IS 0.                                         */
/**************************************************************************/

PINFO FindInfo( label, type )
int label;
int type;
{
    if ( label <= 0 )
        return( NULL );

    if ( ihead == NULL )
        return( ihead = itail = InfoAlloc( label, type ) );

    return( LookupInfo( label, type ) );
}


/**************************************************************************/
/* GLOBAL **************        MakeInfo           ************************/
/**************************************************************************/
/* PURPOSE: MAKE AN INFO NODE AND INITIALIZE IT (LINKING IT TO OTHER INFO */
/*          NODES IF REQUIRED BY ITS TYPE. IF BUILDING A BASIC TYPE THEN  */
/*          ADJUST IT SO THAT ALL TYPE IDENTIFIERS ARE UNIQUE.            */
/**************************************************************************/

void MakeInfo( label, type, ref1, ref2 )
int   label;
int   type;
int   ref1;
int   ref2;
{
    register PINFO i;
    register PINFO i1;
    register PINFO i2;

    switch ( type ) {
	case IF_BASIC:
            type = ref1 + BASIC_TYPE_START;
	    break;

	case IF_UNKNOWN:
	    break;

        default:
            i1 = FindInfo( ref1, IF_NONTYPE );
            i2 = FindInfo( ref2, IF_NONTYPE );
	    break;
        }

    if ( (type == IF_DOUBLE) && flt )
      type = IF_REAL;
    else if ( (type == IF_REAL) && dbl )
      type = IF_DOUBLE;

    i = FindInfo( label, type );
    i->if1line = line;

    AssignPragmas( i );

    i->funct = NULL;                 /* DON'T BOTHER FOR TYPE INFORMATION */
    i->file  = NULL;

    switch ( type ) {
        case IF_FUNCTION:
            i->F_IN  = i1;
            i->F_OUT = i2;
	    i->tname = "FUNCTION";
            break;

        case IF_STREAM:
	    Error2( "IF2gen-MakeInfo", "STREAM DATA TYPE ENCOUNTERED" );
            break;

        case IF_MULTPLE:
            i->A_ELEM = i1;
	    i->tname  = "Multiple";
            break;

        case IF_ARRAY:
            i->A_ELEM = i1;
	    i->tname = "POINTER";
            break;

	case IF_BUFFER:
            i->A_ELEM = i1;
	    i->tname  = "BUFFERP"; 
            break;

        case IF_UNION:
        case IF_RECORD:
	    i->tname = "POINTER";
            i->R_FIRST = i1;
            break;

        case IF_TUPLE: 
        case IF_FIELD:
        case IF_TAG:
            i->L_SUB  = i1;
            i->L_NEXT = i2;
	    i->tname  = "TupleFieldTag";
            break;

	case IF_DOUBLE:
	    i->tname = "double";
	    break;

	case IF_NULL:
	    i->tname = "char";
	    break;

	case IF_CHAR:
	    i->tname = "char";
	    break;

	case IF_REAL:
	    i->tname = "float";
	    break;

	case IF_BOOL:
	    i->tname = "char";
	    break;

	case IF_INTEGER:
	    i->tname = "int";
	    integer  = i;
	    break;

        default:
	    i->tname = "DefaultType";
	    break;
        }
}


/**************************************************************************/
/* LOCAL  **************       PlaceGraph          ************************/
/**************************************************************************/
/* PURPOSE: PLACE A NEW GRAPH NODE AT END OF GRAPH LIST ADDRESSED BY      */ 
/*          glstop. NOTE THAT glstop->gpred ADDRESSES THE LAST GRAPH NODE */
/*          IN THE LIST, BUT IT IS NOT ADJUSTED BY UnlinkGraph.           */
/**************************************************************************/

static void PlaceGraph( g )
PNODE g;
{
    if ( glstop->gsucc == NULL ) {
	glstop->gsucc = g;
	glstop->gpred = g;
	g->gpred      = glstop;
	}
    else
        glstop->gpred = LinkGraph( glstop->gpred, g );
}


/**************************************************************************/
/* LOCAL  **************         PlaceNode         ************************/
/**************************************************************************/
/* PURPOSE: PLACE A NEW NODE IN THE NODE LIST ADDRESSED BY nlstop IN      */
/*          LABEL ORDER (n->label).                                       */
/**************************************************************************/

static void PlaceNode( n )
register PNODE n;
{
    register PNODE nd;

    if ( nprd != NULL )
      if ( nprd->label < n->label ) {
	LinkNode( nprd, n );
	nprd = n;
	return;
	}

    for ( nd = nlstop; nd->nsucc != NULL; nd = nd->nsucc ) /* WHERE? */
	if ( nd->nsucc->label > n->label )
	    break;

    LinkNode( nd, n );

    if ( nprd == NULL || nd == nprd )
      nprd = n;
}


/**************************************************************************/
/* GLOBAL **************         FindGraph         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE nTH GRAPH IN THE GRAPH LIST ADDRESSED BY glstop.   */
/**************************************************************************/

static PNODE FindGraph( n )
int n;
{
    register PNODE g;

    for ( g = glstop->gsucc; g != NULL; g = g->gsucc, n-- )
        if ( n == 0 )
            break;

    return( g );
}


/**************************************************************************/
/* GLOBAL **************         FindNode          ************************/
/**************************************************************************/
/* PURPOSE: FIND AND RETURN THE NODE WITH LABEL label.  IF label IS 0,    */
/*          THE GRAPH NODE OF THE GRAPH BEING CONSTRUCTED IS BEING        */
/*          REFERENCED (LAST ONE IN THE GRAPH LIST ADDRESSED BY glstop),  */
/*          ELSE SEARCH FOR IT IN THE NODE LIST ADDRESSED BY nlstop. IF   */
/*          NOT FOUND, IT IS CREATED, PLACED IN THE LIST IN label ORDER,  */
/*          AND RETURNED.                                                 */
/**************************************************************************/

static PNODE FindNode( label, type )
int label;
int type;
{
    register PNODE n;

    if ( label == 0 )
        return( glstop->gpred );

    /* FIRST CHECK THE FAST NODE LOOKUP HASH TABLE */
    if ( (n = nhash[label%MAX_NHASH]) != NULL )
	if ( n->label == label ) {
            if ( n->type == IFUndefined )
                n->type = type;

            return( n );
	    }

    if ( nprd != NULL )
      if ( nprd->label < label )
	goto DoThePlacement;

    for ( n = nlstop->nsucc; n != NULL; n = n->nsucc )
        if ( n->label == label ) {
            if ( n->type == IFUndefined )
                n->type = type;

	    /* ENTER n INTO THE FAST LOOKUP HASH TABLE */
	    nhash[label%MAX_NHASH] = n;

            return( n );
            }

DoThePlacement:
    PlaceNode( n = NodeAlloc( label, type ) );

    /* ENTER n INTO THE FAST LOOKUP HASH TABLE */
    nhash[label%MAX_NHASH] = n;

    return( n );
}


/**************************************************************************/
/* LOCAL  **************        AssignTags         ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN TAGS FROM THE ASSOCIATION LIST lst TO SUBGRAPHS OF     */
/*          COMPOUND NODE dad. THE GRAPHS OF INTEREST ARE THOSE ADDRESSED */
/*          BY glstop. THESE SUBGRAPH GRAPHS ARE UPDATED TO ADDRESS dad.  */
/*          THE GRAPH FOUND BY THE nth ASSOCIATION LIST ENTRY IS ASSIGNED */
/*          THE TAG n.                                                    */
/**************************************************************************/

static void AssignTags( lst, dad )
PALIST lst;
PNODE  dad;
{
    register PNODE n;
    register int   tag;

    for ( n = glstop->gsucc; n !=NULL; n = n->gsucc )
        n->G_DAD = dad;

    for ( tag = 0; lst != NULL; lst = lst->next, tag++ ) {
        n = FindGraph( lst->datum ); 

        if ( n->G_TAGS == NULL )
            n->G_TAGS = AssocListAlloc( tag );
        else
            n->G_TAGS = LinkAssocLists( n->G_TAGS, AssocListAlloc( tag ) );
        }
}


/**************************************************************************/
/* GLOBAL **************       MakeCompound        ************************/
/**************************************************************************/
/* PURPOSE: MAKE A COMPOUND NODE.  ALL THE SUBGRAPHS HAVE BEEN BUILD AND  */
/*          ARE ADDRESSED BY glstop (IN ORDER OF ENCOUNTER). THE CURRENT  */
/*          NODE LIST ADDRESSED BY nlstop IS REMOVED TO UNCOVER THE SCOPE */
/*          DEFINING THE COMPOUND NODE AND TO COMPLETE CONSTRUCTION OF    */
/*          THE LAST SUBGRAPH.  THEN THE SUBGRAPHS ARE LINKED TO THE      */
/*          COMPOUND NODE, COMPLETING SCOPE EXIT.  THE Iterate AND        */
/*          IfThenElse NODES ARE NOT RECOGNIZED.                          */
/**************************************************************************/

void MakeCompound( label, type, cnt, lst )
int    label;
int    type;
int    cnt;
PALIST lst;
{
    register PNODE n;

    if ( (type == IFIterate) || (type == IFIfThenElse) )
	Error1( "Iterate AND IfThenElse NODEs NOT RECOGNIZED" );

    PopNodeList();

    n          = FindNode( label, type );
    n->if1line = line;

    AssignPragmas( n );

    n->C_SUBS = glstop->gsucc;

    if ( glstop->gsucc != NULL )
        glstop->gsucc->gpred = n;

    n->C_SCNT = cnt;
    n->C_ALST = lst;

    AssignTags( lst, n );

    PopGraphList();
}


/**************************************************************************/
/* GLOBAL **************         MakeNode          ************************/
/**************************************************************************/
/* PURPOSE: MAKE A SIMPLE NODE AND INITIALIZE IT.  IF IT ALREADY EXISTS,  */
/*          THEN JUST INITIALIZE IT. THE ReplaceMulti NODE IS NOT         */
/*          RECOGNIZED.                                                   */
/**************************************************************************/

void MakeNode( label, type )
int   label;
int   type;
{
    register PNODE n;

    if ( type == IFReplaceMulti )
	Error1( "ReplaceMulti NODEs NOT RECOGNIZED" );

    n          = FindNode( label, type );
    n->if1line = line;

    AssignPragmas( n );
}


/**************************************************************************/
/* GLOBAL **************         MakeGraph         ************************/
/**************************************************************************/
/* PURPOSE: MAKE A NEW GRAPH NODE AND INITIALIZE IT.  POP THE CURRENT NODE*/
/*          LIST HEADER OFF THE NODE LIST HEADER STACK SO TO COMPLETE     */
/*          CONSTRUCTION OF PREVIOUS GRAPH.  THEN PUSH A NEW NODE LIST    */
/*          HEADER FOR THE NODE LIST OF THE NEW GRAPH.  IF THE NEW GRAPH  */
/*          IS A FUNCTION, THEN FIND ITS TYPE INFORMATION.  THEN PLACE THE*/
/*          GRAPH NODE AT THE END OF THE CURRENT GRAPH LIST ADDRESSED BY  */
/*          glstop.                                                       */
/**************************************************************************/

void MakeGraph( type, label, name )
int   type;
int   label;
char *name;
{
    register PNODE n;

    n          = NodeAlloc( GRAPH_LABEL, type );
    n->if1line = line;

    PopNodeList();
    PushNodeList();

    if ( type != IFSGraph ) {
        n->G_INFO = FindInfo( label, IF_FUNCTION );
	n->G_NAME = name;
	cfunct    = n;
	}
    else
	n->G_TAGS = NULL;

    AssignPragmas( n );

    PlaceGraph( n );
}


/**************************************************************************/
/* GLOBAL **************         MakeEdge          ************************/
/**************************************************************************/
/* PURPOSE: MAKE AN EDGE AND INITIALIZE IT (INCLUDING FINDING ITS TYPE    */
/*          INFO).  THE EDGE IS LINKED TO ITS DESTINATION NODE (IN iport  */
/*          ORDER) AND ITS SOURCE NODE (IN RANDOM ORDER).                 */
/**************************************************************************/

void MakeEdge( snode, eport, dnode, iport, type )
int   snode, eport;
int   dnode, iport;
int   type;
{
    register PNODE dst;
    register PNODE src;
    register PEDGE e;

    dst  = FindNode( dnode, IFUndefined );
    src  = FindNode( snode, IFUndefined );

    e          = EdgeAlloc( src, eport, dst, iport );
    e->if1line = line;

    AssignPragmas( e );

    /* FOR SOME YET KNOWN REASON, dmark EDGES ARE NOT FREED IN ALL CASES. */
    /* THIS CAN BE FORCED BY SETTING THE CM PRAGMA OF dmark EDGES TO -1.  */
    if ( e->dmark ) /* CANN 10-3 */
      e->cm = -1;   /* CANN 10-3 */

    e->funct = NULL;                 /* DON'T BOTHER FOR EDGE INFORMATION */
    e->file  = NULL;

    LinkImport( dst, e );
    LinkExport( src, e );

    e->info = FindInfo( type, IF_NONTYPE );
}


/**************************************************************************/
/* GLOBAL **************         MakeConst         ************************/
/**************************************************************************/
/* PURPOSE: MAKE A CONSTANT EDGE AND INITIALIZE IT (INCLUDING FINDING ITS */
/*          TYPE INFO). THEN LINK THE EDGE INTO ITS DESTINATION NODE's    */
/*          IMPORT LIST IN label ORDER.                                   */
/**************************************************************************/

void MakeConst( dnode, iport, label, CoNsT )
int    dnode;
int    iport;
int    label;
char  *CoNsT;
{
    register PEDGE c;
    register PNODE dst;

    dst = FindNode( dnode, IFUndefined );
    
    c = EdgeAlloc( NULL, CONST_PORT, dst, iport );

    c->info    = FindInfo( label, IF_NONTYPE );
    c->CoNsT   = CoNsT;
    c->if1line = line;

    AssignPragmas( c );

    c->funct = NULL;                        /* DON'T BOTHER FOR CONSTANTS */
    c->file  = NULL;

    LinkImport( dst, c );
}


/**************************************************************************/
/* GLOBAL **************          MakeAde          ************************/
/**************************************************************************/
/* PURPOSE: DO NOTHING.  WE ASSUME THE INPUT IS TOTALLY ORDERED.          */
/**************************************************************************/

void MakeAde( snode, dnode )
int snode;
int dnode;
{
}


/**************************************************************************/
/* GLOBAL **************          IsStamp          ************************/
/**************************************************************************/
/* PURPOSE: RETURNS TRUE IF STAMP stamp IS DEFINED, ELSE FALSE.           */
/**************************************************************************/

int  IsStamp( stamp )
char stamp;
{
    if ( stamps[ stamp ] == NULL )
	return( FALSE );

    return( TRUE );
}


/**************************************************************************/
/* GLOBAL **************        RemoveStamp        ************************/
/**************************************************************************/
/* PURPOSE: REMOVE STAMP stamp FROM THE STAMP TABLE.                      */
/**************************************************************************/

void RemoveStamp( stamp ) char stamp;  { stamps[ stamp ] = NULL; }


/**************************************************************************/
/* GLOBAL **************         AddStamp          ************************/
/**************************************************************************/
/* PURPOSE: ADD STAMP stamp WITH COMMENTARY s TO THE STAMP TABLE.         */
/**************************************************************************/

void AddStamp( stamp, s ) char stamp; char *s; { stamps[ stamp ] = s; }


/**************************************************************************/
/* GLOBAL **************      GetStampString       ************************/
/**************************************************************************/
/* PURPOSE: RETURN A POINTER TO THE COMMENTARY FOR STAMP stamp.           */
/**************************************************************************/

char *GetStampString( stamp ) char stamp; { return( stamps[ stamp ] ); }
