#include "world.h"


int nchange = 0;                       /* WAS A NODE UNLINKED OR LINKED?  */
int echange = 0;                       /* WAS AN EDGE UNLINKED OR LINKED? */


/**************************************************************************/
/* GLOBAL **************          MyAlloc          ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE AND RETURN A POINTER TO size BYTES OF MEMORY. IF THE */
/*          ALLOCATION FAILS, AN ERROR MESSAGE IS PRINTED AND EXECUTION   */
/*          TERMINATES.                                                   */
/**************************************************************************/

char *MyAlloc( size )
int size;
{
    char *p;

    if ( (p = malloc( size )) == NULL )
        Error1( "MALLOC FAILED" );

    return( p );
}


/**************************************************************************/
/* LOCAL  **************      MyBBlockAlloc        ************************/
/**************************************************************************/
/* PURPOSE: RETURN A POINTER TO A BBLOCK WHICH CAN BECOME EITHER A NODE  */
/*          OR EDGE OR INFO.  IT MAY NEVER BE FREED!!!!                   */
/**************************************************************************/

#define MAX_BBLOCKS 5000

union bblock {
  NODE n;
  EDGE e;
  INFO i;
  };

typedef union bblock BBLOCK, *PBBLOCK;

static PBBLOCK pool;
static int     pidx = MAX_BBLOCKS+100;

static PBBLOCK MyBBlockAlloc()
{
  if ( pidx >= MAX_BBLOCKS ) {
    pool = (PBBLOCK) MyAlloc( sizeof(BBLOCK)*MAX_BBLOCKS );
    pidx = 0;
    }

  return( &(pool[pidx++]) );
}


/**************************************************************************/
/* GLOBAL **************        UsageCount         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE NUMBER OF EXPORT EDGES ATTACHED TO NODE n WITH     */
/*          EXPORT PORT eport.                                            */
/**************************************************************************/

int UsageCount( n, eport )
PNODE n;
int   eport;
{
    register PEDGE e;
    register int   cnt = 0;

    for ( e = n->exp; e != NULL; e = e->esucc )
        if ( e->eport == eport )
            cnt++;

    return( cnt );
}


/**************************************************************************/
/* GLOBAL **************       AreConstsEqual      ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF CONSTANT c1 IS THE SAME AS CONSTANT c2, ELSE   */
/*          RETURN FALSE.                                                 */
/**************************************************************************/

int AreConstsEqual( c1, c2 )
PEDGE c1;
PEDGE c2;
{
    if ( c1->info->type == c2->info->type )
        if ( strcmp( c1->CoNsT, c2->CoNsT ) == 0 )
            return( TRUE );

    return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************       ChangeToConst       ************************/
/**************************************************************************/
/* PURPOSE: BLINDLY CHANGE EDGE d INTO CONSTANT c.                        */
/**************************************************************************/

void ChangeToConst( d, c )
PEDGE d;
PEDGE c;
{
    d->CoNsT = c->CoNsT;
    d->info  = c->info; 
    d->src   = NULL;
    d->esucc = NULL;
    d->epred = NULL;
    d->eport = CONST_PORT;
}


/**************************************************************************/
/* GLOBAL **************     ChangeEdgeToConst     ************************/
/**************************************************************************/
/* PURPOSE: INTELLEGENTLY CHANGE EDGE d INTO CONSTANT c; UNLINKING d FROM */
/*          ITS SOURCE NODE'S EXPORT LIST.                                */
/**************************************************************************/

void ChangeEdgeToConst( d, c )
PEDGE d;
PEDGE c;
{
    UnlinkExport( d );
    ChangeToConst( d, c );
}


/**************************************************************************/
/* GLOBAL **************        FindLastNode       ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE LAST NODE IN THE NODE LIST BEGINNING WITH NODE n;  */
/*          NOTE n ITSELF MIGHT BE THE LAST NODE.                         */
/**************************************************************************/

PNODE FindLastNode( n )
register PNODE n;
{
    if ( n == NULL )
      return( NULL );

    while ( n->nsucc != NULL )
        n = n->nsucc;

    return( n );
}


/**************************************************************************/
/* GLOBAL **************     ChangeImportPorts     ************************/
/**************************************************************************/
/* PURPOSE: FOR ALL IMPORTS OF NODE n CHANGE IMPORT PORT NUMBERS MATCHING */
/*          oport TO nport.                                               */
/**************************************************************************/

void ChangeImportPorts( n, oport, nport )
PNODE n;
int   oport;
int   nport;
{
    register PEDGE i;

    for ( i = n->imp; i != NULL; i = i->isucc )
        if ( i->iport == oport )
            i->iport = nport;
}


/**************************************************************************/
/* GLOBAL **************     ChangeExportPorts    *************************/
/**************************************************************************/
/* PURPOSE: FOR ALL EXPORTS OF NODE n CHANGE EXPORT PORT NUMBERS MATCHING */
/*          oport TO nport.                                               */
/**************************************************************************/

void ChangeExportPorts( n, oport, nport )
PNODE n;
int   oport;
int   nport;
{
    register PEDGE e;

    for ( e = n->exp; e != NULL; e = e->esucc )
        if ( e->eport == oport )
            e->eport = nport;
}


/**************************************************************************/
/* GLOBAL **************        FindExport         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE FIRST EXPORT OF NODE n WITH EXPORT PORT NUMBER     */ 
/*          eport.  IF NOT FOUND, RETURN NULL.                            */
/**************************************************************************/

PEDGE FindExport( n, eport )
PNODE n;
int   eport;
{
    register PEDGE e;

    for ( e = n->exp; e != NULL; e = e->esucc )
        if ( e->eport == eport )
            return( e );

    return( NULL );
}


/**************************************************************************/
/* GLOBAL **************        FindImport         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE IMPORT OF NODE n WITH IMPORT PORT NUMBER iport. IF */
/*          NOT FOUND, RETURN NULL.                                       */
/**************************************************************************/

PEDGE FindImport( n, iport )
PNODE n;
register int iport;
{
    register PEDGE i;

    for ( i = n->imp; i != NULL; i = i->isucc )
        if ( i->iport == iport )
            return( i );

    return( NULL );
}


/**************************************************************************/
/* GLOBAL **************        LinkGraph          ************************/
/**************************************************************************/
/* PURPOSE: LINK GRAPH NODE g TO THE DOUBLE LINK LIST CONTAINING GRAPH    */
/*          NODE pred SO TO DIRECTLY FOLLOW pred.  IF pred IS NULL, THE   */
/*          INSERTION WILL NOT TAKE PLACE; REGUARDLESS, NODE g IS ALWAYS  */
/*          RETURNED.                                                     */
/**************************************************************************/

PNODE LinkGraph( pred, g )
PNODE pred;
PNODE g;
{
    if ( pred == NULL )
        return( g );

    g->gsucc = pred->gsucc;
    g->gpred = pred;

    if ( pred->gsucc != NULL )
        pred->gsucc->gpred = g;

    pred->gsucc = g;

    return( g );
}


/**************************************************************************/
/* GLOBAL **************        UnlinkGraph        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK GRAPH NODE g FROM ITS DOUBLE LINK LIST. IF IT EXISTS,  */
/*          THE PREDECESSOR OF g IS RETURNED, ELSE g'S SUCCESSOR IS       */
/*          RETURNED.                                                     */
/**************************************************************************/

PNODE UnlinkGraph( g )
PNODE g;
{
    register PNODE pred = g->gpred;

    if ( pred != NULL )
        pred->gsucc = g->gsucc;

    if ( g->gsucc != NULL )
        g->gsucc->gpred = pred;

    return( (pred != NULL)? pred : g->gsucc );
}


/**************************************************************************/
/* GLOBAL **************         LinkNode          ************************/
/**************************************************************************/
/* PURPOSE: LINK NODE n TO THE DOUBLE LINK LIST CONTAINING NODE pred SO   */
/*          TO FOLLOW pred. IF pred IS NULL, THE INSERTION IS NOT DONE;   */
/*          REGUARDLESS, NODE n IS RETURNED.                              */
/**************************************************************************/

PNODE LinkNode( pred, n )
PNODE pred;
PNODE n;
{
    nchange = TRUE;

    if ( pred == NULL )
        return( n );

    n->nsucc = pred->nsucc;
    n->npred = pred;

    if ( pred->nsucc != NULL )
        pred->nsucc->npred = n;

    pred->nsucc = n;

    return( n );
}


/**************************************************************************/
/* GLOBAL **************       LinkNodeLists       ************************/
/**************************************************************************/
/* PURPOSE: LINK THE NODE LIST OF GRAPH g2 TO THAT OF GRAPH g1. THE NODE  */
/*          LIST OF g2 IS SET TO NULL.                                    */
/**************************************************************************/

void LinkNodeLists( g1, g2 )
PNODE g1;
PNODE g2;
{
    register PNODE ln;
    register PNODE n;
    register PNODE ns;

    ln = FindLastNode( g1 );

    for ( n = g2->G_NODES; n != NULL; n = ns ) {
        ns = n->nsucc;
        ln = LinkNode( ln, n );
        }

    g2->G_NODES = NULL;
}


/**************************************************************************/
/* GLOBAL **************         UnlinkNode        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK NODE n FROM ITS DOUBLE LINK LIST. IF IT EXISTS, THE    */
/*          PREDECESSOR OF n IS RETURNED, ELSE n'S SUCCESSOR IS RETURNED. */
/**************************************************************************/

PNODE UnlinkNode( n )
PNODE n;
{
    register PNODE pred = n->npred;

    nchange = TRUE;

    if ( pred != NULL )
        pred->nsucc = n->nsucc;

    if ( n->nsucc != NULL )
        n->nsucc->npred = pred;

    return( (pred != NULL)? pred : n->nsucc );
}


/**************************************************************************/
/* GLOBAL **************        LinkImport         ************************/
/**************************************************************************/
/* PURPOSE: ADD IMPORT e TO THE IMPORT LIST of NODE dst IN iport ORDER.   */
/*          THE PREDECESSOR OF THE FIRST IMPORT IS ALWAYS NULL.           */
/**************************************************************************/

void LinkImport( dst, e )
register PNODE dst;
register PEDGE e;
{
    register PEDGE i;

    echange = TRUE;

    e->ipred = NULL;
    e->isucc = NULL;
    e->dst   = dst;

    if ( dst->imp == NULL ) {                       /* IMPORT LIST EMPTY */
        dst->imp = e;
        return;
        }

    if ( dst->imp->iport > e->iport ) {          /* BEFORE FIRST IN LIST */
        e->isucc        = dst->imp;
        dst->imp->ipred = e;
        dst->imp        = e;
        return;
        }

    for( i = dst->imp; i->isucc != NULL; i = i->isucc )       /* WHERE? */
        if ( i->isucc->iport > e->iport )
            break;

    e->isucc  = i->isucc;                               /* LINK AFTER i */
    e->ipred  = i;

    if ( i->isucc != NULL )
        i->isucc->ipred = e;

    i->isucc = e;
}


/**************************************************************************/
/* GLOBAL **************      LinkImportLists      ************************/
/**************************************************************************/
/* PURPOSE: LINK THE IMPORT LIST OF NODE n2 TO THAT OF NODE n1.  THE      */
/*          IMPORT LIST OF n2 IS SET TO NULL.                             */
/**************************************************************************/

void LinkImportLists( n1, n2 )
PNODE n1;
PNODE n2;
{
    register PEDGE i;
    register PEDGE si;

    for ( i = n2->imp; i != NULL; i = si ) {
        si = i->isucc;
        LinkImport( n1, i );
        }

    n2->imp = NULL;
}


/**************************************************************************/
/* GLOBAL **************       UnlinkImport        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK IMPORT i FROM ITS DESTINATION NODE's IMPORT LIST.      */
/**************************************************************************/

void UnlinkImport( i )
PEDGE i;
{
    echange = TRUE;

    if ( i->ipred == NULL ) {                         /* FIRST ONE IN LIST */
        i->dst->imp = i->isucc;

        if ( i->isucc != NULL )
            i->isucc->ipred = NULL;
    } else {
        i->ipred->isucc = i->isucc;

        if ( i->isucc != NULL )
            i->isucc->ipred = i->ipred;
    }
}


/**************************************************************************/
/* GLOBAL **************     LinkExportToEnd       ************************/
/**************************************************************************/
/* PURPOSE: ADD EDGE e TO THE TAIL OF src'S EXPORT LIST.                  */  
/**************************************************************************/


void LinkExportToEnd( src, e )
PNODE src;
PEDGE e;
{
    register PEDGE ee;

    for ( ee = src->exp; ee != NULL; ee = ee->esucc )
      if ( ee->esucc == NULL )
	break;

    e->esucc = NULL;
    e->epred = ee;
    e->src   = src;

    if ( ee == NULL ) {
      src->exp = e;
      return;
      }

    ee->esucc = e;
}


/**************************************************************************/
/* GLOBAL **************        LinkExport         ************************/
/**************************************************************************/
/* PURPOSE: ADD EDGE e TO THE HEAD OF src'S EXPORT LIST. THE PEDECESSOR   */  
/*          OF THE FIRST EXPORT IS ALWAYS NULL.                           */
/**************************************************************************/


void LinkExport( src, e )
PNODE src;
PEDGE e;
{
    e->src = src;

    e->epred = NULL;
    e->esucc = src->exp;

    if ( src->exp != NULL )
        src->exp->epred = e;

    src->exp = e;
}


/**************************************************************************/
/* GLOBAL **************      LinkExportLists      ************************/
/**************************************************************************/
/* PURPOSE: LINK THE EXPORT LIST OF NODE n2 TO THAT OF NODE n1. NODE n2'S */
/*          EXPORT LIST IS SET TO NULL.                                   */
/**************************************************************************/

void LinkExportLists( n1, n2 )
PNODE n1;
PNODE n2;
{
    register PEDGE e2;
    register PEDGE se2;

    /* for ( e2 = n2->exp; e2 != NULL; e2 = se2 ) {
      e2->src = n1;

      if ( (se2 = e2->esucc) == NULL ) 
	break;
      }

    if ( e2 != NULL ) {
      e2->esucc = n1->exp;

      if ( n1->exp != NULL )
	n1->exp->epred = e2;

      n1->exp = n2->exp;
      } */

{
register PEDGE e;
register PEDGE se;
    for ( e = n2->exp; e != NULL; e = se ) {
        se = e->esucc;
        LinkExport( n1, e );
        }
}

    n2->exp = NULL;
}


/**************************************************************************/
/* GLOBAL **************       UnlinkExport        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK EDGE e FROM ITS SOURCE NODE'S EXPORT LIST.  IF e IS A  */
/*          CONSTANT, NOTHING IS DONE.                                    */
/**************************************************************************/

void UnlinkExport( e )
PEDGE e;
{
    if ( IsConst( e ) )
        return;

    if ( e->epred == NULL ) {                         /* FIRST ONE IN LIST */
        e->src->exp = e->esucc;

        if ( e->esucc != NULL )
            e->esucc->epred = NULL;
    } else {
        e->epred->esucc = e->esucc;

        if ( e->esucc != NULL )
            e->esucc->epred = e->epred;
    }
}


/**************************************************************************/
/* GLOBAL **************    ChangeExportsToConst   ************************/
/**************************************************************************/
/* PURPOSE: CHANGE ALL EXPORTS OF NODE n WITH EXPORT PORT NUMBER MATCHING */
/*          eport INTO CONSTANT c.                                        */
/**************************************************************************/

void ChangeExportsToConst( n, eport, c )
PNODE n;
int   eport;
PEDGE c;
{
    register PEDGE e;
    register PEDGE se;

    for ( e = n->exp; e != NULL; e = se ) {
        se = e->esucc;

        if ( e->eport == eport )
            ChangeEdgeToConst( e, c );
        }
}


/**************************************************************************/
/* GLOBAL **************      LinkAssocLists       ************************/
/**************************************************************************/
/* PURPOSE: LINK THE ASSOCIATION LIST lst2 TO lst1 AND RETURN THE RESULT. */
/*          IF THE FIRST LIST IS EMPTY, THE SECOND IS RETURNED.           */
/**************************************************************************/

PALIST LinkAssocLists( lst1, lst2 )
PALIST lst1;
PALIST lst2;
{
    register PALIST l;
    register PALIST prev = NULL;

    for( l = lst1; l != NULL; l = l->next )
        prev = l;

    if ( prev == NULL )
        return( lst2 );

    prev->next = lst2;

    return( lst1 );
}


/**************************************************************************/
/* GLOBAL **************        CopyString         ************************/
/**************************************************************************/
/* PURPOSE: RETURN A COPY OF THE INPUT STRING s.                          */
/**************************************************************************/

char *CopyString( s ) 
char *s;
{
    return( strcpy( MyAlloc( strlen( s ) + 1 ), s ) );
}


/**************************************************************************/
/* GLOBAL **************         Warning1          ************************/
/**************************************************************************/
/* PURPOSE: PRINT A WARNING MESSAGE TO stderr, THEN CONTINUE AS NORMAL.   */
/**************************************************************************/

void Warning1( msg1 )
char *msg1;
{
    fprintf( stderr, "%s: W - %s\n", program, msg1 );
}


/**************************************************************************/
/* GLOBAL **************          Error1           ************************/
/**************************************************************************/
/* PURPOSE: PRINT AN ERROR MEASAGE TO stderr AND ABORT EXECUTION.         */
/**************************************************************************/

void Error1( msg1 )
char *msg1;
{
    fprintf( stderr, "\n%s: E - %s\n", program, msg1 );
    Stop( ERROR );
}


/**************************************************************************/
/* GLOBAL **************          Error2           ************************/
/**************************************************************************/
/* PURPOSE: PRINT TWO ERROR MEASAGES TO stderr AND ABORT EXECUTION.       */
/**************************************************************************/

void Error2( msg1, msg2 )
char *msg1;
char *msg2;
{
    fprintf( stderr, "\n%s: E - %s %s\n", program, msg1, msg2 );
    Stop( ERROR );
}


/**************************************************************************/
/* GLOBAL **************         EdgeAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN EDGE.                     */
/**************************************************************************/

PEDGE EdgeAlloc( src, eport, dst, iport )
PNODE src;
int   eport;
PNODE dst;
int   iport;
{
    register PEDGE   e;
    register PBBLOCK b;

    b = MyBBlockAlloc();
    e = &(b->e);

    e->eport = eport;
    e->src   = src;
    e->iport = iport;
    e->dst   = dst;
    e->dname = NULL;
    e->dope  = NULL;
    e->lvl   = 0;

    e->epred = e->esucc = NULL;
    e->ipred = e->isucc = NULL;

    e->uedge = NULL;

    InitPragmas( e );

    return( e );
}


/**************************************************************************/
/* GLOBAL **************         NodeAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN A NODE.                     */
/**************************************************************************/

PNODE NodeAlloc( label, type )
int   label;
int   type;
{
    register PNODE   n;
    register PBBLOCK b;

    b = MyBBlockAlloc();
    n = &(b->n);

    n->label = label;
    n->type  = type;
    n->size  = -1;

    n->imp   = NULL;
    n->exp   = NULL;

    n->gpred = n->gsucc = NULL;
    n->npred = n->nsucc = NULL;

    n->usucc = NULL;

    InitPragmas( n );

    return( n );
}


/**************************************************************************/
/* GLOBAL **************         InfoAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN INFO NODE.                */
/**************************************************************************/


PINFO InfoAlloc( label, type )
int   label;
int   type;
{
    register PINFO   i;
    register PBBLOCK b;

    b = MyBBlockAlloc();
    i = &(b->i);

    i->info1  = i->info2 = NULL;

    i->label  = label;
    i->type   = type;
    i->touch1 = FALSE;

    InitPragmas( i );

    i->next  = NULL;

    return( i );
}


/**************************************************************************/
/* GLOBAL **************      AssocListAlloc       ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN ASSOCIATION LIST NODE.    */
/**************************************************************************/

PALIST AssocListAlloc( datum )
int datum;
{
    register PALIST l;

    l = (PALIST) MyAlloc( sizeof(ALIST) );

    l->next  = NULL;
    l->datum = datum;

    return( l );
}


/**************************************************************************/
/* GLOBAL **************         LowerCase         ************************/
/**************************************************************************/
/* PURPOSE: RETURN A COPY OF STRING n WITH ALL UPPER CASE LETTERS CHANGED */
/*          TO LOWER CASE.                                                */
/**************************************************************************/

char *LowerCase( n )
char *n;
{
    register char *u;
    register char *p;

    for ( p = u = CopyString( n ); *p != '\0'; p++ )
        if ( (*p >= 'A') && (*p <= 'Z') )
            *p = 'a' + (*p - 'A'); 

    return( u );
}


/**************************************************************************/
/* GLOBAL **************        ImportSwap         ************************/
/**************************************************************************/
/* PURPOSE: SWAP THE IMPORTS OF DYADIC NODE n.                            */
/**************************************************************************/

void ImportSwap( n )
PNODE n;
{
  register PEDGE one;
  register PEDGE two;

  one = n->imp;
  two = one->isucc;

  UnlinkImport( one );
  UnlinkImport( two );

  one->iport = 2;
  two->iport = 1;

  LinkImport( n, two );
  LinkImport( n, one );
}


/**************************************************************************/
/* GLOBAL **************        IsReadOnly         ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF THE ARRAY EXPORTED FROM NODE n AT PORT eport   */
/*          IS READ ONLY.                                                 */
/**************************************************************************/

int IsReadOnly( n, eport )
PNODE n;
int   eport;
{
  register PEDGE e;
  register PNODE sg;

  for ( e = n->exp; e != NULL; e = e->esucc ) {
    if ( e->eport != eport )
      continue;

    if ( !IsArray( e->info ) )
      continue;

    if ( IsCompound( e->dst ) ) {
      for ( sg = e->dst->C_SUBS; sg != NULL; sg = sg->gsucc )
        if ( !IsReadOnly( sg, e->iport ) )
          return( FALSE );

      continue;
      }

    switch ( e->dst->type ) {
      case IFALimL:
      case IFASize:
      case IFALimH:
        break;

      case IFAElement:
        if ( !IsReadOnly( e->dst, 1 ) )
          return( FALSE );

        break;

      default:
        return( FALSE );
      }
    }

  return( TRUE );
}


/**************************************************************************/
/* GLOBAL **************        IsInnerLoop        ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF THE LOOP WITH BODY b IS AN INNER LOOP.         */
/**************************************************************************/

int IsInnerLoop( b )
PNODE b;
{
  register PNODE n;
  register PNODE sg;

  for ( n = b->G_NODES; n != NULL; n = n->nsucc ) {
    if ( IsCompound( n ) ) {
      if ( !IsSelect( n ) )
        return( FALSE );

      for ( sg = n->C_SUBS; sg != NULL; sg = sg->gsucc )
        if ( !IsInnerLoop( sg ) )
          return( FALSE );
      }
    }

  return( TRUE );
}
