/* ------------------------------------------------------------
xy.c

Patrick Miller		Initial Release			11 June 1991

This module assigns (x,y) offsets to each of the nodes.

Two methods are included (although only one is used).  In the first,
each node is added to a level until a node would cause a intra-level
dependency.  In that case, a new level is started with an initial x.
In the second method,
------------------------------------------------------------ */
#include "world.h"

#define SEQUENTIALIZED	(0)	/* Set to (1) for method 1 and (0) for */
				/* method 2. (See above) */

#define XPos(n)	((n)->XMARK)
#define YPos(n)	((n)->YMARK)
#define StartX	(1)
#define StartY	(1)

#if SEQUENTIALIZED
/* ------------------------------------------------------------ */
/* Returns 1 if b is directly dependent on a, 0 otherwise. */
int
DependencyBetween(a,b)
     PNODE	a,b;
{
  PEDGE		E;

  /* There is a dependency iff an output of a is an input of b. */
  /* (Actually, also if an output of b is an input of a, but we are */
  /* assuming dataflow ordering on the nodes. */
  for(E=a->exp;E;E=E->esucc) {
    if ( E->dst == b ) return 1;
  }

  return 0;
}

/* ------------------------------------------------------------ */
/* Assign (x,y) pairs to the nodes (and recursively to their */
/* sub-graphs) in graph G.  Nodes are assigned to a level until a */
/* cross dependency developes, in which case a new level is started. */
void
AssignGraphXY(G)
     PNODE	G;
{
  PNODE		LP;		/* Pointer to first node in level */
  PNODE		N,CheckN,Sub;
  int		X,Y;
  int		Dependency;

  /* Make sure there are nodes to assign to */
  if ( G->G_NODES ) {

    X = StartX;
    Y = StartY;

    LP = G->G_NODES;		/* This is the first node */
    XPos(LP) = X++;
    YPos(LP) = Y;

    /* For the remaining nodes, check to see if there is are any */
    /* dependencies with nodes at the current level.  If there are no */
    /* dependencies, then add it at this level.  Otherwise, start a */
    /* new level. */
    for(N=LP->nsucc;N;N=N->nsucc) {
      /* Check the level for dependecies*/
      Dependency = 0;
      for(CheckN=LP;CheckN!=N;CheckN=CheckN->nsucc) {
	if ( DependencyBetween(CheckN,N) ) {
	  Dependency = 1;
	  break;
	}
      }

      /* If there's a dependency, start a new level */
      if ( Dependency ) {
	X = StartX;
	Y++;
      }
      XPos(N) = X++;
      YPos(N) = Y;
    }

    /* Make sure to follow the graphs contained in compounds */
    for(N=G->G_NODES;N;N=N->nsucc) {
      if ( IsCompound(N) ) {
	X = StartX;
	Y = StartY;
	for(Sub=N->C_SUBS;Sub;Sub=Sub->C_SUBS) {
	  XPos(Sub) = X;
	  YPos(Sub) = Y++;
	  AssignGraphXY(Sub);
	}
      }
    }
  }
}
#else
/* ------------------------------------------------------------ */
/* This function returns the largest value of y for any node that */
/* inputs a value to this node.  This sets a lower bound on the y for */
 /* that node (y+1 is guarantied to be a legal level for N).  */
int
MaxLevel(N)
     PNODE	N;
{
  PEDGE		E;
  int		level = 0;

  for(E=N->imp;E;E=E->isucc) {
    if ( !IsConst(E) && !IsGraph(E->src) && YPos(E->src) > level ) {
      level = YPos(E->src);
    }
  }

  return level;
}

/* ------------------------------------------------------------ */
/* For each node, the Y is determined by looking at the levels of */
/* nodes with direct inputs to N.  The X position is set by choosing */
/* the next free position at that level.  The X array below saves the */
/* current location for the X value for each level.  The program will */
/* not support any more levels than the size of the X array. */
void
ParAssignGraphXY(G)
     PNODE	G;
{
  PNODE		N,Sub;
  int		X[1000],Y;
  int		i;

  /* Make sure there are nodes to assign to */
  if ( G->G_NODES ) {

    /* Initialize the X's */
    for(i=0;i< (sizeof(X)/sizeof(X[0]));i++) X[i] = StartX;

    for(N=G->G_NODES;N;N=N->nsucc) {
      Y = MaxLevel(N)+1;
      XPos(N) = X[Y]++;
      YPos(N) = Y;
    }

    /* Make sure to follow the graphs contained in compounds */
    for(N=G->G_NODES;N;N=N->nsucc) {
      if ( IsCompound(N) ) {
	Y = StartY;
	for(Sub=N->C_SUBS;Sub;Sub=Sub->C_SUBS) {
	  XPos(Sub) = StartX;
	  YPos(Sub) = Y++;
	  ParAssignGraphXY(Sub);
	}
      }
    }
  }
}
#endif

/* ------------------------------------------------------------ */
/* This routine simply assigns (x,y) positions to each of the entries. */
/* The AssignGraphXY() routine will recursively descend the rest of */
/* the graph.*/
void
AssignXY()
{
  PNODE		F;

  for( F=glstop->gsucc; F; F = F->gsucc ) {
    XPos(F) = 0;
    YPos(F) = 0;

    /* Use one of these two alternative ways of assigning (x,y) */
    /* offsets.  The first follows the sequentialized graph.  The */
    /* second will reorder so that parallel actions take place as soon */
    /* as possible.  If the graph has been Dataflow Ordered by the */
    /* LLNL frontend, then the two are equivalent. */
#if SEQUENTIALIZED
    AssignGraphXY(F);
#else
    ParAssignGraphXY(F);
#endif
  }
}
