/* ------------------------------------------------------------
nodetable.c

Patrick Miller		Initial Release			11 June 1991

This module will dump out a tabular representation of the graph.  The
basic form is

  Op-name x y in-arity out-arity
  sub-node-list
  edge-list
  literal-list
  SISAL-source-line
  Pragma-string
  Parent-location
  Jump-location
  Table-position


The table first dumps out all the nodes sorted by their unique global
identifiers.  Then some extra graph info associated with the end of
graph psuedo-nodes and the compounds (which exist as nodes inside a
graph, but are represented as a set of sub-graphs).
------------------------------------------------------------ */
#include "world.h"

/* ------------------------------------------------------------ */
/* Some simple defs for this module. */

#define PROLOGUE	(0)	/* In first half of table */
#define EPILOGUE	(1)	/* Second half of tabel */
#define NameOf(n)	(OpList[(n)->type])

/* ------------------------------------------------------------ */
/* External routines used in the module.  This keeps lint from */
/* complaining. */
extern char* GraphNameOf();
extern char* EndGraphNameOf();
extern char* EdgeNameOf();
extern int GlobalIdCount;
extern void DumpNodeInfo();
extern void DumpLine();
extern char *OpList[];

/* ------------------------------------------------------------ */
/* TailPos is used to predict the location of items in the second half */
/* of the table. */

static int TailPos = 0;

/* ------------------------------------------------------------ */
/* Returns the largest line number of the nodes in a graph */
int
LastLineInGraph(G)
     PNODE	G;
{
  int		MaxLine = G->line;
  PNODE		N;

  for(N=G->G_NODES;N;N=N->nsucc) {
    if (N->line > MaxLine) MaxLine = N->line;
  }

  return MaxLine;
}

/* ------------------------------------------------------------ */
/* Return the largest Y offset of the (x,y) positions of sub-nodes */
int
MaxYInGraph(G)
     PNODE	G;
{
  int		MaxY = 0;
  PNODE		N;

  for(N=G->G_NODES;N;N=N->nsucc) {
    if ( N->YMARK > MaxY ) MaxY = N->YMARK;
  }

  return MaxY;
}

/* ------------------------------------------------------------ */
/* Return the largest input port number of a node */
LargestInputPort(N)
     PNODE	N;
{
  PEDGE		E;
  int		max = 0;

  for(E=N->imp;E;E=E->isucc) {
    if ( E->iport > max ) max = E->iport;
  }

  return max;
}

/* ------------------------------------------------------------ */
/* Return the largest output port number of a node */
int
LargestOutputPort(N)
     PNODE	N;
{
  PEDGE		E;
  int		max = 0;

  for(E=N->exp;E;E=E->esucc) {
    if ( E->eport > max ) max = E->eport;
  }

  return max;
}
/* ------------------------------------------------------------ */
/* Dump a TCL list of edges output by a node */
void
DumpEdges(N,EndId)
     PNODE	N;
     int	EndId;
{
  PEDGE		E;

  for(E=N->exp;E;E=E->esucc) {
    fprintf(output,"\t\t{%d %d %d %d %s 0}\n",
	    N->label,
	    E->eport,
	    IsGraph(E->dst)?(EndId):(E->dst->label),
	    E->iport,
	    EdgeNameOf(E));
  }
}
/* ------------------------------------------------------------ */
/* Dump a TCL list of literals imported by a node */
void
DumpLiterals(N,EndId)
     PNODE	N;
     int	EndId;
{
  PEDGE		E;

  for(E=N->imp;E;E=E->isucc) {
    if ( IsConst(E) ) fprintf(output,"\t\t{%d %d {%s}}\n",
	    IsGraph(N)?(EndId):(N->label),
	    E->iport,
	    E->constant);
  }
}
/* ------------------------------------------------------------ */
/* Dump out entries for all the sub nodes in a graph.  For a compound */
/* node, first dump out its sub-graphs and then dump its entry. */
void
DumpSubNodes(G)
     PNODE	G;
{
  PNODE		N,Sub;
  int		SavePos;

  /* Dump lines for each node in the graph */
  for(N=G->G_NODES;N;N=N->nsucc) {

    /* If the node is a compound, then dump its component graphs and */
    /* then dump the actual compound. */
    if ( IsCompound(N) ) {
      /* For compounds, in the epilogue portion of the table, we will */
      /* be dumping an entry for the compound and two for each */
      /* sub-graph (an entry and an end-entry).  TailPos is predicting */
      /* the location of the compound.  Remember this position and */
      /* update the prediction to point to the predicted locations of */
      /* the graph and end-graph entries. */
      SavePos = TailPos++;

      /* Dump out the sub graphs */
      for(Sub=N->C_SUBS;Sub;Sub=Sub->C_SUBS) {
	DumpLine(
		 GraphNameOf(Sub), /* NAME */
		 Sub->XMARK,	/* X */
		 Sub->YMARK,	/* Y */
		 0,		/* IN */
		 0,		/* OUT */
		 (PNODE)NULL,	/* GRAPH */
		 0,		/* endid */
		 Sub->line,	/* SOURCE */
		 Sub,		/* PRAGS */
		 N->GlobalID,	/* PARENT */
		 TailPos,	/* CALL */
		 Sub->GlobalID	/* OFFSET */
		 );
	TailPos += 2;		/* Skip graph, end-graph predictions */
	DumpSubNodes(Sub);	/* Recursively dump nodes in graph */
      }
      /* Now dump out the compound */
      DumpLine(
	       NameOf(N),	 /* NAME */
	       N->XMARK,	/* X */
	       N->YMARK,	/* Y */
	       LargestInputPort(N), /* IN */
	       LargestOutputPort(N), /* OUT */
	       (PNODE)NULL,	/* GRAPH */
	       0,		/* endid */
	       N->line,		/* SOURCE */
	       N,		/* PRAGS */
	       G->GlobalID,	/* PARENT */
	       SavePos,		/* CALL */
	       N->GlobalID	/* OFFSET */
	       );
    } else {
      /* For normal lines, just dump the line */
      DumpLine(
	       NameOf(N),	 /* NAME */
	       N->XMARK,	/* X */
	       N->YMARK,	/* Y */
	       LargestInputPort(N), /* IN */
	       LargestOutputPort(N), /* OUT */
	       (PNODE)NULL,	/* GRAPH */
	       0,		/* endid */
	       N->line,		/* SOURCE */
	       N,		/* PRAGS */
	       G->GlobalID,	/* PARENT */
	       0,		/* CALL */
	       N->GlobalID	/* OFFSET */
	       );
    }
  }
}
/* ------------------------------------------------------------ */
/* In the epilogue, compound nodes and their sub-graphs are being */
/* dumped.  Normal entries are not included (except end of */
/* functions).  Dump out the compound itself and the graph pairs for */
/* its children.  */
void
EpiDumpSubNodes(G)
     PNODE	G;
{
  PNODE		N,Sub;
  int		ParentID;

  /* Dump lines for each node in the graph */
  for(N=G->G_NODES;N;N=N->nsucc) {

    /* If the node is a compound, then dump the compound followed by */
    /* its component graphs. */
    if ( IsCompound(N) ) {
      /* First dump out the compound */
      DumpLine(
	       NameOf(N),	 /* NAME */
	       0,		 /* X */
	       0,		 /* Y */
	       0,		 /* IN */
	       0,		 /* OUT */
	       N,		/* GRAPH */
	       TailPos,		/* endid */
	       N->line,		/* SOURCE */
	       N,		/* PRAGS */
	       G->GlobalID,	/* PARENT */
	       TailPos,		/* CALL */
	       TailPos		/* OFFSET */
	       );

      ParentID = TailPos;	/* Save this for children */
      TailPos++;		/* Move to next offset */

      /* Dump out the sub graphs */
      for(Sub=N->C_SUBS;Sub;Sub=Sub->C_SUBS) {
	DumpLine(
		 GraphNameOf(Sub), /* NAME */
		 0,		/* X */
		 0,		/* Y */
		 0,		/* IN */
		 LargestOutputPort(Sub),/* OUT */
		 Sub,		/* GRAPH */
		 TailPos+1,	/* endid */
		 Sub->line,	/* SOURCE */
		 Sub,		/* PRAGS */
		 ParentID,	/* PARENT */
		 TailPos,	/* CALL */
		 TailPos	/* OFFSET */
		 );
	TailPos++;		/* Move to next offset */
	DumpLine(
		 EndGraphNameOf(Sub), /* NAME */
		 0,		/* X */
		 MaxYInGraph(Sub)+1,/* Y */
		 LargestInputPort(Sub),/* IN */
		 0,		/* OUT */
		 (PNODE)NULL,	/* GRAPH */
		 0,		/* endid */
		 Sub->line,	/* SOURCE */
		 Sub,		/* PRAGS */
		 ParentID,	/* PARENT */
		 0,		/* CALL */
		 TailPos	/* OFFSET */
		 );
	TailPos++;		/* Move to next offset */

	EpiDumpSubNodes(Sub);
      }
    }
  }
}
/* ------------------------------------------------------------ */
/* For graphs in the prologue, just dump out a header entry and then */
/* dump all the sub-nodes.  In the epilogue, we dump end-of-function */
/* entries and then dump all the included compounds and graphs. */
void
TouchFunction(F,Mode)
     PNODE	F;
     int	Mode;
{
  if ( Mode == PROLOGUE ) {
    DumpLine(
	     GraphNameOf(F),	/* NAME */
	     0,			/* X */
	     0,			/* Y */
	     0,			/* IN */
	     LargestOutputPort(F), /* OUT */
	     F,			/* GRAPH */
	     TailPos++,		/* endid */
	     F->line,		/* SOURCE */
	     F,			/* PRAGS */
	     F->GlobalID,	/* PARENT */
	     F->GlobalID,	/* CALL */
	     F->GlobalID	/* OFFSET */
	     );

    DumpSubNodes(F);
  } else {
    DumpLine(
	     EndGraphNameOf(F),	/* NAME */
	     0,			/* X */
	     MaxYInGraph(F)+1,	/* Y */
	     LargestInputPort(F), /* IN */
	     0,			/* OUT */
	     (PNODE)NULL,	/* GRAPH */
	     TailPos,		/* endid */
	     LastLineInGraph(F),/* SOURCE */
	     F,			/* PRAGS */
	     F->GlobalID,	/* PARENT */
	     0,			/* CALL */
	     TailPos		/* OFFSET */
	     );
    TailPos++;

    EpiDumpSubNodes(F);
  }
}

/* ------------------------------------------------------------ */
/* Foreach function, we must first dump out the list of nodes and */
 /* edges for the graph proper (and interior nodes and edges).  Then, */
 /* we need the clean up nodes. */
void
CreateNodeTable()
{
  PNODE		F;


  /* Build the node table */
  fprintf(output,"set graphtable {\n");

  TailPos = GlobalIdCount+1;
  for( F=glstop->gsucc; F; F = F->gsucc ) TouchFunction(F,PROLOGUE);

  TailPos = GlobalIdCount+1;
  for( F=glstop->gsucc; F; F = F->gsucc ) TouchFunction(F,EPILOGUE);

  fprintf(output,"}\n");
}

/* ------------------------------------------------------------ */
/* This function searchs for an edge with another graph destination */
/* and indicates where a 0 must be inserted in the subnode list. */
int
SearchGraphNode( G )
     PNODE G;
{
  int SubNodeCount, *UsedNode, i;
  PNODE N;
  PEDGE E;

  /* Count the number of subnodes */
  for( SubNodeCount = 0, N=G->G_NODES;N;N=N->nsucc, ++SubNodeCount);

  /* If nodes present add first and final nodes */
  if ( SubNodeCount ) SubNodeCount += 2;
  else return( 0 );

  UsedNode = (int *) malloc( SubNodeCount * sizeof(int) );
  if ( UsedNode == NULL ) {
    fprintf( stderr, "Unable to malloc in SearchGraphNode\n");
    exit(1);
  }

  /* Mark nodes with any edges */
  UsedNode[0] = 1;
  for( i = 1; i < SubNodeCount; i++ ) UsedNode[i] = 0;
  UsedNode[G->label] = 1;
  for( E = G->exp; E; E = E->esucc ) UsedNode[E->dst->label] = 1;

  for( N = G->G_NODES; N; N = N->nsucc ) {
    UsedNode[N->label] = 1;
    for( E = N->exp; E; E = E->esucc )
      UsedNode[IsGraph(E->dst)?(SubNodeCount-1):(E->dst->label)] = 1;
  }

  /* Return first index not marked or zero */
  for( i = 1; i < SubNodeCount; i++ ) if ( UsedNode[i] == 0 ) break;

  free( UsedNode );

  if ( i < SubNodeCount ) return(i);
  else return(0);
}

/* ------------------------------------------------------------ */
/* This routine dumps TCL list entries.  Basically, each argument */
/* describes each of the components of the entry. */
void
DumpLine(NAME,X,Y,IN,OUT,GRAPH,endid,SOURCE,PRAGS,PARENT,CALL,OFFSET)
     char	*NAME;
     int	X,Y,IN,OUT;
     PNODE	GRAPH;
     int	endid;
     int	SOURCE;		/* Source line number */
     PNODE	PRAGS;
     int	PARENT;
     int	CALL;
     int	OFFSET;
{
  static int	OffsetCheck = 0;
  PNODE		N;
  int		SubNodeCount, GraphNodeLoc;

  /* Check OFFSET */
  if ( ++OffsetCheck != OFFSET ) {
    fprintf(stderr,"Badd offset %d\n",OFFSET);
    exit(1);
  }

  /* Open line... */
  fprintf(output,"{");

  /* NAME */
  fprintf(output," {%s}",NAME);

  /* X Y IN OUT */
  fprintf(output," %d %d %d %d",X,Y,IN,OUT);

  if ( GRAPH && IsGraph(GRAPH)  ) {
    /* SUBNODE list */
    GraphNodeLoc = SearchGraphNode( GRAPH );
    fprintf(output," {%d",OFFSET);
    SubNodeCount = (GraphNodeLoc != 0) ? 1 : 0;
    for(N=GRAPH->G_NODES;N;N=N->nsucc) {
      if (--GraphNodeLoc == 0) fprintf(output," 0");
      fprintf(output," %d",N->GlobalID);
      SubNodeCount++;
    }
    fprintf(output," %d",endid);
    fprintf(output,"}");

    /* EDGES */
    fprintf(output,"\n\t{\n");
    DumpEdges(GRAPH,SubNodeCount+1);
    for(N=GRAPH->G_NODES;N;N=N->nsucc) DumpEdges(N,SubNodeCount+1);
    fprintf(output,"\t}\n");

    /* LITERALS */
    fprintf(output,"\t{\n");
    DumpLiterals(GRAPH,SubNodeCount+1);
    for(N=GRAPH->G_NODES;N;N=N->nsucc) DumpLiterals(N,SubNodeCount+1);
    fprintf(output,"\t}\n");
  } else if ( GRAPH && IsCompound(GRAPH) ) {
    /* SUBNODE list */
    fprintf(output," {%d",OFFSET);
    SubNodeCount = 0;
    for(N=GRAPH->C_SUBS;N;N=N->C_SUBS) {
      fprintf(output," %d",N->GlobalID);
      SubNodeCount++;
    }
    fprintf(output,"}");

    /* EDGES */
    /* LITERALS */
    fprintf(output," {} {}");
  } else {
    /* SUBNODE list */
    /* EDGES */
    /* LITERALS */
    fprintf(output," {} {} {}");
  }

  /* SOURCE LINE */
  fprintf(output,"\t%d",SOURCE);

  /* PRAGS */
  fprintf(output," {PG}");

  /* PARENT */
  fprintf(output," %d",PARENT);

  /* CALL NAME */
  if ( CALL ) {
    fprintf(output," %d",CALL);
  } else {
    fprintf(output," {}");
  }

  /* COMMENT */
  fprintf(output,"\t#%d}\n",OFFSET);
}
