/* ------------------------------------------------------------
entries.c

Patrick Miller		Initial Release			11 June 1991

This module has several name utility routines ( GraphNameOf,
EndGraphNameOf, and EdgeNameOf ) as well as routines to assign Global
unique identifiers to each node.  The main action routine
CreateEntryTable creates a TCL list of functions and their offset in
the node table that will be created next.
------------------------------------------------------------ */
#include "world.h"
#include <ctype.h>

int GlobalIdCount = 0;		/* This is used to assign unique */
				/* identifier numbers to each node in */
				/* the graph. */

/* ------------------------------------------------------------ */
/* Routines to get the name of graphs (with and without the End-Of */
/* extension) and edges. */

#define MakeUpperCase(S){char *p;for(p=S;*p;p++)if(islower(*p))*p=toupper(*p);}

char*
GraphNameOf(G)
     PNODE	G;
{
  static char buf[100];

  if ( G->gname ) {
    MakeUpperCase(G->gname);
    sprintf(buf,"%s",G->gname);
  } else {
    sprintf(buf,"Graph%d",G->GlobalID);
  }

  return buf;
}
char*
EndGraphNameOf(G)
     PNODE	G;
{
  static char buf[100];

  if ( G->gname ) {
    MakeUpperCase(G->gname);
    sprintf(buf,"End-Of-%s",G->gname);
  } else {
    sprintf(buf,"End-Of-Graph%d",G->GlobalID);
  }

  return buf;
}

char*
EdgeNameOf(E)
     PEDGE	E;
{
  static char buf[100];

  if ( E->name ) {
    sprintf(buf,"{%s}",E->name);
    MakeUpperCase(buf);
  } else {
    strcpy(buf,"{}");
  }

  return buf;
}

/* ------------------------------------------------------------ */
/* These routines assign unique global ids to nodes by making a post */
/* order traversal of the IF1 graph.  AssignGlobalID() starts with the */
/* function entries and uses AssignGlobalIDToGraph() to walk the rest */
/* of the way down the graph. */
void
AssignGlobalIDToGraph(G)
     PNODE	G;
{
  PNODE		N,Sub;

  G->GlobalID = ++GlobalIdCount;

  for(N=G->G_NODES;N;N=N->nsucc) {
    if ( IsCompound(N) ) {
      for(Sub=N->C_SUBS;Sub;Sub=Sub->C_SUBS) {
	AssignGlobalIDToGraph(Sub);
      }
    }

    N->GlobalID = ++GlobalIdCount;
  }
}

void
AssignGlobalID()
{
  PNODE		F;

  for( F=glstop->gsucc; F; F = F->gsucc ) AssignGlobalIDToGraph(F);
}

/* ------------------------------------------------------------ */
/* CreateEntryTable() first assigns unique IDs to the nodes and then */
/* prints out a list of entries in TCL form.  An entry is simply the */
/* name of the entry followed by the offset in the node table which */
/* will be created by the next step.  The offset is the unique ID. */
void
CreateEntryTable()
{
  PNODE		F;

  AssignGlobalID();

  /* Build the entry table */
  fprintf(output,"set entrytable {\n");

  for( F=glstop->gsucc; F; F = F->gsucc ) {
    fprintf(output,"  {{%s} %d}\n",GraphNameOf(F),F->GlobalID);
  }

  fprintf(output,"}\n");
}
