#include "world.h"

/* *****                                             */
/* ***** SISAL Run Time Support Software             */
/* ***** Parallel dynamic storage allocation section */
/* ***** Author:   R. R. Oldehoeft                   */
/* ***** Modifier: D. C. Cann                        */

/* shared LOCK_TYPE clock; */

struct top { 
  struct top *frwd, *bkwd;
  int size;
  int lsize;
  int PId;
  char status;
  LOCK_TYPE lock;
  };

struct bot { struct top *top_ptr; };

static void DistDeAllocEnQ();


#define   TOPSIZE     SIZE_OF(struct top)
#define   BOTSIZE     SIZE_OF(struct bot)
#define	  SIZETAGS    (TOPSIZE + BOTSIZE)


struct bot *sh_dsorg;
struct top *sh_zero_bl;
struct top *sh_caches;
LOCK_TYPE  *sh_coal_lock;
struct top *sh_btop;

struct bot *pr_dsorg;
struct top *pr_zero_bl;
struct top *pr_cache;
struct top *pr_btop;

static int xfthresh;

#ifdef DSA_DEBUG
int dallocs = 0;
int dfrees  = 0;
int dbytes  = 0;
#endif


void ShutDownDsa()
{
#ifdef DSA_DEBUG
fprintf( stderr, "D - (ShutDownDsa) Allocs %d Frees %d Lost Bytes %d\n", 
         dallocs, dfrees, dbytes );
#endif
}

static void InitDistDeAllocList();


void InitDsa( sh_size, xft )
int sh_size;
int xft;
{
  register struct top *cu, *nx;
  register struct bot *cubot;
  register int         i, inc, roundsize;

  InitDistDeAllocList();

  xfthresh = xft >= 0 ? xft : 0;

  /* MY_SINIT_LOCK(&clock); */

  sh_coal_lock = (LOCK_TYPE*) SharedMalloc( sizeof(LOCK_TYPE) );
  MY_SINIT_LOCK(sh_coal_lock);

  sh_caches  = (struct top*) SharedMalloc( sizeof(struct top) * MAX_PROCS );

  for(i=0; i<NumWorkers; i++){
    sh_caches[i].size  = 0;
    sh_caches[i].lsize = 0;
    sh_caches[i].frwd  = 0;
    sh_caches[i].bkwd  = 0;
    }


  roundsize = ALIGN(int,sh_size);
  sh_zero_bl = (struct top*) SharedMalloc( sizeof(struct top) );
  sh_dsorg = (struct bot *) SharedMalloc( roundsize );

  sh_btop = (struct top *)((PCMCAST)sh_dsorg + roundsize - TOPSIZE);
  sh_btop->status = 'A'; 
  sh_btop->size   = -1;
  sh_btop->frwd   = sh_btop;
  MY_SINIT_LOCK(&(sh_btop->lock));
  sh_btop->bkwd   = sh_btop;

  sh_dsorg->top_ptr = sh_btop;

  cu = (struct top *)((PCMCAST)sh_dsorg + BOTSIZE);

  inc  = ((int) ((PCMCAST)sh_btop - (PCMCAST)sh_dsorg));
  inc -= (inc % ALIGN_SIZE);

  /* Construct last free block and link in */
  cu->frwd   = sh_zero_bl;
  cu->bkwd   = sh_zero_bl;
  cu->status = 'F';

  cubot = (struct bot *)((PCMCAST)sh_btop - BOTSIZE);
  cubot->top_ptr = cu;

  cu->size = (int) (( (PCMCAST)sh_btop - (PCMCAST)cu ) - SIZETAGS);

  MY_SINIT_LOCK(&(cu->lock));
  sh_zero_bl->frwd   = cu;
  sh_zero_bl->status = 'F';
  sh_zero_bl->size   = 0;
  MY_SINIT_LOCK(&(sh_zero_bl->lock));

  sh_zero_bl->bkwd = cu;
}


/* Allocate storage from the boundary tag-managed pool */

static char *btAlloc(size)
register int size;
{
   register struct top *cu, *pr, *back;
   struct top          *newtop;
   char                *addr;
   struct bot          *cubot, *newbot;
   int                  mystart, lsize;
   int                  pID;

   lsize = size;
   size  = ALIGN(int,size);

   GETPROCID(pID);

   IncStorageUsed(pID,(size+SIZETAGS));
   IncStorageWanted(pID,(size));

   /* Get a free block */

   pr = sh_zero_bl;

   back = pr;

   MY_SLOCK(&(pr->lock));

   for ( ;; ) {
     cu = pr->frwd;

     if ( cu == back ) {
       MY_SUNLOCK(&(pr->lock));
       return( 0 );
       }

     MY_SLOCK(&(cu->lock));

     if( cu->size >= size )
       break;

     MY_SUNLOCK(&(pr->lock)); 
     pr = cu;
     }


   if (cu->size - size <= xfthresh + SIZETAGS){
     /* Exact fit (or close enough).  Unlink cu */
     pr->frwd = cu->frwd;

     cu->frwd->bkwd = pr;
     cu->status     = 'A';
     cu->PId        = pID;
     cu->lsize      = lsize;

     addr = (char *) ((PCMCAST)cu + TOPSIZE);
   }else{
     /* Split this large block to satisfy request */
     cubot  = (struct bot *)((PCMCAST)cu + cu->size + TOPSIZE);

     addr   = (char *)((PCMCAST)cubot - size);

     newtop = (struct top *)((PCMCAST)addr - TOPSIZE);
     newbot = (struct bot *)((PCMCAST)newtop - BOTSIZE);

     /* Reduce size of cu and make new bottom tag */
     cu->size -= size + SIZETAGS;

     newbot->top_ptr = cu;

     /* Make tags for new allocated block */
     newtop->size   = size;
     newtop->status = 'A';
     newtop->PId    = pID;
     newtop->lsize  = lsize;

     MY_SINIT_LOCK(&(newtop->lock));

     cubot->top_ptr = newtop;
   }

  MY_SUNLOCK(&(pr->lock));
  MY_SUNLOCK(&(cu->lock));

  return(addr);
}


static int btDeAlloc(ptr)
register struct top *ptr;
{
  register struct top *bl_above, *bl_below, *pr, *cu;
  struct bot          *bot_above, *cubot;
  int                  mystart;
  int                  pID;

  GETPROCID(pID);

  bl_below  = (struct top *)((PCMCAST)ptr + ptr->size + SIZETAGS);
  bot_above = (struct bot *)((PCMCAST)ptr - BOTSIZE);

  /* Become the only coalescing process (sigh) */
  MY_SLOCK(sh_coal_lock);

  /* Attempt to coalesce the free block below with this block */
  if( bl_below != sh_btop ){
    for (;;) {
      MY_SLOCK(&(bl_below->lock));

      if ( bl_below->status != 'F' ) {
        MY_SUNLOCK(&(bl_below->lock)); 
	goto tryabove;
        }

      pr = bl_below->bkwd;
      MY_SUNLOCK(&(bl_below->lock));

      MY_SLOCK(&(pr->lock));
      if( pr->status == 'F' && pr->frwd == bl_below )
	break;
      MY_SUNLOCK(&(pr->lock));
      }

    MY_SLOCK(&(bl_below->lock));
    pr->frwd = bl_below->frwd;

    bl_below->frwd->bkwd = pr;

    cubot = (struct bot *)((PCMCAST)bl_below + bl_below->size + TOPSIZE);
    cubot->top_ptr = ptr;

    ptr->size += bl_below->size + SIZETAGS;

    MY_SUNLOCK(&(pr->lock));
    MY_SUNLOCK(&(bl_below->lock));
    }

tryabove:

  /* Attempt to coalesce with the block above */
  if ( bot_above != sh_dsorg ) {
    for (;;) {
      bl_above = bot_above->top_ptr;

      MY_SLOCK(&(bl_above->lock));
      if( (PCMCAST)ptr == ((PCMCAST)bl_above + bl_above->size + SIZETAGS))
        break;
      MY_SUNLOCK(&(bl_above->lock));
      }

    if ( bl_above->status == 'F' ) {
      bl_above->size += ptr->size + SIZETAGS;

      cubot = (struct bot *)((PCMCAST)ptr + ptr->size + TOPSIZE);
      cubot->top_ptr = bl_above;

      MY_SUNLOCK(&(bl_above->lock));
      MY_SUNLOCK(sh_coal_lock);

      /* Report new block size */
      return;
      }

    MY_SUNLOCK(&(bl_above->lock));
    }

  /* Cannot merge.  Add this block to the free list */
  MY_SUNLOCK(sh_coal_lock);

  pr = sh_zero_bl;

  MY_SLOCK(&(pr->lock));

  cu = pr->frwd;

  if ( pr != cu )
    MY_SLOCK(&(cu->lock));

  pr->frwd = ptr;

  ptr->bkwd   = pr;
  ptr->frwd   = cu;
  ptr->status = 'F';

  cu->bkwd = ptr;

  MY_SUNLOCK(&(pr->lock));

  if ( pr != cu )
    MY_SUNLOCK(&(cu->lock));
}

/* Return blocks from cache p to the boundary tag pool */

static int OldZap(p)
struct top *p;
{
  struct top *q;

#ifdef DSA_DEBUG
  fprintf( stderr, "D - IN OldZAP!!!\n" );
#endif

  do {
    for ( q = p->bkwd; q != 0; q = p->bkwd ) {
      p->bkwd = q->bkwd;
      btDeAlloc(q);
      }

    q = p;
    p = p->frwd;
    btDeAlloc(q);
    }
  while( p != 0 );
}


POINTER Alloc(size)
register int size;
{
  register struct top *cu, *pr, *ptr;
  register char       *addr;
  int                  pID;

  GETPROCID(pID);
  pr = &sh_caches[pID];

  /* Search for a block of exactly the right size. */
  for ( cu = pr->frwd; cu != 0; cu = cu->frwd ){
    if ( cu->lsize == size ) {
      if (cu->bkwd == 0){
        addr = (char *)((PCMCAST)cu + TOPSIZE);
#ifdef DSA_DEBUG
cu->status = 'A';
dallocs++;
/* fprintf( stderr, "ALLOCATION %x %d\n", addr, size ); */
dbytes += size;
#endif
        pr->frwd = cu->frwd;
        return( (POINTER) addr);
        }

      /* There are at least two blocks this size. */
      addr = (char *)((PCMCAST)(cu->bkwd) + TOPSIZE);
#ifdef DSA_DEBUG
cu->bkwd->status = 'A';
/* fprintf( stderr, "ALLOCATION %x %d\n", addr, size ); */
dallocs++;
dbytes += size;
#endif
      cu->bkwd = cu->bkwd->bkwd;
      return( (POINTER) addr);
      }

    pr = cu;
    }

  addr = btAlloc( size );

  if ( addr != 0 ) {
#ifdef DSA_DEBUG
/* fprintf( stderr, "ALLOCATION %x %d\n", addr, size ); */
dallocs++;
dbytes += size;
#endif
    return(  (POINTER) addr );
    }

  /* flush the cache into boundary tag system. NOTE: ALL THE CACHES, */
  /* AND NOT JUST THIS ONE, SHOULD BE FLUSHED.                       */
  DsaHelp();
  
  /* try boundary tag system one more time */
  addr = btAlloc( size );

  if ( addr != 0 ) {
#ifdef DSA_DEBUG
/* fprintf( stderr, "ALLOCATION %x %d\n", addr, size ); */
dallocs++;
dbytes += size;
#endif
    return(  (POINTER) addr );
    }

  SisalError( "Alloc", "ALLOCATION FAILURE: increase -ds value" );
}


/* "Shape up" the dsa system in a last ditch attempt to avoid deadlock */
/* on data memory.                                                     */

int DsaHelp()
{
  register int pID;

  GETPROCID(pID);

  IncDsaHelp(pID);

  if ( sh_caches[pID].frwd != 0 ) {
    OldZap(sh_caches[pID].frwd); 
    sh_caches[pID].frwd = 0;
    }
}


static void TheDeAlloc( x, Seq, Phys )
POINTER x;
int     Seq;
PHYSP   Phys;
{
   register struct top *pr, *cu, *ptr;
   register int size;
   register int pID;

#ifdef DSA_DEBUG
if ( x == NULL ) SisalError( "DeAlloc", "NULL POINTER ON DeAlloc!!!" );
#endif

   ptr  = (struct top *)((PCMCAST)x - TOPSIZE);

   /* IS x PRIVATE? */ 
   if ( ptr->status == 'a' ) {
     pr_DeAlloc( x );
     return;
     }

   size = ptr->lsize;

#ifdef DSA_DEBUG
if ( ptr->status != 'A' ) SisalError( "DeAlloc", "MULTIPLE DEALLOCS!!!" );
ptr->status = 'C';
/* fprintf( stderr, "FREE %x %d\n", x, size ); */
dfrees++;
dbytes -= size;
#endif

   if ( Seq )
     pr  = &sh_caches[ptr->PId];
   else if ( Phys != (PHYSP) NULL ) {
     if ( ptr->PId == GetProcId )
       pr  = &sh_caches[ptr->PId];
     else {
       Phys->PBase = (POINTER) NULL;
       DistDeAllocEnQ( Phys, ptr->PId );
       return;
       }
   } else {
     GETPROCID(pID);
     pr = &sh_caches[ptr->PId = pID];
     }

   cu = pr->frwd;

   for ( ;; ) {
     if ( cu == 0 ) {
       pr->frwd = ptr;
       ptr->bkwd = 0;
       ptr->frwd = 0;
       return;
       }

     if ( cu->lsize == size ) {
       ptr->bkwd = cu->bkwd;
       cu->bkwd  = ptr;
       return;
       }

     pr = cu;
     cu = pr->frwd;
     }
}

void DeAlloc( x )
POINTER x;
{
  TheDeAlloc( x, Sequential, NULL );
}


/* ****************** PRIVATE MEMORY ROUTINES *********************** */

void pr_InitDsa( pr_size )
int pr_size;
{
  register struct top *cu, *nx;
  register struct bot *cubot;
  register int         i, inc, roundsize;

  pr_cache  = (struct top*) PrivateMalloc( sizeof(struct top) );

  pr_cache->size  = 0;
  pr_cache->lsize = 0;
  pr_cache->frwd  = 0;
  pr_cache->bkwd  = 0;

  roundsize = ALIGN(int,pr_size);
  pr_zero_bl = (struct top*) PrivateMalloc( sizeof(struct top) );
  pr_dsorg = (struct bot *) PrivateMalloc( roundsize );

  pr_btop = (struct top *)((PCMCAST)pr_dsorg + roundsize - TOPSIZE);
  pr_btop->status = 'a'; 
  pr_btop->size   = -1;
  pr_btop->frwd   = pr_btop;
  pr_btop->bkwd   = pr_btop;

  pr_dsorg->top_ptr = pr_btop;

  cu = (struct top *)((PCMCAST)pr_dsorg + BOTSIZE);

  inc  = ((int) ((PCMCAST)pr_btop - (PCMCAST)pr_dsorg));
  inc -= (inc % ALIGN_SIZE);

  /* Construct last free block and link in */
  cu->frwd   = pr_zero_bl;
  cu->bkwd   = pr_zero_bl;
  cu->status = 'F';

  cubot = (struct bot *)((PCMCAST)pr_btop - BOTSIZE);
  cubot->top_ptr = cu;

  cu->size = (int) (( (PCMCAST)pr_btop - (PCMCAST)cu ) - SIZETAGS);

  pr_zero_bl->frwd   = cu;
  pr_zero_bl->status = 'F';
  pr_zero_bl->size   = 0;

  pr_zero_bl->bkwd = cu;
}


/* Allocate storage from the boundary tag-managed pool */

static char *pr_btAlloc(size)
register int size;
{
   register struct top *cu, *pr, *back;
   struct top          *newtop;
   char                *addr;
   struct bot          *cubot, *newbot;
   int                  mystart, lsize;
   int                  pID;

   lsize = size;
   size  = ALIGN(int,size);

   GETPROCID(pID);

   IncPRStorageUsed(pID,(size+SIZETAGS));
   IncPRStorageWanted(pID,(size));

   /* Get a free block */

   pr = pr_zero_bl;

   back = pr;

   for ( ;; ) {
     cu = pr->frwd;

     if ( cu == back )
       return( 0 );

     if( cu->size >= size )
       break;

     pr = cu;
     }


   if (cu->size - size <= xfthresh + SIZETAGS){
     /* Exact fit (or close enough).  Unlink cu */
     pr->frwd = cu->frwd;

     cu->frwd->bkwd = pr;
     cu->status     = 'a';
     cu->PId        = pID;
     cu->lsize      = lsize;

     addr = (char *) ((PCMCAST)cu + TOPSIZE);
   }else{
     /* Split this large block to satisfy request */
     cubot  = (struct bot *)((PCMCAST)cu + cu->size + TOPSIZE);

     addr   = (char *)((PCMCAST)cubot - size);

     newtop = (struct top *)((PCMCAST)addr - TOPSIZE);
     newbot = (struct bot *)((PCMCAST)newtop - BOTSIZE);

     /* Reduce size of cu and make new bottom tag */
     cu->size -= size + SIZETAGS;

     newbot->top_ptr = cu;

     /* Make tags for new allocated block */
     newtop->size   = size;
     newtop->status = 'a';
     newtop->PId    = pID;
     newtop->lsize  = lsize;

     cubot->top_ptr = newtop;
   }

  return(addr);
}


static int pr_btDeAlloc(ptr)
register struct top *ptr;
{
  register struct top *bl_above, *bl_below, *pr, *cu;
  struct bot          *bot_above, *cubot;

  bl_below  = (struct top *)((PCMCAST)ptr + ptr->size + SIZETAGS);
  bot_above = (struct bot *)((PCMCAST)ptr - BOTSIZE);

  /* Attempt to coalesce the free block below with this block */
  if( bl_below != pr_btop ){
    for (;;) {
      if ( bl_below->status != 'F' )
	goto tryabove;

      pr = bl_below->bkwd;

      if( pr->status == 'F' && pr->frwd == bl_below )
	break;
      }

    pr->frwd = bl_below->frwd;

    bl_below->frwd->bkwd = pr;

    cubot = (struct bot *)((PCMCAST)bl_below + bl_below->size + TOPSIZE);
    cubot->top_ptr = ptr;

    ptr->size += bl_below->size + SIZETAGS;
    }

tryabove:

  /* Attempt to coalesce with the block above */
  if ( bot_above != pr_dsorg ) {
    for (;;) {
      bl_above = bot_above->top_ptr;

      if( (PCMCAST)ptr == ((PCMCAST)bl_above + bl_above->size + SIZETAGS))
        break;
      }

    if ( bl_above->status == 'F' ) {
      bl_above->size += ptr->size + SIZETAGS;

      cubot = (struct bot *)((PCMCAST)ptr + ptr->size + TOPSIZE);
      cubot->top_ptr = bl_above;

      /* Report new block size */
      return;
      }
    }

  /* Cannot merge.  Add this block to the free list */
  pr = pr_zero_bl;

  cu = pr->frwd;

  pr->frwd = ptr;

  ptr->bkwd   = pr;
  ptr->frwd   = cu;
  ptr->status = 'F';

  cu->bkwd = ptr;
}

/* Return blocks from cache p to the boundary tag pool */

static int pr_OldZap(p)
struct top *p;
{
  struct top *q;

#ifdef DSA_DEBUG
  fprintf( stderr, "D - IN OldZAP!!!\n" );
#endif

  do {
    for ( q = p->bkwd; q != 0; q = p->bkwd ) {
      p->bkwd = q->bkwd;
      pr_btDeAlloc(q);
      }

    q = p;
    p = p->frwd;
    pr_btDeAlloc(q);
    }
  while( p != 0 );
}


POINTER pr_Alloc(size)
register int size;
{
  register struct top *cu, *pr, *ptr;
  register char       *addr;

  pr = pr_cache;

  /* Search for a block of exactly the right size. */
  for ( cu = pr->frwd; cu != 0; cu = cu->frwd ){
    if ( cu->lsize == size ) {
      if (cu->bkwd == 0){
        addr = (char *)((PCMCAST)cu + TOPSIZE);
#ifdef DSA_DEBUG
cu->status = 'a';
dallocs++;
/* fprintf( stderr, "ALLOCATION %x %d\n", addr, size ); */
dbytes += size;
#endif
        pr->frwd = cu->frwd;
        return( (POINTER) addr);
        }

      /* There are at least two blocks this size. */
      addr = (char *)((PCMCAST)(cu->bkwd) + TOPSIZE);
#ifdef DSA_DEBUG
cu->bkwd->status = 'a';
/* fprintf( stderr, "ALLOCATION %x %d\n", addr, size ); */
dallocs++;
dbytes += size;
#endif
      cu->bkwd = cu->bkwd->bkwd;
      return( (POINTER) addr);
      }

    pr = cu;
    }

  addr = pr_btAlloc( size );

  if ( addr != 0 ) {
#ifdef DSA_DEBUG
/* fprintf( stderr, "ALLOCATION %x %d\n", addr, size ); */
dallocs++;
dbytes += size;
#endif
    return(  (POINTER) addr );
    }

  /* flush the cache into boundary tag system. NOTE: ALL THE CACHES, */
  /* AND NOT JUST THIS ONE, SHOULD BE FLUSHED.                       */
  pr_DsaHelp();
  
  /* try boundary tag system one more time */
  addr = pr_btAlloc( size );

  if ( addr != 0 ) {
#ifdef DSA_DEBUG
/* fprintf( stderr, "ALLOCATION %x %d\n", addr, size ); */
dallocs++;
dbytes += size;
#endif
    return(  (POINTER) addr );
    }

  SisalError( "Alloc", "ALLOCATION FAILURE: increase -ds value" );
}


/* "Shape up" the dsa system in a last ditch attempt to avoid deadlock */
/* on data memory.                                                     */

int pr_DsaHelp()
{
  register int pID;

  GETPROCID(pID);

  IncPRDsaHelp(pID);

  if ( pr_cache->frwd != 0 ) {
    pr_OldZap(pr_cache->frwd); 
    pr_cache->frwd = 0;
    }
}


void pr_DeAlloc( x )
POINTER x;
{
   register struct top *pr, *cu, *ptr;
   register int size;
   register int pID;

#ifdef DSA_DEBUG
if ( x == NULL ) SisalError( "pr_DeAlloc", "NULL POINTER ON DeAlloc!!!" );
#endif

   ptr  = (struct top *)((PCMCAST)x - TOPSIZE);

   size = ptr->lsize;

#ifdef DSA_DEBUG
if ( ptr->status != 'a' ) SisalError( "DeAlloc", "MULTIPLE DEALLOCS!!!" );
ptr->status = 'C';
/* fprintf( stderr, "FREE %x %d\n", x, size ); */
dfrees++;
dbytes -= size;
#endif

   pr = pr_cache;

   cu = pr->frwd;

   for ( ;; ) {
     if ( cu == 0 ) {
       pr->frwd = ptr;
       ptr->bkwd = 0;
       ptr->frwd = 0;
       return;
       }

     if ( cu->lsize == size ) {
       ptr->bkwd = cu->bkwd;
       cu->bkwd  = ptr;
       return;
       }

     pr = cu;
     cu = pr->frwd;
     }
}


struct DistDeAllocNode {
  LOCK_TYPE  Mutex;
  PHYSP      Head;
  };

static struct DistDeAllocNode *DistList;


static void InitDistDeAllocList()
{
  register int Index;

  DistList = (struct DistDeAllocNode *) 
	     SharedMalloc( sizeof(struct DistDeAllocNode) * NumWorkers );

  for ( Index = 0; Index < NumWorkers; Index++ ) {
    MY_INIT_LOCK( (&(DistList[Index].Mutex)) );
    DistList[Index].Head = (PHYSP) NULL;
    }
}


static void DistDeAllocEnQ( ThisNode, Pid )
register PHYSP ThisNode;
int Pid;
{
  register struct DistDeAllocNode *DistPtr;

  DistPtr = &(DistList[Pid]);

  MY_LOCK( &(DistPtr->Mutex) );

  ThisNode->Next = DistPtr->Head;
  DistPtr->Head = ThisNode;

  MY_UNLOCK( &(DistPtr->Mutex) );
}


void DoDistDeAllocs()
{
  register PHYSP ThisNode;
  register struct DistDeAllocNode *DistPtr;

  DistPtr = &(DistList[GetProcId]);

  while ( DistPtr->Head != (PHYSP) NULL ) {
    MY_LOCK( &(DistPtr->Mutex) );
    ThisNode = DistPtr->Head;
    DistPtr->Head = ThisNode->Next;
    MY_UNLOCK( &(DistPtr->Mutex) );

    if ( ThisNode->PBase != (POINTER) NULL )
      pr_DeAlloc( ThisNode->PBase );

    TheDeAlloc( (POINTER) ThisNode, FALSE, ThisNode );
    }
}


void PhysDeAlloc( Phys )
PHYSP Phys;
{
   if ( Phys->PBase != (POINTER) NULL ) {
     DistDeAllocEnQ( Phys, Phys->Owner );
     return;
     }

  DeAlloc( (POINTER) Phys );
}
