#include "world.h"


#define MaxLevel  100                 /* MAXIMUM NUMBER OF NESTING LEVELS */

struct level {                        /* LEVEL OCCURRENCE COUNTERS        */
    int lits;
    int edges;
    int simples;
    int ats;
    int comps;
    int graphs;
    };

static struct level levels[MaxLevel];     /* OCCURRENCE COUNT LEVEL STACK */
static int          maxl;                 /* MAXIMUM ENCOUNTERED LEVEL    */
static int          topl;                 /* TOP OF LEVEL STACK           */ 

static int lits;                          /* TOTAL OCCURRENCE COUNTERS    */
static int edges;
static int simples;
static int ats;
static int graphs;
static int comps;
static int rsum;
static int rprod;
static int rcat;
static int rleast;
static int rgreat;

static int gnodes[IF1GraphNodes];
static int snodes[IF1SimpleNodes];
static int atnodes[IF2AtNodes];
static int cnodes[IF1CompoundNodes];

static int convatns  = 0;
static int inatns    = 0;
static int patns     = 0;
static int syncatns  = 0;
static int fsyncatns = 0;
static int incratns  = 0;
static int fincratns = 0;

static void GatherATNodeInfo( n )
PNODE n;
{
  register PEDGE e;

  if ( !IsAtNode( n ) )
    return;

  inatns++;

  switch ( n->type ) {
    case IFAAddHAT:
    case IFAAddLAT:
      convatns++;

      for ( e = n->exp; e != NULL; e = e->esucc )
	if ( e->mark == 'P' ) {
	  patns++;
	  break;
	  }

      if ( n->imp->mark == 'P' )
	syncatns++;
      else
	fsyncatns++;

      if ( IsSGraph( n->imp->src ) )
	if ( IsLoop( n->imp->src->G_DAD ) )
	  for ( e = n->exp; e != NULL; e = e->esucc )
	    if ( IsSGraph( e->dst ) && ( e->iport == n->imp->eport ) ) {
              incratns++;

	      if ( !(n->imp->mark == 'P' && e->mark == 'P') )
		fincratns++;

              break;
	      }

      break;

    case IFACatenateAT:
      convatns++;

      for ( e = n->exp; e != NULL; e = e->esucc )
	if ( e->mark == 'P' ) {
	  patns++;
	  break;
	  }

      if ( n->imp->isucc->mark == 'P' && n->imp->mark == 'P' )
	syncatns++;
      else
	fsyncatns++;

      if ( IsSGraph( n->imp->src ) )
	if ( IsLoop( n->imp->src->G_DAD ) )
	  for ( e = n->exp; e != NULL; e = e->esucc )
	    if ( IsSGraph( e->dst ) && ( e->iport == n->imp->eport ) ) {
              incratns++;

	      if ( !(n->imp->mark == 'P' && e->mark == 'P') )
		fincratns++;

              break;
	      }

      if ( IsSGraph( n->imp->isucc->src ) )
	if ( IsLoop( n->imp->isucc->src->G_DAD ) )
	  for ( e = n->exp; e != NULL; e = e->esucc )
	    if ( IsSGraph( e->dst ) && ( e->iport == n->imp->isucc->eport ) ) {
              incratns++;

	      if ( !(n->imp->isucc->mark == 'P' && e->mark == 'P') )
		fincratns++;

              break;
	      }
      break;

    case IFABuildAT:
    case IFAFillAT:
      convatns++;
      syncatns++;

      for ( e = n->exp; e != NULL; e = e->esucc )
	if ( e->mark == 'P' ) {
	  patns++;
	  break;
	  }

      break;

    case IFAGatherAT:
    case IFFinalValueAT:
    case IFReduceAT:
    case IFRedTreeAT:
    case IFRedLeftAT:
    case IFRedRightAT:
      convatns++;
      syncatns++;

      for ( e = n->exp->dst->G_DAD->exp; e != NULL; e = e->esucc )
	if ( e->eport == n->exp->iport )
	  if ( e->mark == 'P' ) {
	    patns++;
	    break;
	    }

      break;


    default:
      break;
    }
}


static void PrintATNodeInfo()
{
  fprintf( stderr, "\n   * INSERTED AT-NODE INFORMATION\n" );
  fprintf( stderr, " Total Number of AT-NODES: %d\n", inatns );
  fprintf( stderr, " Total Number of CONVERTED AT-NODES: %d\n", convatns );
  fprintf( stderr, " Total Number of SYNC AT-NODES: %d\n", syncatns );
  fprintf( stderr, " Total Number of FAILED SYNC AT-NODES: %d\n", fsyncatns );
  fprintf( stderr, " Total Number of PARENT AT-NODES: %d\n", patns );
  fprintf( stderr, " Total Number of Incremental-Build AT-NODES: %d\n", 
	   incratns );
  fprintf( stderr, " Total Number of FAILED Incremtnal-Build AT-NODES: %d\n",
		     fincratns );
}

/**************************************************************************/
/* LOCAL  **************           Count           ************************/
/**************************************************************************/
/* PURPOSE: COUNT NODES, LITERALS, AND EDGES IN GRAPH g, PRODUCING A PER- */
/*          LEVEL SUMMARY AND TOTALS.                                     */
/**************************************************************************/

static void Count( g )
PNODE g;
{
    register PNODE n;
    register PEDGE i;

    if ( (++topl) >= MaxLevel )
	Error1( "Count: LEVEL STACK OVERFLOW" );

    if ( topl > maxl )
	maxl = topl;

    levels[topl].graphs++;

    gnodes[ g->type - IFSGraph ]++; graphs++;

    for ( i = g->imp; i != NULL; i = i->isucc )
	if ( IsConst( i ) ) {
	    lits++;
	    levels[topl].lits++;
        } else {
	    edges++;
	    levels[topl].edges++;
	    }

    for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
        for ( i = n->imp; i != NULL; i = i->isucc )
	    if ( IsConst( i ) ) {
	        lits++;
	        levels[topl].lits++;
            } else {
	        edges++;
	        levels[topl].edges++;
	        }

	if ( IsSimple( n ) ) {
	    if ( IsPeek( n ) )
	      continue;

            snodes[ n->type - IFAAddH ]++; simples++; levels[topl].simples++;
	    
            if ( IsReduction( n ) )
		switch ( n->imp->CoNsT[0] ) {
		    case REDUCE_SUM:
			rsum++;
			break;

		    case REDUCE_PRODUCT:
			rprod++;
			break;

		    case REDUCE_LEAST:
			rleast++;
			break;

		    case REDUCE_GREATEST:
			rgreat++;
			break;

		    case REDUCE_CATENATE:
			rcat++;
			break;

		    default:
			Error1( "ILLEGAL REDUCTION FUNCTION" );
		    }

	    continue;
	    }

	if ( IsAtNode( n ) ) {
            atnodes[ n->type - IFAAddLAT ]++; ats++; levels[topl].ats++;
	    continue;
	    }

	cnodes[ n->type - IFForall ]++; comps++; levels[topl].comps++;

	for ( g = n->C_SUBS; g != NULL; g = g->gsucc )
            Count( g );
	}

    topl--;
}


/**************************************************************************/
/* LOCAL  **************       WriteCountInfo      ************************/
/**************************************************************************/
/* PURPOSE: WRITE COUNT VALUES TO stderr: TOTALS AND A SUMMARY BY LEVEL.  */
/**************************************************************************/

static void WriteCountInfo( msg )
char *msg;
{
    register struct level *l;
    register int           i;

    fprintf( stderr, "\n   * OCCURRENCE COUNTS %s\n\n", msg );

    fprintf( stderr, " Lits  %4d Edges %4d Smpls %4d Comps %4d Grphs %4d",
		       lits, edges, simples, comps, graphs              );

    fprintf( stderr, " AtNds %4d", ats );

    for ( i = 0; i < IF1GraphNodes; i++ ) {
	if ( (i % 7) == 0 )
	    fprintf( stderr, "\n" );

        fprintf( stderr, " %-5.5s %4d", gnames[i], gnodes[i] );
	}

    for ( i = 0; i < IF1CompoundNodes; i++ ) {
	if ( (i % 7) == 0 )
	    fprintf( stderr, "\n" );

        fprintf( stderr, " %-5.5s %4d", cnames[i], cnodes[i] );
	}

    for ( i = 0; i < IF1SimpleNodes; i++ ) {
	if ( (i % 7) == 0 )
	    fprintf( stderr, "\n" );

        fprintf( stderr, " %-5.5s %4d", snames[i], snodes[i] );
	}

    fprintf( stderr, "\n RSum  %4d RProd %4d RLst  %4d RGrt  %4d RCat  %4d",
		     rsum, rprod, rleast, rgreat, rcat                    );

    for ( i = 0; i < IF2AtNodes; i++ ) {
	if ( (i % 7) == 0 )
	    fprintf( stderr, "\n" );

        fprintf( stderr, " %-6.6s %3d", atnames[i], atnodes[i] );
	}

    fprintf( stderr, "\n" );

    fprintf( stderr, "\n   * SUMMARY BY LEVEL\n\n" );

    for ( i = 0; i <= maxl; i++ ) {
	l = &(levels[i]);

        fprintf( stderr, " Level %4d Lits  %4d Edges %4d Smpls %4d", i, 
			 l->lits, l->edges, l->simples               );

	fprintf( stderr, " AtNds %4d Comps %4d Grphs %4d\n",
		         l->ats, l->comps, l->graphs      );
        }

}


/**************************************************************************/
/* GLOBAL **************     CountNodesAndEdges    ************************/
/**************************************************************************/
/* PURPOSE: COUNT OCCURRENCES OF GRAPH NODES, SUBGRAPH NODES, SIMPLE      */
/*          NODES, COMPOUND NODES, LITERALS, AND EDGES IN ALL FUNCTION    */
/*          GRAPHS.  THE COUNTS ARE PRINTED TO stderr.                    */
/**************************************************************************/

void CountNodesAndEdges( msg )
char *msg;
{
    register struct level *l;
    register PNODE         f;
    register int           i;

    lits = edges  = simples = graphs = comps = ats = 0;
    rsum = rgreat = rleast  = rcat   = rprod = 0;


    topl = maxl =  -1;

    for ( i = 0; i < MaxLevel; i++ ) {
	l = &(levels[i]);

	l->lits  = l->edges  = l->simples = 0;
	l->comps = l->graphs = l->ats     = 0;
	}

    for ( i = 0; i < IF1GraphNodes; i++ )
	gnodes[i] = 0;

    for ( i = 0; i < IF1SimpleNodes; i++ )
	snodes[i] = 0;

    for ( i = 0; i < IF1CompoundNodes; i++ )
	cnodes[i] = 0;

    for ( i = 0; i < IF2AtNodes; i++ )
	atnodes[i] = 0;

    for ( f = glstop->gsucc; f != NULL; f = f->gsucc )
	Count( cfunct = f );

    WriteCountInfo( msg );
}


/**************************************************************************/
/* LOCAL  **************     WriteNotConverted     ************************/
/**************************************************************************/
/* PURPOSE: IF IF1 NODE n EXPORTS AN ARRAY THEN PRINT A FILURE TO CONVERT */
/*          TO AT-NODE MESSAGE TO stderr.  STRING op DEFINES THE SYMBOLIC */
/*          NAME FOR NODE n.  THE ORIGINAL SOURCE FILE POSITION OF n IS   */
/*          PRINTED.                                                      */
/**************************************************************************/

static void WriteNotConverted( n, op )
PNODE  n;
char  *op;
{
    if ( !IsArray( n->exp->info ) )
	return;

    fprintf( stderr, "  AT-NODE CONVERSION FAILURE: %s: %s, %s, %d\n",
		        op, n->file, n->funct, n->line              );
}


/**************************************************************************/
/* LOCAL  **************       IsPMarked           ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF A REFERENCE TO EXPORT eport OF NODE n CARRIES  */
/*          A `P' MARK.                                                   */
/**************************************************************************/

static int IsPMarked( n, eport )
PNODE n;
int   eport;
{
  register PEDGE e;

  for ( e = n->exp; e != NULL; e = e->esucc ) {
    if ( e->eport != eport )
      continue;

    if ( e->mark == 'P' )
      return( TRUE );
    }

  return( FALSE );
}


/**************************************************************************/
/* LOCAL  **************        TraceBuffer        ************************/
/**************************************************************************/
/* PURPOSE: TRACE THE BUFFER EXPORTED FROM NODE n WITH EXPORT PORT eport  */
/*          TO ITS FINAL AT-NODE DESTINATIONS, PRINTING THEIR SYMBOLIC    */
/*          NAMES AND ORIGINAL SOURCE FILE LOCATIONS. MARK PRAGMAS ON     */
/*          RELAVENT IMPORTS ARE PRINTED.                                 */
/**************************************************************************/

void TraceBuffer( n, eport )
PNODE n;
int   eport;
{
    register PEDGE e;
    register PEDGE ee;
    register PNODE a;

    for ( e = n->exp; e != NULL; e = e->esucc ) {
	if ( e->eport != eport )
	    continue;

	a = e->dst;
        switch ( a->type ) {
	    case IFAAddHAT:
	        if ( IsPMarked( a, 1 ) && (UsageCount( a, 1 ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   array_addh(%%mk=%c,%%mk=V): %s, %s, %d\n", 
				 a->imp->mark, a->file, a->funct, a->line    );

                break;

	    case IFAAddLAT:
	        if ( IsPMarked( a, 1 ) && (UsageCount( a, 1 ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   array_addl(%%mk=%c,%%mk=V): %s, %s, %d\n", 
				 a->imp->mark, a->file, a->funct, a->line    );

                break;

	    case IFACatenateAT:
	        if ( IsPMarked( a, 1 ) && (UsageCount( a, 1 ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   %%mk=%c || %%mk=%c: %s, %s, %d\n",
				 a->imp->mark, a->imp->isucc->mark, a->file,
				 a->funct, a->line                        );

                break;

	    case IFAFillAT:
	        if ( IsPMarked( a, 1 ) && (UsageCount( a, 1 ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   array_fill(low,hi,%%mk=V): %s, %s, %d\n", 
				 a->file, a->funct, a->line                 );

                break;

	    case IFABuildAT:
	        if ( IsPMarked( a, 1 ) && (UsageCount( a, 1 ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   array [low,%%mk=V...]: %s, %s, %d\n", 
				 a->file, a->funct, a->line             );

                break;

	    case IFAGatherAT:
	        if ( IsPMarked( a->exp->dst->G_DAD, a->exp->iport ) && 
		     (UsageCount( a->exp->dst->G_DAD, a->exp->iport ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   returns array of %%mk=V: %s, %s, %d", 
				 a->file, a->funct, a->line             );

                if ( a->imp->isucc->isucc->iport == 3 )
		    fprintf( stderr, " FILTER OVERRIDE\n" );
                else
		    fprintf( stderr, "\n" );

                break;

	    case IFFinalValueAT:
	        if ( IsPMarked( a->exp->dst->G_DAD, a->exp->iport ) && 
		     (UsageCount( a->exp->dst->G_DAD, a->exp->iport ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   returns value of %%mk=P: %s, %s, %d\n", 
				 a->file, a->funct, a->line               );

                break;

	    case IFReduceAT:
	        if ( IsPMarked( a->exp->dst->G_DAD, a->exp->iport ) && 
		     (UsageCount( a->exp->dst->G_DAD, a->exp->iport ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   returns value of catenate " );
                goto ReduceFinish;

	    case IFRedLeftAT:
	        if ( IsPMarked( a->exp->dst->G_DAD, a->exp->iport ) && 
		     (UsageCount( a->exp->dst->G_DAD, a->exp->iport ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   returns value of left catenate " );
		goto ReduceFinish;

	    case IFRedRightAT:
	        if ( IsPMarked( a->exp->dst->G_DAD, a->exp->iport ) && 
		     (UsageCount( a->exp->dst->G_DAD, a->exp->iport ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   returns value of right catenate " );
		goto ReduceFinish;

	    case IFRedTreeAT:
	        if ( IsPMarked( a->exp->dst->G_DAD, a->exp->iport ) && 
		     (UsageCount( a->exp->dst->G_DAD, a->exp->iport ) > 1) )
	          fprintf( stderr, "[P_FANOUT]\n" );

		fprintf( stderr, "   returns value of tree catenate " );

ReduceFinish:

		fprintf( stderr, "%%mk=%c: %s, %s, %d\n", 
				 a->imp->isucc->isucc->mark, a->file, a->funct,
				 a->line                                     );

                break;


	    case IFShiftBuffer:
		TraceBuffer( a, 1 );
		break;

	    case IFForall:
		/* DOES THE GENERATE SUBGRAPH SCATTER PARTITIONS OF THE   */
		/* BUFFER INTO THE BODY SUBGRAPH?  IF SO, TRACE THEM INTO */
		/* THE BODY SUBGRAPH!                                     */

		if ( (ee = FindExport( a->F_GEN, e->iport )) != NULL )
		    TraceBuffer( a->F_BODY, ee->dst->exp->iport );

                TraceBuffer( a->F_RET, e->iport );
		break;

	    case IFLoopA:
	    case IFLoopB:
		/* IS THE BUFFER REFERENCED IN THE INITIAL OR BODY        */
		/* SUBGRAPH? IF SO, FOLLOW IT!                            */

		if ( (ee = FindExport( a->L_INIT, e->iport )) != NULL )
		    TraceBuffer( a->L_BODY, ee->iport );
                else if ( IsExport( a->L_BODY, e->iport ) )
		    TraceBuffer( a->L_BODY, e->iport );

                TraceBuffer( a->L_RET, e->iport );
                break;

	    case IFSGraph:
		break;                             /* LOOP CARRIED BUFFER */

	    case IFSelect:        /* SPECIAL BUFFER REMOVAL AND COMBINING */
		if ( IsExport( a->S_CONS, e->iport ) ) {
		  fprintf( stderr, "[S_CONS]\n" );
		  TraceBuffer( a->S_CONS, e->iport );
		  }

		if ( IsExport( a->S_ALT, e->iport ) ) {
		  fprintf( stderr, "[S_ALT]\n" );
		  TraceBuffer( a->S_ALT, e->iport );
		  }

		break;

	    default:
		Error1( "TraceBuffer: ILLEGAL BUFFER DESTINATION NODE" );
		break;
            }
        }
}


/**************************************************************************/
/* LOCAL  **************     WriteGraphPicture     ************************/
/**************************************************************************/
/* PURPOSE: WRITE INFORMATION CONCERNING BUFFER USAGE IN GRAPH g TO       */
/*          stderr.  IF1 TO IF2 AT-NODE CONVERSION FAILURES ARE REPORTED  */
/*          IN THIS ROUTINE.                                              */
/**************************************************************************/

static void WriteGraphPicture( g )
PNODE g;
{
    register PNODE n;

    for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
	GatherATNodeInfo( n );

	if ( IsCompound( n ) ) {
	    for ( g = n->C_SUBS; g != NULL; g = g->gsucc )
		WriteGraphPicture( g );

            continue;
	    }

	switch ( n->type ) {
            case IFAAddH: 
	        if ( !IsArray( n->exp->info ) )
	            break;

		WriteNotConverted( n, "array_addh(...)" );
		break;

	    case IFAAddL:
	        if ( !IsArray( n->exp->info ) )
	            break;

		WriteNotConverted( n, "array_addl(...)" );
		break;

	    case IFACatenate:
	        if ( !IsArray( n->exp->info ) )
	            break;

		WriteNotConverted( n, "||" );
		break;

	    case IFAFill:
	        if ( !IsArray( n->exp->info ) )
	            break;

		WriteNotConverted( n, "array_fill(...)" );
		break;

	    case IFABuild:
	        if ( !IsArray( n->exp->info ) )
	            break;

		WriteNotConverted( n, "array [...]" );
		break;

	    case IFAGather:
	        if ( !IsArray( n->exp->info ) )
	            break;

		WriteNotConverted( n, "returns array of" );
		break;

	    case IFFinalValue:
	        if ( !IsArray( n->exp->info ) )
	            break;

		WriteNotConverted( n, "returns value of" );
		break;

	    case IFReduce:
	        if ( !IsArray( n->exp->info ) )
	            break;

		if ( n->imp->CoNsT[0] != REDUCE_CATENATE )
		    break;

		WriteNotConverted( n, "returns value of catenate" );
		break;

	    case IFRedLeft:
	        if ( !IsArray( n->exp->info ) )
	            break;

		if ( n->imp->CoNsT[0] != REDUCE_CATENATE )
		    break;

		WriteNotConverted( n, "returns value of left catenate" );
		break;

	    case IFRedRight:
	        if ( !IsArray( n->exp->info ) )
	            break;

		if ( n->imp->CoNsT[0] != REDUCE_CATENATE )
		    break;

		WriteNotConverted( n, "returns value of right catenate" );
		break;

	    case IFRedTree:
	        if ( !IsArray( n->exp->info ) )
	            break;

		if ( n->imp->CoNsT[0] != REDUCE_CATENATE )
		    break;

		WriteNotConverted( n, "returns value of tree catenate" );
		break;

	    case IFMemAlloc:
		fprintf( stderr, "  MEMORY BUFFER USAGE TRACE:\n" );
		TraceBuffer( n, 1 );
		break;

	    default:
		break;
	    }
        }
}


/**************************************************************************/
/* GLOBAL **************     WriteIf2memPicture    ************************/
/**************************************************************************/
/* PURPOSE: WRITE INFORMATION CONCERNING BUFFER USAGE IN ALL FUNCTIONS    */
/*          TO stderr; INCLUDING IF1 TO IF2 AT-NODE CONVERSION FAILURES.  */
/**************************************************************************/

void WriteIf2memPicture()
{
    PNODE f;

    fprintf( stderr, "\n   * ARRAY MEMORY MAP\n" );

    for ( f = glstop->gsucc ; f != NULL ; f = f->gsucc ) {
	if ( IsIGraph( f ) )
	    continue;

	fprintf( stderr, "\n FUNCTION %s(...)\n", f->G_NAME );

	WriteGraphPicture( f );
	}
    
    PrintATNodeInfo();
}


/**************************************************************************/
/* LOCAL  **************        PrintWarning       ************************/
/**************************************************************************/
/* PURPOSE: PRINT op'S COPY WARNING TO stderr.                            */
/**************************************************************************/

static void PrintWarning( op, line, funct, file, copy )
char *op;
int   line;
char *funct;
char *file;
int   copy;
{
  if ( funct == NULL ) 
    funct = "???()";

  if ( file == NULL )  
    file = "???.sis";

  if ( copy )
    fprintf( stderr,"%s: W - %s ON LINE %d OF %s IN %s INTRODUCES COPYING\n",
	            program, op, line, funct, file                         );
  else
    fprintf( stderr,"%s: W - %s ON LINE %d OF %s IN %s MAY INTRODUCE COPYING\n",
	            program, op, line, funct, file                            );
}


/**************************************************************************/
/* LOCAL  **************     WriteGraphWarnings     ************************/
/**************************************************************************/
/* PURPOSE: WRITE BUILD COPY WARNINGS FOR GRAPH g TO stderr.              */
/**************************************************************************/

static void WriteGraphWarnings( g )
PNODE g;
{
  register PNODE n;
  register PNODE sg;

  for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
    if ( IsCompound( n ) ) {
      for ( sg = n->C_SUBS; sg != NULL; sg = sg->gsucc )
        WriteGraphWarnings( sg );

      continue;
      }

    switch ( n->type ) {
      case IFAAddHAT:
        if ( n->imp->mark == 'P' )
          break;

        PrintWarning( "array_addh", n->line, n->funct, n->file, TRUE );
	break;

      case IFAAddLAT:
        if ( n->imp->mark == 'P' )
          break;

        PrintWarning( "array_addl", n->line, n->funct, n->file, TRUE );
	break;

      case IFACatenateAT:
        if ( n->imp->isucc->mark == 'P' && n->imp->mark == 'P' )
	  break;

        PrintWarning( "||", n->line, n->funct, n->file, TRUE );
        break;

      case IFAFillAT:
      case IFAGatherAT:
      case IFFinalValueAT:
        break;

      case IFReduceAT:
      case IFRedTreeAT:
      case IFRedLeftAT:
      case IFRedRightAT:
        if ( n->imp->isucc->isucc->mark == 'P' )
          break;

        PrintWarning( "catenate", n->line, n->funct, n->file, TRUE );
        break;

      case IFAAddH: 
        if ( !IsArray( n->exp->info ) )
          break;

        PrintWarning("array_addh", n->line, n->funct, n->file, FALSE );
        break;

      case IFAAddL:
        if ( !IsArray( n->exp->info ) )
          break;

        PrintWarning("array_addl", n->line, n->funct, n->file, FALSE );
        break;

      case IFACatenate:
        if ( !IsArray( n->exp->info ) )
          break;

        PrintWarning( "||", n->line, n->funct, n->file, FALSE );
        break;

      case IFAFill:
        if ( !IsArray( n->exp->info ) )
          break;

        PrintWarning( "array_fill", n->line, n->funct, n->file, FALSE );
        break;

      case IFABuild:
        if ( !IsArray( n->exp->info ) )
          break;

        PrintWarning( "array [..]", n->line, n->funct, n->file, FALSE );
        break;

      case IFAGather:
        if ( !IsArray( n->exp->info ) )
          break;

        PrintWarning( "array of", n->line, n->funct, n->file, FALSE );
        break;

      case IFRedLeft:
      case IFRedRight:
      case IFRedTree:
      case IFReduce:
        if ( !IsArray( n->exp->info ) )
          break;

        if ( n->imp->CoNsT[0] != REDUCE_CATENATE )
          break;

        PrintWarning( "catenate", n->line, n->funct, n->file, FALSE );
	break;

      default:
        break;
        }
      }
}


/**************************************************************************/
/* GLOBAL **************    WriteIf2memWarnings    ************************/
/**************************************************************************/
/* PURPOSE: WRITE AGGREGATE CONSTRUCTION COPY WARNINGS TO stderr.         */
/**************************************************************************/

void WriteIf2memWarnings()
{
  PNODE f;

  for ( f = glstop->gsucc ; f != NULL ; f = f->gsucc ) {
    if ( IsIGraph( f ) )
      continue;

    WriteGraphWarnings( f );
    }
}
