#include "world.h"


/**************************************************************************/
/* GLOBAL **************          MyAlloc          ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE AND RETURN A POINTER TO size BYTES OF MEMORY. IF THE */
/*          ALLOCATION FAILS, AN ERROR MESSAGE IS PRINTED AND EXECUTION   */
/*          TERMINATES.                                                   */
/**************************************************************************/

char *MyAlloc( size )
int size;
{
    char *p;

    if ( (p = malloc( size )) == NULL )
        Error1( "MALLOC FAILED" );

    return( p );
}


/**************************************************************************/
/* LOCAL  **************      MyBBlockAlloc        ************************/
/**************************************************************************/
/* PURPOSE: RETURN A POINTER TO A BBLOCK WHICH CAN BECOME EITHER A NODE  */
/*          OR EDGE OR INFO.  IT MAY NEVER BE FREED!!!!                   */
/**************************************************************************/

#define MAX_BBLOCKS 5000

union bblock {
  NODE n;
  EDGE e;
  INFO i;
  TEMP t;
  };

typedef union bblock BBLOCK, *PBBLOCK;

static PBBLOCK pool;
static int     pidx = MAX_BBLOCKS+100;

static PBBLOCK MyBBlockAlloc()
{
  if ( pidx >= MAX_BBLOCKS ) {
    pool = (PBBLOCK) MyAlloc( sizeof(BBLOCK)*MAX_BBLOCKS );
    pidx = 0;
    }

  return( &(pool[pidx++]) );
}


/**************************************************************************/
/* GLOBAL **************    ChangeExportsToConst   ************************/
/**************************************************************************/
/* PURPOSE: CHANGE ALL EXPORTS OF NODE n WITH EXPORT PORT NUMBER MATCHING */
/*          eport INTO CONSTANT c.                                        */
/**************************************************************************/

void ChangeExportsToConst( n, eport, c )
PNODE n;
int   eport;
PEDGE c;
{
    register PEDGE e;
    register PEDGE se;

    for ( e = n->exp; e != NULL; e = se ) {
	se = e->esucc;

	if ( e->eport == eport ) {
            UnlinkExport( e );

            e->CoNsT = c->CoNsT;
            e->info  = c->info; 
            e->src   = NULL;
            e->esucc = NULL;
            e->epred = NULL;
            e->eport = CONST_PORT;
	    }
	}
}


/**************************************************************************/
/* GLOBAL **************        UsageCount         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE NUMBER OF EXPORT EDGES ATTACHED TO NODE n WITH     */
/*          EXPORT PORT eport.                                            */
/**************************************************************************/

int UsageCount( n, eport )
PNODE n;
int   eport;
{
    register PEDGE e;
    register int   cnt = 0;

    for ( e = n->exp; e != NULL; e = e->esucc )
	if ( e->eport == eport )
	    cnt++;

    return( cnt );
}


/**************************************************************************/
/* GLOBAL **************        FindLastNode       ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE LAST NODE IN THE NODE LIST BEGINNING WITH NODE n;  */
/*          NOTE n ITSELF MIGHT BE THE LAST NODE.                         */
/**************************************************************************/

PNODE FindLastNode( n )
register PNODE n;
{
    while ( n->nsucc != NULL )
	n = n->nsucc;

    return( n );
}


/**************************************************************************/
/* GLOBAL **************        FindExport         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE FIRST EXPORT OF NODE n WITH EXPORT PORT NUMBER     */ 
/*          eport.  IF NOT FOUND, RETURN NULL.                            */
/**************************************************************************/

PEDGE FindExport( n, eport )
PNODE n;
int   eport;
{
    register PEDGE e;

    for ( e = n->exp; e != NULL; e = e->esucc )
	if ( e->eport == eport )
	    return( e );

    return( NULL );
}


/**************************************************************************/
/* GLOBAL **************        FindImport         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE IMPORT OF NODE n WITH IMPORT PORT NUMBER iport. IF */
/*          NOT FOUND, RETURN NULL.                                       */
/**************************************************************************/

PEDGE FindImport( n, iport )
PNODE n;
int   iport;
{
    register PEDGE i;

    for ( i = n->imp; i != NULL; i = i->isucc )
	if ( i->iport == iport )
	    return( i );

    return( NULL );
}


/**************************************************************************/
/* GLOBAL **************        LinkGraph          ************************/
/**************************************************************************/
/* PURPOSE: LINK GRAPH NODE g TO THE DOUBLE LINK LIST CONTAINING GRAPH    */
/*          NODE pred SO TO DIRECTLY FOLLOW pred.  IF pred IS NULL, THE   */
/*          INSERTION WILL NOT TAKE PLACE; REGUARDLESS, NODE g IS ALWAYS  */
/*          RETURNED.                                                     */
/**************************************************************************/

PNODE LinkGraph( pred, g )
PNODE pred;
PNODE g;
{
    if ( pred == NULL )
	return( g );

    g->gsucc = pred->gsucc;
    g->gpred = pred;

    if ( pred->gsucc != NULL )
	pred->gsucc->gpred = g;

    pred->gsucc = g;

    return( g );
}


/**************************************************************************/
/* GLOBAL **************        UnlinkGraph        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK GRAPH NODE g FROM ITS DOUBLE LINK LIST. IF IT EXISTS,  */
/*          THE PREDECESSOR OF g IS RETURNED, ELSE g'S SUCCESSOR IS       */
/*          RETURNED.                                                     */
/**************************************************************************/

PNODE UnlinkGraph( g )
PNODE g;
{
    register PNODE pred = g->gpred;

    if ( pred != NULL )
        pred->gsucc = g->gsucc;

    if ( g->gsucc != NULL )
	g->gsucc->gpred = pred;

    pred     = (pred != NULL)? pred : g->gsucc;
    g->gsucc = NULL;
    g->gpred = NULL;

    return( pred );
}


/**************************************************************************/
/* GLOBAL **************         LinkNode          ************************/
/**************************************************************************/
/* PURPOSE: LINK NODE n TO THE DOUBLE LINK LIST CONTAINING NODE pred SO   */
/*          TO FOLLOW pred. IF pred IS NULL, THE INSERTION IS NOT DONE;   */
/*          REGUARDLESS, NODE n IS RETURNED.                              */
/**************************************************************************/

PNODE LinkNode( pred, n )
PNODE pred;
PNODE n;
{
    if ( pred == NULL )
	return( n );

    n->nsucc = pred->nsucc;
    n->npred = pred;

    if ( pred->nsucc != NULL )
	pred->nsucc->npred = n;

    pred->nsucc = n;

    return( n );
}


/**************************************************************************/
/* GLOBAL **************         UnlinkNode        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK NODE n FROM ITS DOUBLE LINK LIST. IF IT EXISTS, THE    */
/*          PREDECESSOR OF n IS RETURNED, ELSE n'S SUCCESSOR IS RETURNED. */
/**************************************************************************/

PNODE UnlinkNode( n )
PNODE n;
{
    register PNODE pred = n->npred;

    if ( pred != NULL )
        pred->nsucc = n->nsucc;

    if ( n->nsucc != NULL )
	n->nsucc->npred = pred;

    pred     = (pred != NULL)? pred : n->nsucc;
    n->nsucc = NULL;
    n->npred = NULL;

    return( pred );
}


/**************************************************************************/
/* GLOBAL **************        LinkImport         ************************/
/**************************************************************************/
/* PURPOSE: ADD IMPORT e TO THE IMPORT LIST of NODE dst IN iport ORDER.   */
/*          THE PREDECESSOR OF THE FIRST IMPORT IS ALWAYS NULL.           */
/**************************************************************************/

void LinkImport( dst, e )
register PNODE dst;
register PEDGE e;
{
    register PEDGE i;

    e->ipred = NULL;
    e->isucc = NULL;
    e->dst   = dst;

    if ( dst->imp == NULL ) {                       /* IMPORT LIST EMPTY */
        dst->imp = e;
        return;
        }

    if ( dst->imp->iport > e->iport ) {          /* BEFORE FIRST IN LIST */
        e->isucc        = dst->imp;
        dst->imp->ipred = e;
	dst->imp        = e;
        return;
        }

    for( i = dst->imp; i->isucc != NULL; i = i->isucc )       /* WHERE? */
	if ( i->isucc->iport > e->iport )
	    break;

    e->isucc  = i->isucc;                               /* LINK AFTER i */
    e->ipred  = i;

    if ( i->isucc != NULL )
        i->isucc->ipred = e;

    i->isucc = e;
}


/**************************************************************************/
/* GLOBAL **************       UnlinkImport        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK IMPORT i FROM ITS DESTINATION NODE's IMPORT LIST.      */
/**************************************************************************/

void UnlinkImport( i )
PEDGE i;
{
    if ( i->ipred == NULL ) {                         /* FIRST ONE IN LIST */
	i->dst->imp = i->isucc;

	if ( i->isucc != NULL )
	    i->isucc->ipred = NULL;
    } else {
	i->ipred->isucc = i->isucc;

	if ( i->isucc != NULL )
	    i->isucc->ipred = i->ipred;
    }
}


/**************************************************************************/
/* GLOBAL **************        LinkExport         ************************/
/**************************************************************************/
/* PURPOSE: ADD EDGE e TO THE HEAD OF src'S EXPORT LIST. THE PEDECESSOR   */  
/*          OF THE FIRST EXPORT IS ALWAYS NULL.                           */
/**************************************************************************/


void LinkExport( src, e )
PNODE src;
PEDGE e;
{
    e->src = src;

    e->epred = NULL;
    e->esucc = src->exp;

    if ( src->exp != NULL )
	src->exp->epred = e;

    src->exp = e;
}


/**************************************************************************/
/* GLOBAL **************       UnlinkExport        ************************/
/**************************************************************************/
/* PURPOSE: UNLINK EDGE e FROM ITS SOURCE NODE'S EXPORT LIST.  IF e IS A  */
/*          CONSTANT, NOTHING IS DONE.                                    */
/**************************************************************************/

void UnlinkExport( e )
PEDGE e;
{
    if ( IsConst( e ) )
	return;

    if ( e->epred == NULL ) {                         /* FIRST ONE IN LIST */
	e->src->exp = e->esucc;

	if ( e->esucc != NULL )
	    e->esucc->epred = NULL;
    } else {
	e->epred->esucc = e->esucc;

	if ( e->esucc != NULL )
	    e->esucc->epred = e->epred;
    }
}


/**************************************************************************/
/* GLOBAL **************      LinkAssocLists       ************************/
/**************************************************************************/
/* PURPOSE: LINK THE ASSOCIATION LIST lst2 TO lst1 AND RETURN THE RESULT. */
/*          IF THE FIRST LIST IS EMPTY, THE SECOND IS RETURNED.           */
/**************************************************************************/

PALIST LinkAssocLists( lst1, lst2 )
PALIST lst1;
PALIST lst2;
{
    register PALIST l;
    register PALIST prev = NULL;

    for( l = lst1; l != NULL; l = l->next )
	prev = l;

    if ( prev == NULL )
	return( lst2 );

    prev->next = lst2;

    return( lst1 );
}


/**************************************************************************/
/* GLOBAL **************        CopyString         ************************/
/**************************************************************************/
/* PURPOSE: RETURN A COPY OF THE INPUT STRING s.                          */
/**************************************************************************/

char *CopyString( s ) 
char *s;
{
    return( strcpy( MyAlloc( strlen( s ) + 1 ), s ) );
}


/**************************************************************************/
/* GLOBAL **************         Warning1          ************************/
/**************************************************************************/
/* PURPOSE: PRINT A WARNING MESSAGE TO stderr, THEN CONTINUE AS NORMAL.   */
/**************************************************************************/

void Warning1( msg1, line, funct, file )
char *msg1;
int   line;
char *funct;
char *file;
{
    fprintf( stderr, "%s: W - %s (file=%s,funct=%s,line=%d)\n", 
	     program, msg1, file, funct, line );
}


/**************************************************************************/
/* GLOBAL **************          Error1           ************************/
/**************************************************************************/
/* PURPOSE: PRINT AN ERROR MEASAGE TO stderr AND ABORT EXECUTION.         */
/**************************************************************************/

void Error1( msg1 )
char *msg1;
{
    fprintf( stderr, "%s: E - %s\n", program, msg1 );
    Stop( ERROR );
}


/**************************************************************************/
/* GLOBAL **************          Error2           ************************/
/**************************************************************************/
/* PURPOSE: PRINT TWO ERROR MEASAGES TO stderr AND ABORT EXECUTION.       */
/**************************************************************************/

void Error2( msg1, msg2 )
char *msg1;
char *msg2;
{
    fprintf( stderr, "%s: E - %s %s\n", program, msg1, msg2 );
    Stop( ERROR );
}


/**************************************************************************/
/* GLOBAL **************         TempAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN A TEMPORARY                  */
/**************************************************************************/

PTEMP TempAlloc( name, info, mod, fld )
char  *name;
PINFO  info;
int    mod;
int    fld;
{
    register PTEMP   t;
    register PBBLOCK b;
	     char    buf[100];

    /* t = (PTEMP) MyAlloc( sizeof(TEMP) ); */
    b = MyBBlockAlloc();
    t = &(b->t);

    t->DeAl    = FALSE;
    t->status  = UNKNOWN;
    t->next    = NULL;
    t->info    = info;
    t->mod     = mod;
    t->fld     = fld;
    t->pop     = 0;
    t->dist    = 0;
    t->fid     = -1;
    t->ccost   = 0.0;

    sprintf( buf, "%s%d", name, ++tmpid );
    t->name = CopyString( buf );

    return( t );
}


/**************************************************************************/
/* GLOBAL **************         EdgeAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN EDGE.                     */
/**************************************************************************/

PEDGE EdgeAlloc( src, eport, dst, iport )
PNODE src;
int   eport;
PNODE dst;
int   iport;
{
    register PEDGE e;
    PBBLOCK b;

    /* e = (PEDGE) MyAlloc( sizeof(EDGE) ); */
    b = MyBBlockAlloc();
    e = &(b->e);

    e->eport = eport;
    e->src   = src;
    e->iport = iport;
    e->dst   = dst;
    e->temp  = NULL;
    e->rc    = -1;
    e->vtemp = -1;
    e->info  = NULL;
    e->sdbx  = -1;

    e->epred = e->esucc = NULL;
    e->ipred = e->isucc = NULL;

    e->if1line = 0;

    InitPragmas( e );

    return( e );
}


/**************************************************************************/
/* GLOBAL **************         NodeAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN A NODE.                      */
/**************************************************************************/

PNODE NodeAlloc( label, type )
int   label;
int   type;
{
    register PNODE n;
    PBBLOCK b;

    /* n = (PNODE) MyAlloc( sizeof(NODE) ); */
    b = MyBBlockAlloc();
    n = &(b->n);

    n->label = label;
    n->type  = type;

    n->imp   = NULL;
    n->exp   = NULL;
    n->temp  = NULL;
    n->info  = NULL;
    n->next  = NULL;
    n->usucc = NULL;
    n->sdbx  = -1;

    n->flps  = NULL;

    n->gpred = n->gsucc = NULL;
    n->npred = n->nsucc = NULL;

    InitPragmas( n );

    return( n );
}


/**************************************************************************/
/* GLOBAL **************         InfoAlloc         ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN INFO NODE.                */
/**************************************************************************/


PINFO InfoAlloc( label, type )
int   label;
int   type;
{
    register PINFO i;
    PBBLOCK b;

    /* i = (PINFO) MyAlloc( sizeof(INFO) ); */
    b = MyBBlockAlloc();
    i = &(b->i);

    i->info1 = i->info2 = NULL;

    i->label  = label;
    i->type   = type;

    i->touch1 = FALSE;
    i->touch2 = FALSE;
    i->touch3 = FALSE;
    i->touch4 = FALSE;
    i->touch5 = FALSE;
    i->touch6 = FALSE;

    i->fmem   = FALSE;

    i->tname  = NULL;
    i->cname  = NULL;
    i->sname  = NULL;
    i->rname  = NULL;
    i->wname  = NULL;

    i->fname1 = NULL;
    i->fname2 = NULL;

    InitPragmas( i );

    i->next  = NULL;

    return( i );
}


/**************************************************************************/
/* GLOBAL **************      AssocListAlloc       ************************/
/**************************************************************************/
/* PURPOSE: ALLOCATE, INITIALIZE, AND RETURN AN ASSOCIATION LIST NODE.    */
/**************************************************************************/

PALIST AssocListAlloc( datum )
int datum;
{
    register PALIST l;

    l = (PALIST) MyAlloc( sizeof(ALIST) );

    l->next  = NULL;
    l->datum = datum;

    return( l );
}


/**************************************************************************/
/* GLOBAL **************         UpperCase         ************************/
/**************************************************************************/
/* PURPOSE: CONVERT THE LOWER CASE LETTERS OF NAME n TO UPPER CASE. THIS  */
/*          ROUTINE APPEND A '_' PREFIX AND SUFFIX TO n IF underl AND     */
/*          underr ARE TRUE RESPECTIVELY.                                 */
/**************************************************************************/

char *UpperCase( n, underl, underr )
char *n;
int   underl;
int   underr;
{
    register char *p;
             char  buf[100];

    sprintf( buf, "%s%s%s", (underl)? "_" : "", n, (underr)? "_" : "" );

    for ( p = buf; *p != '\0'; p++ )
	if ( (*p >= 'a') && (*p <= 'z') )
	    *p = 'A' + (*p - 'a'); 
        else if ( *p == '.' )
	    *p = '_';

    return( CopyString( buf ) );
}


/**************************************************************************/
/* GLOBAL **************       FindFunction        ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE FUNCTION GRAPH WITH NAME nm.  IF IT IS NOT FOUND,  */
/*          AN ERROR MESSAGE IS PRINTED AND COMPILATION STOPS.            */
/**************************************************************************/

PNODE FindFunction( nm )
char *nm;
{
    register PNODE f;

    for ( f = glstop->gsucc; f != NULL; f = f->gsucc )
	if ( strcmp( nm, f->G_NAME ) == 0 )
	    return( f );

    Error1( "FindFunction FAILED" );
}


/**************************************************************************/
/* GLOBAL **************      LinkExportLists      ************************/
/**************************************************************************/
/* PURPOSE: LINK THE EXPORT LIST OF NODE n2 TO THAT OF NODE n1. NODE n2'S */
/*          EXPORT LIST IS SET TO NULL.                                   */
/**************************************************************************/

void LinkExportLists( n1, n2 )
PNODE n1;
PNODE n2;
{
  register PEDGE e;
  register PEDGE se;

  for ( e = n2->exp; e != NULL; e = se ) {
    se = e->esucc;
    LinkExport( n1, e );
    }

  n2->exp = NULL;
}


void ImportSwap( n )
PNODE n;
{
  register PEDGE one;
  register PEDGE two;

  one = n->imp;
  two = one->isucc;

  UnlinkImport( one );
  UnlinkImport( two );

  one->iport = 2;
  two->iport = 1;

  LinkImport( n, two );
  LinkImport( n, one );
}


int IsDivByZero( n )
PNODE n;
{
  if ( n->type != IFDiv )
    return( FALSE );

  if ( !IsConst( n->imp->isucc ) )
    return( FALSE );

  switch ( n->imp->isucc->info->type ) {
    case IF_DOUBLE:
    case IF_REAL:
      if ( atof( n->imp->isucc->CoNsT ) == 0.0 )
	return( TRUE );

      break;

    case IF_INTEGER:
      if ( atoi( n->imp->isucc->CoNsT ) == 0 )
	return( TRUE );

      break;

    default:
      break;
    }

  return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************          CopyEdge         ************************/
/**************************************************************************/
/* PURPOSE: RETURN A COPY OF EDGE e WITH SOURCE NODE src AND DESTINATION  */
/*          NODE dst.  THE PORT NUMBERS ARE TAKEN FROM e.                 */
/**************************************************************************/

PEDGE CopyEdge( e, src, dst )
PEDGE e;
PNODE src;
PNODE dst;
{
    register PEDGE ee;

    ee = EdgeAlloc( src, e->eport, dst, e->iport );  

    CopyPragmas( e, ee );

    ee->CoNsT   = e->CoNsT;
    ee->info    = e->info;
    ee->if1line = e->if1line;

    return( ee );
}


/**************************************************************************/
/* GLOBAL **************     CopyEdgeAndLink       ************************/
/**************************************************************************/
/* PURPOSE: COPY EDGE e WITH DESTINATION NODE dst AND LINK IT INTO THE    */
/*          GRAPH. THE DESTINATION INPUT PORT NUMBER IS iport.            */
/**************************************************************************/

void CopyEdgeAndLink( e, dst, iport )
PEDGE e;
PNODE dst;
int   iport;
{
  register PEDGE i;

  i = CopyEdge( e, e->src, dst );
  i->iport = iport;

  /* PRESERVE REFERENCE COUNT OPERATIONS */
  i->sr = e->sr;
  i->pm = e->pm;

  if ( !IsConst( e ) )
    LinkExport( e->src, i );

  LinkImport( dst, i );
}


/**************************************************************************/
/* GLOBAL **************      TouchTheUnions       ************************/
/**************************************************************************/
/* PURPOSE: TOUCH ALL UNION TYPES (touch2) REACHED THROUGH TYPE i.        */
/**************************************************************************/

void TouchTheUnions( i )
PINFO i;
{
  register PINFO ii;

  /* WATCH OUT FOR RECURSIVE UNIONS! */
  if ( i->touch2 )
    return;

  switch( i->type ) {
    case IF_UNION:
      i->touch2 = TRUE;

    case IF_RECORD:
      for ( ii = i->R_FIRST; ii != NULL; ii = ii->L_NEXT )
	TouchTheUnions( ii->L_SUB );

      break;

    case IF_ARRAY:
      TouchTheUnions( i->A_ELEM );
      break;

    default:
      break;
    }
}


/**************************************************************************/
/* LOCAL  **************          IsCaller         ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF GRAPH g CALLS FUNCTION who.                    */
/**************************************************************************/

static int IsCaller( g, who )
PNODE  g;
char  *who;
{
  register PNODE  n;
  register PNODE  sg;

  for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
    if ( IsCompound( n ) )
      for ( sg = n->C_SUBS; sg != NULL; sg = sg->gsucc )
        if ( IsCaller( sg, who ) )
	  return( TRUE );

    if ( IsCall( n ) )
      if ( strcmp( n->imp->CoNsT, who ) == 0 )
	return( TRUE );
    }

  return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************         IsCalled          ************************/
/**************************************************************************/
/* PURPOSE: RETURNS TRUE IF who IS CALLED BY A GRAPH IN THE GRAPH LIST.   */
/**************************************************************************/

int IsCalled( who )
char *who;
{
  register PNODE f;

  for ( f = glstop->gsucc; f != NULL; f = f->gsucc )
    if ( IsCaller( f, who ) )
      return( TRUE );

  return( FALSE );
}


/**************************************************************************/
/* LOCAL  **************     ClearTouchFlags       ************************/
/**************************************************************************/
/* PURPOSE: CLEAR ALL THE SYMBOL TABLE TOUCH FLAGS.                       */
/**************************************************************************/

void ClearTouchFlags()
{
  register PINFO i;

  for ( i = ihead; i != NULL; i = i->next ) {
    i->touch1 = FALSE;
    i->touch2 = FALSE;
    i->touch3 = FALSE;
    }
}


/**************************************************************************/
/* GLOBAL **************      FindLastImport       ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE LAST IMPORT TO NODE n.                             */
/**************************************************************************/

PEDGE FindLastImport( n )
PNODE n;
{
  register PEDGE i;

  for ( i = n->imp; i != NULL; i = i->isucc )
    if ( i->isucc == NULL )
      break;

  return( i );
}


/**************************************************************************/
/* GLOBAL **************         LowerCase         ************************/
/**************************************************************************/
/* PURPOSE: CONVERT THE UPPER CASE LETTERS OF NAME n TO LOWER CASE. THIS  */
/*          ROUTINE APPEND A '_' PREFIX AND SUFFIX TO n IF underl AND     */
/*          underr ARE TRUE RESPECTIVELY.                                 */
/**************************************************************************/

char *LowerCase( n, underl, underr )
char *n;
int   underl;
int   underr;
{
    register char *p;
             char  buf[100];

    sprintf( buf, "%s%s%s", (underl)? "_" : "", n, (underr)? "_" : "" );

    for ( p = buf; *p != '\0'; p++ )
	if ( (*p >= 'A') && (*p <= 'Z') )
	    *p = 'a' + (*p - 'A'); 
        else if ( *p == '.' )
	    *p = '_';

    return( CopyString( buf ) );
}


/**************************************************************************/
/* GLOBAL **************      NormalizeNode        ************************/
/**************************************************************************/
/* PURPOSE: IF NODE n IMPORTS A CONSTANT AND n IS A Plus OR Times NODE,   */
/*          THEN MAKE THE CONSTANT THE SECOND IMPORT.                     */
/**************************************************************************/

void NormalizeNode( n )
PNODE n;
{
  switch ( n->type ) {
    case IFPlus:
    case IFTimes:
      break;

    default:
      return;
    }

  if ( IsConst( n->imp ) )
    if ( IsInteger( n->imp->info ) )
      ImportSwap( n );
}


/**************************************************************************/
/* GLOBAL **************       IsPMarked           ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF A REFERENCE TO EXPORT eport OF NODE n CARRIES  */
/*          A `P' MARK.                                                   */
/**************************************************************************/

int IsPMarked( n, eport )
PNODE n;
int   eport;
{
  register PEDGE e;

  for ( e = n->exp; e != NULL; e = e->esucc ) {
    if ( e->eport != eport )
      continue;

    if ( e->pmark )
      return( TRUE );
    }

  return( FALSE );
}


/**************************************************************************/
/* GLOBAL **************      FindGraphNode        ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE GRAPH NODE OWNING NODE n.                          */
/**************************************************************************/

PNODE FindGraphNode( n )
PNODE n;
{
  register PNODE nn;

  for ( nn = n; !IsGraph(nn); nn = nn->npred ) 
    /* DO NOTHING */;

  return( nn );
}
