;;;; psgml-lucid.el --- Part of SGML-editing mode with parsing support
;; $Id: psgml-lucid.el,v 1.12 1994/09/25 16:13:26 lenst Exp $

;; Copyright (C) 1994 Lennart Staflin

;; Author: Lennart Staflin <lenst@lysator.liu.se>
;;	   William M. Perry <wmperry@indiana.edu>

;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


;;;; Commentary:

;;; Part of psgml.el

;;; Menus for use with Lucid Emacs


;;;; Code:

(require 'psgml)

(eval-and-compile
  (autoload 'sgml-do-set-option "psgml-edit"))

(defvar sgml-max-menu-size (/ (* (screen-height) 2) 3)
  "*Max number of entries in Tags and Entities menus before they are split
into several panes.")

;;;; Pop Up Menus

(defun sgml-popup-menu (event title entries)
  "Display a popup menu."
  (setq entries
	(loop for ent in entries collect
	      (vector (car ent)
		      (list 'setq 'value (list 'quote (cdr ent)))
		      t)))
  (let ((value nil)
	(event nil)
	(menudesc
	 (cons
	  title
	  (if (> sgml-max-menu-size (length entries))
	      entries
	    (let* ((n (1+ (/ (1- (length entries)) sgml-max-menu-size)))
		   (l (1+ (/ (1- (length entries)) n))))
	      (loop for i from 1 while entries collect
		    (prog1 (cons
			    (format "%s %d" title i)
			    (subseq entries 0 (min l (length entries))))
		      (setq entries (nthcdr l entries)))))))))
    (popup-menu menudesc)
    (while (popup-menu-up-p)
      (setq event (next-command-event event))
      (cond ((and (menu-event-p event)
		  (or (eq (event-object event) 'abort)
		      (eq (event-object event) 'menu-no-selection-hook)))
	     (signal 'quit nil))
	    ((menu-event-p event)
	     (eval (event-object event)))
	    ((button-release-event-p event) ; don't beep twice
	     nil)
	    (t
	     (beep)
	     (message "please make a choice from the menu."))))
    value))

(defun sgml-fix-x-menu (menudesc)
  "Take a menu for x-popup-menu and return a lucid menu."
  (cons (car menudesc)			; title
	(mapcar (function
		 (lambda (item)
		   ;; item is (string . value) or string
		   (if (stringp item)
		       item
		     (vector (car item)
			     (list 'quote (cdr item))
			     t))))
		(cdr menudesc))))


(defun x-popup-menu (pos menudesc)
  "My hacked up function to do a blocking popup menu..."
  (let ((echo-keystrokes 0)
	event menu)
    (cond
     ((stringp (car menudesc))		; deck of meues
      (setq menu (if (null (cddr menudesc)) ; only one menu
		     (sgml-fix-x-menu (cadr menudesc))
		   (cons (car menudesc)
			 (mapcar (function sgml-fix-x-menu)
				 (cdr menudesc))))))
     ((listp (car menudesc))		; deck of keymaps
      (error "NIY"))
     (t					; keymap
      (error "NIY")))
    (popup-menu menu)
    (cadr
     (catch 'popup-done
       (while (popup-menu-up-p)
	 (setq event (next-command-event event))
	 (cond ((and (menu-event-p event)
		     (eq 'quote (car-safe (event-object event))))
		(throw 'popup-done (event-object event)))
	       ((and (menu-event-p event)
		     (or (eq (event-object event) 'abort)
			 (eq (event-object event) 'menu-no-selection-hook)))
		(signal 'quit nil))
	       ((button-release-event-p event) ; don't beep twice
		nil)
	       (t
		(beep)
		(message "please make a choice from the menu."))))))))



;;;; Lucid menu bar

(defun sgml-install-lucid-menus ()
  "Install lucid menus for psgml mode"
  (set-buffer-menubar (copy-sequence current-menubar))
  (add-menu nil (car sgml-sgml-menu) (cdr sgml-sgml-menu))
  (add-menu nil (car sgml-markup-menu) (cdr sgml-markup-menu))
  (add-menu nil (car sgml-fold-menu) (cdr sgml-fold-menu))
  (add-menu nil (car sgml-dtd-menu) (cdr sgml-dtd-menu)))

(defvar sgml-markup-menu
  '("Markup"
    ["Insert Element" (sgml-element-menu last-command-event) t]
    ["Insert Start-Tag" (sgml-start-tag-menu last-command-event) t]
    ["Insert End-Tag" (sgml-end-tag-menu last-command-event) t]
    ["Tag Region" (sgml-tag-region-menu last-command-event) t]
    ["Insert Attribute" (sgml-attrib-menu last-command-event) t]
    ["Insert Entity" (sgml-entities-menu last-command-event) t]
    ))

(defvar sgml-dtd-menu
  '("DTD"
    ["Parse DTD" sgml-parse-prolog t]
    ["Load Parsed DTD" sgml-load-dtd t]
    ["Save Parsed DTD" sgml-save-dtd t]
    ))

(defvar sgml-fold-menu
  '("Fold"
    ["Fold Element" sgml-fold-element t]
    ["Fold Subelement" sgml-fold-subelement t]
    ["Fold Region" sgml-fold-region t]
    ["Unfold Line" sgml-unfold-line t]
    ["Unfold Element" sgml-unfold-element t]
    ["Unfold All" sgml-unfold-all t]
    ["Expand" sgml-expand-element t]
    ))

(defun sgml-make-options-menu (vars)
  (loop for var in vars 
	for type = (sgml-variable-type var)
	for desc = (sgml-variable-description var)
	collect
	(cond
	 ((eq type 'toggle)
	  (vector desc (list 'setq var (list 'not var))
		  ':style 'toggle ':selected var))
	 ((consp type)
	  (cons desc
		(loop for c in type collect
		      (if (atom c)
			  (vector (prin1-to-string c)
				  (`(setq (, var) (, c)))
				  :style 'toggle
				  :selected (`(eq (, var) '(, c))))
			(vector (car c)
				(`(setq (, var) '(,(cdr c))))
				:style 'toggle
				:selected (`(eq (, var) '(,(cdr c)))))))))
	 (t
	  (vector desc
		  (`(sgml-do-set-option '(, var)))
		  t)))))


(defvar sgml-sgml-menu
  (append
   '("SGML"
     ["Next Data Field"  sgml-next-data-field t]
     ["End Element" sgml-insert-end-tag t]
     ["Show Context" sgml-show-context t]
     ["What Element" sgml-what-element t]
     ["Next Trouble Spot" sgml-next-trouble-spot t]
     ["Edit Attributes" sgml-edit-attributes t]
     ["Change Element Name" sgml-change-element-name t]
     ["Show Valid Tags" sgml-list-valid-tags t]
     ["Show/Hide Warning Log" sgml-show-or-clear-log t]
     ["Validate" sgml-validate t]
     ["Normalize" sgml-normalize t]
     ["Fill Element" sgml-fill-element t])
   (if (or (not (boundp 'emacs-major-version))
	   (and (boundp 'emacs-minor-version)
		(< emacs-minor-version 10)))
       '(
	 ["File Options" sgml-file-options-menu t]
	 ["User Options" sgml-user-options-menu t]
	 )
     (list
      (cons "File Options" (sgml-make-options-menu sgml-file-options))
      (cons "User Options" (sgml-make-options-menu sgml-user-options))))
   '(["Save File Options" sgml-save-options t]
     ["Submit Bug Report" sgml-submit-bug-report t]
     )))


;;;; Custom menus

(defun sgml-build-custom-menus ()
  (and sgml-custom-markup (add-menu-item '("Markup") "------------" nil t))
  (mapcar (function
	   (lambda (x)
	     (add-menu-item '("Markup") (nth 0 x)
			    (list 'sgml-insert-markup (nth 1 x)) t)))
	  sgml-custom-markup)
  (and sgml-custom-dtd (add-menu-item '("DTD") "-------------" nil t))
  (mapcar (function
	   (lambda (x)
	     (add-menu-item '("DTD") (nth 0 x)
			    (list 'apply ''sgml-doctype-insert
				  (list 'quote (cdr x)))
			    t)))
	  sgml-custom-dtd))


;;;; Key definitions

(define-key sgml-mode-map [button3] 'sgml-tags-menu)


;;;; Insert with properties

(defun sgml-insert (props format &rest args)
  (let ((start (point))
	tem)
    (insert (apply (function format)
		   format
		   args))
    (remf props 'rear-nonsticky)	; not useful in Lucid

    ;; Copy face prop from category
    (when (setq tem (getf props 'category))
      (when (setq tem (get tem 'face))
	  (set-face-underline-p (make-face 'underline) t)
	  (setf (getf props 'face) tem)))

    (add-text-properties start (point) props)

    ;; A read-only value of 1 is used for the text after values
    ;; and this should in Lucid be open at the front.
    (if (eq 1 (getf props 'read-only))
	(set-extent-property
	 (extent-at start nil 'read-only)
	 'start-open t))))


;;;; Set face of markup

(defun sgml-set-face-for (start end type)
  (let ((face (cdr (assq type sgml-markup-faces)))
	o)
    (loop for e being the extents from start to end
	  do (when (extent-property e 'type)
	       (cond ((and (null o)
			   (eq type (extent-property e 'type)))
		      (setq o e))
		     (t (delete-extent e)))))

    (cond (o
	   (set-extent-endpoints o start end))
	  (face
	   (setq o (make-extent start end))
	   (set-extent-property o 'type type)
	   (set-extent-property o 'face face)
	   (set-extent-face o face)))))

(defun sgml-set-face-after-change (start end &optional pre-len)
  (when sgml-set-face
    (let ((o (extent-at start nil 'type)))
      (cond
       ((null o))
       ((= start (extent-start-position o))
	(set-extent-endpoints o end (extent-end-position o)))
       (t (delete-extent o))))))

;(defalias 'next-overlay-at 'next-overlay-change) ; fix bug in cl.el

(defun sgml-clear-faces ()
  (interactive)
  (loop for o being the overlays
	if (extent-property o 'type)
	do (delete-extent o)))


;;;; Functions not in Lucid Emacs

(unless (fboundp 'frame-width)
  (defalias 'frame-width 'screen-width))



;;;; Provide

(provide 'psgml-lucid)


;;; psgml-lucid.el ends here
