/*  $Header: shortref.c,v 3.0 88/04/13 16:24:19 jos Exp $ */
/*
 *  This file is part of the Amsterdam SGML Parser.
 *
 *  Copyright: Faculteit Wiskunde en Informatica
 *             Department of Mathematics and Computer Science
 *             Vrije Universiteit Amsterdam
 *             The Netherlands
 *
 *  Authors:   Sylvia van Egmond
 *             Jos Warmer
 */
#include "types.h"
#include "Lpars.h"
#include "conc_syn.h"
#include "group.h"
#include "modes.h"
#include "set.h"
#include "shortref.h"
#include "tags.h"

static Bool       debug = FALSE;
static P_Group    map_group;
#ifdef GENERATOR
static P_Group    use_group;
#endif

typedef struct use_map_struct {
    P_Group   elem_names;
    String    map;
}Usemap, *P_Usemap;

typedef struct map_struct {
    String    name;
    String    sh_refs[NR_SH_REFS];
    P_Set     first_chars;
} Map;

Bool is_shortref(delimiter)
Delimiter delimiter;
{
    return ( (delimiter<SH_REF+NR_SH_REFS) and (delimiter>=SH_REF));
}

Bool map_equal(map1, map2)
P_Map map1;
P_Map map2;
{
    return streq(map1->name, map2->name);
}

P_Map new_map(name)
String  name;
{
    P_Map      map;
    int        i;
    extern int mapcap;

    map = (P_Map)MALLOC( (unsigned) sizeof(Map));
    map->name = name;
#ifdef GENERATOR
    map->first_chars = (P_Set)0;
#else
    map->first_chars = new_set(256);
#endif
    for( i=0; i<NR_SH_REFS; i++){
	map->sh_refs[i] = (String)0;
    }
    if( not group_add_unique(map_group, map, map_equal) ){
	report(SHORT_DOUBLE, NOTFATAL, 0, 0, map->name);
	FREE(map, sizeof(Map));
	return ILL_MAP;
    } else {
	mapcap++;
	return map;
    }
}

#ifdef DOC_PARSER
static void add_first_chars(set, ch)
P_Set	set;
int	ch;
{
    if( ch == 'B' ){
	set_add(set, ' ');
	set_add(set, '\t');
    } else {
	set_add(set, ch);
    }
}
#endif

void define_mapping(map, literal, entity_name)
P_Map  map;
String literal;
String entity_name;
{
    Bool   ok;

    if( map == ILL_MAP ){ return; }
    DEB2("define_mapping : `%s' ==> `%s'\n", literal, entity_name);
    ok = string_to_delimiter(literal, MODE_CON);
    if( (ok == TOK_NOD) or (not is_shortref(ok)) ){
        report(SHORT_UNDEF_ROLE, FATAL, 0, 0, literal);
        return;
    } else if( map->sh_refs[ok-1] != 0 ){
	report(SHORT_ROLE_TWO, FATAL, 0, 0, literal);
	return;
    } else {
        map->sh_refs[ok-1] = entity_name;
        DEB1("short reference role %d\n", ok);
    }
#ifdef DOC_PARSER
    add_first_chars(map->first_chars, literal[0]);
#endif
}

#ifdef DEBUG
void  print_map(file, map)
FILE    *file;
P_Map    map;
{
    int i;

    if( map == ILL_MAP ){ return; }
    fprintf(file, "SHORTREF MAP %s\n", map->name);
    for(i=0; i<NR_SH_REFS; i++){
	if( map->sh_refs[i] == (String)0 ){
	    fprintf(file, "%10d `(nil)'\n", i+1);
	} else {
	    fprintf(file, "%10d `%s'\n", i+1, map->sh_refs[i]);
	}
    }
}
#endif

#ifdef GENERATOR
/*  Checks whether each used map is declared.
 *  Checks whether each element has at most one used map
 */
void maps_ok()
{
    P_Usemap   u;
    P_Iterator it = group_iterator(use_group);

    while( u = (P_Usemap)iterator_next(it) ){
	if( lookup_map(u->map) == ILL_MAP ){
	    report(SHORT_UNDEF, FATAL, 0, 0, u->map);
	}
    }
    fatal_report();
}

void generate_maps(src, ext)
FILE  *src;
FILE  *ext;
{
    P_Iterator   it;
    P_Map        map;
    int          i;

    it = group_iterator(map_group);
    while( map = (P_Map)iterator_next(it) ){
	fprintf(ext,"extern struct map_struct MAP%s;\n", cname(map->name));
	fprintf(src,"Map   MAP%s = { \"%s\", { ", cname(map->name), map->name);
	for(i=0; i<NR_SH_REFS; i++){
	    if( map->sh_refs[i] ){
		fprintf(src, "        \"%s\"", map->sh_refs[i]);
	    } else {
                fprintf(src, "        0");
	    }
	    if( i == (NR_SH_REFS-1) ){
		fprintf(src," }, 0 };\n");
	    } else {
		fprintf(src,",\n");
	    }
	}
    }

    fprintf(src,"P_Map  maps[] = {");
    it = group_iterator(map_group);
    while( map = (P_Map)iterator_next(it) ){
	fprintf(src,"&MAP%s, ", cname(map->name));
    }
    fprintf(src,"ILL_MAP };\n");
}

String element_map_var(elem_name)
String elem_name;
{
    String      map_name;
    static char answer[20] ;

    map_name = element_map(elem_name);
    if( lookup_map(map_name) == ILL_MAP ){
	sprintf(answer, "0");
    } else {
	sprintf(answer,"&MAP%s", map_name);
    }
    return answer;
}

String element_map(elem_name)
String elem_name;
{
    P_Usemap   u;
    P_Iterator it = group_iterator(use_group);
    P_Iterator it2;
    static char answer[10] ;
    String     name;

    while( u = (P_Usemap)iterator_next(it) ){
	it2 = group_iterator(u->elem_names);
	while( name = next_name(it2) ){
	    if(streq(name, elem_name)){
		iterator_delete(it);
		iterator_delete(it2);
		return u->map;
	    }
	}
    }
    sprintf(answer, "");
    return answer;
}
#endif

void usemap(elem_names, map)
P_Group elem_names;
String  map;
{
#ifdef DOC_PARSER
    report(USE_ERROR, FATAL, 0, 0);
    return;
#else
    String      str;
    String      name;
    P_Usemap    u;
    P_Iterator  it = group_iterator(elem_names);

    while( name = next_name(it) ){
	str = element_map(name);
	if( not streq(str, "") ){
	    report(USE_TWICE, FATAL, 0, 0, str, map, name);
	}
    }
    u = (P_Usemap)MALLOC(sizeof(Usemap));
    u->elem_names = elem_names;
    u->map        = strsave(map);
    group_add(use_group, u);
#endif
}

/*****************************************************************/
#ifdef DOC_PARSER

#include "map.ext"
#include "map.i"

typedef struct map_stack_struct {
    P_Map        map;
    Delimiter    gi;
} Stackelem, *P_Stackelem;

#define STACK           Mapstack
#define ST_ELEMENT      P_Stackelem
#define stack_struct    mapstackstruct

#define	create_stack    new_mapstack
#define	push            push_mapstack
#define	pop             pop_mapstack
#define	top             top_mapstack

#include "stack.gh"
#include "stack.gen"

static Mapstack mapstack;
static P_Map    current_map;

void push_map(map, gi)
P_Map      map;
Delimiter  gi;
{
    P_Stackelem st;

    if( map == ILL_MAP ){				/* copy previous map */
	st = top_mapstack(mapstack);
	map = st->map;
    }

    if( gi == TOK_NOD ){				/* manual map */
	st = pop_mapstack(mapstack);
	st->map = map;
	DEB2("push: manual map `%s' for element %s\n",
	      map->name,(((st->gi==0)?("EMPTY"):(starttag_to_string(st->gi)))));
    } else {						/* associated map */
	st = (P_Stackelem)MALLOC(sizeof(Stackelem));
	st->map = map;
	st->gi  = gi ;
	DEB2("push: associated map `%s' for element %s\n",
	      map->name, ((st->gi==0)?("EMPTY"):(starttag_to_string(st->gi))));
    }

    push_mapstack(mapstack, st);
    current_map = map;
}

void pop_map(map, gi)
P_Map      map;
Delimiter  gi;
{
    P_Stackelem st;

    st = pop_mapstack(mapstack);
    DEB2("map %s popped for element %s\n",
	  st->map->name, ((gi==0)?("EMPTY"):(starttag_to_string(gi))));
    FREE(st, sizeof(Stackelem));

    st = top_mapstack(mapstack);
    current_map = st->map;
    DEB2("pop: current map `%s' for element %s\n",
	  st->map->name, ((st->gi==0)?("EMPTY"):(starttag_to_string(st->gi))));
}
#endif

String current_mapping(shortref)
Delimiter shortref;
{
    assert( is_shortref(shortref) );
#ifdef GENERATOR
    return 0;
#else
    return(current_map->sh_refs[shortref-1]);
#endif
}

Bool in_current_map(shortref)
Delimiter shortref;
{
    assert( is_shortref(shortref) );
#ifdef GENERATOR
    return FALSE;
#else
    return( (current_map->sh_refs[shortref-1]) != 0 );
#endif
}

Bool is_current_shortref_char(ch)
int  ch;
{
#ifdef GENERATOR
    return FALSE;
#else
    return set_member(current_map->first_chars, ch);
#endif
}

P_Map lookup_map(map_name)
String map_name;
{
    P_Iterator it;
    P_Map      map;

    it = group_iterator(map_group);
    while( map = (P_Map)iterator_next(it) ){
	if( streq(map->name, map_name) ){
	    iterator_delete(it);
	    return map;
	}
    }
    return ILL_MAP;
}

void init_shortref()
{
#ifdef DOC_PARSER
    int    i;
    P_Map  map;
    int    shref;
    String str;
#endif

    map_group = group_create();
#ifdef GENERATOR
    use_group = group_create();
    new_map("EMPTY");
#endif
#ifdef DOC_PARSER
    mapstack = new_mapstack(ILL_MAP);
    push_map(&MAPEMPTY, 0);
    for(i=0; maps[i] != ILL_MAP; i++){
	map = maps[i];
	map->first_chars = new_set(256);
        group_add(map_group, map);
	for(shref=0; shref<NR_SH_REFS; shref++){
	    if( map->sh_refs[shref] != 0 ){
		str = delimiter_to_string(SH_REF+shref);
		DEB1("predefine shortref %s\n", str);
		add_first_chars(map->first_chars, str[0]);
	    }
	}
    }
#endif
}

#ifdef DEBUG
void debug_shortref(b)
Bool b;
{
    debug = b;
}
#endif
