;;; -*-Scheme-*-
;;;
;;; $Id: macros.scm,v 1.10 1993/04/19 17:19:42 cph Exp $
;;;
;;; Copyright (c) 1993 Massachusetts Institute of Technology
;;;
;;; This material was developed by the Scheme project at the
;;; Massachusetts Institute of Technology, Department of Electrical
;;; Engineering and Computer Science.  Permission to copy this
;;; software, to redistribute it, and to use it for any purpose is
;;; granted, subject to the following restrictions and understandings.
;;;
;;; 1. Any copy made of this software must include this copyright
;;; notice in full.
;;;
;;; 2. Users of this software agree to make their best efforts (a) to
;;; return to the MIT Scheme project any improvements or extensions
;;; that they make, so that these may be included in future releases;
;;; and (b) to inform MIT of noteworthy uses of this software.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4. MIT has made no warrantee or representation that the operation
;;; of this software will be error-free, and MIT is under no
;;; obligation to provide any services, by way of maintenance, update,
;;; or otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this
;;; material, there shall be no use of the name of the Massachusetts
;;; Institute of Technology nor of any adaptation thereof in any
;;; advertising, promotional, or sales literature without prior
;;; written consent from MIT in each case.

;;;; Macros for Scheme Object System

(declare (usual-integrations))

(define (transform:define-class name superclasses . slot-arguments)
  (if (not (or (symbol? name)
	       (and (pair? name)
		    (symbol? (car name)))))
      (serror 'DEFINE-CLASS "Malformed class name:" name))
  (if (not (list? superclasses))
      (serror 'DEFINE-CLASS "Malformed class superclasses:" superclasses))
  (let ((name+options (if (pair? name) name (list name))))
    (let ((options (cdr name+options))
	  (args
	   `(',(car name+options)
	     (LIST ,@superclasses)
	     (LIST
	      ,@(map
		 (lambda (arg)
		   (cond ((symbol? arg)
			  `',arg)
			 ((and (pair? arg)
			       (symbol? (car arg))
			       (list? (cdr arg)))
			  `(LIST ',(car arg)
				 ,@(let loop ((plist (cdr arg)))
				     (cond ((null? plist)
					    '())
					   ((and (symbol? (car plist))
						 (pair? (cdr plist)))
					    (cons* `',(car plist)
						   (cadr plist)
						   (loop (cddr plist))))
					   (else
					    (serror 'DEFINE-CLASS
						    "Malformed slot argument:"
						    arg))))))
			 (else
			  (serror 'DEFINE-CLASS
				  "Malformed slot argument:" arg))))
		 slot-arguments)))))
      (if (not (and (list? options)
		    (for-all? options
		      (lambda (option)
			(or (symbol? option)
			    (and (pair? option)
				 (symbol? (car option))
				 (list? (cdr option))))))))
	  (serror 'DEFINE-CLASS "Malformed class options:" options))
      (let ((metaclass '<CLASS>))
	(for-each (lambda (option)
		    (let ((name (if (pair? option) (car option) option)))
		      (case name
			((METACLASS)
			 (if (not (and (pair? option)
				       (pair? (cdr option))
				       (null? (cddr option))))
			     (serror 'DEFINE-CLASS
				     "Malformed class option:" option))
			 (set! metaclass (cadr option)))
			(else
			 (serror 'DEFINE-CLASS
				 "Unknown class option:" name)))))
		  options)
	`(DEFINE ,(car name+options)
	   ,(if (eq? metaclass '<CLASS>)
		`(MAKE-CLASS ,@args)
		`(MAKE-INSTANCE ,metaclass ,@args)))))))

(define (transform:define-generic-procedure name lambda-list . body)
  (if (not (symbol? name))
      (serror 'DEFINE-GENERIC-PROCEDURE
	      "Malformed generic procedure name:" name))
  (let ((initial-method? (not (null? body))))
    (call-with-values
	(lambda ()
	  (parse-lambda-list lambda-list
			     initial-method?
			     'DEFINE-GENERIC-PROCEDURE))
      (lambda (required optional rest)
	(let ((min-arity (length required)))
	  `(BEGIN
	     (DEFINE ,name
	       (MAKE-GENERIC-PROCEDURE ',name
				       ,min-arity
				       ,(and (not rest)
					     (+ min-arity (length optional)))))
	     ,@(if initial-method?
		   `((DEFINE-METHOD ,name ,lambda-list ,@body))
		   '())))))))

(define (transform:define-method name lambda-list . body)
  (if (not (symbol? name))
      (serror 'DEFINE-METHOD "Malformed method name:" name))
  (call-with-values
      (lambda () (parse-lambda-list lambda-list #t 'DEFINE-METHOD))
    (lambda (required optional rest)
      (call-with-values (lambda () (extract-required-specializers required))
	(lambda (required specializers)
	  `(ADD-METHOD ,name
	     ,(make-method-sexp name required optional rest specializers
				body)))))))

(define (transform:method lambda-list . body)
  (call-with-values (lambda () (parse-lambda-list lambda-list #t 'METHOD))
    (lambda (required optional rest)
      (call-with-values (lambda () (extract-required-specializers required))
	(lambda (required specializers)
	  (make-method-sexp #f required optional rest specializers body))))))

(define (make-method-sexp name required optional rest specializers body)
  (call-with-values (lambda () (ignore-call-next-method? body))
    (lambda (body ignore?)
      `(,(if ignore? 'MAKE-LEAF-METHOD 'MAKE-METHOD)
	(LIST ,@specializers)
	,(make-named-lambda name
			    (if ignore?
				required
				(cons 'CALL-NEXT-METHOD required))
			    optional
			    rest
			    body)))))

(define (ignore-call-next-method? body)
  (let ((ignore?
	 (and (not (null? body))
	      (let loop ((tail body))
		(and (symbol? (car tail))
		     (not (null? (cdr tail)))
		     (or (eq? 'CALL-NEXT-METHOD (car tail))
			 (loop (cdr tail))))))))
    (values (if ignore?
		(let loop ((tail body))
		  (if (eq? 'CALL-NEXT-METHOD (car tail))
		      (cdr tail)
		      (cons (car tail) (loop (cdr tail)))))
		body)
	    ignore?)))

(define (extract-required-specializers required)
  (let loop ((required required) (names '()) (specializers '()))
    (cond ((null? required)
	   (values (reverse! names)
		   (reverse! (let loop ((specializers specializers))
			       (if (and (not (null? specializers))
					(eq? '<OBJECT> (car specializers))
					(not (null? (cdr specializers))))
				   (loop (cdr specializers))
				   specializers)))))
	  ((pair? (car required))
	   (loop (cdr required)
		 (cons (caar required) names)
		 (cons (cadar required) specializers)))
	  (else
	   (loop (cdr required)
		 (cons (car required) names)
		 (cons '<OBJECT> specializers))))))

(define (parse-lambda-list lambda-list allow-specializers? specform)
  specform
  (let ((required '())
	(optional '())
	(rest #f))
    (letrec
	((parse-required
	  (lambda (lambda-list)
	    (cond ((null? lambda-list)
		   (finish))
		  ((pair? lambda-list)
		   (cond ((or (valid-name? (car lambda-list))
			      (and allow-specializers?
				   (pair? (car lambda-list))
				   (valid-name? (caar lambda-list))
				   (pair? (cdar lambda-list))
				   (null? (cddar lambda-list))))
			  (set! required (cons (car lambda-list) required))
			  (parse-required (cdr lambda-list)))
			 ((eq? #!optional (car lambda-list))
			  (parse-optional (cdr lambda-list)))
			 ((eq? #!rest (car lambda-list))
			  (parse-rest (cdr lambda-list)))
			 (else
			  (illegal-element lambda-list))))
		  ((symbol? lambda-list)
		   (set! rest lambda-list)
		   (finish))
		  (else
		   (illegal-tail lambda-list)))))
	 (parse-optional
	  (lambda (lambda-list)
	    (cond ((null? lambda-list)
		   (finish))
		  ((pair? lambda-list)
		   (cond ((valid-name? (car lambda-list))
			  (set! optional (cons (car lambda-list) optional))
			  (parse-optional (cdr lambda-list)))
			 ((eq? #!optional (car lambda-list))
			  (error "#!optional may not recur:" lambda-list))
			 ((eq? #!rest (car lambda-list))
			  (parse-rest (cdr lambda-list)))
			 (else
			  (illegal-element lambda-list))))
		  ((symbol? lambda-list)
		   (set! rest lambda-list)
		   (finish))
		  (else
		   (illegal-tail lambda-list)))))
	 (parse-rest
	  (lambda (lambda-list)
	    (if (and (pair? lambda-list)
		     (null? (cdr lambda-list)))
		(if (valid-name? (car lambda-list))
		    (begin
		      (set! rest (car lambda-list))
		      (finish))
		    (illegal-element lambda-list))
		(illegal-tail lambda-list))))
	 (valid-name?
	  (lambda (element)
	    (and (symbol? element)
		 (not (eq? #!optional element))
		 (not (eq? #!rest element)))))
	 (finish
	  (lambda ()
	    (values (reverse! required)
		    (reverse! optional)
		    rest)))
	 (illegal-tail
	  (lambda (lambda-list)
	    (error "Illegal parameter list tail:" lambda-list)))
	 (illegal-element
	  (lambda (lambda-list)
	    (error "Illegal parameter list element:" (car lambda-list)))))
      (parse-required lambda-list))))

(define (make-named-lambda name required optional rest body)
  (let ((bvl
	 (append required
		 (if (null? optional)
		     '()
		     `(#!OPTIONAL ,@optional))
		 (or rest '()))))
    (if name
	`(NAMED-LAMBDA (,name ,@bvl) ,@body)
	`(LAMBDA ,bvl ,@body))))

(define (serror procedure message . objects)
  procedure
  (apply error message objects))