;;;
;;;              Copyright 1991 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;;
;;; stredit.sc -- a simple widget for text input called StringEditor. The
;;;               name comes from the InterViews equivalent, but we do not
;;;               in any way claim the same or nearly as much functionality.
;;;
;;; Problems:
;;;  2) Currently variable width fonts works because we always redraw the
;;;     entire string plus a few trailing blanks. When deleting these
;;;     blanks erase the old character. Thus, we have no knowledge about
;;;     current xpos etc. This makes it difficult to add stuff like a
;;;     cursor, emacs-editing and blocking writes over the edge. Sigh.
;;;     Maybe we need to redo it with control of the width of each char.
;;;  3) Stupid arbitrary limits like (- ypos 4) or xpos = 5.
;;;
;;; $Id: stredit.sc,v 2.4 91/09/15 00:58:30 johani Exp $

(module stoxstredit)

(include "../include/masks.sch")
(include "../include/objects.sch")
(include "../include/lw-objs.sch")
(include "../include/util.sch")

(define-external dfsm stoxdfsm)
(define-external widget stoxwidget)
(define-external add-new-args stoxutil)

(include "../macros/extsyntax.sc")
(include "../macros/oos.sc")

(define-class (stringeditor . args)
  (locals
   (scr        ((match-arg 'parent 'no-default args) 'screen))
   (curfont    (font 'make 'screen scr 'fontname "cursor"))
   (the-cursor (cursor 'make 'screen scr 'glyph 'source curfont 'mask curfont
		       'source-char 152 'mask-char 152)) ; The "xterm" cursor
   (gcache     (match-arg 'gcache (scr 'gcache) args))
   (gc-draw     (gcache 'gc-draw))
   (gc-clear   (gcache 'gc-clear))
   (input      '())
   (keybd      (scr 'keyboard))
   (lsize      (gc-draw 'querytextextents (s8->s16 "A")))
   (pixelpad   (lsize 'font-descent))
   (width      (match-arg 'width 50 args))
   (ysize      (match-arg 'height 20 args))
   (height     (max ysize (+ 4 (* 2 pixelpad) (lsize 'font-ascent))))
   (xpos       5)
   (wants-result '())
   (the-input "") (the-length 0)
   (ypos (if (= ysize height)		 
	     (+ (lsize 'font-ascent)
		(inexact->exact (round (/ (- ysize (lsize 'font-ascent)) 2))))
	     (+ 1 pixelpad (lsize 'font-ascent)) ))
   (the-objlist (list (imagetext8 'make 5 ypos "")))
   (semantics `(((Ready . KeyPress)
		 Ready
		 ,(lambda (e)
		    (let ((c (decrypt-char e keybd)))
		      (cond ((string? c)
			     (set! the-input (string-append the-input c))
			     (set! the-length (+ the-length 1))
			     ((car the-objlist) 'set-text! the-input)
			     (me 'dfsm-transit 'expose #f) )
			    ((eq? c 'return)
			     (for-each (lambda (o) (o the-input))
				       wants-result) )
			    ((eq? c 'delete)
			     (if (positive? the-length)
				 (begin
				   (set! the-length (- the-length 1))
				   (set! the-input
					 (substring the-input 0 the-length))
				   ((car the-objlist) 'set-text!
						      (string-append the-input
								     "   ") )
				   (me 'draw the-objlist gc-draw)
				   (scr 'flush!) )))
			    ))))
		((Ready . Expose) Ready
				  ,(lambda (e)
				     (me 'draw the-objlist gc-draw) )) ))
   )
  (inherit
   (widget (cons* 'width width 'height height 'value-mask
		  (make-window-value-mask
		   `(background-pixel ,(gcache 'background))
		   `(event-mask ,(make-bitmask 'event 'Exposure 'KeyPress))
		   `(cursor ,the-cursor))
		  'dfsm-data (list 'Ready #f semantics)
		  args)) )
  (methods
   (stox-notify-result (lambda (o)
			 (set! wants-result (cons o wants-result)) ))
   (current-input (lambda () (apply string-append (reverse input))))
   (stox-free (lambda ()
		(the-cursor 'freecursor)
		(me '(widget stox-free)) ))
   )
  (init
   (curfont 'closefont)
   ))

;;; decrypt-char -- try to transform the info in the KeyPress event into
;;;                 something more useful.
(define (decrypt-char event keybd)
  (let ((shifted ((event 'state) 'set? 'shift))
	(ksyms ((keybd 'getkeyboardmapping (event 'detail) 1) 'keysyms)))
    (cond ((< (caar ksyms) 256)		                 ; Latin-1 set.
	   (let ((aux (integer->char (caar ksyms))))
	     (list->string (list (cond ((char-alphabetic? aux)
					(if shifted
					    (if (char-upper-case? aux)
						aux
						(char-upcase aux) )
					    (if (char-lower-case? aux)
						aux
						(char-downcase aux) )))
				       (else
					(if shifted
					    (integer->char (cadar ksyms))
					    aux)))))))
	  ((= #xff00 (bit-and #xff00 (caar ksyms)))       ; Keyboard set.
	   (let ((c (bit-and (caar ksyms) #xff)))
	     ;;(display (format "decrypt-char: c: ~a~%" c))
	     (case c
	       ((9)  'tab)
	       ((10) 'linefeed)
	       ((13) 'return)
	       ((27) 'escape)
	       ((255) 'delete)
	       (else 'unknown))))
	  (else 'unknown))))	   

