;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.

;;;    Author: Magnus Persson <mpersson@stkhlm.dec.com>

;;; $Id: dfsm.sc,v 1.4 91/09/15 00:58:26 johani Exp $

(module stoxdfsm)

(include "../include/util.sch")
(include "../macros/extsyntax.sc")
(include "../macros/oos.sc")

(define-class (dfsm . args)
  (locals
   (dfsm-data (match-arg 'dfsm-data '(Undefined #f ()) args))
   (start-state (car dfsm-data))
   (to-be-logged (cadr dfsm-data))
   (transit-table (caddr dfsm-data))
   ;; Some set operations
   (set-insert (lambda (element set)             ; Inserts into a set
		 (if (memq element set)
		     set
		     (cons element set))))
   (create-set (lambda (l)                       ; Creates a set of an
		 (letrec ((create-set            ; arbitrary list
			   (lambda (l)
			     (if (null? l)
				 '()
				 (set-insert (car l)
					     (create-set (cdr l)))))))
		   (create-set l))))
   ;; State
   (state start-state)
   (input-alphabet (create-set (map cdar transit-table)))
   (state-set (create-set (cons 'Undefined (map caar transit-table))))
   (log '()) )

  (methods
   (dfsm-log (lambda () log))
   (dfsm-alphabet (lambda () input-alphabet))
   (dfsm-states (lambda () state-set))
   (dfsm-state (lambda () state))
   (dfsm-halt (lambda () (set! state 'Undefined)))
   ;; johani -- I'm not sure about the usefulness of the reset-thunk.
   (dfsm-reset (lambda (reset-thunk . reset-log)
		 (set! state start-state)
		 (reset-thunk)
		 (if (and (not (null? reset-log)) (eq? #t (car reset-log)))
		     (begin
		       (set! log '())
		       (set! to-be-logged #t)))
		 #t))
   (dfsm-transit (lambda (input ev-obj)
		   (let* ((input-pair (cons state input))
			  (next (assoc input-pair transit-table)))
		     (if to-be-logged (set! log (cons input-pair log)))
		     (if (not next)
			 (begin
			   (if (not (equal? input-pair
					    '(Undefined . LeaveNotify)) )
			       (begin
				 (scix-msg
               "DFSM: Warning: Not possible to resolve input ~a in state ~a~%"
                         	           input state)))
			   (set! state 'Undefined)
			   #f)
			 (let ((thunk (cddr next)))
			   (set! state (cadr next))
			   (if (not (null? thunk))
			       ((car thunk) ev-obj)
			       #t))))))
   (add-transit-description! (lambda (tr-description)
			       (set! transit-table
				     (append transit-table
					     tr-description))
			       #t))
   ;; Debug
   (dfsm-set-state! (lambda (x) (set! state x)))
   )
  (init
   ;; Some consistency checking could be done here like
   (if (not (memq start-state (me 'dfsm-states)))
       (error 'State "Start state ~a cannot be exited" start-state) )))

;;; Sometimes it is convenient to replay the action of a DFSM -
;;; Of course only the logged actions are replayed.

(define (replay	dfsm)
  (dfsm 'reset (lambda () '()))
  (for-each (lambda (event-name)
	      (dfsm 'transit event-name))
	    (map car (reverse (dfsm 'log)))))
