;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: requests2.sc,v 1.5 91/09/15 01:13:29 johani Exp $

;;; requests2.sc -- the code that constructs the individual requests, i e here
;;;                 we have the translation from the OO-level to the X protocol
;;;                 level.

(module scixreq2 (with scixatypes)); (top-level))

(include "../include/types.sch")
(include "../include/util.sch")
(include "../include/lowlevel.sch")
(include "../include/msgutil.sch")

;;; Request #52: GetFontPath (new format)
(define send-GetFontPath
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 52 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name . ,(d-identity 'GetFontPath))
		     (unused     . ,(d-unused 1))
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 4)) ; Reply length not used
		     (nr-of-strs . ,d-card16)
		     (unused     . ,(d-unused 22))
		     (path . ,(lambda (str dpy)
				(let loop ((nr (backtrack 'nr-of-strs))
					   (total-length 0)
					   (result '()) )
				  (if (zero? nr)
				      (begin
					((d-unused (pad total-length)) str dpy)
					result)
				      (let* ((this-str (list (d-str str dpy)))
					     (this-len (+ 1
							  (string-length
							   (car this-str)))))
					(loop (- nr 1)
					      (+ total-length this-len)
					      (append result this-str) )))))))
		   scr) )))

;;; Request #53: CreatePixmap (new format)
(define send-CreatePixmap
  (let ((fmt (vector a-request a-card8 a-card16
		     a-pixmap a-drawable a-card16 a-card16))
	(dta (vector 53 #f 4 #f #f #f #f)) )
    (lambda (pmap scr . rest)
      (vector-set! dta 1 (pmap 'depth))
      (vector-set! dta 3 pmap)
      (vector-set! dta 4 (pmap 'drawable))
      (vector-set! dta 5 (pmap 'width))
      (vector-set! dta 6 (pmap 'height))
      (scr 'scix-xas 6 fmt dta rest) )))

;;; Request #54: FreePixmap (new format)
(define send-FreePixmap
  (let ((fmt (vector a-request a-card8 a-card16 a-pixmap))
	(dta (vector 54 0 2 #f)) )
    (lambda (pmap scr . rest)
      (vector-set! dta 3 pmap)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #55: CreateGC (new format)
(define send-CreateGC
  (let ((fmt (vector a-request a-card8 a-card16 a-gcontext
		     a-drawable a-bitmask a-listofvalue))
	(dta (vector 55 0 #f #f #f #f #f)) )
   (lambda (gc data scr . rest)
      (vector-set! dta 2 (+ 4 (data 'length)))
      (vector-set! dta 3 gc)
      (vector-set! dta 4 (gc 'drawable))
      (vector-set! dta 5 data)
      (vector-set! dta 6 data)
      (scr 'scix-xas 6 fmt dta rest) )))

;;; Request #56: ChangeGC (new format)
(define send-ChangeGC
  (let ((fmt (vector a-request a-card8 a-card16
		     a-gcontext a-bitmask a-listofvalue))
	(dta (vector 56 0 #f #f #f #f)) )
    (lambda (gc data scr . rest)
      (vector-set! dta 2 (+ 3 (data 'length)))
      (vector-set! dta 3 gc)
      (vector-set! dta 4 data)
      (vector-set! dta 5 data)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #57: CopyGC (new format)
(define send-CopyGC
  (let ((fmt (vector a-request a-card8 a-card16
		     a-gcontext a-gcontext a-bitmask))
	(dta (vector 57 0 4 #f #f #f)) )
    (lambda (src-gc dst-gc data scr . rest)
      (vector-set! dta 3 src-gc)
      (vector-set! dta 4 dst-gc)
      (vector-set! dta 5 data)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #58: SetDashes (new format)
(define send-SetDashes
  (let ((fmt (vector a-request a-card8
		     a-card16 a-gcontext a-card16 a-card16 a-intlist))
	(dta (vector 58 0 #f #f #f #f #f)) )
    (lambda (gc dash-offset dash-list scr . rest)
      (let ((n (length dash-list)))
	(vector-set! dta 2 (+ 3 (/ (+ n (pad n)) 4))); Request length
	(vector-set! dta 3 gc)
	(vector-set! dta 4 dash-offset)
	(vector-set! dta 5 n)		    ; Length of dashes
	(vector-set! dta 6 (list-pad4 dash-list))
	(scr 'scix-xas 6 fmt dta rest) ))))

;;; Request #59: SetClipRectangles (new format)
(define send-SetClipRectangles
  (let ((fmt (vector a-request a-card8 a-card16 a-gcontext a-int16 a-int16
		     (a-list a-rectangle)))
	(dta (vector 59 #f #f #f #f #f #f)) )
    (lambda (ordering gc clip-x-orig clip-y-orig rectangle-list scr . rest) 
      (vector-set! dta 1 (lookup-constant ordering '((UnSorted . 0)
						     (YSorted  . 1)
						     (YXSorted . 2)
						     (YXBanded . 3) )))
      (vector-set! dta 2 (+ 3 (* 2 (length rectangle-list))))
      (vector-set! dta 3 gc)
      (vector-set! dta 4 clip-x-orig)
      (vector-set! dta 5 clip-x-orig)
      (vector-set! dta 6 rectangle-list)
      (scr 'scix-xas 6 fmt dta rest) )))

;;; Request #60: FreeGC (new format)
(define send-FreeGC
  (let ((fmt (vector a-request a-card8 a-card16 a-gcontext))
	(dta (vector 60 0 2 #f)) )
    (lambda (gc scr . rest)
      (vector-set! dta 3 gc)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #61: ClearArea (new format)
(define send-ClearArea
  (let ((fmt (vector a-request a-bool a-card16 a-window a-int16 a-int16
		     a-card16 a-card16))
	(dta (vector 61 #f 4 #f #f #f #f #f)) )
    (lambda (exposures w x y width height scr . rest)
      (vector-set! dta 1 exposures)
      (vector-set! dta 3 w)
      (vector-set! dta 4 x)
      (vector-set! dta 5 y)
      (vector-set! dta 6 width)
      (vector-set! dta 7 height)
      (scr 'scix-xas 7 fmt dta rest) )))

;;; Request #62: CopyArea (new format)
(define send-CopyArea
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-drawable
		     a-gcontext a-gcontext a-int16 a-int16 a-int16 a-int16 
		     a-card16 a-card16))
	(dta (vector 62 0 7 #f #f #f #f #f #f #f #f #f)) )
    (lambda (src-d dst-d gc src-x src-y dst-x dst-y width height scr . rest)
      (vector-set! dta 3 src-d)
      (vector-set! dta 4 dst-d)
      (vector-set! dta 5 gc)
      (vector-set! dta 6 src-x)
      (vector-set! dta 7 src-y)
      (vector-set! dta 8 dst-x)
      (vector-set! dta 9 dst-y)
      (vector-set! dta 10 width)
      (vector-set! dta 11 height)
      (scr 'scix-xas 11 fmt dta rest) )))

;;; Request #63: CopyPlane (new format)
(define send-CopyPlane
  (let ((fmt (vector a-request a-card8
		     a-card16 a-drawable a-drawable a-gcontext a-int16
		     a-int16 a-int16 a-int16 a-card16 a-card16 a-card32))
	(dta (vector 63 0 8 #f #f #f #f #f #f #f #f #f #f)) )
    (lambda (src-d dst-d gc src-x
		   src-y dst-x dst-y width height bit-plane scr . rest)
      (vector-set! dta 3 src-d)
      (vector-set! dta 4 dst-d)
      (vector-set! dta 5 gc)
      (vector-set! dta 6 src-x)
      (vector-set! dta 7 src-y)
      (vector-set! dta 8 dst-x)
      (vector-set! dta 9 dst-y)
      (vector-set! dta 10 width)
      (vector-set! dta 11 height)
      (vector-set! dta 12 bit-plane)
      (scr 'scix-xas 12 fmt dta rest) )))

;;; Note: None of the "graphics objects" requests deal with the actual graphics
;;; objects. Instead the objects supply all the parameters to the requests.
;;; This is inconsistent with the rest of the requests but is done to keep the
;;; graphics objects as simple as possible.

;;; A compound assembler type constructor.
(define (a-list type)
  (lambda (l buffer dpy)
    (for-each (lambda (ls)
		(type ls buffer dpy) )
	      l)))

(define (a-point l buffer dpy)
  (a-int16 (car l) buffer dpy)
  (a-int16 (cadr l) buffer dpy) )

(define (a-rectangle l buffer dpy)
  (a-int16 (car l) buffer dpy)
  (a-int16 (cadr l) buffer dpy)
  (a-card16 (caddr l) buffer dpy)
  (a-card16 (cadddr l) buffer dpy) )

(define (a-arc l buffer dpy)
  (a-int16 (car l) buffer dpy)
  (a-int16  (cadr l) buffer dpy)
  (a-card16 (caddr l) buffer dpy)
  (a-card16 (cadddr l) buffer dpy)
  (a-int16  (list-ref l 4) buffer dpy)
  (a-int16  (list-ref l 5) buffer dpy) )

;;; Request #64: PolyPoint (new format)
(define send-PolyPoint
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable
		     a-gcontext (a-list a-point)))
	 (dta (vector 64 #f #f #f #f #f)) )
    (lambda (drawable gc c-mode point-list scr . rest)
      (vector-set! dta 1 (lookup-constant c-mode '((Origin . 0)
						    (Previous . 1) )))
      (vector-set! dta 2 (+ 3 (length point-list)))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 point-list)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #65: PolyLine (new format)
(define send-PolyLine
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-gcontext
		     (a-list a-point)))
	(dta (vector 65 #f #f #f #f #f)) )
    (lambda (drawable gc c-mode point-list scr . rest)
      (vector-set! dta 1 (lookup-constant c-mode '((Origin . 0)
						   (Previous . 1) )))
      (vector-set! dta 2 (+ 3 (length point-list)))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 point-list)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #66: PolySegment (new format)
(define send-PolySegment
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-gcontext
		     (a-list (lambda (seg-l buffer dpy)
			       (a-int16 (car seg-l) buffer dpy)
			       (a-int16 (cadr seg-l) buffer dpy)
			       (a-int16 (caddr seg-l) buffer dpy)
			       (a-int16 (cadddr seg-l) buffer dpy)))))
	(dta (vector 66 0 #f #f #f #f)) )
    (lambda (drawable gc seg-list scr . rest)
      (vector-set! dta 2 (+ 3 (* 2 (length seg-list))))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 seg-list)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #67: PolyRectangle (new format)
(define send-PolyRectangle
  (let ((fmt (vector a-request a-card8 a-card16
		     a-drawable a-gcontext (a-list a-rectangle)))
	(dta (vector 67 0 #f #f #f #f)) )
    (lambda (drawable gc rect-list scr . rest)
      (vector-set! dta 2 (+ 3 (* 2 (length rect-list))))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 rect-list)
      (scr 'scix-xas 5 fmt dta rest))))

;;; Request #68: PolyArc (new format)
(define send-PolyArc
  (let ((fmt (vector a-request a-card8 a-card16
		     a-drawable a-gcontext (a-list a-arc)))
	(dta (vector 68 0 #f #f #f #f)) )
    (lambda (drawable gc arc-list scr . rest)
      (vector-set! dta 2 (+ 3 (* 3 (length arc-list))))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 arc-list)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #69: FillPoly (new format)
(define send-FillPoly
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable
		     a-gcontext a-card8 a-card8 a-card16 (a-list a-point)))
	(dta (vector 69 0 #f #f #f #f #f 0 #f)) )
    (lambda (drawable gc shape coord-mode point-list scr . rest)
      (vector-set! dta 2 (+ 4 (length point-list)))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 (lookup-constant shape '((Complex . 0)
						  (Nonconvex . 1)
						  (Convex . 2) )))
      (vector-set! dta 6 (lookup-constant coord-mode '((Origin . 0)
						       (Previous . 1) )))
      (vector-set! dta 8 point-list)
      (scr 'scix-xas 8 fmt dta rest) )))

;;; Request #70: PolyFillRectangle (new format)
(define send-PolyFillRectangle
  (let ((fmt (vector a-request a-card8 a-card16
		     a-drawable a-gcontext (a-list a-rectangle)))
	(dta (vector 70 0 #f #f #f #f)) )
    (lambda (drawable gc rect-list scr . rest)
      (vector-set! dta 2 (+ 3 (* 2 (length rect-list))))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 rect-list)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #71: PolyFillArc (new format)
(define send-PolyFillArc
  (let ((fmt (vector a-request a-card8 a-card16
		     a-drawable a-gcontext (a-list a-arc)))
	(dta (vector 71 0 #f #f #f #f)) )
    (lambda (drawable gc arc-list scr . rest)
      (vector-set! dta 2 (+ 3 (* 3 (length arc-list))))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 arc-list)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #72: PutImage (new format)
(define send-PutImage
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-gcontext
		     a-card16 a-card16 a-int16 a-int16 a-card8 a-card8
		     a-card16 a-intlist))
	(dta (vector 72 #f #f #f #f #f #f #f #f #f #f 0 #f))
	(n #f) )
    (lambda (image-format drawable gc width height dst-x dst-y
			  left-pad byte-list scr . rest)
      (set! n (length byte-list))
      (vector-set! dta 1 (lookup-constant image-format '((Bitmap . 0)
							 (XYPixmap . 1)
							 (ZPixmap . 2) )))
      (vector-set! dta 2 (+ 6 (/ (+ n (pad n)) 4)))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 width)
      (vector-set! dta 6 height)
      (vector-set! dta 7 dst-x)
      (vector-set! dta 8 dst-y)
      (vector-set! dta 9 left-pad)
      (vector-set! dta 10 (if (eq? image-format 'Bitmap) 
			      1        ; The depth can be calculated
			      (drawable 'depth)))
      (vector-set! dta 12 (list-pad4 byte-list))
      (scr 'scix-xas 12 fmt dta rest) )))

;;; Request #73: GetImage (new format)
;;; NOTE: Padding is not stripped from the data list returned, as we don't
;;; know how much is present. Does the caller know?
(define send-GetImage
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-int16 a-int16
		     a-card16 a-card16 a-card32))
	(dta (vector 73 #f 5 #f #f #f #f #f #f)) )
    (lambda (format drawable x y width height plane-mask scr . rest)
      (vector-set! dta 1 (lookup-constant format '((XYPixmap . 1)
						   (ZPixmap . 2))))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 x)
      (vector-set! dta 5 y)
      (vector-set! dta 6 width)
      (vector-set! dta 7 height)
      (vector-set! dta 8 plane-mask)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 8 fmt dta rest)
		   `((reply-name . ,(d-identity 'GetImage))
		     (depth      . ,d-card8)
		     (seq-nr     . ,d-card16)
		     (reply-len  . ,d-card32)
		     (visual     . ,(d-const #f d-visualid '((0 . None))))
		     (unused     . ,(d-unused 20))
		     (data       . ,(lambda (str dpy)
				      ((d-list
					d-card8 (* 4 (backtrack 'reply-len)) )
				       str dpy))))
		   scr) )))

;;; Request #74: PolyText8 (new format)
(define send-PolyText8
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-gcontext
		     a-int16 a-int16 a-intlist))
	(dta (vector 74 0 #f #f #f #f #f #f)) )
    (lambda (drawable gc x y textobj scr . rest)
      (vector-set! dta 2 (+ 4 (textobj 'length)))    ; Request length
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 x)
      (vector-set! dta 6 y)
      (vector-set! dta 7 (textobj 'items))
      (scr 'scix-xas 7 fmt dta rest) )))

;;; Request #75: PolyText16 (new format)
(define send-PolyText16
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-gcontext
		     a-int16 a-int16 a-intlist))
	(dta (vector 75 0 #f #f #f #f #f #f)) )
    (lambda (drawable gc x y textobj scr . rest)
      (vector-set! dta 2 (+ 4 (textobj 'length)))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 x)
      (vector-set! dta 6 y)
      (vector-set! dta 7 (textobj 'items)) ; Should use strings...
      (scr 'scix-xas 7 fmt dta rest) )))

;;; Request #76: ImageText8 (new format)
(define send-ImageText8
  (let ((fmt (vector a-request a-card8 a-card16
		     a-drawable a-gcontext a-int16 a-int16 a-string8))
	(dta (vector 76 #f #f #f #f #f #f #f))
	(n #f) )
    (lambda (drawable gc x y str scr . rest)
      (set! n (string-length str))
      (vector-set! dta 1 n)
      (vector-set! dta 2 (+ 4 (/ (+ n (pad n)) 4))) ; Request len
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 x)
      (vector-set! dta 6 y)
      (vector-set! dta 7 str)
      (scr 'scix-xas 7 fmt dta rest) )))

;;; Request #77: ImageText16 (new format)
;;;                          -- Note: STRING16 is represented as an ordinary
;;;                             STRING8 with two bytes per CHAR16. If some
;;;                             other representation (like a list of pairs for
;;;                             instance) should turn out to be better then
;;;                             change it.
;;;
(define send-ImageText16
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-gcontext
		     a-int16 a-int16 a-string8))
	(dta (vector 77 #f #f #f #f #f #f #f))
	(two-n #f) )
    (lambda (drawable gc x y str scr . rest)
      (set! two-n (string-length str))
      (vector-set! dta 1 (/ two-n 2))
      (vector-set! dta 2 (+ 4 (/ (+ two-n (pad two-n)) 4)))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 gc)
      (vector-set! dta 5 x)
      (vector-set! dta 6 y)
      (vector-set! dta 7 str)                 ; a-string8 pads.
      (scr 'scix-xas 7 fmt dta rest) )))

;;; Request #78: CreateColormap (new format)
(define send-CreateColormap
  (let ((fmt (vector a-request
		     a-card8 a-card16 a-colormap a-window a-visualid))
	(dta (vector 78 #f 4 #f #f #f)) )
    (lambda (alloc cmap visual scr . rest)
      (vector-set! dta 1 (lookup-constant alloc '((None . 0)
						  (All  . 1) )))
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 (scr 'root))
      (vector-set! dta 5 visual)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #79: FreeColormap (new format)
(define send-FreeColormap
  (let ((fmt (vector a-request a-card8 a-card16 a-colormap))
	(dta (vector 79 0 2 #f)) )
    (lambda (cmap scr . rest)
      (vector-set! dta 3 cmap)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #80: CopyColormapAndFree (new format)
(define send-CopyColormapAndFree
  (let ((fmt (vector a-request a-card8 a-card16 a-colormap a-colormap))
	(dta (vector 80 0 3 #f #f)) )
    (lambda (cmap src-cmap scr . rest)
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 src-cmap)
      (scr 'scix-xas 4 fmt dta rest) )))

;;; Request #81: InstallColormap (new format)
(define send-InstallColormap
  (let ((fmt (vector a-request a-card8 a-card16 a-colormap))
	(dta (vector 81 0 2 #f)) )
    (lambda (cmap scr . rest)
      (vector-set! dta 3 cmap)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #82: UninstallColormap (new format)
(define send-UninstallColormap
  (let ((fmt (vector a-request a-card8 a-card16 a-colormap))
	(dta (vector 82 0 2 #f)) )
    (lambda (cmap scr . rest)
      (vector-set! dta 3 cmap)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #83: ListInstalledColormaps (new format)
(define send-ListInstalledColormaps
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 83 0 2 #f)) )
    (lambda (scr . rest)
      (vector-set! dta 3 (scr 'root))
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name  . ,(d-identity 'ListInstalledColormaps))
		     (unused      . ,d-card8)
		     (seq-nr      . ,d-card16)
		     (unused      . ,(d-unused 4)) ; Reply length not used
		     (nr-of-cmaps . ,d-card16)
		     (unused      . ,(d-unused 22))
		     (cmaps       . ,(lambda (str dpy)
				       ((d-list d-colormap
						(backtrack 'nr-of-cmaps) )
					str dpy))))
		   scr) )))

;;; Request #84: AllocColor (new format)
(define send-AllocColor
  (let ((fmt (vector a-request a-card8 a-card16 a-colormap a-card16
		     a-card16 a-card16 a-card16))
	(dta (vector 84 0 4 #f #f #f #f 0)) )
    (lambda (cmap red green blue scr . rest)
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 red)
      (vector-set! dta 5 green)
      (vector-set! dta 6 blue)
      (msg-handler 'scix-wait-for-reply 
		   (scr 'scix-xas 7 fmt dta rest)
		   `((reply-name . ,(d-identity 'AllocColor))
		     (unused     . ,(d-unused 1))
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 4)) ; Reply length = 0
		     (red        . ,d-card16)
		     (green      . ,d-card16)
		     (blue       . ,d-card16)
		     (unused     . ,(d-unused 2))
		     (pixel      . ,d-card32)
		     (unused     . ,(d-unused 12)) )
		   scr) )))

;;; Request #85: AllocNamedColor (new format)
(define send-AllocNamedColor
  (let ((fmt (vector a-request a-card8 a-card16 a-colormap a-card16
		     a-card16 a-string8))
	(dta (vector 85 0 #f #f #f 0 #f))
	(n #f) )
    (lambda (cmap name scr . rest)
      (set! n (string-length name))
      (vector-set! dta 2 (+ 3 (/ (+ n (pad n)) 4)))
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 n)
      (vector-set! dta 6 name)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 6 fmt dta rest)
		   `((reply-name   . ,(d-identity 'AllocNamedColor))
		     (unused       . ,(d-unused 1))
		     (seq-nr       . ,d-card16)
		     (unused       . ,(d-unused 4)) ; Reply length = 0
		     (pixel        . ,d-card32)
		     (exact-red    . ,d-card16)
		     (exact-green  . ,d-card16)
		     (exact-blue   . ,d-card16)
		     (visual-red   . ,d-card16)
		     (visual-green . ,d-card16)
		     (visual-blue  . ,d-card16)
		     (unused       . ,(d-unused 8)) )
		   scr) )))

;;; Request #86: AllocColorCells (new format)
;;; Note: see comment at method alloccolorcells in the colormap object.
(define send-AllocColorCells
  (let ((fmt (vector a-request a-bool a-card16 a-colormap a-card16 a-card16))
	(dta (vector 86 #f 3 #f #f #f)) )
    (lambda (cont cmap colors planes scr . rest)
      (vector-set! dta 1 cont)
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 colors)
      (vector-set! dta 5 planes)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 5 fmt dta rest)
		   `((reply-name           . ,(d-identity 'AllocColorCells))
		     (unused               . ,(d-unused 1))
		     (seq-nr               . ,d-card16)
		     (unused               . ,(d-unused 4)) ; Replylen not used
		     (nr-of-cards-in-pix   . ,d-card16)
		     (nr-of-cards-in-masks . ,d-card16)
		     (unused               . ,(d-unused 20))
		     (pixels               . ,(lambda (str dpy)
						((d-list d-card32
							 (backtrack
							  'nr-of-cards-in-pix))
						 str dpy)))
		     (masks                . ,(lambda (str dpy)
						((d-list d-card32
							 (backtrack
							'nr-of-cards-in-masks))
						 str dpy))))
		   scr) )))

;;; Request #87: AllocColorPlanes (new format)
(define send-AllocColorPlanes
  (let ((fmt (vector a-request a-bool a-card16 a-colormap a-card16 a-card16
		     a-card16 a-card16))
	(dta (vector 87 #f 4 #f #f #f #f #f)) )
    (lambda (cont cmap colors reds greens blues scr . rest)
      (vector-set! dta 1 cont)
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 colors)
      (vector-set! dta 5 reds)
      (vector-set! dta 6 greens)
      (vector-set! dta 7 blues)
      (msg-handler 'scix-wait-for-reply 
		   (scr 'scix-xas 7 fmt dta rest)
		   `((reply-name    . ,(d-identity 'AllocColorPlanes))
		     (unused        . ,d-card8)
		     (seq-nr        . ,d-card16)
		     (unused        . ,d-card32) ; Reply length not used
		     (nr-of-card32s . ,d-card16)
		     (unused        . ,d-card16)
		     (red-mask      . ,d-card32)
		     (green-mask    . ,d-card32)
		     (blue-mask     . ,d-card32)
		     (unused        . ,(d-unused 8))
		     (pixels        . ,(lambda (str dpy)
					 ((d-list d-card32
						  (backtrack 'nr-of-card32s) )
					  str dpy))))
		   scr) )))

;;; Request #88: FreeColors (new format)
(define send-FreeColors
  (let ((fmt (vector a-request
		     a-card8 a-card16 a-colormap a-card32 (a-list a-card32)))
	(dta (vector 88 0 #f #f #f #f)) )
    (lambda (cmap plane-mask pixels scr . rest)
      (vector-set! dta 2 (+ 3 (length pixels)))
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 plane-mask)
      (vector-set! dta 5 pixels)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #89: StoreColors (new format)
(define send-StoreColors
  (let ((fmt (vector a-request a-card8 a-card16 a-colormap
		     (a-list (lambda (c buffer dpy)
			       (a-color c buffer dpy)
			       (a-card16 (c 'red) buffer dpy)
			       (a-card16 (c 'green) buffer dpy)
			       (a-card16 (c 'blue) buffer dpy)
			       (a-card8  (c 'do-mask) buffer dpy)
			       (a-card8 0 buffer dpy) ))))
	(dta (vector 89 0 #f #f #f)) )
    (lambda (cmap colors scr . rest)
      (vector-set! dta 2 (+ 2 (* 3 (length colors))))
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 colors)
      (scr 'scix-xas 4 fmt dta rest) )))

;;; Request #90: StoreNamedColor (new format)
(define send-StoreNamedColor
  (let ((fmt (vector a-request a-card8 a-card16
		     a-colormap a-color a-card16 a-card16 a-string8))
	(dta (vector 90 #f #f #f #f #f 0 #f))
	(name-len #f) )
    (lambda (cmap color name scr . rest)
      (set! name-len (string-length name))
      (vector-set! dta 1 (color 'do-mask))
      (vector-set! dta 2 (+ 4 (/ (+ name-len (pad name-len)) 4)))
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 color)
      (vector-set! dta 5 name-len)
      (vector-set! dta 7 name)
      (scr 'scix-xas 7 fmt dta rest) )))

;;; Request #91: QueryColors (new format)
(define send-QueryColors
  (let ((fmt (vector a-request a-card8 a-card16 a-colormap (a-list a-color)))
	(dta (vector 91 0 #f #f #f)) )
    (lambda (cmap colors scr . rest)
      (vector-set! dta 2 (+ 2 (length colors)))
      (vector-set! dta 3 cmap)
      (vector-set! dta 4 colors)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 4 fmt dta rest)
		   `((reply-name . ,(d-identity 'QueryColors))
		     (unused     . ,d-card8)
		     (seq-nr     . ,d-card16)
		     (unused     . ,d-card32) ; Reply length not used
		     (nr-of-RGBs . ,d-card16)
		     (unused     . ,(d-unused 22))
		     (colors     . ,(lambda (str dpy)
				      ((d-list (lambda (str dpy)
						 (let* ((red
							 (d-card16 str dpy))
							(green
							 (d-card16 str dpy))
							(blue
							 (d-card16 str dpy))
							(this-RGB (list red
									green
									blue)))
						   ((d-unused 2) str dpy)
						   this-RGB))
					       (backtrack 'nr-of-RGBs) )
				       str dpy))))
		 scr) )))

;;; Request #92: LookupColor (new format)
(define send-LookupColor
  (let ((fmt (vector a-request a-card8
		     a-card16 a-colormap a-card16 a-card16 a-string8))
	(dta (vector 92 0 #f #f #f 0 #f)) )
    (lambda (name scr . rest)
      (let ((n (string-length name)))
	(vector-set! dta 2 (+ 3 (/ (+ n (pad n)) 4)))
	(vector-set! dta 3 (scr 'default-colormap))
	(vector-set! dta 4 n)
	(vector-set! dta 6 name)
	(msg-handler 'scix-wait-for-reply 
		     (scr 'scix-xas 6 fmt dta rest)
		     `((reply-name   . ,(d-identity 'LookupColor))
		       (unused       . ,(d-unused 1))
		       (seq-nr       . ,d-card16)
		       (unused       . ,(d-unused 4)) ; Reply length = 0
		       (exact-red    . ,d-card16)
		       (exact-green  . ,d-card16)
		       (exact-blue   . ,d-card16)
		       (visual-red   . ,d-card16)
		       (visual-green . ,d-card16)
		       (visual-blue  . ,d-card16)
		       (unused       . ,(d-unused 12)) )
		     scr) ))))

;;; Request #93: CreateCursor (new format)
(define send-CreateCursor
  (let ((fmt (vector a-request a-card8 a-card16 a-cursor a-pixmap a-pixmap
		     a-card16 a-card16 a-card16 a-card16 a-card16 a-card16
		     a-card16 a-card16) )
	(dta (vector 93 0 8 #f #f #f #f #f #f #f #f #f #f #f)) )
    (lambda (cursor source mask fore back x y scr . rest)
      (vector-set! dta 3 cursor)
      (vector-set! dta 4 source)
      (vector-set! dta 5 (if (eq? mask 'None)
			     0
			     mask))
      (vector-set! dta 6 (fore 'red))
      (vector-set! dta 7 (fore 'green))
      (vector-set! dta 8 (fore 'blue))
      (vector-set! dta 9 (back 'red))
      (vector-set! dta 10 (back 'green))
      (vector-set! dta 11 (back 'blue))
      (vector-set! dta 12 x)
      (vector-set! dta 13 y)
      (scr 'scix-xas 13 fmt dta rest) )))

;;; Request #94: CreateGlyphCursor (new format)
(define send-CreateGlyphCursor
  (let ((fmt (vector a-request a-card8 a-card16 a-cursor a-font a-font
		     a-card16 a-card16 a-card16 a-card16
		     a-card16 a-card16 a-card16 a-card16))
	(dta (vector 94 0 8 #f #f #f #f #f #f #f #f #f #f #f)) )
    (lambda (cursor source-font mask-font
		    source-char mask-char fore back scr . rest)
      (vector-set! dta 3 cursor)
      (vector-set! dta 4 source-font)
      (vector-set! dta 5 (if (eq? mask-font 'None)
			     0
			     mask-font))
      (vector-set! dta 6 source-char)
      (vector-set! dta 7 mask-char)
      (vector-set! dta 8 (fore 'red))
      (vector-set! dta 9 (fore 'green))
      (vector-set! dta 10 (fore 'blue))
      (vector-set! dta 11 (back 'red))
      (vector-set! dta 12 (back 'green))
      (vector-set! dta 13 (back 'blue))
      (scr 'scix-xas 13 fmt dta rest) )))

;;; Request #95: FreeCursor (new format)
(define send-FreeCursor
  (let ((fmt (vector a-request a-card8 a-card16 a-cursor))
	(dta (vector 95 0 2 #f)) )
    (lambda (cursor scr . rest)
      (vector-set! dta 3 cursor)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #96: RecolorCursor (new format)
(define send-RecolorCursor
  (let ((fmt (vector a-request a-card8 a-card16 a-cursor
		     a-card16 a-card16 a-card16 a-card16 a-card16 a-card16))
	(dta (vector 96 0 5 #f #f #f #f #f #f #f)) )
    (lambda (cursor fore back scr . rest)
      (vector-set! dta 3 cursor)
      (vector-set! dta 4 (fore 'red))
      (vector-set! dta 5 (fore 'green))
      (vector-set! dta 6 (fore 'blue))
      (vector-set! dta 7 (back 'red))
      (vector-set! dta 8 (back 'green))
      (vector-set! dta 9 (back 'blue))
      (scr 'scix-xas 9 fmt dta rest) )))

;;; Request #97: QueryBestSize (new format)
;;; Note: What is the use of the width and height parameters?
(define send-QueryBestSize
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable a-card16 a-card16))
	(dta (vector 97 #f 3 #f #f #f)) )
    (lambda (cl drawable width height scr . rest)
      (vector-set! dta 1 (lookup-constant cl '((Cursor . 0)
					       (Tile . 1)
					       (Stipple . 2) )))
      (vector-set! dta 3 drawable)
      (vector-set! dta 4 width)
      (vector-set! dta 5 height)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 5 fmt dta rest)
		   `((reply-name . ,(d-identity 'QueryBestSize))
		     (unused     . ,(d-unused 1))
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 4)) ; Reply length = 0
		     (width      . ,d-card16)
		     (height     . ,d-card16)
		     (unused     . ,(d-unused 20)) )
		   scr) )))

;;; Request #98: QueryExtension (new format)
(define send-QueryExtension
  (let ((fmt (vector a-request a-card8 a-card16 a-card16 a-card16 a-string8))
	(dta (vector 98 0 #f #f 0 #f)) )
    (lambda (name scr . rest)
      (let ((len (string-length name)))
	(vector-set! dta 2 (+ 2 (/ (+ len (pad len)) 4)))
	(vector-set! dta 3 len)
	(vector-set! dta 5 name)
	(msg-handler 'scix-wait-for-reply
		     (scr 'scix-xas 5 fmt dta rest)
		     `((reply        . ,(d-identity 'QueryExtension))
		       (unused       . ,(d-unused 1))
		       (seq-nr       . ,d-card16)
		       (unused       . ,(d-unused 4)) ; Reply length = 0
		       (present      . ,d-bool)
		       (major-opcode . ,d-card8)
		       (first-event  . ,d-card8)
		       (first-error  . ,d-card8)
		       (unused       . ,(d-unused 20)) )
		     scr) ))))

;;; Request #99: ListExtensions (new format)
(define send-ListExtensions
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 99 0 1)) )
    (lambda (scr . rest)
      (msg-handler
       'scix-wait-for-reply
       (scr 'scix-xas 2 fmt dta rest)
       `((reply-name . ,(d-identity 'ListExtensions))
	 (nr-of-strs . ,d-card8)
	 (seq-nr     . ,d-card16)
	 (unused     . ,(d-unused 28)) ; Reply length not used
	 (names      . ,(lambda (str dpy)
			  (let loop ((nr (backtrack 'nr-of-strs))
				     (total-length 0)
				     (result '()))
			    (if (zero? nr)
				(begin
				  ((d-unused (pad total-length)) str dpy)
				  result)
				(let* ((this-str (list (d-str str dpy)))
				       (this-len (+ 1 (string-length
						       (car this-str)) )))
				  (loop (- nr 1)
					(+ total-length this-len)
					(append result this-str) )))))))
		   scr) )))

;;; Request #100: ChangeKeyboardMapping (new format)
;;;(send-ChangeKeyboardMapping 64 '((12312 12312 142414 124124)
;;;                                 (13451 435435 34534 435345)
;;;                                 (34234 324342 32432 234324)) scr . rest)
(define send-ChangeKeyboardMapping
  (let ((fmt (vector a-request a-card8 a-card16
		     a-keycode a-card8 a-card16 (a-list (a-list a-keysym))))
	(dta (vector 100 #f #f #f #f 0 #f)) )
    (lambda (first-keycode keysyms scr . rest)
      (let ((keycode-count (length keysyms))
	    (keysyms-per-keycode (length (car keysyms))) )
	(vector-set! dta 1 keycode-count)
	(vector-set! dta 2 (+ 2 (* keycode-count keysyms-per-keycode)))
	(vector-set! dta 3 first-keycode)
	(vector-set! dta 4 keysyms-per-keycode)
	(vector-set! dta 6 keysyms)
	(scr 'scix-xas 6 fmt dta rest) ))))

;;; Request #101: GetKeyboardMapping (new format)
(define send-GetKeyboardMapping
  (let ((fmt (vector a-request a-card8 a-card16 a-keycode a-card8 a-card16))
	(dta (vector 101 0 2 #f #f 0)) )
    (lambda (first-keycode count scr . rest)
      (vector-set! dta 3 first-keycode)
      (vector-set! dta 4 count)
      (msg-handler 'scix-wait-for-reply 
		   (scr 'scix-xas 5 fmt dta rest)
		   `((reply-name          . ,(d-identity 'GetKeyboardMapping))
		     (keysyms-per-keycode . ,d-card8)
		     (seq-nr              . ,d-card16)
		     (reply-len           . ,d-card32)
		     (unused              . ,(d-unused 24))
		     (keysyms             . ,(lambda (str dpy)
					       ((d-list
						 (d-list
						  d-keysym 
						  (backtrack
						   'keysyms-per-keycode))
						 (/ (backtrack 'reply-len)
						    (backtrack
						     'keysyms-per-keycode) ))
						str dpy))))
		   scr) )))

;;; Request #102: ChangeKeyboardControl (new format)
(define send-ChangeKeyboardControl
  (let ((fmt (vector a-request a-card8 a-card16 a-bitmask a-listofvalue))
	(dta (vector 102 0 #f #f #f)) )
    (lambda (data scr . rest)
      (vector-set! dta 2 (+ 2 (data 'length)))    ; Request length
      (vector-set! dta 3 data)
      (vector-set! dta 4 data)
      (scr 'scix-xas 4 fmt dta rest) )))

;;; Request #103: GetKeyboardControl (new format)
(define send-GetKeyboardControl
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 103 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name         . ,(d-identity 'GetKeyboardControl))
		     (global-auto-repeat . ,(d-const #t d-card8 '((0 . Off)
								  (1 . On) )))
		     (seq-nr             . ,d-card16)
		     (unused             . ,d-card32) ; Reply length = 5
		     (led-mask           . ,d-card32)
		     (key-click-percent  . ,d-card8)
		     (bell-percent       . ,d-card8)
		     (bell-pitch         . ,d-card16)
		     (bell-duration      . ,d-card16)
		     (unused             . ,d-card16)
		     (auto-repeats       . ,(d-list d-card8 32)) )
		   scr) )))

;;; Request #104: Bell (new format)
(define send-Bell
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 104 #f 1)) )
    (lambda (percent scr . rest)
      (vector-set! dta 1 percent)
      (scr 'scix-xas 2 fmt dta rest) )))

;;; Request #105: ChangePointerControl (new format)
(define send-ChangePointerControl
  (let ((fmt (vector a-request
		     a-card8 a-card16 a-int16 a-int16 a-int16 a-bool a-bool))
	(dta (vector 105 0 3 #f #f #f #f #f)) )
    (lambda (acc-num acc-den thr do-acc do-thr scr . rest)
      (vector-set! dta 3 acc-num)
      (vector-set! dta 4 acc-den)
      (vector-set! dta 5 thr)
      (vector-set! dta 6 do-acc)
      (vector-set! dta 7 do-thr)
      (scr 'scix-xas 7 fmt dta rest) )))

;;; Request #106: GetPointerControl (new format)
(define send-GetPointerControl
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 106 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name            . ,(d-identity 'GetPointerControl))
		     (unused                   . ,(d-unused 1))
		     (seq-nr                   . ,d-card16)
		     (unused                   . ,(d-unused 4)) ; Reply len = 0
		     (acceleration-numerator   . ,d-card16)
		     (acceleration-denominator . ,d-card16)
		     (threshold                . ,d-card16)
		     (unused                   . ,(d-unused 18)) )
		   scr) )))

;;; Request #107: SetScreenSaver (new format)
(define send-SetScreenSaver
  (let ((fmt (vector a-request a-card8
		     a-card16 a-int16 a-int16 a-card8 a-card8 a-card16))
	(dta (vector 107 0 3 #f #f #f #f 0))
	(alist '((No . 0) (Yes . 1) (Default . 2))) )
    (lambda (timeout interval prefer-blanking allow-exposures scr . rest)
      (vector-set! dta 3 timeout)
      (vector-set! dta 4 interval)
      (vector-set! dta 5 (lookup-constant prefer-blanking alist))
      (vector-set! dta 6 (lookup-constant allow-exposures alist))
      (scr 'scix-xas 7 fmt dta rest) )))

;;; Request #108: GetScreenSaver (new format)
(define send-GetScreenSaver
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 108 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply 
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name      . ,(d-identity 'GetScreenSaver))
		     (unused          . ,d-card8)
		     (seq-nr          . ,d-card16)
		     (unused          . ,d-card32) ; Reply length = 0
		     (timeout         . ,d-card16)
		     (interval        . ,d-card16)
		     (prefer-blanking . ,(d-const #t d-card8 '((0 . No)
							       (1 . Yes) )))
		     (allow-exposures . ,(d-const #t d-card8 '((0 . No)
							       (1 . Yes) )))
		     (unused          . ,(d-unused 18)) )
		   scr) )))

;;; Request #109: ChangeHosts (new format)
;;; Note: This is ugly, but there is no better place for it. Maybe bugs left
(define send-ChangeHosts
  (let ((fmt (vector a-request a-card8
		     a-card16 a-card8 a-card8 a-card16 (a-list a-card8)))
	(dta (vector 109 #f #f #f 0 #f #f)) )
    (lambda (mode family addr scr . rest)
      (let* ((addr (if (eq? family 'DECnet)
		       (list (bit-and (cadr addr)
				      255) ; Mask off LSByte of node-nr
			     (bit-or (bit-rsh (car addr)
					      2) ; Area part of byte2
				     (bit-rsh (cadr addr) 6) ))
		       addr))
	     (addr-len (length addr)) )
	(vector-set! dta 1 (lookup-constant mode '((Insert . 0)
						   (Delete . 1) )))
	(vector-set! dta 2 (+ 2 (/ (+ addr-len (pad addr-len)) 4)))
	(vector-set! dta 3 (lookup-constant family '((Internet . 0)
						     (DECnet   . 1)
						     (Chaos    . 2))))
	(vector-set! dta 5 addr-len)
	(vector-set! dta 6 addr)
	(scr 'scix-xas 6 fmt dta rest) ))))

;;; Request #110: ListHosts (new format)
;;; Note: some kind of trouble with DECnet addresses
(define send-ListHosts
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 110 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name  . ,(d-identity 'ListHosts))
		     (mode        . ,(d-const #t d-card8 '((0 . Disabled)
							   (1 . Enabled) )))
		     (seq-nr      . ,d-card16)
		     (unused      . ,d-card32)
		     (nr-of-hosts . ,d-card16)
		     (unused      . ,(d-unused 22))
		     (hosts       . ,(lambda (str dpy)
				       ((d-list d-host
						(backtrack 'nr-of-hosts))
					str dpy) )))
		   scr) )))

;;; Request #111: SetAccessControl (new format)
(define send-SetAccessControl
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 111 #f 1)) )
    (lambda (mode scr . rest)
      (vector-set! dta 1 (lookup-constant mode '((Disable . 0)
						 (Enable  . 1))))
      (scr 'scix-xas 2 fmt dta rest) )))

;;; Request #112: SetCloseDownMode (new format)
(define send-SetCloseDownMode
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 112 #f 1)) )
    (lambda (mode scr . rest)
      (vector-set! dta 1 (lookup-constant mode '((Destroy         . 0)
						 (RetainPermanent . 1)
						 (RetainTemporary . 2) )))
      (scr 'scix-xas 2 fmt dta rest) )))

;;; Request #113: KillClient (new format)
(define send-KillClient
  (let ((fmt (vector a-request a-card8 a-card16 a-resource))
	(dta (vector 113 0 2 #f)) )
    (lambda (resource scr . rest)
      (vector-set! dta 3 (lookup-constant resource '((AllTemporary . 0))))
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #114: RotateProperties (new format)
(define send-RotateProperties
  (let ((fmt (vector a-request a-card8
		     a-card16 a-window a-card16 a-int16 (a-list a-atom)))
	(dta (vector 114 0 #f #f #f #f #f)) )
    (lambda (w delta properties scr . rest)
      (let ((properties (map (lambda (p)
			       ((scr 'atombox) 'lookup-id p) )
			     properties) ))
	(vector-set! dta 2 (+ 3 (length properties)))
	(vector-set! dta 3 w)
	(vector-set! dta 4 (length properties))
	(vector-set! dta 5 delta)
	(vector-set! dta 6 properties)
	(scr 'scix-xas 6 fmt dta rest) ))))

;;; Request #115: ForceScreenSaver (new format)
(define send-ForceScreenSaver
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 115 #f 1)) )
    (lambda (mode scr . rest)
      (vector-set! dta 1 (lookup-constant mode '((Reset    . 0)
						 (Activate . 1))))
      (scr 'scix-xas 2 fmt dta rest) )))

;;; Request #116: SetPointerMapping (new format)
(define send-SetPointerMapping
  (let ((fmt (vector a-request a-card8 a-card16 (a-list a-card8)))
	(dta (vector 116 #f #f #f)) )
    (lambda (p-map scr . rest)
      (let ((n (length p-map)))
	(vector-set! dta 1 n)
	(vector-set! dta 2 (+ 1 (/ (+ n (pad n)) 4)))
	(vector-set! dta 3 (list-pad4 p-map))
	(msg-handler 'scix-wait-for-reply
		     (scr 'scix-xas 3 fmt dta rest)
		     `((reply-name . ,(d-identity 'SetPointerMapping))
		       (status     . ,(d-const #t d-card8 '((0 . Success)
							    (1 . Busy) )))
		       (seq-nr     . ,d-card16)
		       (unused     . ,(d-unused 28)) ) ; Reply length = 0
		     scr)))))

;;; Request #117: GetPointerMapping (new format)
(define send-GetPointerMapping
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 117 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply 
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name    . ,(d-identity 'GetPointerMapping))
		     (length-of-map . ,d-card8)
		     (seq-nr        . ,d-card16)
		     (reply-len     . ,d-card32)
		     (unused        . ,(d-unused 24))
		     (pmap          . ,(lambda (str dpy)
					 ((d-list d-card8 
						  (backtrack 'length-of-map))
					  str dpy)))
		     (unused        . ,(lambda (str dpy)
					 ((d-unused (pad (backtrack
							  'length-of-map)))
					  str dpy))))
		   scr) )))

;;; Request #118: SetModifierMapping (new format)
;;; Note: keycodes is a list of lists, one for each modifier.
(define send-SetModifierMapping
  (let ((fmt (vector a-request a-card8 a-card16
		     (a-list (lambda (keycode-list buffer dpy)
			       (for-each (lambda (keycode)
					   (a-keycode keycode buffer dpy))
					 keycode-list)))))
	(dta (vector 118 #f #f #f)) )
    (lambda (keycodes scr . rest)
      (vector-set! dta 1 (length (car keycodes)))
      (vector-set! dta 2 (+ 1 (* 2 (length (car keycodes)))))
      (vector-set! dta 3 keycodes)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name . ,(d-identity 'SetModifierMapping))
		     (status     . ,(d-const #t d-card8 '((0 . Success)
							  (1 . Busy)
							  (2 . Failed) )))
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 28)) ) ; Reply length = 0
		   scr) )))

;;; Request #119: GetModifierMapping (new format)
(define send-GetModifierMapping
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 119 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name          . ,(d-identity 'GetModifierMapping))
		     (keycodes-per-modifier . ,d-card8)
		     (seq-nr                . ,d-card16)
		     (unused              . ,(d-unused 28)) ; Repl len not used
		     (keycodes . ,(lambda (str dpy)
				    ((d-list d-keycode
					     (* 8 
						(backtrack
						 'keycodes-per-modifier)))
						str dpy))))
		 scr) )))

;;; Request #127: NoOperation (new format)
(define send-NoOperation
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 127 0 1)) )
    (lambda (scr . rest)
      (scr 'scix-xas 2 fmt dta rest) )))
