;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: msg-hndl.sc,v 1.3 91/09/15 01:13:24 johani Exp $

(module scixmh (top-level msg-handler))

(include "../include/util.sch")
(include "../include/lowlevel.sch")
(include "../include/types.sch")

(define-external identity          scixmu)
(define-external backtrack         scixmu)
(define-external make-error        scixmu)
(define-external make-event        scixmu)
(define-external make-reply        scixmu)
(define-external send-GetInputFocus scixreq1)

;;; Utilities
(define-in-line (x-error? message)
  (eq? 'error (message 'object-class)) )

(define-in-line (x-event? message)
  (eq? 'event (message 'object-class)) )

(define-in-line (x-reply? message)
  (eq? 'reply (message 'object-class)) )

;;; The SCIX message handler.

;;; Note: This is written in a highly non-functional way in an attempt
;;;       to minimize garbage generation.

(define msg-handler
  (let ((msg-list '())
	(handling-event #f)
	(strange-stuff '()))		; Debug.
    ;; xdas -- the X protocol disassembler. xdas will take a string containing
    ;;	       output from the server and process it according to the following
    ;;         rules:
    ;;         If the first byte is 1 then this is a x-reply? to a request.
    ;;         If the first byte is 0 then this is an error. The error type is
    ;;         in the second byte.
    ;;         If the first byte is 2-34 then this is an event.
    ;;         
    ;;         xdas returns an appropriate object, or 'not-enough if the
    ;;         message doesn't contain enough information for the specified
    ;;         type, If the message claims to be longer than 64kb, an error is
    ;;         signaled.
    ;;
    (define xdas
      (let ((type 0) (masked-type 0) (bytes-left 0) (errno 0) (tot-len 0))
	(lambda (str reply-type dpy)
	  (set! type (get-next-byte! str))
	  (set! masked-type (bit-and type #x7f)) ; Mask off SendEvent-flag
	  (cond ((zero? type)			 ; This is an error
		 (set! bytes-left (- (c-input-string-length str)
				     (c-input-string-pos str) ))
		 (set! errno (peek-next-byte str))
		 (if (< bytes-left 31)	; Partial message read?
		     (begin
		       (set-c-input-string-pos! str
						(- (c-input-string-pos str)
						   1) )
		       'not-enough)
		     (make-error str dpy) ))
		((= type 1)			 ; This is a reply
		 (set! bytes-left (- (c-input-string-length str)
				     (c-input-string-pos str) ))
		 (set! tot-len
		       (+ 32
			  (* 4
			     (c-unsigned-ref (c-input-string-string str)
					     (+ (c-input-string-pos str)
						3)))))
		 (if (> tot-len 65536)
		     (error 'xdas
	       "Got a message longer than 64kb from server, can't handle it."))
		 (if (< bytes-left (- tot-len 1))  ; Partial message read?
		     (begin 
		       (set-c-input-string-pos! str
						(- (c-input-string-pos str)
						   1) )
		       'not-enough)
		     (make-reply reply-type str dpy) ))
		(else			            ; This is an event
		 (set! bytes-left (- (c-input-string-length str)
				     (c-input-string-pos str) ))
		 (if (< bytes-left 31)
		     (begin
		       (set-c-input-string-pos! str
						(- (c-input-string-pos str)
						   1) )
		       'not-enough)
		     (make-event type str dpy)) )))))

    ;; wait-for-reply -- reads the messages from the server in quest for a
    ;;                   reply with sequence number seq-nr, which is returned.
    ;;                   Possible events and errors are dispatched to the
    ;;                   corresponding handler. Other replys are treated as
    ;;                   errors (this might be wrong...).
    ;;
    ;; Note: wait-for-reply starts by telling unprocessed-msgs? that the next
    ;;       reply found will be of reply-type. Then it walks through msg-list
    ;;       in search of this reply. It feels a bit insecure to leave the
    ;;       reply on its own like this, but as message arrival is guarenteed
    ;;       to be in order it should not give any problems.
    ;;
    (define wait-for-reply
      (let ((reply #f) (r #f))
	(lambda (seq-nr reply-type dpy)
	  (set! reply
		(if handling-event
		    (mutex-wait-for-reply seq-nr reply-type dpy)
		    (begin
		      (set! r (find-msg-with-seq-nr seq-nr reply-type))
		      (if r
			  r
			  (let loop ((found-msgs (unprocessed-msgs? dpy
							       reply-type)))
			    (if (not found-msgs)
				;; Did not find a message
				(loop (unprocessed-msgs? dpy reply-type))
				;; Found a message
				(begin
				  (set! r (find-msg-with-seq-nr seq-nr
								reply-type))
				  (if (not r)		; Didn't find it, sorry
				      (loop (unprocessed-msgs? dpy reply-type))
				      r))))))))
	  ((dpy 'scix-id-vector) 'flush-deferred!)
	  reply)))

    ;; mutex-wait-for-reply -- returns the reply with the wanted sequence
    ;;                         number without dispatching of events that are
    ;;                         queued
    (define mutex-wait-for-reply
      (let ((msg #f))
	(lambda (seq-nr reply-type dpy)
	  (let loop ((found-msgs #t))
	    (if (not found-msgs)
		;; Did not find a message
		(loop (unprocessed-msgs? dpy reply-type))
		;; Found a message
		(let iloop ((msg-l msg-list))
		  (if (null? msg-l)
		      (loop (unprocessed-msgs? dpy reply-type))
		      (begin
			(set! msg (car msg-l))
			(cond ((x-reply? msg)
			       (if (= (msg 'seq-nr) seq-nr)
				   (begin
				     ;; set! is needed if we try to delete
				     ;; the first element in the list, as
				     ;; remq! can't handle that.
				     (set! msg-list (remq! msg msg-list))
				     msg)
				   (begin
				     (error 'msg-handler
       "mutex-wait-for-reply: Warning: Saw a lost reply: ~a when fetching ~a~%"
                                             (cons (msg 'reply-name)
						   (msg 'seq-nr) )
					     (cons ((cadar reply-type) 'foo
								       'foo)
						   seq-nr))
				     (flush-buffer)
				     (set! msg-list (remq! msg msg-list))
				     (iloop (cdr msg-l)) )))
			      ((x-error? msg)
			       (if (= seq-nr (msg 'seq-nr)) ; Did we get an
				   (begin	    ; error instead of a reply?
				     (set! msg-list (remq! msg msg-list))
				     (scix-msg
				    "MSG-HANDLER: Warning. Found a ~a error~%"
				      (msg 'error-name) )
				     msg)
				   (if (< seq-nr (msg 'seq-nr)); later error?
				       (error 'mutex-wait-for-reply
			"Did not find a reply or error to round-trip request")
				       (iloop (cdr msg-l)) )))
			      (else (iloop (cdr msg-l))) )))))))))

    (define find-msg-with-seq-nr
      (let ((msg #f))
	(lambda (seq-nr reply-type)
	  (let loop ()
	    (if (null? msg-list)
		#f				; Did not find it, sorry
		(begin
		  (set! msg (car msg-list))
		  (cond ((x-event? msg)
			 (event-dispatch)
			 (loop) )
			((x-reply? msg)
			 (if (= seq-nr (msg 'seq-nr))
			     (begin
			       (set! msg-list (remq! msg msg-list))
			       msg)
			     (begin
			       (error 'msg-handler
      "find-msg-with-seq-nr: Warning: Saw a lost reply: ~a when fetching ~a~%"
                                       (cons (msg 'reply-name)
					     (msg 'seq-nr) )
				       (cons reply-type seq-nr) )
			       (flush-buffer)
			       (set! strange-stuff (cons msg strange-stuff))
			       (set! msg-list (remq! msg msg-list))
			       (loop) )))
			((x-error? msg)
			 (if (= seq-nr (msg 'seq-nr)) ; Did we get an error 
			     (begin		      ; instead of a reply?
			       (set! msg-list (cdr msg-list))
			       (scix-msg
				"MSG-HANDLER: Warning. Found a ~a error~%"
				(msg 'error-name) )
			       msg)
			     (if (< seq-nr (msg 'seq-nr)) ; A later error?
				 (error 'find-msg-with-seq-nr
		    "Did not find a reply or error to round-trip request")
				 (begin
				   (error-dispatch)
				   (loop) ))))
			(else 
			 (set! strange-stuff (cons msg strange-stuff))
			 (error 'find-msg-with-seq-nr
				"Unknown kind of server message")))))))))

    (define event-dispatch
      (let ((recipient #f))
	(lambda ()
	  (set! recipient ((car msg-list) 'recipient))
	  (set! handling-event #t)	; Flag that we are handling an event
	  (recipient 'scix-event-dispatch msg-handler recipient)
	  (set! handling-event #f) )))
	
    (define error-dispatch
      (let ((error #f))
	(lambda ()
	  (set! error (car msg-list))
	  ((error 'display) 'scix-error-dispatch error)
	  (set! msg-list (cdr msg-list)) )))

    ;; next-event! -- used by the event-handling routines (callbacks and the
    ;;               event-dispatch method in windows).
    (define next-event!
      (let ((e #f))
	(lambda ()
	  (set! e (car msg-list))
	  (set! msg-list (cdr msg-list))
	  e)))

    ;; my-events! -- returns a list with the consecutive events belonging to o.
    ;;               msg-list is stripped of these events.
    (define my-events!
      (let ((msg #f))
	(lambda (o)
	  (let loop ((its-events '()) (msg-l msg-list))
	    (if (null? msg-l)
		(begin
		  (set! msg-list msg-l)
		  its-events)
		(begin
		  (set! msg (car msg-l))
		  (if (and (x-event? msg) (eq? o (msg 'recipient)))
		      (loop (append its-events (list msg)) (cdr msg-l))
		      (begin
			(set! msg-list msg-l)
			its-events))))))))

    ;; my-events -- returns a list with the consecutive events belonging to o.
    ;;              msg-list remains intact.
    (define my-events 
      (let ((msg #f))
	(lambda (o)
	  (let loop ((its-events '()) (msg-l msg-list))
	    (if (null? msg-l)
		its-events
		(begin
		  (set! msg (car msg-l))
		  (if (and (x-event? msg) (eq? o (msg 'recipient)))
		      (loop (append its-events (list msg)) (cdr msg-l))
		      its-events)))))))

    ;; put-event! -- used by the event-handling routines to push an event back
    ;;              onto the unprocessed list of messages.
    (define (put-event! event)
      (if (x-event? event)
	  (set! msg-list (cons event msg-list))
	  (error 'put-event "Argument was not an event") ))

    ;; unprocessed-msgs? -- read data from the server and queue any new
    ;;                      messages in msg-list. Return #t if any new
    ;;                      messages were found.
    ;;
    ;; Note1: unprocessed-msgs? flushes the dpy with the philosophy that it is
    ;;        stupid to ask for new input before you have shown the result of
    ;;        the last.
    ;; Note2: reply-type is used for the additional purpose of indicating that
    ;;        the unread data in the inputbuffer should be discarded.
    ;;
    (define (unprocessed-msgs? dpy reply-type)
      (dpy 'flush!)
      (non-blocking-io (dpy 'scix-socket))
      (let loop ((str (readfromserver dpy)))
	(if (null? str)
	    (blocking-io (dpy 'scix-socket)) ; No, didn't find anything - block
	    (begin
	      (let ((r (xdas str reply-type dpy)))
		(if (eq? r 'not-enough)     ; Partial message, update state
		    (begin
		      (blocking-io (dpy 'scix-socket))
		      (let iloop ()
			(set! str (readmorefromserver dpy str))
			(set! r (xdas str reply-type dpy))
			(if (eq? r 'not-enough)
			  (iloop) ))
		      (non-blocking-io (dpy 'scix-socket)) ))
		(set! msg-list (append msg-list (list r))) ; Queue parsed msg
		(if (not (end-of-input-string? str))
		    (loop str)                         ; Parse rest of string
		    (begin
		      (blocking-io (dpy 'scix-socket))
		      #t)))))))		             ; Yes, did find something
    
    ;; mainloop -- Terminate? is a function of no arguments
    ;;             that returns #t if the loop should terminate. Calc (if
    ;;             present) should be a procedure of one argument that does
    ;;             whatever the application wants done when not handling
    ;;             events. The argument is a procedure of no arguments that
    ;;             should be called from within the calc procedure now and then
    ;;             to handled events. Note that if a calc-procedure is supplied
    ;;             it is up to the calc-procedure itself to terminate, the
    ;;             terminate? predicate is only used when no "idle"-calculation
    ;;             is supplied.
    ;;             Example of use, first without a calc-procedure:
    ;;             (mainloop (lambda ()
    ;;                         (button 'button-state) ) ; a toggle-button
    ;;                       (list scr) )               ; with states #t and #f
    ;;
    ;;             With a simple calc-procedure:
    ;;             (mainloop #f                         ; terminate? unused
    ;;                       (list scr)
    ;;                       (lambda (handle-events)    ; calc-procedure
    ;;                         (let loop ()
    ;;                           (do-some-calculating)
    ;;                           (handle-events)
    ;;                           (loop) )))
    ;;
    ;;             Note: mainloop starts and ends with flushing all buffered
    ;;             requests in the displays provided.
    (define (mainloop terminate? dpy-list . calc)
      (let* ((sd-l (map (lambda (d) (d 'scix-socket)) dpy-list))
	     (nfds (+ (apply max sd-l) 1))
	     (select-mask 0) )
	(define (act-on-event)
	  (let loop ((dpy-list dpy-list))
	    (if (not (null? dpy-list))
		(if (unprocessed-msgs? (car dpy-list) #f) ; Any msgs on dpy?
		    (let iloop ()                         ; Yes
		      (if (not (null? msg-list))
			  (begin
			    (if (x-event? (car msg-list))
				(event-dispatch)
				(error-dispatch) )
			    (iloop) )
			  (loop (cdr dpy-list)) ))))))    ; No
      
	(for-each (lambda (sd)		; Set up argument for select
		    (set! select-mask (bit-or select-mask
					      (bit-lsh 1 sd) )))
		  sd-l)
	(act-on-event)			; Handle any pending events.
	(if (null? calc)
	    ;; Have nothing to do while waiting for events.
	    (let loop ()
	      (if (not (terminate?))
		  (begin
		    (select nfds select-mask); NOTE: select used only for
		    (act-on-event)	; blocking, returned info is not used.
		    (loop) )
		  'done))
	    ;; Have something to do while waiting for events.
	    ((car calc) act-on-event) )))

    ;; process-events -- dispatch any queued up events on these screens.
    (define (process-events . screens)
      (let loop ((screens screens))
	(if (not (null? screens))
	    (if (unprocessed-msgs? (car screens) #f); Any messages on this dpy?
		(let iloop ()                       ; Yes
		  (if (not (null? msg-list))
		      (begin
			(if (x-event? (car msg-list))
			    (event-dispatch)
			    (error-dispatch) )
			(iloop) )
		      (loop (cdr screens)) ))))))    ; No

    (define (delete-lower! seq-nr)
      (let loop ((msg-l msg-list))
	(cond ((null? msg-l) (set! msg-list '()))
	      ((>= ((car msg-l) 'seq-nr) seq-nr) (set! msg-list msg-l))
	      (else (loop (cdr msg-l))) )))
    
    ;; The msg-handler's interface to the world:
    (define (me msg . args)
      (cond ((eq? msg 'next-event!) (if (null? msg-list)
					#f
					(next-event!)))
	    ;; Non-destructive version of next-event!.
	    ((eq? msg 'next-event) (if (null? msg-list)
				       #f
				       (car msg-list)))
	    ((eq? msg 'my-events!) (my-events! (car args)))
	    ;; Non-destructive version of my-events!.
	    ((eq? msg 'my-events) (my-events (car args)))
	    ((eq? msg 'put-event!) (put-event! (car args)))
	    ((eq? msg 'scix-unprocessed-msgs?) (apply unprocessed-msgs? args))
	    ((eq? msg 'scix-msg-list) msg-list)
	    ((eq? msg 'sync!) (apply
			       (lambda (dpy discard)
				 (let ((old-flag handling-event))
				   (me 'flush!)
				   (if discard
				       (begin
					(set! handling-event #t) 
					(delete-lower!
					 ((send-GetInputFocus dpy) 'seq-nr) ))
				       (send-GetInputFocus dpy))
				   (set! handling-event old-flag)
				   #t))
			       args))
	    ((eq? msg 'flush!) (set! msg-list '()))
	    ((eq? msg 'scix-wait-for-reply) (apply wait-for-reply args))
	    ((eq? msg 'mainloop) (apply mainloop args))
	    ((eq? msg 'scix-process-events) (apply process-events args))
	    ((eq? msg 'scix-strange-stuff) strange-stuff)
	    ((eq? msg 'object-class) 'msg-handler)
	    ((eq? msg 'object-system) 'custom)
	    ((eq? msg 'supported-messages)
	     '(next-event! next-event my-events! my-events put-event!
			   scix-unprocessed-msgs? scix-msg-list flush!
			   scix-wait-for-reply mainloop scix-strange-stuff
			   sync!))
	    ((eq? msg 'me) (lambda () me))
	    (else (error 'msg-handler "Unknown message: ~a"
			 (cons msg args))) ))
    me))

