;;;
;;;              Copyright 1990 Joacim Halen and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;; $Id: oos.sc,v 1.1 91/07/09 16:13:37 johani Exp $
;;;

;;; Two utilities:
(extend-syntax (match-arg)
  ((match-arg tag default arg-list)
   (match-arg-aux tag (lambda () default) arg-list) ))

(extend-syntax (match-arg!)
  ((match-arg! tag default arg-list)
   (match-arg-aux! tag (lambda () default) arg-list) ))

;;;
;;; Protocol: when inheriting instances of other object systems the
;;;           the foreign instance must honour the messages
;;;           (x 'supported-messages) => list of all supported messages
;;;           (x 'attach-object)      => a pair (<class-name> . <fetcher>)
;;;                                      where <class-name> is a symbol and
;;;                                      <fetcher> is a procedure that given
;;;                                      a message returns the corresponding
;;;                                      method object. 
(extend-syntax (define-class-aux
		 class-vars
		 class-methods
		 locals
		 inherit
		 methods
		 init)
  ;; With single inheritance
  ((define-class-aux class-head
     (class-vars (class-var class-var-val) ...)
     (class-methods (class-method-name class-method-proc) ...)
     (locals (local-var local-var-val) ...)
     (inherit super)
     (methods (method-no method-name method-proc) ...)
     (init init-forms ...))
   
   (with ((super-obj (if (pair? 'super)
			 (cons 'apply
			       (cons (car 'super)
				     (list (cons 'cons
						 (cons ''make-super ; Yep: ''
						       (cdr 'super))))))
			 'super))
	  (class-name (car 'class-head))
	  (constructor (string->symbol (format "MAKE-~a" (car 'class-head))))
	  (params (cdr 'class-head))
	  (object-class-proc (gensym))
	  ((class-method ...) (map (lambda (x) (gensym))
				   '(class-method-name ...)))
	  (local-methods (gensym)) )
     
     (define class-name
       (let ((local-methods '(method-name ...))
	     (object-class-proc (lambda () 'class-name)) )
	 
	 ;; The method lookup
	 (define (msg-lookup msg)
	   (cond ((eq? msg 'method-name) method-no)
		 ...
		 ((eq? msg 'me) 0)
		 ((eq? msg 'inner-me) 1)
		 ((eq? msg 'supported-messages) 2)
		 ((eq? msg 'attach-object) 3)
		 ((eq? msg 'object-class) 4)
		 (else #f) ))
	 
	 (define constructor
	   (lambda params
	     (let* ((inner-me #f) (me #f) (local-var local-var-val) ...
				  (super-struct super-obj) )
	       (letrec 
	       ;; The vector of method objects. Note that the posi-
	       ;; tions 0 to 5 are reserved for methods internal to
	       ;; the object system.
		   ((method-vec
		     (vector (lambda () me)       ; #0: me
			     (lambda () inner-me) ; #1: inner-me
			     (lambda ()           ; #2: supported-messages
			       (append local-methods
				       ((cdr super-pair) 'supported-messages)))
			     (lambda ()           ; #3: attach-object
			       (cons 'class-name fetcher) )
			     object-class-proc    ; 'object-class
			     #f                   ; this space for rent
			     method-proc ...))    ; 
	       
		    (p1  (cons #f #f))
		    (p2  (cons #f #f))
		    (p3  (cons #f #f))
		    (cache (list p1 p2 p3)) )
		 
		 (set-cdr! (last-pair cache) cache)
		 
		 (define super-pair
		   (if (pair? super-struct)
		       (cons (car super-struct) (cadddr super-struct))
		       (super-struct 'attach-object) ))
	       
		 ;; The dispatcher that represents the object from outside
		 (set! me (let ((method #f))
			    (lambda (msg . args)
			      (cond ((eq? msg (car p1)) (apply (cdr p1) args))
				    ((eq? msg (car p2)) (apply (cdr p2) args))
				    ((eq? msg (car p3)) (apply (cdr p3) args))
				    (else
				     (set! method (fetcher msg))
				     (if (procedure? method)
					 (begin
					   (set-car! (car cache) msg)
					   (set-cdr! (car cache) method)
					   (set! cache (cdr cache))
					   (apply method args) )
					 (error 'class-name
				  "INSTANCE: Unknown message: ~a with args: ~a"
						msg args) ))))))
		 
		 ;; The method object fetcher:
		 (define fetcher
		   (let ((super-fetcher (cdr super-pair)) (r #f))
		     (lambda (msg)
		       (if (pair? msg)   ; Explicit dispatch to superclass.
			   (if (eq? (car msg) (car super-pair))
			       (super-fetcher (cadr msg))
			       (error 'class-name
			      "INSTANCE: Unknown superclass: ~a in message: ~a"
				      (car msg) msg))			   
			   ;; Local dispatch if possible, else defer to super.
			   (begin
			     (set! r (msg-lookup msg))
			     (if (number? r)
				 (vector-ref method-vec r)
				 (super-fetcher msg) ))))))

		 ;; This is what the constructor returns:
		 (list 'class-name                  ; #1: Name of class
		       me                           ; #2: The dispatcher
		       (lambda (im)                 ; #3: Announce the
			 (set! inner-me im)         ;     innermost dispatcher
			 (if (pair? super-struct)
			     ((caddr super-struct) im) ))
		       fetcher                      ; #4: The local fetcher
		       (lambda ()                   ; #5: The init thunk
			 (if (pair? super-struct)
			     ((car (cddddr super-struct))) )
			 init-forms ...
			 #f)) )) ))
	 
	 (letrec ((class-method class-method-proc)
		  ...
		  (me (lambda (msg . args)
			(cond ((eq? msg 'class-method-name)
			       (apply class-method args) )
			      ...
			      ((eq? msg 'make)
			       (apply me (cons 'make-internal args)))
			      ((eq? msg 'make-internal)
			       (let ((instance (apply constructor args)))
				 ;; Setup inner-me.
				 ((caddr instance) (cadr instance))
				 ((car (cddddr instance))) ; Eval init thunk.
				 (cadr instance) ))  ; Return the dispatcher.
			      ((eq? msg 'make-super)
			       (apply constructor args))
			      ((eq? msg 'me) ((lambda () me)))
			      ((eq? msg 'object-class)
			       (cons 'class-object 'class-name))
			      (else
			       (error 'class-name
				     "CLASS: Unknown message: ~a with args: ~a"
				      msg args))))))
	   me) ))))
  
  ;; With multiple inheritance
  ((define-class-aux class-head
     (class-vars (class-var class-var-val) ...)
     (class-methods (class-method-name class-method-proc) ...)
     (locals (local-var local-var-val) ...)
     (inherit super1 super2 ...)
     (methods (method-no method-name method-proc) ...)
     (init init-forms ...))
   
   (with (((super-obj ...)
	   (map (lambda (x)
		  (if (pair? x)
		      (cons 'apply
			    (cons (car x)
				  (list (cons 'cons
					      (cons ''make-super ; Yep: ''
						    (cdr x))))))
		      x))
		'(super1 super2 ...)))
	  (class-name (car 'class-head))
	  (constructor (string->symbol (format "MAKE-~a" (car 'class-head))))
	  (params (cdr 'class-head))
	  (object-class-proc (gensym))
	  ((class-method ...) (map (lambda (x) (gensym))
				   '(class-method-name ...)))
	  (local-methods (gensym)) )
     
     (define class-name
       (let ((local-methods '(method-name ...))
	     (object-class-proc (lambda () 'class-name)) )
	 
	 ;; The method lookup
	 (define (msg-lookup msg)
	   (cond ((eq? msg 'method-name) method-no)
		 ...
		 ((eq? msg 'me) 0)
		 ((eq? msg 'inner-me) 1)
		 ((eq? msg 'supported-messages) 2)
		 ((eq? msg 'attach-object) 3)
		 ((eq? msg 'object-class) 4)
		 (else #f) ))
	 
	 (define constructor
	   (lambda params
	     (let* ((inner-me #f) (me #f) (local-var local-var-val) ...
				  (supers (list super-obj ...)) )
	       (letrec 
	       ;; The vector of method objects. Note that the posi-
	       ;; tions 0 to 5 are reserved for methods internal to
	       ;; the object system.
		   ((method-vec
		     (vector (lambda () me)       ; #0: me
			     (lambda () inner-me) ; #1: inner-me
			     (lambda ()           ; #2: supported-messages
			       (append local-methods
				       (apply append
					      (map
					       (lambda (fp)
						 (((cdr fp)
						   'supported-messages)))
					       super-fetcher-pairs))))
			     (lambda ()           ; #3: attach-object
			       (cons 'class-name fetcher) )
			     object-class-proc
			     #f
			     method-proc ...))
	       
		    (p1  (cons #f #f))
		    (p2  (cons #f #f))
		    (p3  (cons #f #f))
		    (cache (list p1 p2 p3)) )
		 
		 (set-cdr! (last-pair cache) cache)
		 
		 ;; The dispatcher that represents the object from outside
		 (set! me (let ((method #f))
			    (lambda (msg . args)
			      (cond ((eq? msg (car p1)) (apply (cdr p1) args))
				    ((eq? msg (car p2)) (apply (cdr p2) args))
				    ((eq? msg (car p3)) (apply (cdr p3) args))
				    (else
				     (set! method (fetcher msg))
				     (if (procedure? method)
					 (begin
					   (set-car! (car cache) msg)
					   (set-cdr! (car cache) method)
					   (set! cache (cdr cache))
					   (apply method args) )
					 (error 'class-name
				 "INSTANCE: Unknown message: ~a with args: ~a"
				                msg args) ))))))
		 
		 (define super-fetcher-pairs 
		   (map (lambda (s)
			  (if (pair? s)
			      (cons (car s) (cadddr s))
			      (s 'attach-object) ))
			supers))
	       
		 ;; The method object fetcher
		 (define fetcher
		   (let ((super-pair #f) (r #f))
		     (lambda (msg)
		       (if (pair? msg)
			   ;; Explicit dispatch to superclass.
			   (begin
			     (set! super-pair
				   (assq (car msg) super-fetcher-pairs))
			     (if super-pair
				 ((cdr super-pair) (cadr msg))
				 (error 'class-name
			      "INSTANCE: Unknown superclass: ~a in message: ~a"
				        (car msg) msg)))
			   ;; Local dispatch if possible, else traverse tree.
			   (begin
			     (set! r (msg-lookup msg))
			     (cond ((number? r) (vector-ref method-vec r))
				   ((null? super-fetcher-pairs) #f)
				   (else
				    (let loop ((ff (cdar super-fetcher-pairs)) 
					       (rf (cdr super-fetcher-pairs)) )
				      (set! r (ff msg))
				      (if (procedure? r) 
					  r
					  (if (null? rf)
					      #f
					      (loop (cdar rf)
						    (cdr rf)) ))))))))))
		 
		 ;; This is what the constructor returns:
		 (list 'class-name                  ; #1: Name of class
		       me                           ; #2: The dispatcher
		       (lambda (im)                 ; #3: Announce the
			 (set! inner-me im)         ;     innermost dispatcher
			 (for-each (lambda (superior)
				     (if (pair? superior)
					 ((caddr superior) im) ))
				   supers) )
		       fetcher                      ; #4: The local fetcher
		       (lambda ()                   ; #5: The init thunk
			 (for-each (lambda (superior)
				     (if (pair? superior)
					 ((car (cddddr superior))) ))
				   supers)
			 init-forms ...
			 #f)) )) ))
	 
	 (letrec ((class-method class-method-proc)
		  ...
		  (me (lambda (msg . args)
			(cond ((eq? msg 'class-method-name)
			       (apply class-method args) )
			      ...
			      ((eq? msg 'make)
			       (apply me (cons 'make-internal args)))
			      ((eq? msg 'make-internal)
			       (let ((instance (apply constructor args)))
				 ;; Setup inner-me.
				 ((caddr instance) (cadr instance))
				 ((car (cddddr instance))) ; Eval initthunk.
				 (cadr instance) ))  ; Return the dispatcher.
			      ((eq? msg 'make-super)
			       (apply constructor args))
			      ((eq? msg 'me) ((lambda () me)))
			      ((eq? msg 'object-class)
			       (cons 'class-object 'class-name))
			      (else
			       (error 'class-name
				      "CLASS: Unknown message: ~a with args: ~a"
				      msg args))))))
	   me) ))))
  
  ;; Without inheritance
  ((define-class-aux class-head
     (class-vars (class-var class-var-val) ...)
     (class-methods (class-method-name class-method-proc) ...)
     (locals (local-var local-var-val) ...)
     (methods (method-no method-name method-proc) ...)
     (init init-forms ...))
   
   (with ((class-name (car 'class-head))
	  (constructor (string->symbol (format "MAKE-~a" (car 'class-head))))
	  (params (cdr 'class-head))
	  (object-class-proc (gensym))
	  ((class-method ...) (map (lambda (x) (gensym))
				   '(class-method-name ...)))
	  (local-methods (gensym)) )
     
     (define class-name
       (let ((local-methods '(method-name ...))
	     (object-class-proc (lambda () 'class-name)) )
	 
	 ;; The method lookup
	 (define (msg-lookup msg)
	   (cond ((eq? msg 'method-name) method-no)
		 ...
		 ((eq? msg 'me) 0)
		 ((eq? msg 'inner-me) 1)
		 ((eq? msg 'supported-messages) 2)
		 ((eq? msg 'attach-object) 3)
		 ((eq? msg 'object-class) 4)
		 (else #f) ))
	 
	 (define constructor
	   (lambda params
	     (let* ((inner-me #f) (me #f) (index #f) 
				  (local-var local-var-val) ...)
	       (letrec
		   ;; The vector of method objects. Note that the posi-
		   ;; tions 0 to 5 are reserved for methods internal to
		   ;; the object system.
		   ((method-vec
		     (vector (lambda () me)            ; #0: me
			     (lambda () inner-me)      ; #1: inner-me
			     (lambda () local-methods) ; #2: supported-messages
			     (lambda ()                ; #3: attach-object
			       (cons 'class-name fetcher) )
			     object-class-proc
			     #f
			     method-proc ...)) )
		 
		 ;; The method object fetcher
		 (define (fetcher msg)
		   (set! index (msg-lookup msg))
		   (if (number? index)
		       (vector-ref method-vec index)
		       #f))
		 
		 ;; The dispatcher that represents the object from outside.
		 (set! me (lambda (msg . args)
			    (set! index (msg-lookup msg))
			    (if (number? index)
				(apply (vector-ref method-vec index) args)
				(error 'class-name
				 "INSTANCE: Unknown message: ~a with args ~a"
				       msg args))))
		 
		 ;; This is what the constructor returns:
		 (list 'class-name                  ; #1: Name of class
		       me                           ; #2: The dispatcher
		       (lambda (im)                 ; #3: Announce the
			 (set! inner-me im) )       ;     innermost dispatcher
		       fetcher                      ; #4: The local fetcher
		       (lambda ()                   ; #5: The init thunk
			 init-forms ...
			 #f)) ))) )
	 
	 (letrec ((class-method class-method-proc)
		  ...
		  (me (lambda (msg . args)
			(cond ((eq? msg 'class-method-name)
			       (apply class-method args) )
			      ...
			      ((eq? msg 'make)
			       (apply me (cons 'make-internal args)))
			      ((eq? msg 'make-internal)
			       (let ((r (apply constructor args)))
				 ((caddr r) (cadr r)) ; Setup inner-me.
				 ((car (cddddr r))) ; Evaluate the initthunk.
				 (cadr r) ))        ; Return the dispatcher.
			      ((eq? msg 'make-super)
			       (apply constructor args))
			      ((eq? msg 'me) ((lambda () me)))
			      ((eq? msg 'object-class)
			       (cons 'class-object 'class-name))
			      (else
			       (error 'class-name
   			             "CLASS: Unknown message: ~a with args: ~a"
				      msg args))))))
	   me) ))) )
  
  ) ; End of extend-syntax

(extend-syntax (define-class)
  ((define-class class-head clause ...)
   (with (((proc-clause ...)
	   (let ((class-vars (assq 'class-vars '(clause ...)))
		 (class-methods (assq 'class-methods '(clause ...)))
		 (locals (assq 'locals '(clause ...)))
		 (inherit (assq 'inherit '(clause ...)))
		 (methods (assq 'methods '(clause ...)))
		 (init (assq 'init '(clause ...))))
	     `(,(if class-vars class-vars '(class-vars))
	       ,(if class-methods class-methods '(class-methods))
	       ,(if locals locals '(locals))
	       ,@(if inherit (list inherit) '())
;;;		 ,(if inherit inherit '(inherit))
	       ,(if methods (cons 'methods
				  (map (lambda (l n)
					 (cons n l) )
				       (cdr methods)
				       ((lambda (n offset)
					  (letrec ((iter
						    (lambda (k list)
						      (if (= k offset)
							  list
							  (iter 
							   (- k 1)
							   (cons k list))))))
					    (iter (+ n offset) '()) ))
					(- (length methods) 1) 5) ))
		    '(methods) )
	       ,(if init init '(init)) ))))
     
     (define-class-aux class-head proc-clause ...))))

;;;(extend-syntax (define-class-aux)
;;;  ((define-class-aux class-head clause ...)
;;;   '(define-class-aux class-head clause ...)))


