;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: win-obj.sc,v 1.7 90/06/26 09:56:57 johani Exp $

;;; The SCIX window object.

(module scixwin (top-level make-window))

(include "../include/lw-objs.sch")
(include "../include/requests.sch")
(include "../include/util.sch")
(include "../include/masks.sch")

(define-external make-drawable scixobj)
(define-external msg-handler scixmh)

(include "../macros/extsyntax.sc")
(include "../macros/defclass.sc")

(eval-when (compile eval load)
  (define-class (window width height depth x y parent
			border-width class visual scr &optional id)
    (locals
     (value-mask (make-window-value-mask)) (child-list '()) )
    (inherit
     (drawable width height depth scr (if (null? id)
					  #f
					  (car id))))
    (methods
     ;; Simple selectors...
     (x (lambda () x))
     (y (lambda () y))
     (parent (lambda () parent))
     (border-width (lambda () border-width))
     (class (lambda () class))
     (visual (lambda () visual))
     (mask (lambda () value-mask))
     ;; ...and mutators. Be careful with these. There be dragons here!
     (set-x! (lambda (arg) (set! x arg)))
     (set-y! (lambda (arg) (set! y arg)))
     (set-parent! (lambda (arg)
		    (if parent
			(parent 'scix-denounce-child me) )
		    (set! parent arg)
		    (parent 'scix-announce-child me) ))
     (set-border-width! (lambda (arg) (set! border-width arg)))
     (set-class! (lambda (arg) (set! class arg)))
     (set-visual! (lambda (arg) (set! visual arg)))

     ;; All children to a window announce their precence to the parent
     ;; through the message 'scix-announce-child. This is necessary to maintain
     ;; consistency over operations like 'destroywindow and 'reparentwindow.
     (scix-announce-child (lambda (child)
			  ;;(format #t "~a got a 'scix-announce-child for ~a~%"
			  ;;	    (cons (me 'object-class) (me 'id))
			  ;;	    (cons (child 'object-class) (child 'id)) )
			  ;;(flush-buffer)
			    (set! child-list (cons child child-list)) ))

     ;; 'scix-denounce-child is used to maintain consistency when 
     ;; destroying a non-top-level window and when reparenting.
     (scix-denounce-child (lambda (child)
			  ;;(format #t "~a got a 'scix-denounce-child for ~a~%"
			  ;;	    (cons (me 'object-class) (me 'id))
			  ;;	    (cons (child 'object-class) (child 'id)) )
			  ;;(flush-buffer)
			    (set! child-list (remq! child child-list))))
     
     (scix-child-list (lambda () child-list)) ; Debug

     (scix-announce-destroy (lambda ()
			      ;;(format #t "~a got a 'scix-announce-destroy~%"
			      ;;      (cons (me 'object-class) (me 'id)) )
			      ;;(flush-buffer)
			      (me 'scix-denounce-id!)
			      (map (lambda (o)
				     (o 'scix-announce-destroy) )
				   child-list)))

     ;; 'scix-set-manager redirects events occuring in this window to the
     ;; event-handler provided as the parameter. This mechanism makes it
     ;; easy to have a common manager for several windows, which is sometimes
     ;; attractive.
     (scix-set-manager (lambda (manager)
			 (me 'scix-denounce-id!)
			 (me 'scix-announce-id! manager) ))

     ;; Methods mapped directly on X requests
     (createwindow (lambda (data . rest)	    ; #1: CreateWindow
		     (value-mask 'or-mask! data)
		     (send-createwindow me data scr rest) ))
     (changewindowattributes (lambda (data . rest)  ; #2: ChangeWindowAtributes
			       (value-mask 'or-mask! data)
			       (send-changewindowattributes me data scr) ))
     (getwindowattributes (lambda rest              ; #3: GetWindowAttributes
			    (send-getwindowattributes me scr rest) ))
     (destroywindow (lambda rest		    ; #4: DestroyWindow
		      (me 'scix-denounce-id!)
		      (parent 'scix-denounce-child me)
		      (map (lambda (o)
			     (o 'scix-announce-destroy) )
			   child-list)
		      (send-destroywindow me scr rest) ))
     (destroysubwindows (lambda rest		    ; #5: DestroySubWindows
			  (map (lambda (o)
				 (o 'scix-announce-destroy) )
			       child-list)
			  (send-destroysubwindows me scr rest) ))
     (changesaveset (lambda (mode . rest)               ; #6: ChangeSaveSet
		      (send-changesaveset me mode scr rest) ))
     (reparentwindow (lambda (par xpos ypos . rest)	; #7: ReparentWindow
		       (if parent
			   (parent 'scix-denounce-child me) )
		       (set! parent par)
		       (set! x xpos)
		       (set! y ypos)
		       (parent 'scix-announce-child me)
		       (send-reparentwindow me parent scr rest) ))
     (mapwindow (lambda rest			        ; #8: MapWindow
		  (send-mapwindow me scr rest) ))
     (mapsubwindows (lambda rest			; #9: MapSubWindows
		      (send-mapsubwindows me scr rest) ))
     (unmapwindow (lambda rest			        ; #10: UnmapWindow
		    (send-unmapwindow me scr rest) ))
     (unmapsubwindows (lambda rest		        ; #11: UnmapSubWindows
			(send-unmapsubwindows me scr rest) ))
     (configurewindow (lambda (data . rest)             ; #12: ConfigureWindow
			(send-configurewindow me data scr rest) ))
     (circulatewindow (lambda (direction . rest)        ; #13: CirculateWindow
			(send-circulatewindow me direction scr rest) ))
     (querytree (lambda rest		                ; #15: QueryTree
		  (send-querytree me scr rest) ))
     (changeproperty (lambda (mode prop type            ; #18: ChangeProperty
				   format data . rest)
		       (send-changeproperty me mode prop type
					    format data scr rest)))
     (deleteproperty (lambda (prop . rest)              ; #19: DeleteProperty
		       (send-deleteproperty me prop scr rest) ))
     (getproperty (lambda (del prop type                ; #20: GetProperty
			       l-offset l-len . rest)
		    (send-getproperty del me prop type
				      l-offset l-len scr rest) ))
     (listproperties (lambda rest                       ; #21: ListProperties
		       (send-listproperties me scr rest) ))
     (rotateproperties (lambda (delta props . rest)    ; #114: RotateProperties
			 (send-rotateproperties me delta
						props scr rest) ))
     (grabbutton (lambda (owner-ev ev-mask ptr-mode     ; #28: GrabButton
				   kbd-mode confine-to
				   cursor button modifiers . rest)
		   (send-grabbutton owner-ev me ev-mask ptr-mode
				    kbd-mode confine-to cursor
				    button modifiers scr rest)))
     (ungrabbutton (lambda (button modifiers . rest)    ; #29: UngrabButton
		     (send-ungrabbutton button me modifiers scr rest) ))
     (grabkey (lambda (owner-ev mods key ptr-mode       ; #33: GrabKey
				kbd-mode . rest)
		(send-grabkey owner-ev me mods key ptr-mode
				 kbd-mode scr rest)))
     (ungrabkey (lambda (key modifiers . rest)          ; #34: UngrabKey
		  (send-ungrabkey key me modifiers scr rest) ))
     (querypointer (lambda rest		                ; #38: QueryPointer
		     (send-querypointer me scr rest) ))
     (getmotionevents (lambda (start stop . rest)       ; #39: GetMotionEvents
			(send-getmotionevents me start stop scr rest) ))
     (translatecoordinates (lambda (src-win src-x   ; #40: TranslateCoordinates
					    src-y . rest) 
			     (send-translatecoordinates src-win me
							src-x src-y scr rest)))
     (cleararea (lambda (exposures x y width height . rest) ; #61: ClearArea
		  (send-cleararea exposures me x y
				  width height scr rest) ))
     ; Any methods like clear here should rather be installed with 
     ; insert-method!
     ; (clear (lambda (exposures)	; A shortcut to clear entire window
     ;	  (send-cleararea exposures me 0 0 width height scr rest) ))
     
     ) ; End of methods
    (init (if (or (null? id)		; i e top-level SCIX object
		  (number? (car id)) )	; i e id given by server
	      (begin
		(me 'scix-announce-id! me)
		(if parent              ; i e this is not the root window
		    (parent 'scix-announce-child me) ))))

    )  ;; End of define-class
  )    ;; End of eval-when
