;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: requests2.sc,v 1.3 90/06/26 09:54:06 johani Exp $

;;; requests2.sc -- the code that constructs the individual requests, i e here
;;;                 we have the translation from the OO-level to the X protocol
;;;                 level.

(module scixreq2); (top-level))

(include "../include/opcodes.sch")
(include "../include/types.sch")
(include "../include/util.sch")
(include "../include/lowlevel.sch")
(include "../include/msgutil.sch")

(define-external msg-handler scixmh)

;;; Request #52: GetFontPath
(define (send-GetFontPath scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetFontPath)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 1) )	; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr 
		 `((reply-name . ,(d-identity 'GetFontPath))
		   (unused . ,(d-unused 1))
		   (seq-nr . ,d-card16)
		   (unused . ,(d-unused 4)) ; Reply length not used
		   (nr-of-strs . ,d-card16)
		   (unused . ,(d-unused 22))
		   (path . ,(lambda (str dpy)
			    (let loop ((nr (backtrack 'nr-of-strs))
				       (total-length 0)
				       (result '()) )
			      (if (zero? nr)
				  (begin
				    ((d-unused (pad total-length)) str dpy)
				    result)
				  (let* ((this-str (list (d-str str dpy)))
					 (this-len (+ 1
						      (string-length
						       (car this-str)))))
				    (loop (- nr 1)
					  (+ total-length this-len)
					  (append result this-str) )))))))
		 scr) ))

;;; Request #53: CreatePixmap
(define (send-CreatePixmap pmap scr . rest)
  (scr 'scix-xas `((,a-request  . ,CreatePixmap)
		   (,a-card8    . ,(pmap 'depth))
		   (,a-card16   . 4)			; Request length
		   (,a-pixmap   . ,pmap)
		   (,a-drawable . ,(pmap 'drawable))
		   (,a-card16   . ,(pmap 'width))
		   (,a-card16   . ,(pmap 'height)) )
       rest))

;;; Request #54: FreePixmap
(define (send-FreePixmap pmap scr . rest)
  (scr 'scix-xas `((,a-request . ,FreePixmap)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; Request length
		   (,a-pixmap  . ,pmap) )
       rest))

;;; Request #55: CreateGC
(define (send-CreateGC gc data scr . rest)
  (scr 'scix-xas `((,a-request     . ,CreateGC)
		   (,a-card8       . 0)		; 1 unused byte
		   (,a-card16      . ,(+ 4 (data 'length)))
		   (,a-gcontext    . ,gc)
		   (,a-drawable    . ,(gc 'drawable))
		   (,a-bitmask     . ,data)
		   (,a-listofvalue . ,data) )
       rest))

;;; Request #56: ChangeGC
(define (send-ChangeGC gc data scr . rest)
  (scr 'scix-xas `((,a-request     . ,ChangeGC)
		   (,a-card8       . 0)			; 1 unused byte
		   (,a-card16      . ,(+ 3 (data 'length)))
		   (,a-gcontext    . ,gc)
		   (,a-bitmask     . ,data)
		   (,a-listofvalue . ,data) )
       rest))

;;; Request #57: CopyGC
(define (send-CopyGC src-gc dst-gc data scr . rest)
  (scr 'scix-xas `((,a-request  . ,CopyGC)
		   (,a-card8    . 0)		; 1 unused byte
		   (,a-card16   . 4)		; req-len is constant
		   (,a-gcontext . ,src-gc)
		   (,a-gcontext . ,dst-gc)
		   (,a-bitmask  . ,data) )
       rest))

;;; Request #58: SetDashes
(define (send-SetDashes gc dash-offset dash-list scr . rest)
  (let ((n (length dash-list)))
    (scr 'scix-xas `((,a-request  . ,SetDashes)
		     (,a-card8    . 0)		    ; 1 unused byte
		     (,a-card16   . ,(+ 3 (/ (+ n (pad n)) 4))); Request length
		     (,a-gcontext . ,gc)
		     (,a-card16   . ,dash-offset)
		     (,a-card16   . ,n)		    ; Length of dashes
		     (,a-intlist  . ,(list-pad4 dash-list)) )
	 rest)))

;;; Request #59: SetClipRectangles
(define (send-SetClipRectangles ordering gc clip-x-orig clip-y-orig 
				rectangle-list scr . rest) 
  (let ((ord (lookup-constant ordering '((UnSorted . 0)
					 (YSorted  . 1)
					 (YXSorted . 2)
					 (YXBanded . 3) ))))
    (scr 'scix-xas `((,a-request  . ,SetClipRectangles)
		     (,a-card8    . ,ord)
		     (,a-card16   . ,(+ 3 (* 2 (length rectangle-list))))
		     (,a-gcontext . ,gc)
		     (,a-int16    . ,clip-x-orig)
		     (,a-int16    . ,clip-x-orig)
		     (,(a-list a-rectangle) . ,rectangle-list) )
	 rest)))

;;; Request #60: FreeGC
(define (send-FreeGC gc scr . rest)
  (scr 'scix-xas `((,a-request  . ,FreeGC)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . 2)			; Request length
		   (,a-gcontext . ,gc) )
       rest))

;;; Request #61: ClearArea
(define (send-ClearArea exposures w x y width height scr . rest)
  (scr 'scix-xas `((,a-request . ,ClearArea)
		   (,a-bool    . ,exposures)
		   (,a-card16  . 4)			; Request length
		   (,a-window  . ,w)
		   (,a-int16   . ,x)
		   (,a-int16   . ,y)
		   (,a-card16  . ,width)
		   (,a-card16  . ,height) )
       rest))

;;; Request #62: CopyArea
(define (send-CopyArea src-d dst-d gc src-x src-y
		       dst-x dst-y width height scr . rest)
  (scr 'scix-xas `((,a-request  . ,CopyArea)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . 7)			; Request length
		   (,a-drawable . ,src-d)
		   (,a-drawable . ,dst-d)
		   (,a-gcontext . ,gc)
		   (,a-int16    . ,src-x)
		   (,a-int16    . ,src-y)
		   (,a-int16    . ,dst-x)
		   (,a-int16    . ,dst-y)
		   (,a-card16   . ,width)
		   (,a-card16   . ,height) )
       rest))

 ;;; Request #63: CopyPlane
(define (send-CopyPlane src-d dst-d gc src-x src-y
			dst-x dst-y width height bit-plane scr . rest)
  (scr 'scix-xas `((,a-request  . ,CopyPlane)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . 8)			; Request length
		   (,a-drawable . ,src-d)
		   (,a-drawable . ,dst-d)
		   (,a-gcontext . ,gc)
		   (,a-int16    . ,src-x)
		   (,a-int16    . ,src-y)
		   (,a-int16    . ,dst-x)
		   (,a-int16    . ,dst-y)
		   (,a-card16   . ,width)
		   (,a-card16   . ,height)
		   (,a-card32   . ,bit-plane) )
       rest))

;;; Request #64: PolyPoint
;;; Note: None of the "graphics objects" requests deal with the actual graphics
;;; objects. Instead the objects supply all the parameters to the requests.
;;; This is inconsistent with the rest of the requests but is done to keep the
;;; graphics objects as simple as possible.
;;;
(define (send-PolyPoint drawable gc c-mode point-list scr . rest)
  (scr 'scix-xas `((,a-request        . ,PolyPoint)
		   (,a-card8          . ,(lookup-constant c-mode
							  '((Origin . 0)
							    (Previous . 1) )))
		   (,a-card16         . ,(+ 3 (length point-list)))
		   (,a-drawable       . ,drawable)
		   (,a-gcontext       . ,gc)
		   (,(a-list a-point) . ,point-list) )
       rest))

;;; Request #65: PolyLine
(define (send-PolyLine drawable gc c-mode point-list scr . rest)
  (scr 'scix-xas `((,a-request        . ,PolyLine)
		   (,a-card8          . ,(lookup-constant c-mode
							  '((Origin . 0)
							    (Previous . 1) )))
		   (,a-card16         . ,(+ 3 (length point-list)))
		   (,a-drawable       . ,drawable)
		   (,a-gcontext       . ,gc)
		   (,(a-list a-point) . ,point-list) )
       rest))

;;; Request #66: PolySegment
(define (send-PolySegment drawable gc seg-list scr . rest)
  (scr 'scix-xas `((,a-request  . ,PolySegment)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . ,(+ 3 (* 2 (length seg-list))))
		   (,a-drawable . ,drawable)
		   (,a-gcontext . ,gc)
		   (,(a-list (lambda (seg-l buffer)
			       (a-int16 (car seg-l) buffer)
			       (a-int16 (cadr seg-l) buffer)
			       (a-int16 (caddr seg-l) buffer)
			       (a-int16 (cadddr seg-l) buffer))) . ,seg-list) )
       rest))

;;; Request #67: PolyRectangle
(define (send-PolyRectangle drawable gc rect-list scr . rest)
  (scr 'scix-xas `((,a-request            . ,PolyRectangle)
		   (,a-card8              . 0)		; 1 unused byte
		   (,a-card16             . ,(+ 3 (* 2 (length rect-list))))
		   (,a-drawable           . ,drawable)
		   (,a-gcontext           . ,gc)
		   (,(a-list a-rectangle) . ,rect-list) )
       rest))

;;; Request #68: PolyArc
(define (send-PolyArc drawable gc arc-list scr . rest)
  (scr 'scix-xas `((,a-request      . ,PolyArc)
		   (,a-card8        . 0)			; 1 unused byte
		   (,a-card16       . ,(+ 3 (* 3 (length arc-list))))
		   (,a-drawable     . ,drawable)
		   (,a-gcontext     . ,gc)
		   (,(a-list a-arc) . ,arc-list) )
       rest))

;;; Request #69: FillPoly
(define (send-FillPoly drawable gc shape coord-mode point-list scr . rest)
  (let ((sh (lookup-constant shape '((Complex . 0)
				     (Nonconvex . 1)
				     (Convex . 2) )))
	(cm (lookup-constant coord-mode '((Origin . 0)
					  (Previous . 1) ))))
    (scr 'scix-xas `((,a-request        . ,FillPoly)
		     (,a-card8          . 0)		; 1 unused byte
		     (,a-card16         . ,(+ 4 (length point-list)))
		     (,a-drawable       . ,drawable)
		     (,a-gcontext       . ,gc)
		     (,a-card8          . ,sh)
		     (,a-card8          . ,cm)
		     (,a-card16         . 0)		; 2 unused bytes
		     (,(a-list a-point) . ,point-list) )
	 rest)))

;;; Request #70: PolyFillRectangle
(define (send-PolyFillRectangle drawable gc rect-list scr . rest)
  (scr 'scix-xas `((,a-request            . ,PolyFillRectangle)
		   (,a-card8              . 0)		; 1 unused byte
		   (,a-card16             . ,(+ 3 (* 2 (length rect-list))))
		   (,a-drawable           . ,drawable)
		   (,a-gcontext           . ,gc)
		   (,(a-list a-rectangle) . ,rect-list) )
       rest))

;;; Request #71: PolyFillArc
(define (send-PolyFillArc drawable gc arc-list scr . rest)
  (scr 'scix-xas `((,a-request      . ,PolyFillArc)
		   (,a-card8        . 0)		; 1 unused byte
		   (,a-card16       . ,(+ 3 (* 3 (length arc-list))))
		   (,a-drawable     . ,drawable)
		   (,a-gcontext     . ,gc)
		   (,(a-list a-arc) . ,arc-list) )
       rest))

;;; Request #72: PutImage
;;; Note: Not tested.
(define (send-PutImage image-format drawable gc width height dst-x dst-y
		       left-pad byte-list scr . rest)
  (let ((n (length byte-list))
	(fmt (lookup-constant image-format '((Bitmap . 0)
					     (XYPixmap . 1)
					     (ZPixmap . 2) ))))
    (scr 'scix-xas `((,a-request  . ,PutImage)
		     (,a-card8    . ,fmt)
		     (,a-card16   . ,(+ 6 (/ (+ n (pad n)) 4)))
		     (,a-drawable . ,drawable)
		     (,a-gcontext . ,gc)
		     (,a-card16   . ,width)
		     (,a-card16   . ,height)
		     (,a-int16    . ,dst-x)
		     (,a-int16    . ,dst-y)
		     (,a-card8    . ,left-pad)
		     (,a-card8    . ,(if (eq? image-format 'Bitmap) 
					 1        ; The depth can be calculated
					 (drawable 'depth)))
		     (,a-card16   . 0)			; 2 unused bytes
		     (,a-intlist  . ,(list-pad4 byte-list)) )
	 rest)))

;;; Request #73: GetImage
;;; NOTE: Padding is not stripped from the data list returned, as we don't
;;; know how much is present. Does the caller know?
(define (send-GetImage format drawable x y width height plane-mask scr . rest)
  (let* ((fmt (lookup-constant format '((XYPixmap . 1) (ZPixmap . 2))))
	 (seq-nr (scr 'scix-xas `((,a-request  . ,GetImage)
				  (,a-card8    . ,fmt)
				  (,a-card16   . 5)	; Request length
				  (,a-drawable . ,drawable)
				  (,a-int16    . ,x)
				  (,a-int16    . ,y)
				  (,a-card16   . ,width)
				  (,a-card16   . ,height)
				  (,a-card32   . ,plane-mask) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'GetImage))
		   (depth      . ,d-card8)
		   (seq-nr     . ,d-card16)
		   (reply-len  . ,d-card32)
		   (visual     . ,(d-const #f d-visualid '((0 . None))))
		   (unused     . ,(d-unused 20))
		   (data       . ,(lambda (str dpy)
				    ((d-list
				      d-card8 (* 4 (backtrack 'reply-len)) )
				     str dpy))))
		 scr) ))

;;; Request #74: PolyText8
(define (send-PolyText8 drawable gc x y textobj scr . rest)
  (scr 'scix-xas `((,a-request  . ,PolyText8)
		   (,a-card8    . 0)		               ; 1 unused byte
		   (,a-card16   . ,(+ 4 (textobj 'length)))    ; Request length
		   (,a-drawable . ,drawable)
		   (,a-gcontext . ,gc)
		   (,a-int16    . ,x)
		   (,a-int16    . ,y)
		   (,a-intlist  . ,(textobj 'items)) )
       rest))

;;; Request #75: PolyText16
(define (send-PolyText16 drawable gc x y textobj scr . rest)
  (scr 'scix-xas `((,a-request  . ,PolyText16)
		   (,a-card8    . 0)		               ; 1 unused byte
		   (,a-card16   . ,(+ 4 (textobj 'length)))
		   (,a-drawable . ,drawable)
		   (,a-gcontext . ,gc)
		   (,a-int16    . ,x)
		   (,a-int16    . ,y)
		   (,a-intlist  . ,(textobj 'items)) ) ; Should use strings...
       rest))

;;; Request #76: ImageText8
(define (send-ImageText8 drawable gc x y str scr . rest)
  (let ((n (string-length str)))
    (scr 'scix-xas `((,a-request  . ,ImageText8)
		     (,a-card8    . ,n)
		     (,a-card16   . ,(+ 4 (/ (+ n (pad n)) 4))) ; Request len
		     (,a-drawable . ,drawable)
		     (,a-gcontext . ,gc)
		     (,a-int16    . ,x)
		     (,a-int16    . ,y)
		     (,a-string8  . ,str) )
	 rest)))

;;; Request #77: ImageText16 -- Note: STRING16 is represented as an ordinary
;;;                             STRING8 with two bytes per CHAR16. If some
;;;                             other representation (like a list of pairs for
;;;                             instance) should turn out to be better then
;;;                             change it.
;;;
(define (send-ImageText16 drawable gc x y str scr . rest)
  (let ((two-n (string-length str)))
    (scr 'scix-xas `((,a-request  . ,ImageText16)
		     (,a-card8    . ,(/ two-n 2))
		     (,a-card16   . ,(+ 4 (/ (+ two-n (pad two-n)) 4)))
		     (,a-drawable . ,drawable)
		     (,a-gcontext . ,gc)
		     (,a-int16    . ,x)
		     (,a-int16    . ,y)
		     (,a-string8  . ,str) )                 ; a-string8 pads.
	 rest)))

;;; Request #78: CreateColormap
(define (send-CreateColormap alloc cmap visual scr . rest)
  (scr 'scix-xas `((,a-request  . ,CreateColormap)
		   (,a-card8    . ,(lookup-constant alloc '((None . 0)
							    (All  . 1) )))
		   (,a-card16   . 4)		; Request length
		   (,a-colormap . ,cmap)
		   (,a-window   . ,(scr 'root))
		   (,a-visualid . ,visual) )
       rest))

;;; Request #79: FreeColormap
(define (send-FreeColormap cmap scr . rest)
  (scr 'scix-xas `((,a-request  . ,FreeColormap)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . 2)			; Request length
		   (,a-colormap . ,cmap) )
       rest))

;;; Request #80: CopyColormapAndFree
(define (send-CopyColormapAndFree cmap src-cmap scr . rest)
  (scr 'scix-xas `((,a-request  . ,CopyColormapAndFree)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . 3)			; Request length
		   (,a-colormap . ,cmap)
		   (,a-colormap . ,src-cmap) )
       rest))

;;; Request #81: InstallColormap
(define (send-InstallColormap cmap scr . rest)
  (scr 'scix-xas `((,a-request  . ,InstallColormap)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . 2)			; Request length
		   (,a-colormap . ,cmap) )
       rest))

;;; Request #82: UninstallColormap
(define (send-UninstallColormap cmap scr . rest)
  (scr 'scix-xas `((,a-request  . ,UninstallColormap)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . 2)			; Request length
		   (,a-colormap . ,cmap) )
       rest))

;;; Request #83: ListInstalledColormaps
(define (send-ListInstalledColormaps scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,ListInstalledColormaps)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 2)	; Request length
				 (,a-window  . ,(scr 'root)) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr 
		 `((reply-name  . ,(d-identity 'ListInstalledColormaps))
		   (unused      . ,d-card8)
		   (seq-nr      . ,d-card16)
		   (unused      . ,(d-unused 4)) ; Reply length not used
		   (nr-of-cmaps . ,d-card16)
		   (unused      . ,(d-unused 22))
		   (cmaps       . ,(lambda (str dpy)
				     ((d-list d-colormap
					      (backtrack 'nr-of-cmaps) )
				      str dpy))))
		 scr) ))

;;; Request #84: AllocColor
(define (send-AllocColor cmap red green blue scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request  . ,AllocColor)
				 (,a-card8    . 0)	; 1 unused byte
				 (,a-card16   . 4)	; Request length
				 (,a-colormap . ,cmap)
				 (,a-card16   . ,red)
				 (,a-card16   . ,green)
				 (,a-card16   . ,blue)
				 (,a-card16   . 0) )	; 2 unused bytes
		     rest)))
    (msg-handler 'scix-wait-for-reply 
		 seq-nr
		 `((reply-name . ,(d-identity 'AllocColor))
		   (unused     . ,(d-unused 1))
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 4)) ; Reply length = 0
		   (red        . ,d-card16)
		   (green      . ,d-card16)
		   (blue       . ,d-card16)
		   (unused     . ,(d-unused 2))
		   (pixel      . ,d-card32)
		   (unused     . ,(d-unused 12)) )
		 scr) ))

;;; Request #85: AllocNamedColor
(define (send-AllocNamedColor cmap name scr . rest)
  (let* ((n (string-length name))
	 (seq-nr (scr 'scix-xas `((,a-request  . ,AllocNamedColor)
				  (,a-card8    . 0)	; 1 unused byte
				  (,a-card16   . ,(+ 3 (/ (+ n (pad n)) 4)))
				  (,a-colormap . ,cmap)
				  (,a-card16   . ,n)
				  (,a-card16   . 0)	; 2 unused bytes
				  (,a-string8  . ,name) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name   . ,(d-identity 'AllocNamedColor))
		   (unused       . ,(d-unused 1))
		   (seq-nr       . ,d-card16)
		   (unused       . ,(d-unused 4)) ; Reply length = 0
		   (pixel        . ,d-card32)
		   (exact-red    . ,d-card16)
		   (exact-green  . ,d-card16)
		   (exact-blue   . ,d-card16)
		   (visual-red   . ,d-card16)
		   (visual-green . ,d-card16)
		   (visual-blue  . ,d-card16)
		   (unused       . ,(d-unused 8)) )
		 scr) ))

;;; Request #86: AllocColorCells
;;; Note: see comment at method alloccolorcells in the colormap object.
(define (send-AllocColorCells cont cmap colors planes scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request  . ,AllocColorCells)
				 (,a-bool     . ,cont)
				 (,a-card16   . 3)	; Request length
				 (,a-colormap . ,cmap)
				 (,a-card16   . ,colors)
				 (,a-card16   . ,planes) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name           . ,(d-identity 'AllocColorCells))
		   (unused               . ,(d-unused 1))
		   (seq-nr               . ,d-card16)
		   (unused               . ,(d-unused 4)) ; Reply len not used
		   (nr-of-cards-in-pix   . ,d-card16)
		   (nr-of-cards-in-masks . ,d-card16)
		   (unused               . ,(d-unused 20))
		   (pixels               . ,(lambda (str dpy)
					      ((d-list d-card32
						       (backtrack
							'nr-of-cards-in-pix) )
					       str dpy)))
		   (masks                . ,(lambda (str dpy)
					      ((d-list d-card32
						       (backtrack
							'nr-of-cards-in-masks))
					       str dpy))))
		 scr) ))

;;; Request #87: AllocColorPlanes
(define (send-AllocColorPlanes cont cmap colors reds greens blues scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request  . ,AllocColorPlanes)
				 (,a-bool     . ,cont)
				 (,a-card16   . 4)
				 (,a-colormap . ,cmap)
				 (,a-card16   . ,colors)
				 (,a-card16   . ,reds)
				 (,a-card16   . ,greens)
				 (,a-card16   . ,blues) )
		     rest)))
    (msg-handler 'scix-wait-for-reply 
		 seq-nr
		 `((reply-name    . ,(d-identity 'AllocColorPlanes))
		   (unused        . ,d-card8)
		   (seq-nr        . ,d-card16)
		   (unused        . ,d-card32) ; Reply length not used
		   (nr-of-card32s . ,d-card16)
		   (unused        . ,d-card16)
		   (red-mask      . ,d-card32)
		   (green-mask    . ,d-card32)
		   (blue-mask     . ,d-card32)
		   (unused        . ,(d-unused 8))
		   (pixels        . ,(lambda (str dpy)
				       ((d-list d-card32
						(backtrack 'nr-of-card32s) )
					str dpy))))
		 scr) ))

;;; Request #88: FreeColors
(define (send-FreeColors cmap plane-mask pixels scr . rest)
  (scr 'scix-xas `((,a-request         . ,FreeColors)
		   (,a-card8           . 0)			; 1 unused byte
		   (,a-card16          . ,(+ 3 (length pixels)))
		   (,a-colormap        . ,cmap)
		   (,a-card32          . ,plane-mask)
		   (,(a-list a-card32) . ,pixels) )
       rest))

;;; Request #89: StoreColors
(define (send-StoreColors cmap colors scr . rest)
  (scr 'scix-xas `((,a-request  . ,StoreColors)
		   (,a-card8    . 0)
		   (,a-card16   . ,(+ 2 (* 3 (length colors))))
		   (,a-colormap . ,cmap)
		   (,(a-list (lambda (c str)
			       (a-color c str)
			       (a-card16 (c 'red) str)
			       (a-card16 (c 'green) str)
			       (a-card16 (c 'blue) str)
			       (a-card8  (c 'do-mask) str)
			       (a-card8 0 str) )) . ,colors))
       rest))

;;; Request #90: StoreNamedColor
(define (send-StoreNamedColor cmap color name scr . rest)
  (let ((name-len (string-length name)))
    (scr 'scix-xas `((,a-request  . ,StoreNamedColor)
		     (,a-card8    . ,(color 'do-mask))
		     (,a-card16   . ,(+ 4 (/ (+ name-len (pad name-len)) 4)))
		     (,a-colormap . ,cmap)
		     (,a-color    . ,color)
		     (,a-card16   . ,name-len)
		     (,a-card16   . 0)			; 2 unused bytes
		     (,a-string8  . ,name) )
	 rest)))

;;; Request #91: QueryColors 
(define (send-QueryColors cmap colors scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request        . ,QueryColors)
				 (,a-card8          . 0)       ; 1 unused byte
				 (,a-card16         . ,(+ 2 (length colors)))
				 (,a-colormap       . ,cmap)
				 (,(a-list a-color) . ,colors) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'QueryColors))
		   (unused     . ,d-card8)
		   (seq-nr     . ,d-card16)
		   (unused     . ,d-card32) ; Reply length not used
		   (nr-of-RGBs . ,d-card16)
		   (unused     . ,(d-unused 22))
		   (colors     . ,(lambda (str dpy)
				    ((d-list (lambda (str dpy)
					       (let* ((red (d-card16 str dpy))
						      (green
						       (d-card16 str dpy))
						      (blue (d-card16 str dpy))
						      (this-RGB (list red
								      green
								      blue)))
						 ((d-unused 2) str dpy)
						 this-RGB))
					     (backtrack 'nr-of-RGBs) )
				     str dpy))))
		 scr) ))

;;; Request #92: LookupColor
(define (send-LookupColor name scr . rest)
  (let* ((n (string-length name))
	 (seq-nr (scr 'scix-xas `((,a-request  . ,LookupColor)
				  (,a-card8    . 0)	; 1 unused byte
				  (,a-card16   . ,(+ 3 (/ (+ n (pad n)) 4)))
				  (,a-colormap . ,(scr 'default-colormap))
				  (,a-card16   . ,n)
				  (,a-card16   . 0)	; 2 unused bytes
				  (,a-string8  . ,name) )
		      rest)))
    (msg-handler 'scix-wait-for-reply 
		 seq-nr
		 `((reply-name   . ,(d-identity 'LookupColor))
		   (unused       . ,(d-unused 1))
		   (seq-nr       . ,d-card16)
		   (unused       . ,(d-unused 4)) ; Reply length = 0
		   (exact-red    . ,d-card16)
		   (exact-green  . ,d-card16)
		   (exact-blue   . ,d-card16)
		   (visual-red   . ,d-card16)
		   (visual-green . ,d-card16)
		   (visual-blue  . ,d-card16)
		   (unused       . ,(d-unused 12)) )
		 scr) ))

;;; Request #93: CreateCursor
(define (send-CreateCursor cursor source mask fore back x y scr . rest)
  (scr 'scix-xas `((,a-request . ,CreateCursor)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 8)			; Request length
		   (,a-cursor  . ,cursor)
		   (,a-pixmap  . ,source)
		   (,a-pixmap  . ,(if (eq? mask 'None)
				      0
				      mask))
		   (,a-card16  . ,(fore 'red))
		   (,a-card16  . ,(fore 'green))
		   (,a-card16  . ,(fore 'blue))
		   (,a-card16  . ,(back 'red))
		   (,a-card16  . ,(back 'green))
		   (,a-card16  . ,(back 'blue))
		   (,a-card16  . ,x)
		   (,a-card16  . ,y) )
       rest))

;;; Request #94: CreateGlyphCursor
(define (send-CreateGlyphCursor cursor source-font mask-font
				source-char mask-char fore back scr . rest)
  (scr 'scix-xas `((,a-request . ,CreateGlyphCursor)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 8)			; Request length
		   (,a-cursor  . ,cursor)
		   (,a-font    . ,source-font)
		   (,a-font    . ,(if (eq? mask-font 'None)
				      0
				      mask-font))
		   (,a-card16  . ,source-char)
		   (,a-card16  . ,mask-char)
		   (,a-card16  . ,(fore 'red))
		   (,a-card16  . ,(fore 'green))
		   (,a-card16  . ,(fore 'blue))
		   (,a-card16  . ,(back 'red))
		   (,a-card16  . ,(back 'green))
		   (,a-card16  . ,(back 'blue) ))
       rest))

;;; Request #95: FreeCursor
(define (send-FreeCursor cursor scr . rest)
  (scr 'scix-xas `((,a-request . ,FreeCursor)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; Request length
		   (,a-cursor  . ,cursor) )
       rest))

;;; Request #96: RecolorCursor
(define (send-RecolorCursor cursor fore back scr . rest)
  (scr 'scix-xas `((,a-request . ,RecolorCursor)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 5)			; Request length
		   (,a-cursor  . ,cursor)
		   (,a-card16  . ,(fore 'red))
		   (,a-card16  . ,(fore 'green))
		   (,a-card16  . ,(fore 'blue))
		   (,a-card16  . ,(back 'red))
		   (,a-card16  . ,(back 'green))
		   (,a-card16  . ,(back 'blue) ))
       rest))

;;; Request #97: QueryBestSize
;;; Note: What is the use of the width and height parameters?
(define (send-QueryBestSize cl drawable width height scr . rest)
  (let* ((class (lookup-constant cl '((Cursor . 0)
				      (Tile . 1)
				      (Stipple . 2) )))
	 (seq-nr (scr 'scix-xas `((,a-request  . ,QueryBestSize)
				  (,a-card8    . ,class)
				  (,a-card16   . 3)	      ; Request length
				  (,a-drawable . ,drawable)
				  (,a-card16   . ,width)
				  (,a-card16   . ,height) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'QueryBestSize))
		   (unused     . ,(d-unused 1))
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 4)) ; Reply length = 0
		   (width      . ,d-card16)
		   (height     . ,d-card16)
		   (unused     . ,(d-unused 20)) )
		 scr) ))

;;; Request #98: QueryExtension
(define (send-QueryExtension name scr . rest)
  (let* ((len (string-length name))
	 (seq-nr (scr 'scix-xas `((,a-request . ,QueryExtension)
				  (,a-card8   . 0)	; 1 unused byte
				  (,a-card16  . ,(+ 2 (/ (+ len (pad len)) 4)))
				  (,a-card16  . ,len)
				  (,a-card16  . 0)	; 2 unused bytes
				  (,a-string8 . ,name) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply        . ,(d-identity 'QueryExtension))
		   (unused       . ,(d-unused 1))
		   (seq-nr       . ,d-card16)
		   (unused       . ,(d-unused 4)) ; Reply length = 0
		   (present      . ,d-bool)
		   (major-opcode . ,d-card8)
		   (first-event  . ,d-card8)
		   (first-error  . ,d-card8)
		   (unused       . ,(d-unused 20)) )
		 scr) ))

;;; Request #99: ListExtensions
(define (send-ListExtensions scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,ListExtensions)
				 (,a-card8   . 0)	   ; 1 unused byte
				 (,a-card16  . 1) )        ; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'ListExtensions))
		   (nr-of-strs . ,d-card8)
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 28)) ; Reply length not used
		   (names      . ,(lambda (str dpy)
				    (let loop ((nr (backtrack 'nr-of-strs))
					       (total-length 0)
					       (result '()))
				      (if (zero? nr)
					  (begin
					    ((d-unused (pad total-length))
					     str dpy)
					    result)
					  (let* ((this-str (list
							    (d-str str dpy)))
						 (this-len
						  (+ 1 (string-length
							(car this-str)) )))
					    (loop (- nr 1)
						  (+ total-length this-len)
						  (append result
							  this-str) )))))))
		 scr) ))

;;; Request #100: ChangeKeyboardMapping
;;; Note that key-count and keysyms/keycode are calculated from other params.
;;;(send-ChangeKeyboardMapping 64 '((12312 12312 142414 124124)
;;;                                 (13451 435435 34534 435345)
;;;                                 (34234 324342 32432 234324)) scr . rest)
(define (send-ChangeKeyboardMapping first-keycode keysyms scr . rest)
  (let ((keycode-count (length keysyms))
	(keysyms-per-keycode (length (car keysyms))) )
    (scr 'scix-xas `((,a-request . ,ChangeKeyboardMapping)
		     (,a-card8   . ,keycode-count)
		     (,a-card16  . ,(+ 2 (* keycode-count
					    keysyms-per-keycode)))
		     (,a-keycode . ,first-keycode)
		     (,a-card8   . ,keysyms-per-keycode)
		     (,a-card16  . 0)			; 2 unused bytes
		     (,(a-list (a-list a-keysym)) . ,keysyms) )
	 rest)))

;;; Request #101: GetKeyboardMapping
(define (send-GetKeyboardMapping first-keycode count scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetKeyboardMapping)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 2)	; Request length
				 (,a-keycode . ,first-keycode)
				 (,a-card8   . ,count)
				 (,a-card16  . 0) )   ; 2 unused bytes
		     rest)))
    (msg-handler 'scix-wait-for-reply 
		 seq-nr
		 `((reply-name          . ,(d-identity 'GetKeyboardMapping))
		   (keysyms-per-keycode . ,d-card8)
		   (seq-nr              . ,d-card16)
		   (reply-len           . ,d-card32)
		   (unused              . ,(d-unused 24))
		   (keysyms             . ,(lambda (str dpy)
					     ((d-list
					       (d-list d-keysym 
						       (backtrack
							'keysyms-per-keycode))
					       (/ (backtrack 'reply-len)
						  (backtrack
						   'keysyms-per-keycode) ))
					      str dpy))))
		 scr) ))

;;; Request #102: ChangeKeyboardControl
(define (send-ChangeKeyboardControl data scr . rest)
  (scr 'scix-xas `((,a-request     . ,ChangeKeyboardControl)
		   (,a-card8       . 0)		         ; 1 unused byte
		   (,a-card16      . ,(+ 2 (data 'length)))    ; Request length
		   (,a-bitmask     . ,data)
		   (,a-listofvalue . ,data) )
       rest))

;;; Request #103: GetKeyboardControl
(define (send-GetKeyboardControl scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetKeyboardControl)
				 (,a-card8   . 0)   	; 1 unused byte
				 (,a-card16  . 1) )	; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr 
		 `((reply-name         . ,(d-identity 'GetKeyboardControl))
		   (global-auto-repeat . ,(d-const #t d-card8 '((0 . Off)
							      (1 . On) )))
		   (seq-nr             . ,d-card16)
		   (unused             . ,d-card32) ; Reply length = 5
		   (led-mask           . ,d-card32)
		   (key-click-percent  . ,d-card8)
		   (bell-percent       . ,d-card8)
		   (bell-pitch         . ,d-card16)
		   (bell-duration      . ,d-card16)
		   (unused             . ,d-card16)
		   (auto-repeats       . ,(d-list d-card8 32)) )
		 scr) ))

;;; Request #104: Bell
(define (send-Bell percent scr . rest)
  (scr 'scix-xas `((,a-request . ,Bell)
		   (,a-int8    . ,percent)
		   (,a-card16  . 1) )			; Request length
       rest))

;;; Request #105: ChangePointerControl
(define (send-ChangePointerControl acc-num acc-den thr
				   do-acc do-thr scr . rest)
  (scr 'scix-xas `((,a-request . ,ChangePointerControl)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 3)			; Request length
		   (,a-int16   . ,acc-num)
		   (,a-int16   . ,acc-den)
		   (,a-int16   . ,thr)
		   (,a-bool    . ,do-acc)
		   (,a-bool    . ,do-thr) )
       rest))

;;; Request #106: GetPointerControl
(define (send-GetPointerControl scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetPointerControl)
				 (,a-card8   . 0)        ; 1 unused byte
				 (,a-card16  . 1) )      ; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name              . ,(d-identity 'GetPointerControl))
		   (unused                   . ,(d-unused 1))
		   (seq-nr                   . ,d-card16)
		   (unused                   . ,(d-unused 4)) ; Reply len = 0
		   (acceleration-numerator   . ,d-card16)
		   (acceleration-denominator . ,d-card16)
		   (threshold                . ,d-card16)
		   (unused                   . ,(d-unused 18)) )
		 scr) ))

;;; Request #107: SetScreenSaver
(define (send-SetScreenSaver timeout interval prefer-blanking
			     allow-exposures scr . rest)
  (let ((alist '((No . 0) (Yes . 1) (Default . 2))))
    (scr 'scix-xas `((,a-request . ,SetScreenSaver)
		     (,a-card8   . 0)			; 1 unused byte
		     (,a-card16  . 3)			; Request length
		     (,a-int16   . ,timeout)
		     (,a-int16   . ,interval)
		     (,a-card8   . ,(lookup-constant prefer-blanking alist))
		     (,a-card8   . ,(lookup-constant allow-exposures alist))
		     (,a-card16  . 0) )    		; 2 unused bytes
	 rest)))

;;; Request #108: GetScreenSaver
(define (send-GetScreenSaver scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetScreenSaver)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 1) )	; request length
		     rest)))
    (msg-handler 'scix-wait-for-reply 
		 seq-nr 
		 `((reply-name      . ,(d-identity 'GetScreenSaver))
		   (unused          . ,d-card8)
		   (seq-nr          . ,d-card16)
		   (unused          . ,d-card32) ; Reply length = 0
		   (timeout         . ,d-card16)
		   (interval        . ,d-card16)
		   (prefer-blanking . ,(d-const #t d-card8 '((0 . No)
							     (1 . Yes) )))
		   (allow-exposures . ,(d-const #t d-card8 '((0 . No)
							     (1 . Yes) )))
		   (unused          . ,(d-unused 18)) )
		 scr) ))

;;; Request #109: ChangeHosts
;;; Note: This looks ugly, but there is no better place for it. Maybe bugs left
(define (send-ChangeHosts mode family addr scr . rest)
  (let* ((addr (if (eq? family 'DECnet)
		   (list (bit-and (cadr addr) 255) ; Mask off LSByte of node-nr
			 (bit-or (bit-rsh (car addr) 2) ; Area part of byte2
				 (bit-rsh (cadr addr) 6) ))
		   addr))
	 (addr-len (length addr)) )
    (scr 'scix-xas `((,a-request . ,ChangeHosts)
		     (,a-card8   . ,(lookup-constant mode '((Insert . 0)
							    (Delete . 1) )))
		     (,a-card16  . ,(+ 2 (/ (+ addr-len (pad addr-len)) 4)))
		     (,a-card8   . ,(lookup-constant family '((Internet . 0)
							      (DECnet   . 1)
							      (Chaos    . 2))))
		     (,a-card8   . 0)		; 1 unused byte
		     (,a-card16  . ,addr-len)
		     (,(a-list a-card8) . ,addr) )
	 rest)))

;;; Request #110: ListHosts
;;; Note: some kind of trouble with DECnet addresses
(define (send-ListHosts scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,ListHosts)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 1) )	; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name  . ,(d-identity 'ListHosts))
		   (mode        . ,(d-const #t d-card8 '((0 . Disabled)
							 (1 . Enabled) )))
		   (seq-nr      . ,d-card16)
		   (unused      . ,d-card32)
		   (nr-of-hosts . ,d-card16)
		   (unused      . ,(d-unused 22))
		   (hosts       . ,(lambda (str dpy)
				     ((d-list d-host (backtrack 'nr-of-hosts))
				      str dpy) )))
		 scr) ))

;;; Request #111: SetAccessControl
(define (send-SetAccessControl mode scr . rest)
  (let ((md (lookup-constant mode '((Disable . 0)
				    (Enable  . 1)))))
    (scr 'scix-xas `((,a-request . ,SetAccessControl)
		     (,a-card8   . ,md)
		     (,a-card16  . 1) )		; Request length
	 rest)))

;;; Request #112: SetCloseDownMode
(define (send-SetCloseDownMode mode scr . rest)
  (let ((md (lookup-constant mode '((Destroy         . 0)
				    (RetainPermanent . 1)
				    (RetainTemporary . 2) ))))
    (scr 'scix-xas `((,a-request . ,SetCloseDownMode)
		     (,a-card8   . ,md)
		     (,a-card16  . 1) )		; Request length
	 rest)))

;;; Request #113: KillClient
;;; Note: We use the type window here for the resource, as there is no generic
;;;       type for all resources. It is not possible to use a-card32, because
;;;       it does not now about objects.
(define (send-KillClient resource scr . rest)
  (scr 'scix-xas `((,a-request  . ,KillClient)
		   (,a-card8    . 0)			; 1 unused byte
		   (,a-card16   . 2)			; Request length
		   (,a-resource . ,(lookup-constant resource
						    '((AllTemporary . 0)))) )
       rest))

;;; Request #114: RotateProperties
(define (send-RotateProperties w delta properties scr . rest)
  (let* ((atombox (scr 'atombox))
	 (properties (map (lambda (p)
			    (atombox 'lookup-id p) )
			  properties) ))
    (scr 'scix-xas `((,a-request       . ,RotateProperties)
		     (,a-card8         . 0)		      ; 1 unused byte
		     (,a-card16        . ,(+ 3 (length properties))) ; Req len
		     (,a-window        . ,w)
		     (,a-card16        . ,(length properties))
		     (,a-int16         . ,delta)
		     (,(a-list a-atom) . ,properties) )
	 rest)))

;;; Request #115: ForceScreenSaver
(define (send-ForceScreenSaver mode scr . rest)
  (let ((onoff (lookup-constant mode '((Reset    . 0)
				       (Activate . 1)))))
    (scr 'scix-xas `((,a-request . ,ForceScreenSaver)
		     (,a-card8   . ,onoff)
		     (,a-card16  . 1) )
	 rest)))

;;; Request #116: SetPointerMapping
(define (send-SetPointerMapping p-map scr . rest)
  (let* ((n (length p-map))
	 (seq-nr (scr 'scix-xas `((,a-request . ,SetPointerMapping)
				  (,a-card8   . ,n)
				  (,a-card16  . ,(+ 1 (/ (+ n (pad n)) 4)))
				  (,(a-list a-card8) . ,(list-pad4 p-map)) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'SetPointerMapping))
		   (status     . ,(d-const #t d-card8 '((0 . Success)
							(1 . Busy) )))
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 28)) ) ; Reply length = 0
		 scr) ))

;;; Request #117: GetPointerMapping
(define (send-GetPointerMapping scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetPointerMapping)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 1) )    	; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply 
		 seq-nr
		 `((reply-name    . ,(d-identity 'GetPointerMapping))
		   (length-of-map . ,d-card8)
		   (seq-nr        . ,d-card16)
		   (reply-len     . ,d-card32)
		   (unused        . ,(d-unused 24))
		   (pmap          . ,(lambda (str dpy)
				       ((d-list d-card8 
						(backtrack 'length-of-map))
					str dpy)))
		   (unused        . ,(lambda (str dpy)
				       ((d-unused (pad (backtrack
							'length-of-map)))
					str dpy))))
		 scr) ))

;;; Request #118: SetModifierMapping
;;; Note: keycodes is a list of lists, one for each modifier.
(define (send-SetModifierMapping keycodes scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,SetModifierMapping)
				 (,a-card8   . ,(length (car keycodes)))
				 (,a-card16  . ,(+ 1 (* 2 (length
							   (car keycodes)))))
				 (,(a-list
				    (lambda (keycode-list buffer)
				      (for-each (lambda (keycode)
						  (a-keycode keycode buffer))
						keycode-list))) . ,keycodes) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'SetModifierMapping))
		   (status     . ,(d-const #t d-card8 '((0 . Success)
							(1 . Busy)
							(2 . Failed) )))
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 28)) ) ; Reply length = 0
		 scr) ))

;;; Request #119: GetModifierMapping
(define (send-GetModifierMapping scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetModifierMapping)
				 (,a-card8   . 0)       ; 1 unused byte
				 (,a-card16  . 1) )     ; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name            . ,(d-identity 'GetModifierMapping))
		   (keycodes-per-modifier . ,d-card8)
		   (seq-nr                . ,d-card16)
		   (unused                . ,(d-unused 28)) ; Repl len not used
		   (keycodes              . ,(lambda (str dpy)
					       ((d-list
						 d-keycode
						 (* 8 
						    (backtrack
						     'keycodes-per-modifier)))
						str dpy))))
		 scr) ))

;;; Request #127: NoOperation
(define (send-NoOperation scr . rest)
  (scr 'scix-xas `((,a-request . ,NoOperation)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 1) )  			; Request length
       rest))

