; "dynamic.scm", DYNAMIC data type for Scheme
; Copyright (c) 1992, Andrew Wilcox

;(MAKE-DYNAMIC obj)                                     [Procedure]
;
;Create and return a new ``dynamic'' whose global value is obj.
;
;(DYNAMIC? obj)                                         [Procedure]
;
;Return true if and only if obj is a dynamic.  No object satisfying DYNAMIC?
;satisfies any of the other standard type predicates.
;
;(DYNAMIC-REF dyn)                                      [Procedure]
;
;Return the value of the given dynamic in the current dynamic environment.
;
;(DYNAMIC-SET! dyn obj)                                 [Procedure]
;
;Change the value of the given dynamic to obj in the current dynamic
;environment.  The returned value is unspecified.
;
;(CALL-WITH-DYNAMIC-BINDING dyn obj thunk)              [Procedure]
;
;Invoke and return the value of the given thunk in a new, nested dynamic
;environment in which the given dynamic has been bound to a new location
;whose initial contents are the value obj.  This dynamic environment has
;precisely the same extent as the invocation of the thunk and is thus
;captured by continuations created within that invocation and re-established
;by those continuations when they are invoked.
;
;
;There was also a DYNAMIC-BIND macro which I haven't implemented.

(require 'record)
(require 'dynamic-wind)

(define dynamic-environment-rtd
  (make-record-type "dynamic environment" '(dynamic value parent)))
(define make-dynamic-environment
  (record-constructor dynamic-environment-rtd))
(define dynamic-environment:dynamic
  (record-accessor dynamic-environment-rtd 'dynamic))
(define dynamic-environment:value
  (record-accessor dynamic-environment-rtd 'value))
(define dynamic-environment:set-value!
  (record-modifier dynamic-environment-rtd 'value))
(define dynamic-environment:parent
  (record-accessor dynamic-environment-rtd 'parent))

(define *current-dynamic-environment* #f)
(define (extend-current-dynamic-environment dynamic obj)
  (set! *current-dynamic-environment*
	(make-dynamic-environment dynamic obj
				  *current-dynamic-environment*)))

(define dynamic-rtd (make-record-type "dynamic" '()))
(define make-dynamic
  (let ((dynamic-constructor (record-constructor dynamic-rtd)))
    (lambda (obj)
      (let ((dynamic (dynamic-constructor)))
	(extend-current-dynamic-environment dynamic obj)
	dynamic))))

(define dynamic? (record-predicate dynamic-rtd))
(define (guarantee-dynamic dynamic)
  (or (dynamic? dynamic)
      (slib:error "Not a dynamic" dynamic)))

(define dynamic:errmsg
  "No value defined for this dynamic in the current dynamic environment")

(define (dynamic-ref dynamic)
  (guarantee-dynamic dynamic)
  (let loop ((env *current-dynamic-environment*))
    (cond ((not env)
	   (error dynamic:errmsg dynamic))
	  ((eq? (dynamic-environment:dynamic env) dynamic)
	   (dynamic-environment:value env))
	  (else
	   (loop (dynamic-environment:parent env))))))

(define (dynamic-set! dynamic obj)
  (guarantee-dynamic dynamic)
  (let loop ((env *current-dynamic-environment*))
    (cond ((not env)
	   (error dynamic:errmsg dynamic))
	  ((eq? (dynamic-environment:dynamic env) dynamic)
	   (dynamic-environment:set-value! env obj))
	  (else
	   (loop (dynamic-environment:parent env))))))

(define (call-with-dynamic-binding dynamic obj thunk)
  (let ((out-thunk-env #f)
	(in-thunk-env (make-dynamic-environment
		       dynamic obj
		       *current-dynamic-environment*)))
    (dynamic-wind (lambda ()
		    (set! out-thunk-env *current-dynamic-environment*)
		    (set! *current-dynamic-environment* in-thunk-env))
		  thunk
		  (lambda ()
		    (set! in-thunk-env *current-dynamic-environment*)
		    (set! *current-dynamic-environment* out-thunk-env)))))
