;;;"alist.scm", alist functions for Scheme.
;;;Copyright (c) 1992, Aubrey Jaffer

;Alist functions provide utilities for treating a list of key-value
;pairs as an associative database.  These functions take an equality
;predicate, pred, as an argument.  This predicate should be
;repeatable, symmetric, and transitive.

;Alist functions can be used with a secondary index method like hash
;tables for improved performance.

;  (predicate->asso pred)				procedure

;Returns an association function (like ASSQ, ASSV, or ASSOC)
;corresponding to pred.  The returned function returns a key-value
;pair whose key is pred equal to its first argument or #f if no key in
;the alist is pred equal to the first argument.

;  (alist-inquirer pred)				procedure

;Returns a procedure of 2 arguments, alist and key, which returns the
;value associated with key in alist or #f if key does not appear in
;alist.

;  (alist-associator pred)				procedure

;Returns a procedure of 3 arguments, alist, key, and value, which
;returns an alist with key and value associated.  Any previous value
;associated with key will be lost.  This returned procedure may or may
;not have side effects on its alist argument.  An example of correct
;usage is:

;(define put (alist-associator string-ci=?))
;(define alist '())
;(set! alist (put alist "Foo" 9))

;  (alist-remover pred)					procedure

;Returns a procedure of 2 arguments, alist and key, which returns an
;alist with an association whose key is key removed. This returned
;procedure may or may not have side effects on its alist argument.  An
;example of correct usage is:

;(define rem (alist-remover string-ci=?))
;(set! alist (rem alist "fOO"))

(define (predicate->asso pred)
  (cond ((eq? eq? pred) assq)
	((eq? eqv? pred) assv)
	((eq? equal? pred) assoc)
	(else (lambda (key alist)
		(let l ((al alist))
		  (cond ((null? al) #f)
			((pred key (caar al)) (car al))
			(else (l (cdr al)))))))))

(define (alist-inquirer pred)
  (let ((assofun (predicate->asso pred)))
    (lambda (alist key)
      (let ((pair (assofun key alist)))
	(and pair (cdr pair))))))

(define (alist-associator pred)
  (let ((assofun (predicate->asso pred)))
    (lambda (alist key val)
      (let* ((pair (assofun key alist)))
	(cond (pair (set-cdr! pair val)
		    alist)
	      (else (cons (cons key val) alist)))))))

(define (alist-remover pred)
  (lambda (alist key)
    (cond ((null? alist) alist)
	  ((pred key (caar alist)) (cdr alist))
	  ((null? (cdr alist)) alist)
	  ((pred key (caadr alist))
	   (set-cdr! alist (cddr alist)) alist)
	  (else
	   (let l ((al (cdr alist)))
	     (cond ((null? (cdr al)) alist)
		   ((pred key (caadr al))
		    (set-cdr! al (cddr al)) alist)
		   (else (l (cdr al)))))))))
