;;; -*- Scheme -*-
;;;; sc-macro.scm: Chris Hanson's Syntactic Closures macro implementation.

;  (macro:expand <expression>)				procedure

;Returns scheme code with the macros and derived expression types of
;<expression> expanded to primitive expression types.

;  (macro:eval <expression>)				procedure

;Returns the value of <expression> in the current top level
;environment.  <expression> can contain macro definitions.  Side
;effects of <expression> will effect the top level environment.

;  (macro:load <filename>)				procedure

;Filename should be a string.  If filename names an existing file, the
;macro:load procedure reads Scheme source code expressions and
;definintions from the file and evaluates them sequentially.  These
;source code expressions and definitions may contain macro
;definitions.  The macro:load procedure does not affect the values
;returned by current-input-port and current-output-port.

;;;;--------------------------------------------------------------
;;;; Syntaxer Output Interface

(define syntax-error slib:error)

(define impl-error slib:error)

(define (append-map procedure . lists)
  (apply append (apply map (cons procedure lists))))

(define *counter* 0)

(define (make-name-generator)
  (let ((suffix-promise
	 (make-promise
	  (lambda ()
	    (string-append "."
			   (number->string (begin
					     (set! *counter* (+ *counter* 1))
					     *counter*)))))))
    (lambda (identifier)
      (string->symbol
       (string-append "."
		      (symbol->string (identifier->symbol identifier))
		      (promise:force suffix-promise))))))

(define (output/variable name)
  name)

(define (output/constant datum)
  `(QUOTE ,datum))

(define (output/assignment name value)
  `(SET! ,name ,value))

(define (output/top-level-definition name value)
  `(DEFINE ,name ,value))

(define (output/conditional predicate consequent alternative)
  `(IF ,predicate ,consequent ,alternative))

(define (output/sequence expressions)
  (if (null? (cdr expressions))
      (car expressions)
      `(BEGIN ,@expressions)))

(define (output/combination operator operands)
  `(,operator ,@operands))

(define (output/lambda pattern body)
  `(LAMBDA ,pattern ,body))

(define (output/delay expression)
  `(DELAY ,expression))

(define (output/unassigned)
  `'*UNASSIGNED*)

(define (output/unspecific)
  `'*UNSPECIFIC*)

(require 'promise)			; Portable support for force and delay.
(require 'record)
(require 'eval)
(require 'synchk)			; Syntax checker.

;;; This file is the macro expander proper.
(load (in-vicinity (library-vicinity) "synclo" (scheme-file-suffix)))

;;; These files define the R4RS syntactic environment.
(load (in-vicinity (library-vicinity) "r4rsyn" (scheme-file-suffix)))
(load (in-vicinity (library-vicinity) "synrul" (scheme-file-suffix)))

;;; OK, time to build the databases.
(initialize-scheme-syntactic-environment!)

;;; MACRO:EXPAND is for you to use.  It takes an R4RS expression, macro-expands
;;; it, and returns the result of the macro expansion.
(define (macro:expand expression)
  (set! *counter* 0)
  (compile/top-level (list expression) scheme-syntactic-environment))

;;; Here are EVAL and LOAD which expand macros.  You can replace the
;;; implementation's eval and load with them if you like.
(define base:eval slib:eval)
(define base:load load)

(define (macro:eval x)
  (base:eval (macro:expand x)))

(define (macro:load <pathname>)
  (call-with-input-file <pathname>
    (lambda (port)
      (do ((o (read port) (read port)))
	  ((eof-object? o))
	(macro:eval o)))))

(provide 'macro)			;Here because we may have
					;(require 'sc-macro)
