/* Scheme implementation intended for JACAL.
   Copyright (C) 1990, 1991 Aubrey Jaffer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

The author can be reached at jaffer@ai.mit.edu or
Aubrey Jaffer, 84 Pleasant St., Wakefield MA 01880
*/

#include "scm.h"
#ifdef FLOATS
#include <math.h>

static char s_makrect[]="make-rectangular",s_makpolar[]="make-polar",
	    s_magnitude[]="magnitude",s_angle[]="angle",
	    s_imag_part[]="imag-part",s_in2ex[]="inexact->exact";
#endif
static char s_max[]="max",s_min[]="min";
static char s_sum[]="+",s_difference[]="-",s_product[]="*",s_divide[]="/";

static char s_list_tail[]="list-tail";
static char s_str2list[]="string->list";
static char s_st_copy[]="string-copy", s_st_fill[]="string-fill!";
static char s_vect2list[]="vector->list", s_ve_fill[]="vector-fill!";

SCM list_tail(lst,k)
SCM lst, k;
{
	register long i;
	ASSERT(INUMP(k),k,ARG2,s_list_tail);
	i = INUM(k);
	while (i-- > 0) {
		ASSERT(NIMP(lst) && CONSP(lst),lst,ARG1,s_list_tail);
		lst=CDR(lst);
	}
	return lst;
}
#ifdef STR_EXTENSIONS
static char s_subml[] = "substring-move-left!";
SCM subml(str1, start1, args)
     SCM str1, start1, args;
{
  SCM end1, str2, start2;
  long i,j,e;
  ASSERT(3 == ilength(args),args,WNA,s_subml);
  end1 = CAR(args); args = CDR(args);
  str2 = CAR(args); args = CDR(args);
  start2 = CAR(args);
  ASSERT(NIMP(str1) && STRINGP(str1),str1,ARG1,s_subml);
  ASSERT(INUMP(start1),start1,ARG2,s_subml);
  ASSERT(INUMP(end1),end1,ARG3,s_subml);
  ASSERT(NIMP(str2) && STRINGP(str2),str2,ARG4,s_subml);  
  ASSERT(INUMP(start2),start2,ARG5,s_subml);
  i=INUM(start1),j=INUM(start2),e=INUM(end1);
  ASSERT(i <= LENGTH(str1) && i >= 0,start1,OUTOFRANGE,s_subml);
  ASSERT(j <= LENGTH(str2) && j >= 0,start2,OUTOFRANGE,s_subml);
  ASSERT(e <= LENGTH(str1) && e >= 0,end1,OUTOFRANGE,s_subml);
  ASSERT(e-i+j <= LENGTH(str2), start2,OUTOFRANGE,s_subml);
  while(i<e) CHARS(str2)[j++]=CHARS(str1)[i++];
  return UNSPECIFIED;
}
static char s_submr[] = "substring-move-right!";
SCM submr(str1, start1, args)
     SCM str1, start1, args;
{
  SCM end1, str2, start2;
  long i,j,e;
  ASSERT(3 == ilength(args),args,WNA,s_subml);
  end1 = CAR(args); args = CDR(args);
  str2 = CAR(args); args = CDR(args);
  start2 = CAR(args);
  ASSERT(NIMP(str1) && STRINGP(str1),str1,ARG1,s_subml);
  ASSERT(INUMP(start1),start1,ARG2,s_subml);
  ASSERT(INUMP(end1),end1,ARG3,s_subml);
  ASSERT(NIMP(str2) && STRINGP(str2),str2,ARG4,s_subml);  
  ASSERT(INUMP(start2),start2,ARG5,s_subml);
  i=INUM(start1),j=INUM(start2),e=INUM(end1);
  ASSERT(i <= LENGTH(str1) && i >= 0,start1,OUTOFRANGE,s_subml);
  ASSERT(j <= LENGTH(str2) && j >= 0,start2,OUTOFRANGE,s_subml);
  ASSERT(e <= LENGTH(str1) && e >= 0,end1,OUTOFRANGE,s_subml);
  ASSERT((j=e-i+j) <= LENGTH(str2), start2,OUTOFRANGE,s_subml);
  while(i<e) CHARS(str2)[--j]=CHARS(str1)[--e];
  return UNSPECIFIED;
}
#endif
SCM string2list(str)
SCM str;
{
	long i;
	SCM res = EOL;
	unsigned char *src;
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_str2list);
	src = (unsigned char *)CHARS(str);
	for(i=LENGTH(str)-1;i>=0;i--) res = cons(MAKICHR(src[i]),res);
	return res;
}
SCM string_copy(str)
SCM str;
{
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_st_copy);
	return makfromstr(CHARS(str),(size_t)LENGTH(str));
}
SCM string_fill(str,chr)
SCM str,chr;
{
	register char *dst,c;
	register long k;
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_st_fill);
	ASSERT(ICHRP(chr),chr,ARG2,s_st_fill);
	c = ICHR(chr);
	dst = CHARS(str);
	for(k=LENGTH(str)-1;k>=0;k--) dst[k] = c;
	return UNSPECIFIED;
}
SCM vector2list(v)
SCM v;
{
	SCM res = EOL;
	long i;
	SCM *data;
	ASSERT(NIMP(v) && VECTORP(v),v,ARG1,s_vect2list);
	data=VELTS(v);
	for(i=LENGTH(v)-1;i>=0;i--) res = cons(data[i],res);
	return res;
}
SCM vector_fill(v,fill)
SCM v,fill;
{
	register long i;
	register SCM *data;
	ASSERT(NIMP(v) && VECTORP(v),v,ARG1,s_ve_fill);
	data = VELTS(v);
	for(i=LENGTH(v)-1;i>=0;i--) data[i] = fill;
	return UNSPECIFIED;
}

#ifdef FLOATS
#define NUMBERP(x) (INUMP(x) || (NIMP(x) && INEXP(x)))
#else
#define NUMBERP(x) INUMP(x)
#endif

SCM max(x,y)
SCM x,y;
{
  if UNBNDP(y) {
    ASSERT(NUMBERP(x),x,ARG1,s_max);
    return x;
  }
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && REALP(x),x,ARG1,s_max);
    if UNBNDP(y) return x;
    ASSERT(NIMP(y) && REALP(y),x,ARG2,s_max);
    return (REALPART(x) < REALPART(y)) ? y : x;
  }
#else
  ASSERT(INUMP(x),x,ARG1,s_max);
#endif
  ASSERT(INUMP(y),y,ARG2,s_max);
  return ((long)x < (long)y) ? y : x;
}

SCM min(x,y)
SCM x,y;
{
  if UNBNDP(y) {
    ASSERT(NUMBERP(x),x,ARG1,s_min);
    return x;
  }
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && REALP(x),x,ARG1,s_min);
    if UNBNDP(y) return x;
    ASSERT(NIMP(y) && REALP(y),x,ARG2,s_min);
    return (REALPART(x) > REALPART(y)) ? y : x;
  }
#else
  ASSERT(INUMP(x),x,ARG1,s_min);
#endif
  ASSERT(INUMP(y),y,ARG2,s_min);
  return ((long)x > (long)y) ? y : x;
}
SCM sum(x,y)
     SCM x,y;
{
  if UNBNDP(y) {
    if UNBNDP(x) return INUM0;
    ASSERT(NUMBERP(x),x,ARG1,s_sum);
    return x;
  }
#ifdef FLOATS
  if NINUMP(x) {
    double i=0.0;
    ASSERT(NIMP(x) && INEXP(x),x,ARG1,s_sum);
    ASSERT(NIMP(y) && INEXP(y),x,ARG2,s_sum);
    if CPLXP(x) i = IMAG(x);
    if CPLXP(y) i += IMAG(y);
#ifdef SINGLES
    if (!SINGP(x))
      return makdbl(REAL(x) + REALPART(y),i);
    else if SINGP(y) return makflo(FLO(x)+FLO(y));
    else return makdbl(FLO(x) + REAL(y),i);
#else
    return makdbl(REAL(x) + REAL(y),i);
#endif /*SINGLES*/
  }
#else
    ASSERT(INUMP(x),x,ARG1,s_sum);
#endif
    ASSERT(INUMP(y),y,ARG2,s_sum);
  {
    long z;
    z = INUM(x)+INUM(y);
    y = MAKINUM(z);
    ASSERT(INUM(y) == z,y,OVFLOW,s_sum);
    return y;
  }
}
SCM difference(x,y)
SCM x,y;
{
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && INEXP(x),x,ARG1,s_difference);
    if UNBNDP(y)
      if SINGP(x) return makflo(-FLO(x));
      else
	return makdbl(-REAL(x),CPLXP(x)?-IMAG(x):0.0);
    ASSERT(NIMP(y) && INEXP(y),x,ARG2,s_difference);
#ifdef SINGLES
    if SINGP(x)
      if SINGP(y) return makflo(FLO(x)-FLO(y));
      else
	return makdbl(FLO(x)-REAL(y),CPLXP(y)?-IMAG(y):0.0);
    if SINGP(y)
      return makdbl(REAL(x)-FLO(y),CPLXP(x)?IMAG(x):0.0);
#endif
    if CPLXP(x)
      if CPLXP(y)
	return makdbl(REAL(x)-REAL(y), IMAG(x)-IMAG(y));
      else
	return makdbl(REAL(x)-REAL(y), IMAG(x));
    return makdbl(REAL(x)-REAL(y), CPLXP(y)?-IMAG(y):0.0);
  }
#else
  ASSERT(INUMP(x),x,ARG1,s_difference);
#endif
  if UNBNDP(y) {
    y = x;
    x = INUM0;
  }
  else ASSERT(INUMP(y),y,ARG2,s_difference);
  x = INUM(x)-INUM(y);
  y = MAKINUM(x);
  ASSERT(INUM(y) == x,y,OVFLOW,s_difference);
  return y;
}

SCM product(x,y)
     SCM x,y;
{
  if UNBNDP(y) {
    if UNBNDP(x) return MAKINUM(1L);
    ASSERT(NUMBERP(x),x,ARG1,s_product);
    return x;
  }
#ifdef FLOATS
  if (NINUMP(x)) {
    ASSERT(NIMP(x) && INEXP(x),x,ARG1,s_product);
    ASSERT(NIMP(y) && INEXP(y),x,ARG2,s_product);
#ifdef SINGLES
    if SINGP(x)
      if SINGP(y) return makflo(FLO(x)*FLO(y));
      else {SCM t=x;x=y;y=t;goto fp1;}
    if SINGP(y)
    fp1:
    return makdbl(FLO(y)*REAL(x),CPLXP(x)?IMAG(x)*FLO(y):0.0);
#endif
    if CPLXP(x)
      if CPLXP(y)
	return makdbl(REAL(x)*REAL(y)-IMAG(x)*IMAG(y),
		      REAL(x)*IMAG(y)+IMAG(x)*REAL(y));
      else 
	return makdbl(REAL(x)*REAL(y), IMAG(x)*REAL(y));
    return makdbl(REAL(x)*REAL(y), CPLXP(y)?REAL(x)*IMAG(y):0.0);
  }
#else
  ASSERT(INUMP(x),x,ARG2,s_product);
#endif
  ASSERT(INUMP(y),y,ARG1,s_product);
  {
    long i, j, k;
    i = INUM(x);
    if (0 == i) return INUM0;
    j = INUM(y);
    k = i * j;
    y = MAKINUM(k);
    ASSERT((k == INUM(y)) && (k/i == j),y,OVFLOW,s_product);
    return y;
  }
}
SCM divide(x,y)
SCM x,y;
{
#ifdef FLOATS
  if NINUMP(x){
    double d;
    ASSERT(NIMP(x) && INEXP(x),x,ARG1,s_divide);
    if UNBNDP(y) {
      if SINGP(x) return makflo(1.0/FLO(x));
      if REALP(x) return makdbl(1.0/REAL(x),0.0);
      {
	double r=REAL(x),i=IMAG(x);
	d=r*r+i*i;
	return makdbl(r/d,-i/d);
      }
    }
    else ASSERT(NIMP(y) && INEXP(y),x,ARG2,s_divide);
    if CPLXP(y) {
      double r=REAL(y),i=IMAG(y),a=REALPART(x);
      d=r*r+i*i;
      if CPLXP(x)
	return makdbl((a*r+IMAG(x)*i)/d,(IMAG(x)*r-a*i)/d);
      return makdbl((a*r)/d,(-a*i)/d);
    }
#ifdef SINGLES
    if SINGP(x)
      if SINGP(y) return makflo(FLO(x)/FLO(y));
      else return makdbl(FLO(x)/REAL(y),0.0);
#endif
    d=REALPART(y);
    return makdbl(REAL(x)/d, CPLXP(x)?IMAG(x)/d:0.0);
  }
#else
  ASSERT(INUMP(x),x,ARG1,s_divide);
#endif
  if UNBNDP(y) {
    ASSERT(((x== MAKINUM(1L)) || (x== MAKINUM(-1L))),
	   x,OVFLOW,s_divide);
    return x;
  }
  ASSERT(INUMP(y),y,ARG2,s_divide);
  {
    long z;
    z = INUM(y);
    ASSERT(z && !(INUM(x)%z),y,OVFLOW,s_divide);
    z = INUM(x)/z;
    y = MAKINUM(z);
    ASSERT(INUM(y) == z,y,OVFLOW,s_divide);
    return y;
  }
}

#ifdef FLOATS
int dblprec=55;
#ifdef SINGLES
int floprec=25;
#endif

SCM istr2flo(str,len)
     char *str;
     long len;
{
  char *p = str;
  int c,j=0,prec=0,point= -999;
  double r=0.0,n=0.0;		/* r is real part; n is current part */
 lp:
  if (len > 1)
    if ((p[1]=='i') || (p[1]=='I')) {
      if (len!=2) return BOOL_F;
      switch (p[0]) {
      case '-': return makdbl(r,-1.0);
      case '+': return makdbl(r,1.0);
      default: return BOOL_F;
      }
    }
  if (len > 0)
    switch (p[0]) {
    case '-': case '+': j++;
    default:;
    }
  while(j < len)
    switch(c = p[j++]) {
    case '.':
      if (len == 1) return BOOL_F;
      if (point> -900) return BOOL_F;
      point = 0;
      continue;
    case 'e': case 'E': case 'd': case 'D': case 'l': case 'L': prec=2;
    case 's': case 'S': case 'f': case 'F':
      prec=1;
      {
	int exp=0,sgn=1;
	if ((len-j) > 1) switch (p[j]) {
	case '-': sgn= -1;
	case '+': j++;
	default:;
	}
	while(j < len) switch(c = p[j++]) {
	case DIGITS: exp = exp * 10 + c - '0'; continue;
	case '+': case '-': case 'i': case 'I': j--; goto out;
	default: return BOOL_F;
	}
      out:
	point=((point< -900)?0:point)-exp*sgn;
	if (point<0) for(;point;point++) n*=10.0;
      outq: continue;
      }
    case '/': {
      int exp=0;
      if (point >= -900) return BOOL_F;
      while(j < len) {
	switch(c = p[j++]) {
	case DIGITS: exp = exp * 10 + c - '0'; continue;
	default: return BOOL_F;
	case '+': case '-': case 'i': case 'I': j--;
	}
	break;
      }
      if (!exp) return BOOL_F;
      n /= exp;
      goto outq;
    }
    case '+': case '-':
      p += --j;len -= j;j = 0;
      if (point>0) while(point--) n/=10.0;
      r=n;n=0.0;point= -999;prec=3;
      goto lp;
    case 'i': case 'I':
      if (j!=len) return BOOL_F;
      if (point>0) while(point--) n/=10.0;
      return makdbl(r,(p[0]=='-')?-n:n);
    case '#': c = '0';
    case DIGITS:
      point++;
      n = n * 10.0 + c - '0';
      continue;
    default: return BOOL_F;
    }
  if (point>0) while(point--) n/=10.0;
  if (p[0]=='-') n = -n;
  switch (prec) {
#ifdef SINGLES
  case 0: if (n == (float) n)
  case 1: return makflo(n);
    else
#endif
  default:
  case 2: return makdbl(n,0.0);
  case 3: return BOOL_F;
  }
}
#ifdef ENGNOT
#define MANTRAD 1000.0
#define EXPINC 3
#else
#define MANTRAD 10.0
#define EXPINC 1
#endif
size_t idbl2str(f,prec,ec,str)
     double f;
     int prec;
     char ec,*str;
{
  register size_t i=1;
  register int exp=0,c;
  if (f == 0.0) {
    str[0]='0';
    str[i++]='.';
    str[i++]='0';
    return i;
  }
  str[0]=(f < 0.0)?(f= -f,'-'):'+';
  while(f >= MANTRAD) {exp++;f /= MANTRAD;}
  while(f < 1.0) {exp--;f *= MANTRAD;}
  c = floor(f);
#ifdef ENGNOT
  i += iint2str((long)c,10,&str[i]);
#else
  str[i++] = c+'0';
#endif
  str[i++] = '.';
  {
    double M =
      pow((double)FLORADIX,
	  -floor(prec - 1 - log(f)/log((double)FLORADIX)))/2;
    f -= c;
    do {
      f *= 10;
      c = floor(f);
      f -= c;
      M *= 10;
      str[i++] = c + '0';
/* printf("prec= %d f= %g c= %d M= %g i= %d\n",prec,f,c,M,i); */
    } while ((f >= M) && (f <= 1 - M));
    if (f >= .5) str[i-1]++;
  }
  if (exp) {
    str[i++] = ec;
    i += iint2str((long)exp*EXPINC,10,&str[i]);
  }
  return i;
}
size_t iflo2str(exp,str)
     SCM exp;
     char *str;
{
  size_t i;
#ifdef SINGLES
  if SINGP(exp)
    return idbl2str(FLO(exp),floprec,'s',str);
#endif
  i = idbl2str(REAL(exp),dblprec,'d',str);
  if CPLXP(exp) {
    i += idbl2str(IMAG(exp),dblprec,'d',&str[i]);
    str[i++] = 'i';
  }
  return i;
}

double truncate(x)
     double x;
{
  if (x<0.0) return -floor(-x);
  return floor(x);
}
double round(x)
     double x;
{
  return floor(x+0.5);
}
SCM expt(z1,z2)
     SCM z1,z2;
{
  return(makdbl(pow(REAL(z1),REAL(z2)),0.0));
}
SCM makrect(x,y)
     SCM x,y;
{
  ASSERT(NIMP(x) && REALP(x),x,ARG1,s_makrect);
  ASSERT(NIMP(y) && REALP(y),y,ARG2,s_makrect);
  return makdbl(REALPART(x), REALPART(y));
}
SCM makpolar(x,y)
     SCM x,y;
{
  double s,e;
  ASSERT(NIMP(x) && REALP(x),x,ARG1,s_makpolar);
  ASSERT(NIMP(y) && REALP(y),y,ARG2,s_makpolar);
  s=REALPART(x);
  e=REALPART(y);
  return makdbl(s*cos(e),s*sin(e));
}
SCM imag_part(z)
     SCM z;
{
  ASSERT(NIMP(z) && INEXP(z),z,ARG1,s_imag_part);
  if CPLXP(z) return makdbl(IMAG(z),0.0);
  return makflo(0.0);
}
SCM magnitude(z)
     SCM z;
{
  ASSERT(NIMP(z) && INEXP(z),z,ARG1,s_magnitude);
  if CPLXP(z)
    {
      double i=IMAG(z),r=REAL(z);
      return(makdbl(sqrt(i*i+r*r),0.0));
    }
  return z;
}
SCM angle(z)
     SCM z;
{
  ASSERT(NIMP(z) && INEXP(z),z,ARG1,s_angle);
  if CPLXP(z) return(makdbl(atan2(IMAG(z),REAL(z)),0.0));
  return makdbl(1.0,0.0);
}
double floident(z)
double z;
{
  return z;
}
SCM in2ex(z)
     SCM z;
{
  ASSERT(NIMP(z) && REALP(z),z,ARG1,s_in2ex);
  return MAKINUM((long)floor(REALPART(z)+0.5));
}
#endif /* FLOATS */

char s_getenv[]="getenv";
char s_system[]="system";

SCM lsystem(cmd)
SCM cmd;
{
	ASSERT(NIMP(cmd) && STRINGP(cmd),cmd,ARG1,s_system);
	return MAKINUM(system(CHARS(cmd)));
}
char *getenv();
SCM lgetenv(nam)
SCM nam;
{
	char *val;
	ASSERT(NIMP(nam) && STRINGP(nam),nam,ARG1,s_getenv);
	val = getenv(CHARS(nam));
	if (!val) return BOOL_F;
	return makfromstr(val, strlen(val));
}
SCM softtype()
{
#ifdef MSDOS
  return intern("msdos", sizeof "msdos" -1L);
#endif
#ifdef vms
  return intern("vms", sizeof "vms" -1L);
#endif
#ifdef unix
  return intern("unix", sizeof "unix" -1L);
#endif
}

#ifdef L_tmpdir
SCM ltmpnam()
{
  char *name = tmpnam(NULL);
  if (name)
    return makfromstr(name, strlen(name));
  return BOOL_F;
}
#else
SCM ltmpnam()
{
  SCM name = makfromstr("/usr/tmp/scmXXXXXX",(size_t)18);
  if (mktemp(CHARS(name))) return name;
  return BOOL_F;
}
#endif /* L_tmpdir */

#ifdef vms
#include <descrip.h>
#include <ssdef.h>
static char s_ed[]="ed";
SCM ed(fname)
SCM fname;
{
	struct dsc$descriptor_s d;
	ASSERT(NIMP(fname) && STRINGP(fname),fname,ARG1,s_ed);
	d.dsc$b_dtype = DSC$K_DTYPE_T;
	d.dsc$b_class = DSC$K_CLASS_S;
	d.dsc$w_length = LENGTH(fname);
	d.dsc$a_pointer = CHARS(fname);
	DEFER_SIGINT;
	edt$edit(&d);
	ALLOW_SIGINT;
	return(fname);
}
SCM vms_debug()
{
	lib$signal(SS$_DEBUG);
	return;
}
#endif

static iproc subr0s[]={
	{"tmpnam",ltmpnam},
	{"software-type",softtype},
#ifdef vms
	{"vms-debug",vms_debug},
#endif
	{0,0}};

static iproc subr1s[]={
#ifdef FLOATS
	{s_imag_part,imag_part},
	{s_magnitude,magnitude},
	{s_angle,angle},
	{s_in2ex,in2ex},
#endif
	{s_str2list,string2list},
	{"list->string",string},
	{s_st_copy,string_copy},
	{"list->vector",vector},
	{s_vect2list,vector2list},
	{s_system,lsystem},
	{s_getenv,lgetenv},
#ifdef vms
	{s_ed,ed},
#endif
	{0,0}};

static iproc asubrs[]={
	{s_max,max},
	{s_min,min},
	{s_sum,sum},
	{s_product,product},
	{0,0}};

static iproc subr2s[]={
#ifdef FLOATS
	{"expt",expt},
	{s_makrect,makrect},
	{s_makpolar,makpolar},
#endif
	{s_list_tail,list_tail},
#ifndef PURE_FUNCTIONAL
	{s_ve_fill,vector_fill},
	{s_st_fill,string_fill},
#endif
	{0,0}};

static iproc subr2os[]={
	{s_difference,difference},
	{s_divide,divide},
	{0,0}};

#ifdef STR_EXTENSIONS
static iproc lsubr2s[]={
  {s_subml,subml},
  {s_submr,submr},
  {0,0}};
#endif

#ifdef FLOATS
static dblproc cxrs[] = {
	{"floor",floor},
	{"ceiling",ceil},
	{"truncate",truncate},
	{"round",round},
	{"exp",exp},
	{"log",log},
	{"sin",sin},
	{"cos",cos},
	{"tan",tan},
	{"asin",asin},
	{"acos",acos},
	{"atan",atan},
	{"sqrt",sqrt},
	{"exact->inexact",floident},
	{0,0}};
#endif

void init_scl()
{
	VCELL(intern("char-code-limit",-15L))=MAKINUM(char_code_limit);
	VCELL(intern("most-positive-fixnum",-20L))
	  = MAKINUM(most_positive_fixnum);
	VCELL(intern("most-negative-fixnum",-20L))
	  = MAKINUM(most_negative_fixnum);
#ifdef FLOATS
	{
	  double d=1.0/FLORADIX;
	  dblprec = 1;
	  while (1+d != 1.0) {
	    d /= FLORADIX;
	    dblprec++;
	  }
	}
#ifdef SINGLES
	{
	  float f=1.0/FLORADIX;
	  floprec = 1;
	  while ((float)(1+f) != 1.0) {
	    f /= FLORADIX;
	    floprec++;
	  }
	}
#endif /* SINGLES */
  init_iprocs(cxrs, tc7_cxr);
#endif /* FLOATS */
  init_iprocs(subr0s, tc7_subr_0);
  init_iprocs(subr1s,tc7_subr_1);
  init_iprocs(subr2os,tc7_subr_2o);
  init_iprocs(subr2s,tc7_subr_2);
  init_iprocs(asubrs, tc7_asubr);
#ifdef STR_EXTENSIONS
  init_iprocs(lsubr2s,tc7_lsubr_2);
#endif
}
