/* Scheme implementation intended for JACAL.
   Copyright (C) 1989, 1990 Aubrey Jaffer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

The author can be reached at jaffer@ai.mit.edu or
Aubrey Jaffer, 84 Pleasant St., Wakefield MA 01880
*/

#include <ctype.h>
#include "scm.h"
 
SCM s_setcar, s_setcdr;
SCM s_length, s_append, s_reverse, s_list_ref;
SCM s_memq, s_member, s_assq, s_assoc;
 
SCM s_symbol2string, s_string2symbol;
 
SCM s_exactp, s_inexactp;
SCM s_zerop, s_positivep, s_negativep, s_oddp, s_evenp, s_max, s_min;
SCM s_eqp, s_lessp, s_greaterp, s_lesseqp, s_greatereqp;
SCM s_sum, s_difference, s_product, s_abs, s_quotient, s_remainder, s_modulo;
SCM s_gcd, s_lcm, s_number2string, s_string2number;
 
SCM s_char_lessp, s_char_ci_eq, s_char_ci_lessp;
SCM s_char_alphap, s_char_nump, s_char_whitep, s_char_upperp, s_char_lowerp;
SCM s_char2int, s_int2char, s_char_upcase, s_char_downcase;
 
SCM s_string, s_make_string, s_string_length, s_string_ref, s_string_set;
SCM s_string_equal, s_string_ci_equal, s_string_lessp, s_string_ci_lessp;
SCM s_substring, s_string_append;
 
extern SCM s_make_vector;
SCM s_vector, s_vector_length, s_vector_ref, s_vector_set;
 
SCM lnot(x)
SCM x;
{
	return FALSEP(x) ? BOOL_T : BOOL_F;
}
SCM booleanp(obj)
SCM obj;
{
	if (obj == BOOL_F) return BOOL_T;
	if (obj == BOOL_T) return BOOL_T;
	return BOOL_F;
}
SCM eq(x,y)
SCM x,y;
{
	if (x == y) return BOOL_T;
	else return BOOL_F;
}
 
SCM equal(), string_equal();
 
SCM vector_equal(x,y)
SCM x,y;
{
	register long i;
	for(i=LENGTH(x)-1;i>=0;i--)
		if FALSEP(equal(VELTS(x)[i],VELTS(y)[i])) return BOOL_F;
	return BOOL_T;
}
 
SCM equal(x,y)
SCM x,y;
{
tailrecurse:
	if (x == y) return BOOL_T;
	if IMP(x) return BOOL_F;
	if IMP(y) return BOOL_F;
	if (CONSP(x)&&CONSP(y)) {
		if (BOOL_F == equal(CAR(x),CAR(y))) return BOOL_F;
		x = CDR(x);
		y = CDR(y);
		goto tailrecurse;
	}
	if (TYP6(x) != TYP6(y)) return BOOL_F;
	if STRINGP(x) return (LENGTH(x) == LENGTH(y)) ? string_equal(x,y) : BOOL_F;
	if VECTORP(x) return (LENGTH(x) == LENGTH(y)) ? vector_equal(x,y) : BOOL_F;
	return BOOL_F;
}
 
SCM consp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return CONSP(x) ? BOOL_T : BOOL_F;
}
SCM cxr(x)
SCM x;
{
	err("cxr called",x);
}
SCM setcar(pair,value)
SCM pair, value;
{
	ASSERT(NIMP(pair)&&CONSP(pair),pair,ARG1,s_setcar);
	CAR(pair) = value;
	return UNSPECIFIED;
}
SCM setcdr(pair,value)
SCM pair, value;
{
	ASSERT(NIMP(pair)&&CONSP(pair),pair,ARG1,s_setcdr);
	CDR(pair) = value;
	return UNSPECIFIED;
}
SCM nullp(x)
SCM x;
{
	return NULLP(x) ? BOOL_T : BOOL_F;
}
long ilength(x)
SCM x;
{
	register long i=0;
	register SCM sx=x;
	do {
		if IMP(x) return NULLP(x) ? i : -1;
		if NCONSP(x) return -1;
		x = CDR(x);
		i++;
		if IMP(x) return NULLP(x) ? i : -1;
		if NCONSP(x) return -1;
		x = CDR(x);
		i++;
		sx=CDR(sx);
	}
	while (x != sx);
	return -1;
}
SCM listp(x)
SCM x;
{
	if (ilength(x)<0) return BOOL_F;
	else return BOOL_T;
}
SCM list(objs)
SCM objs;
{
	return objs;
}
SCM length(x)
SCM x;
{
	register SCM i=MAKINUM((long)ilength(x));
	ASSERT(i>=INUM0,i,ARG1,s_length);
	return i;
}
SCM append(args)
SCM args;
{
	SCM res = EOL;
	register SCM *lloc = &res, arg;
	if IMP(args) {
		ASSERT(NULLP(args),args,ARG1,s_append);
		return res;
		}
	ASSERT(CONSP(args),args,ARG1,s_append);
	while (1) {
		arg = CAR(args);
		args = CDR(args);
		if IMP(args) {
			*lloc = arg;
			ASSERT(NULLP(args),args,ARG1,s_append);
			return res;
		}
		ASSERT(CONSP(args),args,ARG1,s_append);
		for(;NIMP(arg);arg = CDR(arg)) {
			ASSERT(CONSP(arg),args,ARG1,s_append);
			*lloc = cons(CAR(arg),EOL);
			lloc = &CDR(*lloc);
		}
	}
}
SCM reverse(lst)
SCM lst;
{
	SCM res = EOL;
	register SCM p = lst;
	for(;NIMP(p);p = CDR(p)) {
		ASSERT(CONSP(p),lst,ARG1,s_reverse);
		res = cons(CAR(p),res);
	}
	ASSERT(NULLP(p),lst,ARG1,s_reverse);
	return res;
}
SCM list_ref(lst,k)
SCM lst, k;
{
	register long i;
	ASSERT(INUMP(k),k,ARG2,s_list_ref);
	i = INUM(k);
	while (i-- > 0) {
		ASSERT(NIMP(lst)&&CONSP(lst),lst,ARG1,s_list_ref);
		lst=CDR(lst);
	}
	ASSERT(NIMP(lst)&&CONSP(lst),lst,ARG1,s_list_ref);
	return CAR(lst);
}
SCM memq(x,lst)
SCM x,lst;
{
	for(;NIMP(lst);lst = CDR(lst)) {
		ASSERT(CONSP(lst),lst,ARG2,s_memq);
		if (CAR(lst) == x) return lst;
	}
	ASSERT(NULLP(lst),lst,ARG2,s_memq);
	return BOOL_F;
}
SCM member(x,lst)
SCM x,lst;
{
	for(;NIMP(lst);lst = CDR(lst)) {
		ASSERT(CONSP(lst),lst,ARG2,s_member);
		if (equal(CAR(lst),x) == BOOL_T) return lst;
	}
	ASSERT(NULLP(lst),lst,ARG2,s_member);
	return BOOL_F;
}
SCM assq(x,alist)
SCM x,alist;
{
	SCM tmp;
	for(;NIMP(alist);alist=CDR(alist)) {
		ASSERT(CONSP(alist),alist,ARG2,s_assq);
		tmp = CAR(alist);
		ASSERT(CONSP(tmp),alist,ARG2,s_assq);
		if (CAR(tmp) == x) return tmp;
	}
	ASSERT(NULLP(alist),alist,ARG2,s_assq);
	return BOOL_F;
}
SCM assoc(x,alist)
SCM x,alist;
{
	SCM tmp;
	for(;NIMP(alist);alist=CDR(alist)) {
		ASSERT(CONSP(alist),alist,ARG2,s_assoc);
		tmp = CAR(alist);
		ASSERT(CONSP(tmp),alist,ARG2,s_assoc);
		if (equal(CAR(tmp),x) == BOOL_T) return tmp;
	}
	ASSERT(NULLP(alist),alist,ARG2,s_assoc);
	return BOOL_F;
}
 
SCM symbolp(x)
SCM x;
{
	if ISYMP(x) return BOOL_T;
	if IMP(x) return BOOL_F;
	return SYMBOLP(x) ? BOOL_T : BOOL_F;
}
SCM symbol2string(s)
SCM s;
{
	if (!ISYMP(s)) {
		ASSERT(NIMP(s) && SYMBOLP(s),s,ARG1,s_symbol2string);
		return NAMESTR(s);
	}
	return makfromstr(ISYMCHARS(s), strlen(ISYMCHARS(s)));
}
SCM string2symbol(s)
SCM s;
{
	ASSERT(NIMP(s)&&STRINGP(s),s,ARG1,s_string2symbol);
	return intern(CHARS(s),LENGTH(s));
}
 
SCM numberp(x)
SCM x;
{
	return INUMP(x) ? BOOL_T : BOOL_F;
}
SCM exactp(x)
SCM x;
{
	ASSERT(INUMP(x),x,ARG1,s_exactp);
	return BOOL_T;
}
SCM inexactp(x)
SCM x;
{
	ASSERT(INUMP(x),x,ARG1,s_inexactp);
	return BOOL_F;
}
SCM numident(x)
SCM x;
{
	ASSERT(INUMP(x),x,"conversion to integer expected number",BOOL_F);
	return x;
}
SCM eqp(x,y,args)
SCM x,y,args;
{
	ASSERT(INUMP(x),x,ARG1,s_eqp);
	while (1) {
		ASSERT(INUMP(y),y,ARG2,s_eqp);
		if ((long)x != (long)y) return BOOL_F;
		if NULLP(args) return BOOL_T;
		y = CAR(args);
		args = CDR(args);
	}
}
SCM lessp(x,y,args)
SCM x,y,args;
{
	ASSERT(INUMP(x),x,ARG1,s_lessp);
	while (1) {
		ASSERT(INUMP(y),y,ARG2,s_lessp);
		if ((long)x >= (long)y) return BOOL_F;
		if NULLP(args) return BOOL_T;
		x = y;
		y = CAR(args);
		args = CDR(args);
	}
}
SCM greaterp(x,y,args)
SCM x,y,args;
{
	ASSERT(INUMP(x),x,ARG1,s_greaterp);
	while (1) {
		ASSERT(INUMP(y),y,ARG2,s_greaterp);
		if ((long)x <= (long)y) return BOOL_F;
		if NULLP(args) return BOOL_T;
		y = CAR(args);
		x = y;
		args = CDR(args);
	}
}
SCM lesseqp(x,y,args)
SCM x,y,args;
{
	ASSERT(INUMP(x),x,ARG1,s_lesseqp);
	while (1) {
		ASSERT(INUMP(y),y,ARG2,s_lesseqp);
		if ((long)x > (long)y) return BOOL_F;
		if NULLP(args) return BOOL_T;
		x = y;
		y = CAR(args);
		args = CDR(args);
	}
}
SCM greatereqp(x,y,args)
SCM x,y,args;
{
	ASSERT(INUMP(x),x,ARG1,s_greatereqp);
	while (1) {
		ASSERT(INUMP(y),y,ARG2,s_greatereqp);
		if ((long)x < (long)y) return BOOL_F;
		if NULLP(args) return BOOL_T;
		x = y;
		y = CAR(args);
		args = CDR(args);
	}
}
SCM zerop(z)
SCM z;
{
	ASSERT(INUMP(z),z,ARG1,s_zerop);
	return (z==INUM0) ? BOOL_T : BOOL_F;
}
SCM positivep(x)
SCM x;
{
	ASSERT(INUMP(x),x,ARG1,s_positivep);
	return (INUM(x)>0) ? BOOL_T : BOOL_F;
}
SCM negativep(x)
SCM x;
{
	ASSERT(INUMP(x),x,ARG1,s_negativep);
	return (0>INUM(x)) ? BOOL_T : BOOL_F;
}
SCM oddp(n)
SCM n;
{
	ASSERT(INUMP(n),n,ARG1,s_oddp);
	return ((long)n &4) ? BOOL_T : BOOL_F;
}
SCM evenp(n)
SCM n;
{
	ASSERT(INUMP(n),n,ARG1,s_evenp);
	return ((long)n &4) ? BOOL_F : BOOL_T;
}
SCM lmax(x,y)
SCM x,y;
{
	ASSERT(INUMP(x),x,ARG1,s_max);
	if UNBNDP(y) return x;
	ASSERT(INUMP(y),y,ARG2,s_max);
	return ((long)x<(long)y) ? y : x;
}
SCM lmin(x,y)
SCM x,y;
{
	ASSERT(INUMP(x),x,ARG1,s_min);
	if UNBNDP(y) return x;
	ASSERT(INUMP(y),y,ARG2,s_min);
	return ((long)x>(long)y) ? y : x;
}
SCM sum(x,y)
SCM x,y;
{
	register long z;
	register SCM res;
	if UNBNDP(y) {
		y = INUM0;
		if UNBNDP(x) return y;
	}
	else ASSERT(INUMP(y),y,ARG2,s_sum);
	ASSERT(INUMP(x),x,ARG1,s_sum);
	z = INUM(x)+INUM(y);
	res = MAKINUM(z);
	ASSERT(INUM(res) == z,res,OVERFLOW,s_sum);
	return res;
}
SCM product(n1,n2)
SCM n1,n2;
{
	register long x,y,z;
	register SCM res;
	if UNBNDP(n2) {
		n2 = MAKINUM(1L);
		if UNBNDP(n1) return n2;
	}
	else ASSERT(INUMP(n2),n2,ARG2,s_product);
	ASSERT(INUMP(n1),n1,ARG1,s_product);
	x = INUM(n1);
	if (0 == x) return INUM0;
	y = INUM(n2);
	z = x * y;
	res = MAKINUM(z);
	ASSERT((z == INUM(res)) && (z/x == y),res,OVERFLOW,s_product);
	return res;
}
SCM difference(x,y)
SCM x,y;
{
	register long z;
	register SCM res;
	if UNBNDP(y) {
/*		if UNBNDP(x) return INUM0; */
		y = x;
		x = INUM0;
	}
	else ASSERT(INUMP(x),x,ARG1,s_difference);
	ASSERT(INUMP(y),y,ARG2,s_difference);
	z = INUM(x)-INUM(y);
	res = MAKINUM(z);
	ASSERT(INUM(res) == z,res,OVERFLOW,s_difference);
	return res;
}
SCM absval(x)
SCM x;
{
	register SCM res;
	register long z = INUM(x);
	ASSERT(INUMP(x),x,ARG1,s_abs);
	if (z<0) z = -z;
	res = MAKINUM(z);
	ASSERT(res>>2 == z,res,OVERFLOW,s_abs);
	return res;
}
 
#if (((-1)%2 == -1) && ((-1)%(-2) == -1) && (1%2 == 1) && (1%(-2) == 1))
#else
#define BADIVSGNS
#endif
 
SCM quotient(x,y)
SCM x,y;
{
	register SCM res;
	register long z;
	ASSERT(INUMP(x),x,ARG1,s_quotient);
	ASSERT(INUMP(y),y,ARG2,s_quotient);
	z = INUM(y);
	ASSERT(z,y,OVERFLOW,s_quotient);
	z = INUM(x)/z;
#ifdef BADIVSGNS
	{
		long t = INUM(x)%INUM(y);
		if (t == 0) ;
		else if (t < 0)
			if (x < 0) ;
			else z--;
		else if (x < 0) z++;
	}
#endif
	res = MAKINUM(z);
	ASSERT(INUM(res) == z,res,OVERFLOW,s_quotient);
	return res;
}
SCM remainder(x,y)
SCM x,y;
{
	register long z;
	ASSERT(INUMP(x),x,ARG1,s_remainder);
	ASSERT(INUMP(y),y,ARG2,s_remainder);
	z = INUM(y);
	ASSERT(z,y,OVERFLOW,s_remainder);
	z = INUM(x)%z;
#ifdef BADIVSGNS
	if (z == 0) ;
	else if (z < 0)
		if (x < 0) ;
		else z += INUM(y);
	else if (x < 0) z -= INUM(y);
#endif
	return MAKINUM(z);
}
SCM modulo(n1,n2)
SCM n1,n2;
{
	register long y,z;
	ASSERT(INUMP(n1),n1,ARG1,s_modulo);
	ASSERT(INUMP(n2),n2,ARG2,s_modulo);
	y = INUM(n2);
	ASSERT(y,n2,OVERFLOW,s_modulo);
	z = INUM(n1)%y;
	return MAKINUM(y<0 ? (z>0) ? z+y : z
			   : (z<0) ? z+y : z);
}
SCM lgcd(n1,n2)
SCM n1,n2;
{
	register long u,v,k,t;
	if UNBNDP(n2) return UNBNDP(n1) ? INUM0 : n1;
	ASSERT(INUMP(n1),n1,ARG1,s_gcd);
	ASSERT(INUMP(n2),n2,ARG2,s_gcd);
	u = INUM(n1);
	if (u<0) u = -u;
	v = INUM(n2);
	if (v<0) v = -v;
	else if (0 == v) return MAKINUM(u);
	if (0 == u) return MAKINUM(v);
	for (k = 1;(!(u&1)) && (!(v&1));k <<= 1,u >>= 1,v >>= 1);
	if (u&1) {
		t = -v;
		goto b4;
	}
	else t = u;
b3:
	t >>= 1;
b4:
	if (!(t&1)) goto b3;
	if (t>0) u = t;
	else v = -t;
	t = u-v;
	if (t != 0) goto b3;
	u = u*k;
	v = MAKINUM(u);
	ASSERT((v>>2) == u,v,OVERFLOW,s_gcd);
	return v;
}
SCM llcm(n1,n2)
SCM n1,n2;
{
	register SCM res;
	register long q,z,x = INUM(n1);
	if UNBNDP(n2) {
		n2 = MAKINUM(1L);
		if UNBNDP(n1) return n2;
	}
	q = INUM(lgcd(n1,n2));
	if ((x == 0) || (n2 == INUM0)) return INUM0;
	q = INUM(n2)/q;
	z = x*q;
	ASSERT(z/q == x,n1,OVERFLOW,s_lcm);
	if (z < 0) z = -z;
	res = MAKINUM(z);
	ASSERT((res>>2) == z,res,OVERFLOW,s_lcm);
	return res;
}
SCM number2string(x,radix)
SCM x,radix;
{
	register char *p,d;
	register int i=1,rad;
	register long n;
	if UNBNDP(radix) radix=MAKINUM(10L);
	else ASSERT(INUMP(radix),radix,ARG2,s_number2string);
	ASSERT(INUMP(x),x,ARG1,s_number2string);
	rad = INUM(radix);
	n = INUM(x);
	if (n < 0) {
		n = -n;
		i++;
	}
	for (n /= rad;n > 0;n /= rad) i++;
	n = INUM(x);
	x = makstr((long)i);
	p = CHARS(x);
	if (n < 0) {
		n = -n;
		*p++ = '-';
		i--;
	}
	for (i--;i >= 0;i--) {
		d = n % rad;
		n /= rad;
		p[i] = d + ((d < 10) ? '0' : 'a');
	}
	return x;
}
SCM string2number(str,radix)
SCM str,radix;
{
	register char *p;
	char sgn = 0;
	register int c,rad,i = 0;
	register long n = 0;
	long len,res;
	if UNBNDP(radix) radix=MAKINUM(10L);
	else ASSERT(INUMP(radix),radix,ARG2,s_string2number);
	ASSERT(NIMP(str)&&STRINGP(str),str,ARG1,s_string2number);
	rad = INUM(radix);
	p = CHARS(str);
	len = LENGTH(str);
	if (p[0] == '-') {
		sgn = 1;
		i++;
	}
	while(i < len) switch(c = p[i++]) {
	case DIGITS:
		c = c - '0';
		goto accumulate;
	case 'A':
	case 'B':
	case 'C':
	case 'D':
	case 'E':
	case 'F':
		c = c-'A'+10;
		goto accumulate;
	case 'a':
	case 'b':
	case 'c':
	case 'd':
	case 'e':
	case 'f':
		c = c-'a'+10;
accumulate:
		if ((c<0)||(c>=rad)) return BOOL_F;
		res = n;
		n = n * rad - c;
		if ((n + c)/rad != res) return BOOL_F;
		continue;
	default:
		return BOOL_F;
	}
	if (!sgn) n = -n;
	res = MAKINUM(n);
	if (INUM(res) != n) return BOOL_F;
	return res;
}
 
SCM charp(x)
SCM x;
{
	return ICHRP(x) ? BOOL_T : BOOL_F;
}
SCM char_lessp(x,y)
SCM x,y;
{
	ASSERT(ICHRP(x),x,ARG1,s_char_lessp);
	ASSERT(ICHRP(y),y,ARG2,s_char_lessp);
	return (ICHR(x) < ICHR(y)) ? BOOL_T : BOOL_F;
}
SCM char_ci_eq(x,y)
SCM x,y;
{
	ASSERT(ICHRP(x),x,ARG1,s_char_ci_eq);
	ASSERT(ICHRP(y),y,ARG2,s_char_ci_eq);
	return (upcase[ICHR(x)] == upcase[ICHR(y)]) ? BOOL_T : BOOL_F;
}
SCM char_ci_lessp(x,y)
SCM x,y;
{
	ASSERT(ICHRP(x),x,ARG1,s_char_ci_lessp);
	ASSERT(ICHRP(y),y,ARG2,s_char_ci_lessp);
	return (upcase[ICHR(x)] < upcase[ICHR(y)]) ? BOOL_T : BOOL_F;
}
SCM char_alphap(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char_alphap);
	return (isalpha(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char_nump(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char_nump);
	return (isdigit(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char_whitep(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char_whitep);
	return (isspace(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char_upperp(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char_upperp);
	return (isupper(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char_lowerp(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char_lowerp);
	return (islower(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char2int(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char2int);
	return MAKINUM((long)ICHR(chr));
}
SCM int2char(n)
SCM n;
{
	ASSERT(INUMP(n),n,ARG1,s_int2char);
	return MAKICHR(INUM(n));
}
SCM char_upcase(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char_upcase);
	return MAKICHR(upcase[ICHR(chr)]);
}
SCM char_downcase(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char_downcase);
	return MAKICHR(downcase[ICHR(chr)]);
}
 
SCM stringp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return STRINGP(x) ? BOOL_T : BOOL_F;
}
SCM make_string(k,chr)
SCM k,chr;
{
	SCM res;
	register char *dst;
	register long i;
	ASSERT(INUMP(k),k,ARG1,s_make_string);
	i = INUM(k);
	if (i == 0) return nullstr;
	res = makstr(i);
	dst = CHARS(res);
	if ICHRP(chr) for(i--;i>=0;i--) dst[i] = ICHR(chr);
	return res;
}
SCM string(chrs)
SCM chrs;
{
	SCM res;
	register char *data;
	register long i = ilength(chrs);
	ASSERT(i>=0,chrs,ARG1,s_string);
	if (i == 0) return nullstr;
	res = makstr(i);
	data = CHARS(res);
	for(;NNULLP(chrs);chrs=CDR(chrs)) *data++ = ICHR(CAR(chrs));
	return res;
}
SCM string_length(str)
SCM str;
{
	ASSERT(NIMP(str)&&STRINGP(str),str,ARG1,s_string_length);
	return MAKINUM(LENGTH(str));
}
SCM string_ref(str,k)
SCM str,k;
{
	ASSERT(NIMP(str)&&STRINGP(str),str,ARG1,s_string_ref);
	ASSERT(INUMP(k),k,ARG2,s_string_ref);
	ASSERT(INUM(k) < LENGTH(str),k,OUTOFRANGE,s_string_ref);
	return MAKICHR(CHARS(str)[INUM(k)]);
}
SCM string_set(str,k,chr)
SCM str,k,chr;
{
	ASSERT(NIMP(str)&&STRINGP(str),str,ARG1,s_string_set);
	ASSERT(INUMP(k),k,ARG2,s_string_set);
	ASSERT(ICHRP(chr),chr,ARG3,s_string_set);
	ASSERT(INUM(k) < LENGTH(str),k,OUTOFRANGE,s_string_set);
	CHARS(str)[INUM(k)] = ICHR(chr);
	return UNSPECIFIED;
}
SCM string_equal(s1, s2)
SCM s1, s2;
{
	register int i;
	register char *c1, *c2;
	ASSERT(NIMP(s1)&&STRINGP(s1),s1,ARG1,s_string_equal);
	ASSERT(NIMP(s2)&&STRINGP(s2),s2,ARG2,s_string_equal);
	i = LENGTH(s2);
	if (LENGTH(s1) != i) return BOOL_F;
	c1 = CHARS(s1);
	c2 = CHARS(s2);
	for(;i>0;i--) if(*c1++ != *c2++) return BOOL_F;
	return BOOL_T;
}
SCM string_ci_equal(s1, s2)
SCM s1, s2;
{
	register int i;
	register char *c1, *c2;
	ASSERT(NIMP(s1)&&STRINGP(s1),s1,ARG1,s_string_ci_equal);
	ASSERT(NIMP(s2)&&STRINGP(s2),s2,ARG2,s_string_ci_equal);
	i = LENGTH(s2);
	if (LENGTH(s1) != i) return BOOL_F;
	c1 = CHARS(s1);
	c2 = CHARS(s2);
	for(;i>0;i--) if(upcase[*c1++] != upcase[*c2++]) return BOOL_F;
	return BOOL_T;
}
SCM string_lessp(s1, s2)
SCM s1, s2;
{
	register long i,len;
	register char *c1, *c2, c;
	ASSERT(NIMP(s1)&&STRINGP(s1),s1,ARG1,s_string_lessp);
	ASSERT(NIMP(s2)&&STRINGP(s2),s2,ARG2,s_string_lessp);
	len = LENGTH(s1);
	i = LENGTH(s2);
	if (len>i) i=len;
	c1 = CHARS(s1);
	c2 = CHARS(s2);
	for(i=0;i<len;i++) {
		c = (*c1++ - *c2++);
		if (c>0) return BOOL_F;
		if (c<0) return BOOL_T;
	}
	return (len != LENGTH(s2)) ? BOOL_T : BOOL_F;
}
SCM string_ci_lessp(s1, s2)
SCM s1, s2;
{
	register long i,len;
	register char *c1, *c2, c;
	ASSERT(NIMP(s1)&&STRINGP(s1),s1,ARG1,s_string_ci_lessp);
	ASSERT(NIMP(s2)&&STRINGP(s2),s2,ARG2,s_string_ci_lessp);
	len = LENGTH(s1);
	i = LENGTH(s2);
	if (len>i) i=len;
	c1 = CHARS(s1);
	c2 = CHARS(s2);
	for(i=0;i<len;i++) {
		c = (upcase[*c1++] - upcase[*c2++]);
		if (c>0) return BOOL_F;
		if (c<0) return BOOL_T;
	}
	return (len != LENGTH(s2)) ? BOOL_T : BOOL_F;
}
SCM substring(str,start,end)
SCM str,start,end;
{
	register long l;
	ASSERT(NIMP(str)&&STRINGP(str),str,ARG1,s_substring);
	ASSERT(INUMP(start),start,ARG2,s_substring);
	ASSERT(INUMP(end),end,ARG3,s_substring);
	ASSERT(INUM(start) < LENGTH(str),start,OUTOFRANGE,s_substring);
	ASSERT(INUM(end) <= LENGTH(str),end,OUTOFRANGE,s_substring);
	l=INUM(end)-INUM(start);
	ASSERT(l>=0,MAKINUM((long)l),OUTOFRANGE,s_substring);
	if (l == 0) return nullstr;
	return makfromstr(&CHARS(str)[INUM(start)],l);
}
SCM string_append(args)
SCM args;
{
	SCM res;
	register long i=0;
	register SCM l,s;
	register char *data;
	for(l=args;NIMP(l);) {
		ASSERT(CONSP(l),l,ARG1,s_string_append);
		s = CAR(l);
		ASSERT(NIMP(s)&&STRINGP(s),s,ARG1,s_string_append);
		i += LENGTH(s);
		l=CDR(l);
	}
	ASSERT(NULLP(l),args,ARG1,s_string_append);
	if (i == 0) return nullstr;
	res = makstr(i);
	data = CHARS(res);
	for(l=args;NIMP(l);l=CDR(l)) {
		s = CAR(l);
		for(i=0;i<LENGTH(s);i++) *data++ = CHARS(s)[i];
	}
	return res;
}
 
SCM vectorp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return VECTORP(x) ? BOOL_T : BOOL_F;
}
SCM vector_length(v)
SCM v;
{
	ASSERT(NIMP(v)&&VECTORP(v),v,ARG1,s_vector_length);
	return MAKINUM(LENGTH(v));
}
SCM vector(l)
SCM l;
{
	SCM res;
	register SCM *data;
	register long i = ilength(l);
	ASSERT(i>=0,l,ARG1,s_vector);
	if (i == 0) return nullvect;
	res = make_vector(MAKINUM((long)i),UNSPECIFIED);
	data = VELTS(res);
	for(;NIMP(l);l=CDR(l)) *data++ = CAR(l);
	return res;
}
SCM vector_ref(v, k)
SCM v,k;
{
	ASSERT(NIMP(v)&&VECTORP(v),v,ARG1,s_vector_ref);
	ASSERT(INUMP(k),k,ARG2,s_vector_ref);
	ASSERT((INUM(k) < LENGTH(v)),k,OUTOFRANGE,s_vector_ref);
	return VELTS(v)[((long) INUM(k))];
}
SCM vector_set(v,k,obj)
SCM v,k,obj;
{
	ASSERT(NIMP(v)&&VECTORP(v),v,ARG1,s_vector_set);
	ASSERT(INUMP(k),k,ARG2,s_vector_set);
	ASSERT((INUM(k) < LENGTH(v)),k,OUTOFRANGE,s_vector_set);
	VELTS(v)[((long) INUM(k))] = obj;
	return UNSPECIFIED;
}
 
init_subrs()
{
	init_subr("not",tc6_subr_1,lnot);
	init_subr("boolean?",tc6_subr_1,booleanp);
 
	init_subr("eq?",tc6_subr_2,eq);
	init_subr("eqv?",tc6_subr_2,eq);
	init_subr("equal?",tc6_subr_2,equal);
 
	init_subr("pair?",tc6_subr_1,consp);
	init_subr("cons",tc6_subr_2,cons);
	init_subr("car",tc6_cxr,cxr);
	init_subr("cdr",tc6_cxr,cxr);
	s_setcar=init_subr("set-car!",tc6_subr_2,setcar);
	s_setcdr=init_subr("set-cdr!",tc6_subr_2,setcdr);
	init_subr("caar",tc6_cxr,cxr);
	init_subr("cadr",tc6_cxr,cxr);
	init_subr("cdar",tc6_cxr,cxr);
	init_subr("cddr",tc6_cxr,cxr);
	init_subr("caaar",tc6_cxr,cxr);
	init_subr("caadr",tc6_cxr,cxr);
	init_subr("cadar",tc6_cxr,cxr);
	init_subr("caddr",tc6_cxr,cxr);
	init_subr("cdaar",tc6_cxr,cxr);
	init_subr("cdadr",tc6_cxr,cxr);
	init_subr("cddar",tc6_cxr,cxr);
	init_subr("cdddr",tc6_cxr,cxr);
	init_subr("caaaar",tc6_cxr,cxr);
	init_subr("caaadr",tc6_cxr,cxr);
	init_subr("caadar",tc6_cxr,cxr);
	init_subr("caaddr",tc6_cxr,cxr);
	init_subr("cadaar",tc6_cxr,cxr);
	init_subr("cadadr",tc6_cxr,cxr);
	init_subr("caddar",tc6_cxr,cxr);
	init_subr("cadddr",tc6_cxr,cxr);
	init_subr("cdaaar",tc6_cxr,cxr);
	init_subr("cdaadr",tc6_cxr,cxr);
	init_subr("cdadar",tc6_cxr,cxr);
	init_subr("cdaddr",tc6_cxr,cxr);
	init_subr("cddaar",tc6_cxr,cxr);
	init_subr("cddadr",tc6_cxr,cxr);
	init_subr("cdddar",tc6_cxr,cxr);
	init_subr("cddddr",tc6_cxr,cxr);
	init_subr("null?",tc6_subr_1,nullp);
	init_subr("list?",tc6_subr_1,listp);
	init_subr("list",tc6_lsubr,list);
	s_length=init_subr("length",tc6_subr_1,length);
	s_append=init_subr("append",tc6_lsubr,append);
	s_reverse=init_subr("reverse",tc6_subr_1,reverse);
	s_list_ref=init_subr("list-ref",tc6_subr_2,list_ref);
	s_memq=init_subr("memq",tc6_subr_2,memq);
	init_subr("memv",tc6_subr_2,memq);
	s_member=init_subr("member",tc6_subr_2,member);
	s_assq=init_subr("assq",tc6_subr_2,assq);
	init_subr("assv",tc6_subr_2,assq);
	s_assoc=init_subr("assoc",tc6_subr_2,assoc);
 
	init_subr("symbol?",tc6_subr_1,symbolp);
	s_symbol2string=init_subr("symbol->string",tc6_subr_1,symbol2string);
	s_string2symbol=init_subr("string->symbol",tc6_subr_1,string2symbol);
 
	init_subr("number?",tc6_subr_1,numberp);
	init_subr("complex?",tc6_subr_1,numberp);
	init_subr("real?",tc6_subr_1,numberp);
	init_subr("rational?",tc6_subr_1,numberp);
	init_subr("integer?",tc6_subr_1,numberp);
	s_exactp=init_subr("exact?",tc6_subr_1,exactp);
	s_inexactp=init_subr("inexact?",tc6_subr_1,inexactp);
	init_subr("floor",tc6_subr_1,numident);
	init_subr("ceiling",tc6_subr_1,numident);
	init_subr("truncate",tc6_subr_1,numident);
	init_subr("round",tc6_subr_1,numident);
 
	s_eqp=init_subr("=",tc6_lsubr_2,eqp);
	s_lessp=init_subr("<",tc6_lsubr_2,lessp);
	s_greaterp=init_subr(">",tc6_lsubr_2,greaterp);
	s_lesseqp=init_subr("<=",tc6_lsubr_2,lesseqp);
	s_greatereqp=init_subr(">=",tc6_lsubr_2,greatereqp);
	s_zerop=init_subr("zero?",tc6_subr_1,zerop);
	s_positivep=init_subr("positive?",tc6_subr_1,positivep);
	s_negativep=init_subr("negative?",tc6_subr_1,negativep);
	s_oddp=init_subr("odd?",tc6_subr_1,oddp);
	s_evenp=init_subr("even?",tc6_subr_1,evenp);
	s_max=init_subr("max",tc6_asubr,lmax);
	s_min=init_subr("min",tc6_asubr,lmin);
	s_sum=init_subr("+",tc6_asubr,sum);
	s_product=init_subr("*",tc6_asubr,product);
	s_difference=init_subr("-",tc6_subr_2,difference);
	s_abs=init_subr("abs",tc6_subr_1,absval);
	s_quotient=init_subr("quotient",tc6_subr_2,quotient);
	s_remainder=init_subr("remainder",tc6_subr_2,remainder);
	s_modulo=init_subr("modulo",tc6_subr_2,modulo);
	s_gcd=init_subr("gcd",tc6_asubr,lgcd);
	s_lcm=init_subr("lcm",tc6_asubr,llcm);
	s_number2string=init_subr("number->string",tc6_subr_2,number2string);
	s_string2number=init_subr("string->number",tc6_subr_2,string2number);
 
	init_subr("char?",tc6_subr_1,charp);
	init_subr("char=?",tc6_subr_2,eq);
	s_char_lessp=init_subr("char<?",tc6_subr_2,char_lessp);
	init_subr("char>?",tc6_subr_2x,char_lessp);
	init_subr("char<=?",tc6_subr_2xn,char_lessp);
	init_subr("char>=?",tc6_subr_2n,char_lessp);
	s_char_ci_eq=init_subr("char-ci=?",tc6_subr_2,char_ci_eq);
	s_char_ci_lessp=init_subr("char-ci<?",tc6_subr_2,char_ci_lessp);
	init_subr("char-ci>?",tc6_subr_2x,char_ci_lessp);
	init_subr("char-ci<=?",tc6_subr_2xn,char_ci_lessp);
	init_subr("char-ci>=?",tc6_subr_2n,char_ci_lessp);
	s_char_alphap=init_subr("char-alphabetic?",tc6_lsubr,char_alphap);
	s_char_nump=init_subr("char-numeric?",tc6_subr_1,char_nump);
	s_char_whitep=init_subr("char-whitespace?",tc6_subr_1,char_whitep);
	s_char_upperp=init_subr("char-upper-case?",tc6_subr_1,char_upperp);
	s_char_lowerp=init_subr("char-lower-case?",tc6_subr_1,char_lowerp);
	s_char2int=init_subr("char->integer",tc6_subr_1,char2int);
	s_int2char=init_subr("integer->char",tc6_subr_1,int2char);
	s_char_upcase=init_subr("char-upcase",tc6_subr_1,char_upcase);
	s_char_downcase=init_subr("char-downcase",tc6_subr_1,char_downcase);
 
	init_subr("string?",tc6_subr_1,stringp);
	s_make_string=init_subr("make-string",tc6_subr_2,make_string);
	init_subr("string",tc6_lsubr,string);
	s_string_length=init_subr("string-length",tc6_subr_1,string_length);
	s_string_ref=init_subr("string-ref",tc6_subr_2,string_ref);
	s_string_set=init_subr("string-set!",tc6_subr_3,string_set);
	s_string_equal=init_subr("string=?",tc6_subr_2,string_equal);
	s_string_ci_equal=init_subr("string-ci=?",tc6_subr_2,string_ci_equal);
	s_string_lessp=init_subr("string<?",tc6_subr_2,string_lessp);
	init_subr("string>?",tc6_subr_2x,string_lessp);
	init_subr("string<=?",tc6_subr_2xn,string_lessp);
	init_subr("string>=?",tc6_subr_2n,string_lessp);
	s_string_ci_lessp=init_subr("string-ci<?",tc6_subr_2,string_ci_lessp);
	init_subr("string-ci>?",tc6_subr_2x,string_ci_lessp);
	init_subr("string-ci<=?",tc6_subr_2xn,string_ci_lessp);
	init_subr("string-ci>=?",tc6_subr_2n,string_ci_lessp);
	s_substring=init_subr("substring",tc6_subr_3,substring);
	s_string_append=init_subr("string-append",tc6_lsubr,string_append);
 
	init_subr("vector?",tc6_subr_1,vectorp);
	s_make_vector=init_subr("make-vector",tc6_subr_2,make_vector);
	init_subr("vector",tc6_lsubr,vector);
	s_vector_length=init_subr("vector-length",tc6_subr_1,vector_length);
	s_vector_ref=init_subr("vector-ref",tc6_subr_2,vector_ref);
	s_vector_set=init_subr("vector-set!",tc6_subr_3,vector_set);
}
