;;;	Copyright (c) 1990 by Aubrey Jaffer, All rights reserved.

;;; How to start up SCHEME and SCL
(defun scheme ()
  (in-package "SCHEME")
  (unuse-package (package-use-list *package*))
  (use-package "SCHEME-DEFS")
  (format 'lisp:t ";;; running Scheme"))
(defun scl ()
  (in-package "USER")
  (do-external-symbols (var (find-package "SCHEME-DEFS"))
		       (shadowing-import var))
  (format 'lisp:t ";;; running scl"))
(defconstant base-language 'common-lisp)

;;; This file should be compiled.  With some compilers all the
;;; variable renaming can cause confusion.  If you have problems
;;; after compiling this file, try getting a fresh lisp before loading
;;; the bin file.

(lisp:in-package "SCHEME-DEFS")
(unuse-package (package-use-list *package*))
(lisp:use-package "LISP")
(provide 'scheme)
(lisp:proclaim '(optimize (speed 3) (compilation-speed 0)))

;;; These definitions implement a language called SCL which runs in
;;; both Common-Lisp and Scheme.  The language definitions are:
;;; Guy Lewis Steele Jr. Common Lisp: The Language.  Digital Press,
;;; Burlington MA, 1984.
;;; William Clinger and Jonathan Rees, editors. Revised^3.99
;;; Report on the Algorithmic Language Scheme.  DRAFT August 31, 1989.
;;; In this implementation, Scheme and Common Lisp code can be freely
;;; intermixed.
;;; Some of the routines are adapted from Jonathan Rees' Pseudo-Scheme.
;;; ;;;Comments give section of Report where construct is described.

;;; These symbols are the name conflicts between Scheme and Common Lisp.
;;; Common Lisp code needs to use lisp:t, lisp:nil, and lisp:string
;;; for the type 'string.  +, *, -, and / are shadowed because of
;;; their values being the last expression typed...  The other
;;; funtions determine whether they are being called with scheme or
;;; Common Lisp arguments and behave accordingly.  Read, read-char,
;;; and peek-char, if given 0 or 1 arguments, will return an
;;; eof-object rather than error.  This represents a minor
;;; incompatability with Common Lisp (whose default is to error).
;;; Other functions should work correctly with Common Lisp code.

(shadow '(t nil member assoc + * - / make-string string
	    map read read-char peek-char write make-list))
(defconstant syntactic-keywords
  '(=> and begin case cond define delay do else if lambda let let*
       letrec or quasiquote quote set! unquote unquote-splicing))
(defconstant symbols-shared-with-lisp
  '(< <= = > >= abs acos and append apply asin atan
      caaaar caaadr caaar caadar caaddr caadr caar cadaar cadadr
      cadar caddar cadddr caddr cadr car cdaaar cdaadr
      cdaar cdadar cdadr cdar cddaar cddadr cddar cdddar cddddr
      cdddr cddr cdr ceiling char-downcase char-upcase cons
      cos denominator error exp expt floor funcall gcd
      identity lcm length list load log max
      min not numerator quote rationalize reverse round
      sin sqrt tan truncate vector write-char))
(defconstant symbols-not-shared-with-lisp
  '(* + - / angle assoc assq assv begin boolean?
      call-with-current-continuation call-with-input-file
      call-with-output-file call/cc char->integer
      char-alphabetic? char-ci<=? char-ci<? char-ci=? char-ci>=?
      char-ci>? char-lower-case? char-numeric? char-ready?
      char-upper-case? char-whitespace? char<=? char<? char=?
      char>=? char>? char? close-input-port
      close-output-port complex? current-input-port
      current-output-port display eof-object?
      eq? equal? eqv? even? exact->inexact exact? for-each force
      imag-part inexact->exact inexact? input-port?
      integer->char integer? last-pair list->string list->vector
      list-copy list-ref list-tail list? magnitude make-list
      make-polar make-rectangular make-string make-vector map member
      memq memv modulo negative? newline null? number->string
      number? odd? open-input-file open-output-file output-port?
      pair? peek-char positive? procedure? quotient rational?
      read read-char real-part real? reduce-init remainder
      set-car! set-cdr! string string->list string->number
      string->symbol string-append string-ci<=? string-ci<?
      string-ci=? string-ci>=? string-ci>? string-copy
      string-fill! string-length string-ref string-set!
      string<=? string<? string=? string>=? string>? string?
      substring symbol->string symbol?
      transcript-off transcript-on vector->list vector-fill!
      vector-length vector-ref vector-set! vector?
      with-input-from-file with-output-to-file write zero?))
(export syntactic-keywords)
(export symbols-shared-with-lisp)
(export symbols-not-shared-with-lisp)
(defconstant eof-object 'eof-object)
;;; closures-might-be-conses adapted from
;;; Pseudoscheme, Copyright (c) 1989 Jonathan Rees
(defconstant closures-might-be-conses?
  #.(or (consp (eval '#'(lambda (x) x))) ;VAX LISP 2.1 or Franz cl 1.3
	(consp (let ((g (gensym)))
		 (eval `(progn (defun ,g () 0) #',g)))) ;Symbolics
	(consp (compile '() '(lambda (x) x))) ;just for kicks
	(consp (funcall (compile '() '(lambda (x) ;VAX LISP 2.2
					#'(lambda () (prog1 x (incf x)))))
			0))))
;;; syntax
(defun sharp-f-read-macro (stream sub-char arg)
  (declare (ignore stream sub-char arg))
  '())
(defun sharp-t-read-macro (stream sub-char arg)
  (declare (ignore stream sub-char arg))
  'lisp:t)
(set-dispatch-macro-character #\# #\f #'sharp-f-read-macro)
(set-dispatch-macro-character #\# #\t #'sharp-t-read-macro)
(set-dispatch-macro-character
 #\# #\b #'(lambda (stream sub-char arg)
	     (declare (ignore sub-char arg))
	     (let ((*read-base* 2.))
	       (lisp:read stream 'lisp:t '() 'lisp:t))))
(set-dispatch-macro-character
 #\# #\o #'(lambda (stream sub-char arg)
	     (declare (ignore sub-char arg))
	     (let ((*read-base* 8.))
	       (lisp:read stream 'lisp:t '() 'lisp:t))))
(set-dispatch-macro-character
 #\# #\d #'(lambda (stream sub-char arg)
	     (declare (ignore sub-char arg))
	     (let ((*read-base* 10.))
	       (lisp:read stream 'lisp:t '() 'lisp:t))))
(set-dispatch-macro-character
 #\# #\x #'(lambda (stream sub-char arg)
	     (declare (ignore sub-char arg))
	     (let ((*read-base* 16.))
	       (lisp:read stream 'lisp:t '() 'lisp:t))))
(set-dispatch-macro-character
 #\# #\e #'(lambda (stream sub-char arg)
	     (declare (ignore sub-char arg))
	     (rationalize (lisp:read stream 'lisp:t '() 'lisp:t))))
(set-dispatch-macro-character
 #\# #\i #'(lambda (stream sub-char arg)
	     (declare (ignore sub-char arg))
	     (float (lisp:read stream 'lisp:t '() 'lisp:t))))

;;; The symbols for all standard Scheme functions have both their
;;; values and function-values defined (to the same procedure).  If
;;; the user wishes to redefine one, She should use (define ...)
;;; rather than (set! ...) the first time (in order to set up a macro
;;; which funcalls the value of this symbol).  Expressions other than
;;; lambda or symbols in the first position of combination require
;;; (funcall (exp-which-yields-proc) args)

;;;4.1.2

;;;4.1.4
(defmacro lambda (formals &rest body)
  (labels ((lispize-formals
	    (formals)
	    (cond ((consp formals)
		   (cons (car formals) (lispize-formals (cdr formals))))
		  ((null formals) '())
		  ((symbolp formals) (list '&rest formals))
		  ((error " Bad lambda formals list ~s" formals)))))
	  `#'(lambda ,(lispize-formals formals) ,@body)))
;;;4.1.5
			;IF is same
;;;4.1.6
(defmacro set! (&rest args) (cons 'setq args)) ;value unspecified
;;;4.2.1
		;Optional => not supported
;;;      CASE is same except use LISP:T or OTHERWISE instead of ELSE.
;;;4.2.2
;;; let and let* are the same except that functions defined here need
;;; FUNCALL or that the function name be defined previously
(defun funcallize (call env) (declare (ignore env)) (cons 'funcall call))
(defmacro letrec (bindings . body)
  (let ((nvs (mapcar #'(lambda (b)
			 (setf (macro-function (car b)) #'funcallize)
			 (gensym))
		     bindings)))
    `(let ,(mapcar #'(lambda (b) (list (car b) :unspecified))
		   bindings)
       (let ,(mapcar #'(lambda (nv b) (list nv (second b)))
		     nvs
		     bindings)
	 ,@(mapcar #'(lambda (b nv) (list 'set! (first b) nv))
		   bindings
		   nvs))
       ,@body)))
;;;4.2.3
(defmacro begin (&rest forms) (cons 'progn forms))
;;;4.2.4
			;same but functions bound need funcall
;;;4.2.5
(defmacro delay (exp) `(make-promise (lambda () ,exp)))
;;;4.2.6 "`", ",", and ",@" work;
;;;      quasiquote, unquote, and unquote-splicing do not.
;;;5.2
(defmacro define (vformal . body)
  (labels ((tls (sym val)
		(setf (macro-function sym) #'funcallize)
		`(progn (setq ,sym ,(if (and (symbolp val) (not (boundp val)))
					(list 'function val)
				      val))
			(setf (macro-function ',sym) #'funcallize)
			',sym))
	   (lispize-formals
	    (formals)
	    (cond ((consp formals)
		   (cons (car formals) (lispize-formals (cdr formals))))
		  ((null formals) '())
		  ((symbolp formals) (list '&rest formals))
		  ((error " Bad lambda formals list ~s" formals))))
	   (tld (sym argl body)
		(setf (macro-function sym) #'funcallize)
		`(progn (defparameter ,sym
			  #'(lambda ,(lispize-formals argl) ,@body))
			(setf (macro-function ',sym) #'funcallize)
			',sym)))
	  (cond ((not (symbolp vformal))
		 (tld (car vformal) (cdr vformal) body))
		((null body)(error "define ~s ~s" vformal body))
		((not (null (cdr body)))
		 (error "define ~s ~s" vformal body))
		((and (consp (car body)) (eq 'lambda (caar body)))
		 (tld vformal (cadar body) (cddar body)))
		((tls vformal (car body))))))
;;;6.1
(defconstant else 'lisp:t)		;if you redefine this, you are a loser
(defun boolean? (obj) (or (eq '() obj) (eq lisp:t obj)))
;;;6.2
(defun equal? (obj1 obj2)
  (labels ((equal?
	    (obj1 obj2)
	    (cond ((eql obj1 obj2) lisp:t)
		  ((consp obj1)
		   (and (consp obj2)
			(equal? (car obj1) (car obj2))
			(equal? (cdr obj1) (cdr obj2))))
		  ((simple-string-p obj1)
		   (and (simple-string-p obj2)
			(string= (the simple-string obj1)
				 (the simple-string obj2))))
		  ((simple-vector-p obj1)
		   (and (simple-vector-p obj2)
			(let ((z (length (the simple-vector obj1))))
			  (declare (fixnum z))
			  (and (= z (length (the simple-vector obj2)))
			       (do ((i 0 (lisp:+ i 1)))
				   ((= i z) lisp:t)
				   (declare (fixnum i))
				   (if (not (equal? (svref obj1 i)
						    (svref obj2 i)))
				       (return '())))))))
		  (lisp:t '()))))
	  (equal? obj1 obj2)))
(setf (symbol-function 'eqv?) #'eql)
(setf (symbol-function 'eq?) #'eq)
;;;6.3
(defun pair? (obj) (and (consp obj) (not (functionp obj))))
(or closures-might-be-conses?
    (setf (symbol-function 'pair?) #'consp))
(setf (symbol-function 'set-car!) #'rplaca) ;value unspecified
(setf (symbol-function 'set-cdr!) #'rplacd) ;value unspecified
(setf (symbol-function 'null?) #'null)
(defun list? (obj) (or (null obj) (and (pair? obj) (null (cdr (last obj))))))
(defun list-tail (list k) (nthcdr k (the list list)))
(defun list-ref (list k) (nth k (the list list)))
(defun memq (obj list) (lisp:member obj (the list list) :test #'eq))
(setf (symbol-function 'memv) #'lisp:member)
(defun member (obj list) (lisp:member obj list :test #'equal?))
(defun assq (obj alist) (lisp:assoc obj alist :test #'eq))
(setf (symbol-function 'assv) #'lisp:assoc)
(defun assoc (obj alist) (lisp:assoc obj alist :test #'equal?))
;;;6.4
(defun symbol? (obj)
  (and (symbolp obj) (not (null obj))
       (not (eq obj 'lisp:t)) (not (eq obj eof-object))))
(setf (symbol-function 'symbol->string) #'symbol-name)
(setf (symbol-function 'string->symbol) #'intern)
;;;6.5.5
(setf (symbol-function 'number?) #'numberp)
(setf (symbol-function 'complex?) #'numberp)
(defun real? (obj)
  (and (numberp obj) (not (complexp obj))))
(setf (symbol-function 'rational?) #'rationalp)
(setf (symbol-function 'integer?) #'integerp)
(setf (symbol-function 'exact?) #'rationalp)
(setf (symbol-function 'inexact?) #'floatp)
(setf (symbol-function 'zero?) #'zerop)
(setf (symbol-function 'positive?) #'plusp)
(setf (symbol-function 'negative?) #'minusp)
(setf (symbol-function 'odd?) #'oddp)
(setf (symbol-function 'even?) #'evenp)
(setf (symbol-function '+) #'lisp:+)
(setf (symbol-function '*) #'lisp:*)
(setf (symbol-function '-) #'lisp:-)
(setf (symbol-function '/) #'lisp:/)
(defun quotient (n1 n2) (values (truncate n1 n2)))
(setf (symbol-function 'remainder) #'rem)
(setf (symbol-function 'modulo) #'mod)
(setf (symbol-function 'make-rectangular) #'complex)
(defun make-polar (x3 x4) (lisp:* x3 (cis x4)))
(setf (symbol-function 'real-part) #'realpart)
(setf (symbol-function 'imag-part) #'imagpart)
(setf (symbol-function 'magnitude) #'abs)
(setf (symbol-function 'angle) #'phase)
(setf (symbol-function 'exact->inexact) #'float)
(setf (symbol-function 'inexact->exact) #'rationalize)
(defun number->string (number &optional (radix 10))
	 (write-to-string number :base radix))
(defun string->number (str &optional (radix 10))
	 (parse-integer str :radix radix))
;;;6.6
(setf (symbol-function 'char?) #'characterp)
(setf (symbol-function 'char=?) #'char=)
(setf (symbol-function 'char<?) #'char<)
(setf (symbol-function 'char>?) #'char>)
(setf (symbol-function 'char<=?) #'char<=)
(setf (symbol-function 'char>=?) #'char>=)
(setf (symbol-function 'char-ci=?) #'char-equal)
(setf (symbol-function 'char-ci<?) #'char-lessp)
(setf (symbol-function 'char-ci>?) #'char-greaterp)
(setf (symbol-function 'char-ci<=?) #'char-not-greaterp)
(setf (symbol-function 'char-ci>=?) #'char-not-lessp)
(setf (symbol-function 'char-alphabetic?) #'alpha-char-p)
(setf (symbol-function 'char-numeric?) #'digit-char-p)
(defun char-whitespace? (char)
  (or (char= char #\space)
      (not (graphic-char-p char))))
(setf (symbol-function 'char-upper-case?) #'upper-case-p)
(setf (symbol-function 'char-lower-case?) #'lower-case-p)
(setf (symbol-function 'char->integer) #'char-code)
(setf (symbol-function 'integer->char) #'code-char)
(setf (symbol-function 'char-upcase) #'char-upcase)
(setf (symbol-function 'char-downcase) #'char-downcase)
;;;6.7
(setf (symbol-function 'string?) #'simple-string-p)
(defun make-string (k &optional (char #\?) &rest restargs)
  (if restargs (apply #'lisp:make-string k char restargs)
    (lisp:make-string k :initial-element char)))
(defun string (&rest char...)
  (if (or (cdr char...) (characterp (car char...))) ;Scheme string
      (coerce char... 'lisp:string)
    (apply #'lisp:string char...)))	;Common Lisp string
(setf (symbol-function 'string-length) #'length)
(setf (symbol-function 'string-ref) #'schar)
(defun string-set! (string k chr)
  (setf (char string k) chr)
  :unspecified)
(setf (symbol-function 'string=?) #'string=)
(setf (symbol-function 'string-ci=?) #'string-equal)
(setf (symbol-function 'string<?) #'string<)
(setf (symbol-function 'string>?) #'string>)
(setf (symbol-function 'string<=?) #'string<=)
(setf (symbol-function 'string>=?) #'string>=)
(setf (symbol-function 'string-ci<?) #'string-lessp)
(setf (symbol-function 'string-ci>?) #'string-greaterp)
(setf (symbol-function 'string-ci<=?) #'string-not-greaterp)
(setf (symbol-function 'string-ci>=?) #'string-not-lessp)
(setf (symbol-function 'substring) #'subseq)
(defun string-append (&rest strings)
  (apply #'concatenate 'simple-string strings))
(defun list->string (chars)
  (coerce chars 'simple-string))
(defun string->list (string)
  (coerce string 'list))
(setf (symbol-function 'string-copy) #'copy-seq)
(defun string-fill! (string char)
  (fill string char)
  :unspecified)
;;;6.8
(defun vector? (obj)
  (and (simple-vector-p obj)
       ;; Structures are vectors in Symbolics, Exploder, and CLISP.
       #+(or tops-20 Lispm)
       (not (typep obj 'lisp::structure))
       ;; Strings are simple vectors in CLISP (this is a bug)
       #+tops-20
       (not (lisp:stringp obj))))
(defun make-vector (k &optional (fill '()))
  (make-sequence 'vector k :initial-element fill))
(setf (symbol-function 'vector-length) #'length)
(setf (symbol-function 'vector-ref) #'svref)
(defun vector-set! (vec k obj)
  (setf (svref vec k) obj)
  :unspecified)
(defun vector->list (vec)
  (coerce (the simple-vector vec) 'list))
(defun list->vector (list)
  (coerce (the list list) 'simple-vector))
(defun vector-fill! (vec val)
  (fill (the simple-vector vec) val)
  :unspecified)
;;;6.9
(defun procedure? (obj)
  (and (functionp obj)
       (not (symbolp obj))
       (or closures-might-be-conses?
	   (not (consp obj)))))
(defun map (&rest args)
  (if (symbolp (car args))
      (apply #'lisp:map args)
    (apply #'mapcar args)))
(setf (symbol-function 'for-each) #'mapc)
(defun make-promise (proc)
  (let ((already-run? '())
	(result '()))
    (lambda ()
      (cond ((not already-run?)
	     (setq result (funcall proc))
	     (setq already-run? 'lisp:t)))
      result)))
(setf (symbol-function 'force) #'funcall)
;;; this involves an upward funarg.  If that bothers you change it to a macro.
(defun call-with-current-continuation (fcn)
  (let ((tag (list '())))
    (catch tag
      (funcall fcn (lambda (value)
		     (throw tag value))))))
(setf (symbol-function 'call/cc) #'call-with-current-continuation)
;;;6.10.1
(defun call-with-input-file (string proc)
  (with-open-file (port (merge-pathnames string) :direction :input)
		  (funcall proc port)))
(defun call-with-output-file (string proc)
  (with-open-file (port (merge-pathnames string)
			:direction :output
			:if-exists :new-version)
		  (funcall proc port)))
(defun input-port? (obj)
	 (and (streamp obj) (input-stream-p obj)))
(defun output-port? (obj)
	 (and (streamp obj) (output-stream-p obj)))
(defun current-input-port () *standard-input*)
(defun current-output-port () *standard-output*)
(defun with-input-from-file (string thunk)
  (with-open-file (*standard-input* (merge-pathnames string)
				    :direction :input)
		  (funcall thunk)))
(defun with-output-to-file (string thunk)
  (with-open-file (*standard-output* (merge-pathnames string)
				     :direction :output
				     :if-exists :new-version)
		  (funcall thunk)))
(defun open-input-file (string)
	 (open (merge-pathnames string) :direction :input))
(defun open-output-file (string)
	 (open (merge-pathnames string) :direction :output))
(setf (symbol-function 'close-input-port) #'close)
(setf (symbol-function 'close-output-port) #'close)
;;;6.10.2
(defun read (&optional (port *standard-input*) &rest restargs)
  (if restargs				;common lisp call
      (apply #'lisp:read port restargs)
    (lisp:read port '() eof-object)))
(defun read-char (&optional (port *standard-input*) &rest restargs)
  (if restargs				;common lisp call
      (apply #'lisp:read-char port restargs)
    (lisp:read-char port '() eof-object)))
(defun peek-char (&optional (port *standard-input*) &rest restargs)
  (if restargs				;common lisp call
      (apply #'lisp:peek-char port restargs)
    (lisp:peek-char '() port '() eof-object)))
(defun eof-object? (obj)
	 (eq obj eof-object))
(defun char-ready? (&optional (port *standard-input*))
  (or (listen port)
      (let ((char (read-char-no-hang port '() eof-object)))
	(cond ((characterp char) (unread-char char port) 'lisp:t)
					;in case char just became available
	      (char)))))

;;;6.10.3
(defun write (obj &optional (port *standard-output*) &rest restargs)
  (if restargs				;common lisp call
      (apply #'lisp:write port restargs)
    (let ((*print-array* lisp:t))	;for #(...)
      (cond ((null obj) (princ "()" port))
	    ((eq obj lisp:t) (princ "#T" port))
	    (lisp:t (prin1 obj port)))
      :unspecified)))

(defun display (obj &optional (port *standard-output*))
  (let ((*print-array* lisp:t))		;for #(...)
    (cond ((null obj) (princ "()" port))
	  ((eq obj lisp:t) (princ "#T" port))
	  (lisp:t (princ obj port))))
  :unspecified)
(setf (symbol-function 'newline) #'terpri)
;;;6.10.4
(defun transcript-on (filespec)
  (dribble filespec)
  :unspecified)
(defun transcript-off()
  (dribble)
  :unspecified)

;;; EXTENSIONS
(defun make-list (k &optional (obj :unspecified) &rest restargs)
  (if restargs (apply #'lisp:make-list k obj restargs)
    (lisp:make-list k :initial-element obj)))
(setf (symbol-function 'list-copy) #'copy-seq)
(setf (symbol-function 'last-pair) #'last)
(defun reduce-init (proc init l) (reduce proc l :initial-value init))

;;; FIXUPS
;funcall
(defun set-values-to-functions (l)
  (cond ((null l))
	(lisp:t (set (car l) (symbol-function (car l)))
		(set-values-to-functions (cdr l)))))
(set-values-to-functions symbols-shared-with-lisp)
(set-values-to-functions symbols-not-shared-with-lisp)
