;;; Unix wait
;;; Copyright (c) 1993 by Olin Shivers.

;;; (wait) => [pid status] or [#f #f]
;;; (wait pid) => status or #f
;;;     This one spins in a (wait-any) loop until it gets PID. Other
;;;     pid/status pairs that are reaped are saved away in a defunct-pid
;;;     table.
;;;  Either form of wait first consults the defunct-pid table to see if
;;;  a/the process has been reaped already. If so, the pid is removed from 
;;;  the table and its saved status returned immediately.

;;; Requires WAIT-ANY (wait(2)) and %WAIT-PID

(define (wait . maybe-pid)
  (if (null? maybe-pid)

      ;; Wait for anybody.
      (receive (pid status) (defunct-pid-remove-any)
	(if pid (values pid status)
	    (wait-any)))

      ;; Wait for a specific pid (and reap others while we wait).
      (let ((pid (car maybe-pid)))
	(if (or (not (integer? pid))
		(not (null? (cdr maybe-pid))))
	    (error "Bad arg(s) to wait" wait maybe-pid)

	    (or (lookup&remove-defunct-pid pid) ; Already reaped it.
		(let lp () ; Wait & reap.
		  (receive (pid2 status) (wait-any)
		    (cond ((not pid2) #f)
			  ((= pid2 pid) status)
			  (else
			   (add-defunct-pid-entry! pid2 status)
			   (lp))))))))))

(define (wait-all)
  (let lp ()
    (receive (pid status) (wait)
      (if pid (lp)))))

(define (wait-pid/no-reap pid)
  (or (lookup&remove-defunct-pid pid)
      (%wait-pid pid)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Defunct pid table routine -- (WAIT pid) tries to migrate other defunct
;;; pids from the kernel proc table into scsh.

;;; (length . alist)
(define *defunct-pid-table* (cons 0 `()))

(define (defunct-pid-remove-any)
  (let ((alist (cdr *defunct-pid-table*)))
    (if (null? alist) (values #f #f)
	(let ((entry (car alist)))
	  (set-car! *defunct-pid-table* (- (car *defunct-pid-table*) 1))
	  (set-cdr! *defunct-pid-table* (cdr alist))
	  (values (car entry) (cdr entry))))))

(define (flush-defunct-pid-table)
  (set! *defunct-pid-table* (cons 0 '())))

(define (lookup&remove-defunct-pid pid)
  (let ((alist (cdr *defunct-pid-table*)))
    (cond ((assv pid alist) =>
	   (lambda (entry)
	     (set-car! *defunct-pid-table* (- (car *defunct-pid-table*) 1))
	     (set-cdr! *defunct-pid-table*
		       (delete (lambda (x) (= (car x) pid)) ; DELQ!, not DEL.
			       (cdr alist)))
	     (cdr entry)))
	  (else #f))))

(define (add-defunct-pid-entry! pid status)
  (set-car! *defunct-pid-table* (+ (car *defunct-pid-table*) 1))
  (set-cdr! *defunct-pid-table* (cons (cons pid status)
				      (cdr *defunct-pid-table*))))
