;;; Unix wait
;;; Copyright (c) 1993, 1994 by Olin Shivers.

;;; THIS DOCUMENTATION NEEDS WORK.

;;; (wait pid [flags]) => status or #f
;;;
;;; FLAGS (default 0) is the exclusive or of the following:
;;;     wait/poll	
;;;		Return [#f #f] immediately if there are no 
;;;		unwaited children available. 
;;; 	wait/stopped-children
;;; 		Report on suspended children as well.
;;;
;;;     If the process hasn't terminated (or suspended, if wait/stopped 
;;; 	is set) and wait/poll is set, return #f. If the process has
;;;	already been reported, then raise an error exception.

;;; WAIT waits for a specific pid by spinning in a (wait-any) until it gets
;;; the requested PID. Other pid/status pairs that are reaped are saved away
;;; in a defunct-pid table.  Either form of wait first consults the
;;; defunct-pid table to see if a/the process has been reaped already. If so,
;;; the pid is removed from the table and its saved status returned
;;; immediately. 
;;;
;;; "Reaping" a process means to move it from the kernel's
;;; process table to scsh's internal "defunct pid" table.

;;; (wait-any [flags]) => [pid status]
;;;     [#f #f] => non-blocking, none ready.
;;;     [#f #t] => no more.

;;; (wait-process-group [proc-group flags]) => [pid status]
;;;     [#f #f] => non-blocking, none ready.

;;; (wait         pid [flags]) => status
;;; (wait/reap    pid [flags]) => status
;;; (wait/no-reap pid [flags]) => status
;;;     Return #f if non-blocking, none ready.
;;;     WAIT is synonym for WAIT/REAP.

;;; (flush-zombies) => bool
;;; (reap-zombies)  => bool
;;;     Return #t if no more outstanding children; #f if some still live.

(define (wait/reap pid . maybe-flags)
  (reap-zombies)
  (apply wait/no-reap pid maybe-flags))

(define wait wait/reap)	; Keep the process table clean.

(define (wait/no-reap pid . maybe-flags)
  (check-arg integer? pid wait/no-reap)
  (let ((flags (check-arg integer? (optional-arg maybe-flags 0) wait/no-reap)))
    (or (lookup&remove-defunct-pid pid)		; Already reaped it.
	(%wait-pid flags))))			; Directly wait for PID.

(define (wait-any . maybe-flags)
  (let ((flags (check-arg integer? (optional-arg maybe-flags 0) wait-any)))
    (receive (pid status) (defunct-pid-remove-any)	; Check internal table.
      (if pid (values pid status)			
	  (%wait-any flags)))))				; Really wait.


;;; (wait-process-group [proc-group flags])
;;; 
;;; If you are doing process-group waits, you do *not* want to use reaping
;;; waits, since the reaper loses process-group information.

(define (wait-process-group . args)
  (receive (proc-group flags) (parse-optionals args 0 0)
    (%wait-process-group (check-arg integer? proc-group wait-process-group)
			 (check-arg integer? flags      wait-process-group))))


;;; Flush all zombies.
;;; Returns #t if there are no more outstanding children, 
;;; #f if some still live.

(define (flush-zombies)
  (flush-defunct-pid-table)
  (let lp ()
    (receive (pid status) (%wait-any wait/poll)
      (if pid (lp) status))))


;;; Move any zombies into scsh's internal table.
;;; Returns #t if there are no more outstanding children, 
;;; #f if some still live.

(define (reap-zombies)
  (let lp ()
    (receive (pid status) (%wait-any wait/poll)
      (if pid
	  (begin (add-defunct-pid-entry! pid status)
		 (lp))
	  status))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Defunct pid table routines -- (WAIT/REAP pid) tries to migrate other 
;;; defunct pids from the kernel proc table into scsh.

;;; (length . alist)
(define *defunct-pid-table* (cons 0 `()))

(define (defunct-pid-remove-any)
  (let ((alist (cdr *defunct-pid-table*)))
    (if (null? alist) (values #f #f)
	(let ((entry (car alist)))
	  (set-car! *defunct-pid-table* (- (car *defunct-pid-table*) 1))
	  (set-cdr! *defunct-pid-table* (cdr alist))
	  (values (car entry) (cdr entry))))))

(define (flush-defunct-pid-table)
  (set! *defunct-pid-table* (cons 0 '())))

(define (lookup&remove-defunct-pid pid)
  (let ((alist (cdr *defunct-pid-table*)))
    (cond ((assv pid alist) =>
	   (lambda (entry)
	     (set-car! *defunct-pid-table* (- (car *defunct-pid-table*) 1))
	     (set-cdr! *defunct-pid-table*
		       (delete (lambda (x) (= (car x) pid)) ; DELQ!, not DEL.
			       (cdr alist)))
	     (cdr entry)))
	  (else #f))))

(define (add-defunct-pid-entry! pid status)
  (set-car! *defunct-pid-table* (+ (car *defunct-pid-table*) 1))
  (set-cdr! *defunct-pid-table* (cons (cons pid status)
				      (cdr *defunct-pid-table*))))



;;; (%wait-any flags) (%wait-pid pid flags) (%wait-process-group pgrp flags)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Direct interfaces to waitpid(2) call.
;;; [#f #f] means no processes ready on a non-blocking wait.
;;; [#f #t] means no waitable process on wait-any.

(define-foreign %wait-pid/errno (wait_pid (integer pid) (integer options))
  (to-scheme integer errno_or_false) ; error flag
  integer  ; process' id
  integer) ; process' status

(define (%wait-pid pid flags)
  (let lp ()
    (receive (err pid status) (%wait-pid/errno pid flags)
      (if err
	  (if (= err errno/intr) (lp)
	      (errno-error err %wait-pid))
	  (and (not (zero? pid)) status)))))	; pid=0 => none ready.


(define (%wait-any flags)
  (let lp ()
    (receive (err pid status) (%wait-pid/errno -1 flags)
      (cond (err (cond ((= err errno/child) (values #f #t))	; No more.
		       ((= err errno/intr)  (lp))
		       (else (errno-error err %wait-any))))
	    ((zero? pid) (values #f #f))			; None ready.
	    (else (values pid status))))))

(define (%wait-process-group pgrp flags)
  (let lp ()
    (receive (err pid status) (%waid-pid/errno (- pgrp) flags)
      (cond (err (if (= err errno/intr) (lp)
		     (errno-error err %wait-process-group pgrp flags)))
	    ((zero? pid) (values #f #f))	; None ready.
	    (else (values pid status))))))
