;(eval-when (compile) (proclaim '(optimize (speed 3) (safety 0) (space 2))))

; port internal repr:  ('&in/out-port CL-stream max-col . curr-col)
(defvar ptport nil)
(proclaim '(special schport colpair linelen pmode linecount))
(proclaim '(fixnum linelen linecount))

(defun transcript-on (file)
      (progn (if ptport (close ptport))
	     (or (setq ptport (open file :direction :output))
		 (raise (list 'SE%print '|transcript-on:|
		    '|Sorry, no ports available|))) ; implicit (reset)
	     (setq old-prompt (lookupinbase 'scheme-prompt))
	     (rplacd (baselocation 'scheme-prompt) "::> ")
	     (cons '&out-port 
		(cons ptport (cddr (standard-output))))))

(defun transcript-off ()
      (cond (ptport (close ptport)
	       (rplacd (baselocation 'scheme-prompt) old-prompt)
	       (setq ptport nil))
	    (t (raise (list 'SE%print '|transcript-off:|
			 '|Transcript mode is not on|)))))

(defun open-input-file (file)
	(let ((port (ignore-errors (open file :direction :input
					      :if-does-not-exist nil))))
	     (if port
		 `(&in-port ,port)
		 (raise (list 'SE%io '|open-input-file:|
			      '|File does not exist|)))))

(defun open-output-file (file)
	(let ((port (ignore-errors (open file :direction :output))))
	     (if port
		 `(&out-port ,port 132 . 1)
		 (raise (list 'SE%io '|open-output-file:|
			      '|Cannot open file|)))))

(defun open-append-file (file)
	(let ((port (ignore-errors (open file :direction :output
					      :if-exists :append))))
	     (if port
		 `(&out-port ,port 132 . 1)
		 (raise (list 'SE%io '|open-append-file:|
			      '|Cannot open file|)))))

(defun port? (x)
	(memq (car x) '(&in-port &out-port)))
(defun input-port? (x)
	(eq (car x) '&in-port))
(defun output-port? (x)
	(eq (car x) '&out-port))

(defun lport (port)
      (if (and (consp port) (memq (car port) '(&in-port &out-port)))
	  (cadr port)
	  (raise (list 'SE%io '|input/output:| '|Bad port|))))
(defun iport (port)
      (if (and (consp port) (eq (car port) '&in-port))
	  (cadr port)
	  (raise (list 'SE%io '|input:| '|Bad input port|))))
(defun oport (port)
      (if (and (consp port) (eq (car port) '&out-port))
	  (cadr port)
	  (raise (list 'SE%io '|output:| '|Bad output port|))))

(defun schclose (port)
      (cond ((member (cadr port)
		     (list *standard-input* *standard-output*))
	     port)
	    (t (close (lport port))(rplaca port '&closed-port))))

(defun eof () '*end-of-file*)

(defun standard-input ()
	(lookupinbase 'standard-input))

(defun standard-output ()
	(lookupinbase 'standard-output))

(setq schpoport `(&out-port ,*standard-output* 74 . 1))

(defun modeprint (x port mode)
      (let ((val (handler-case
		    (progn 
		       (setq schport (oport port))
		       (setq colpair (cddr port))
		       (setq linelen (car colpair))
		       (setq pmode mode)
		       (setq linecount 
			  (if	(eq schport *standard-output*)
				10
				-1))
		       (list (catch 'nomore (objprint x))))
		    (stream-error ()
			   (standardprint "Output error to stream ")
			   (standardprint port)
			   (new-line schpoport)
			   () ))))
	 (cond
	    ((null val) (schclose port) (reset))
	    (t (car val)))))
			       
(defun print-sys-type (type)
	  (colprint (cdr (assq (car type)
			        (lookupinbase 'unprintables)))
	      'print))

(defun sys-type? (typed-object)
      (and (or (eq pmode 'display) (eq pmode 'print))
	   (assq (car typed-object) 
		 (lookupinbase 'unprintables))))

(defun colprint (atm mode)
      (prog (len)
	    (cond ((plusp linelen)
		   (setq len (length (princ-to-string atm)))
		   (if (and (or (eq mode 'display) (eq mode 'display&)) 
			    (stringp atm))
		       (setq len (+ len 2)))
		   (cond ((> (cdr (rplacd colpair (+ (cdr colpair) len)))
			     linelen)
			  (terpri schport)
			  (if ptport (terpri ptport))
			  (rplacd colpair len)
			  (cond ((zerop (setq linecount (1- linecount)))
				 (princ "MORE? ")
				 (clear-input)
				 (cond ((memq (prog1 (read-char)
						     (clear-input))
					      '(#\n #\q #\N #\Q))
					(throw 'nomore nil))
				       (t (setq linecount 10)
					  (terpri)))))))))
	    (cond ((or (eq mode 'display) (eq mode 'display&))
		   (prin1 atm schport)
		   (if ptport (prin1 atm ptport)))
		  (t
		   (princ atm schport)
		   (if ptport (princ atm ptport))))))

(defun objprint (x)
      (cond ((atom x)	   (colprint x pmode))
	    ((sys-type? x) (print-sys-type x))
	    (t		   (colprint "(" 'print)
			   (objprint (car x))
			   (objprint-cdr (cdr x)))))

(defun objprint-cdr (x)
      (cond ((null x) (colprint ")" 'print))
	    ((atom x) 
	     (colprint " . " 'print)
	     (colprint x pmode)
	     (colprint ")" 'print))
	    ((sys-type? x)
	     (colprint " . " 'print)
	     (print-sys-type x)
	     (colprint ")" 'print))
	    (t (colprint " " 'print)
	       (objprint (car x))
	       (objprint-cdr (cdr x)))))

(defun schread (&optional (port (standard-input)))
	(let ((val (safe-read 'read port)))
	     (cond ((and (eq port keyboard) ptport)
		    (princ val ptport)
		    (terpri ptport)))
	     val))

(defun schread-char (&optional (port (standard-input)))
        (safe-read 'read-char port))

(defun schunread-char (x &optional (port (standard-input)))
	(unread-char x (iport port)))

(defun prompt-read (pr)
        (standardprint pr) (standardprint " ")
        (standardread))

(defun display (x &optional (port (standard-output)))
	(modeprint x port 'display))

(defun display& (x &optional (port (standard-output)))
	(modeprint x port 'display&))

(defun schprint& (x &optional (port (standard-output)))
	(modeprint x port 'print&))

(defun schprint (x &optional (port (standard-output)))
	(modeprint x port 'print))

(defun new-line (&optional (port (standard-output)))
	(terpri (oport port))
	(if ptport (terpri ptport))
        (rplacd (cddr port) 1)
        nil)

(defun flush-input (&optional (port (standard-input)))
        (clear-input (iport port)))

(defun flush-output (&optional (port (standard-output)))
        (force-output (oport port)))

(defun schwrite-char (x &optional (port (standard-output)))
	(if (characterp x)
	    (modeprint (symbol-name x) port 'print)
	    (raise (list 'SE%io '|write-char:| '|Argument is not a character:|'
			 x))))

(defun char-ready? (&optional (port (standard-input)))
	(listen (iport port)))

(setq oldprompt ">>> ")

;(defun pretty (l &optional (port (standard-output)))
;      (cond ((atom l) (raise (list 'SE%pretty 0 t '|pretty:|
;				'|argument not a list:| l))))
;      (prog (x gtype gb atm outport errlst oldpoport)
;	    (setq oldpoport *standard-output*)
;	    (setq *standard-output* (oport port))
;	   loop
;	    (setq atm (car l))
;	    (setq x (assq atm (lookupinbase 'defined-forms)))
;	    ($prpr (cond (x (cdr x)) ;??jg??-$prpr
;			 ((setq gtype (global-namespacetype atm))
;			  (setq gb (global-binding atm))
;			  (cond ((eq gtype 'system-function)
;				 `(function-alias ',atm ',gb))
;				((proc? gb)
;				 (setq errlst (cons atm errlst))
;				 `(quote (,atm not-at-all-pretty)))
;				(t `(set! ,atm ',gb))))
;			 ((setq gb (lookupinbase atm))
;			  `(set! ,atm ',gb))
;			 (t (setq errlst (cons atm errlst))
;			    `(quote (,atm not-globally-bound)))))
;	    (setq l (cdr l))
;	    (terpri)
;	    (terpri)
;	    (cond (l (go loop)))
;	    (setq *standard-output* oldpoport)
;	    (return t)))

(defun current-column (&optional (p (standard-output)))
      (oport p)
      (cdddr p))

(defun line-length (&optional (p (standard-output)))
      (oport p)
      (caddr p))

(defun set-line-length! (len &optional (p (standard-output)))
      (oport p)
      (rplaca (cddr p) len)
      len))

(defun standardread ()
      (let ((port keyboard))
	   (let ((val (read (iport port) nil (eof))))
		(cond (ptport (print val ptport) (terpri ptport)))
		val)))

(defun standardprint (x)
      (modeprint x schpoport 'print))

(defun safe-read (readfn port)
      (let ((val (handler-case (list (apply readfn (list (iport port) nil (eof))))
			       (stream-error ()
				       (standardprint readfn)
				       (standardprint ": Read error with port ")
				       (standardprint port)
				       (standardprint ".")
				       (new-line schpoport)
				       ()))))
	 (rplacd colpair 0)
	 (cond
	    ((null val) (handler-case (read-char (iport port) nil (eof))
				      (stream-error ()
					     (standardprint "Cannot read further from port ")
					     (standardprint port)
					     (standardprint ".")
					     (new-line schpoport)
					     ()))
			(if (not (eq keyboard port))
			    (progn (schclose port) (reset))))
	    (t (car val)))))


; Pretty-printer improvements by Jeff Brennan:

; Here's the printmacro for def.  The original $prdf had some special code
; for lambda and nlambda.

;(proclaim '(special $outport$ x)
;    for-each
;    expand-fe
;)

;(eval-when (compile eval)
;   (defun expand-fe (form)
;      (prog (vars body)
;            (return
;             (cons (cond ((memq (car form)
;                                (quote
;                                 (map mapc
;                                      mapcan
;                                      mapcar
;                                      mapcon
;                                      mapconc
;                                      maplist)))
;                          (car form))
;                         (t 'mapc))
;                   (progn (setq vars (car form))
;                          (if (atom vars) (setq vars (list vars)))
;                          (cons (cons 'function
;                                    (cons
;                                        (cons 'lambda
;                                             (cons vars
;                                                   (setq body
;                                                         (Cnth (cddr ; ??jg??
;                                                                form)
;                                                               (length
;                                                                vars)))))
;					nil))
;                                (ldiff (cdr form) body))))))))

;(defmacro for-each (&rest l)
;	(expand-fe l))

;(defun printdef (l lmar rmar)
;    (cond ((and (zerop lmar)		; only if we're really printing a defn
;                (zerop rmar)
;                (cadr l)
;                (atom (cadr l))
;                (caddr l)
;                (null (cdddr l))
;                (eq (caaddr l) 'lambda)
;                (null (cdr (last (caddr l)))))
;           (princ '|(| $outport$)
;           (princ (car l) $outport$)
;           (princ '| | $outport$)
;           (princ (cadr l) $outport$)
;           (terpri $outport$)
;           (princ '|  (| $outport$)
;           (princ (caaddr l) $outport$)
;           (princ '| | $outport$)
;           (princ (cadaddr l) $outport$)
;           (terpri $outport$)
;           (for-each x (cddaddr l) ($prdf x 4 0)) ; ??jg??
;           (princ '|))| $outport$)
;           t)))

;(setf (get 'def 'printmacro) 'printdef)

;(defun printmkmac (l lmar rmar)
;    (cond ((and (zerop lmar)
;	        (zerop rmar)
;		(cadr l))
;           (princ '|(| $outport$)
;           (princ (car l) $outport$)
;           (princ '| | $outport$)
;           (princ (cadr l) $outport$)
;           (terpri $outport$)
;	   ($prdf (caddr l) 7 0) ; ??jg??
;	   (princ '|)| $outport$)
;	   t)))


;;
;;	simpler version which
;;	should look nice for lambda's also.(inside mapcar's) -dhl
;;

;(defun print-lambda (l lmar rmar)
;  (prog (col)
;	(cond ((cdr (last l)) (return nil)))
;	(setq col (1+ lmar))
;	(princ '|(| $outport$)
;	(princ (car l) $outport$)
;       	(princ '| | $outport$)
;        (print (cadr l) $outport$)
;	(mapc '(lambda (x)
;		 ($prdf x (+ lmar 2) rmar))
;			(cddr l))
;        (princ '|)| $outport$)
;	(terpri $outport$)
;	(tab lmar $outport$)
;	(return t)))

;(setf (get 'macro 'printmacro) 'printdef)
;(setf (get 'define 'printmacro) 'printdef)
;(setf (get 'mkmac 'printmacro) 'printmkmac)
;(setf (get 'lambda 'printmacro) 'print-lambda)
;(setf (get 'mulambda 'printmacro) 'print-lambda)


