; -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; File module.scm / Copyright (c) 1991 Jonathan Rees / See file COPYING

;;;; Signatures, program environments, modules

;(schi::set-file-context! si:fdefine-file-pathname schi::scheme-translator-context)

; Signatures

(define signature-rtd
  (make-record-type 'signature '(id names aux-names)))

(define make-signature
  (record-constructor signature-rtd '(id names aux-names)))

(define signature-names (record-accessor signature-rtd 'names))
(define signature-aux-names (record-accessor signature-rtd 'aux-names))

; SIGNATURE-REF returns one of
;   #F       if the name is not exported
;   PUBLIC   if exported as a value
;   PRIVATE  if exported as an auxiliary value

;+++ This can be slow if SIG exports many variables (as does the r^4
; signature).  If this becomes a problem, change it so that it does a
; table lookup (after some threshold size?).

(define (signature-ref sig name)
  (cond ((memq name (signature-names sig)) 'public)
	(else #f)))

(define (signature-ref-aux sig name)
  (cond ((memq name (signature-names sig)) 'public)
	((memq name (signature-aux-names sig)) 'private)
	(else #f)))


; Program (i.e. top-level) environments contain macro definitions.

(define program-env-rtd
  (make-record-type 'program-env '(id use-list table package)))
(define program-env-id       (record-accessor program-env-rtd 'id))
(define program-env-use-list (record-accessor program-env-rtd 'use-list))
(define program-env-table    (record-accessor program-env-rtd 'table))
(define program-env-package  (record-accessor program-env-rtd 'package))
(define program-env? (record-predicate program-env-rtd))

(define make-program-env
  (let ((create (record-constructor program-env-rtd
				    '(id use-list table package))))
    (lambda (id use-list)
      (let ((env
	     (create id
		     use-list
		     (make-table)
		     (make-package-using id (map module-package use-list)))))
	(init-environment-for-syntax! env)
	env))))

(define-record-discloser program-env-rtd
  (lambda (r) (list "Program-env" (program-env-id r))))

; Careful, name need not be a symbol

(define (program-env-lookup program-env name)
  (or (table-ref (program-env-table program-env) name)
      (let ((q? (and (symbol? name)
		     (qualified-symbol? name))))
	(or (and (not q?)
		 (let loop ((mods (program-env-use-list program-env)))
		   (and (not (null? mods))
			(or (module-ref (car mods) name)
			    (loop (cdr mods))))))
	    ;; SIDE EFFECT!  Not so good.
	    (let ((node (make-program-variable
			 name
			 (if q?
			     name
			     (scheme-hacks:intern-renaming-perhaps
			      (name->string name)
			      (program-env-package program-env))))))
	      (table-set! (program-env-table program-env) name node)
	      node)))))

(define (program-env-define! program-env name binding)
  (table-set! (program-env-table program-env) name binding))

(define client-lookup program-env-lookup)   ;for classify
(define client-define! program-env-define!) ;for classify


; Get the environment in which to evaluate transformer procedure expressions.

(define environment-for-syntax-key
  (list 'environment-for-syntax-key))  ;any unique id

(define (get-environment-for-syntax env)
  (force (lookup env environment-for-syntax-key)))

(define (define-transformer-env! env t-env-promise)
  (define! env environment-for-syntax-key t-env-promise))

(define (init-environment-for-syntax! env)
  (define-transformer-env! env
    (delay (make-program-env
	    (string->symbol
	     (string-append (symbol->string (program-env-id env))
			    "[META]"))
	    (list revised^4-scheme-module)))))


; A module is a pair <signature, program-environment>.
; Pavel Curtis would prefer to call these things "interfaces".

(define module-rtd
  (make-record-type 'module '(id sig program-env package)))
(define make-module
  (let ((create
	 (record-constructor module-rtd '(id sig program-env package))))
    (lambda (id sig env)
      (create id sig env
	      (make-package-exporting
	           id
		   (let ((ppackage (program-env-package env)))
		     (map (lambda (name)
			    (scheme-hacks:intern-renaming-perhaps
		                 (symbol->string name) ppackage))
			  (signature-names sig))))))))


(define module-id        (record-accessor module-rtd 'id))
(define module-signature (record-accessor module-rtd 'sig))
(define module-program-env (record-accessor module-rtd 'program-env))
(define module-package   (record-accessor module-rtd 'package))

(define-record-discloser module-rtd
  (lambda (r) (list "Module" (module-id r))))

(define (module-ref mod name)
  (if (eq? (signature-ref (module-signature mod) name)
	   'public)
      (program-env-lookup (module-program-env mod) name)
      #f))
