/* linked lists module */

#include <stdio.h>
#include "mlists.h"

#ifndef lint
static char SccsId[] = "@(#)mlists.c	1.2 1/9/90";
#endif

/* list memory allocator struct (must be initialized) */
Mem listAlloc = (Mem)NULL;
     
/* Initialize the memory allocator (to have nblocks list nodes) used by
 * the list routines.  The allocator, listAlloc, is global. You can
 * share the allocator with other data types or use another type's
 * allocator by setting listAlloc appropriately (in this case, do not
 * call listInit). */
void listInit(nblocks)
     int nblocks;
{
    listAlloc = memNew(ALIGNSIZE(sizeof (listnode_t)), nblocks);
}

/* Return an empty list */
List listNew()
{
   List new;

   new = (List)memBlock(listAlloc);
   *new = (list_t)NULL;
   return new;
}

static list_t listNewNode(item)
     Ptr item;
{
   list_t new;

   new = (list_t)memBlock(listAlloc);
   new->item = item;
   new->next = (list_t)NULL;
   return new;
}

static void listFreeNode(node, userfree)
     list_t node;
     F_VOID userfree;
{
   if (node) {
      if (userfree && node->item) (*userfree)(node->item);
      memFreeBlock((Ptr)node, listAlloc);
   }
}

/* Add item to the front of l */
void listPush(item, l)
     Ptr item;
     List l;
{
    list_t new;

    new = listNewNode(item);
    new->next = *l;
    *l = new;
}

/* Add item to the front of l and return a pointer to item */
Ptr *listRef(item, l)
     Ptr item;
     List l;
{
    list_t new;
    new = listNewNode(item);
    new->next = *l;
    *l = new;

    return &(new->item);
}

/* Return the first item in l or NULL if l is empty. */
Ptr listPeek(l)
     List l;
{
    return listEmpty(l) ? (Ptr)NULL : (*l)->item;
}

/* Remove the first item in the list and return it or NULL if l is empty. */
Ptr listPop(l)
     List l;
{
    list_t temp;
    Ptr stuff;

    if (listEmpty(l)) return (Ptr)NULL;
   
    temp = *l;
    *l = temp->next;
    stuff = temp->item;
    memFreeBlock((Ptr)temp, listAlloc);
    return stuff;
}

/* return the list starting with the next item in l or NULL if l has one or
 * less items. */
List listNext(l)
     List l;
{
    List next;

    if (listEmpty(l)) return (List)NULL;

    next = &((*l)->next);
    if (listEmpty(next)) return (List)NULL;
    return next;
}

/* Apply f with (item, arg2) to each item in l */
void listApply(l, f, arg2)
     List l;
     F_VOID f;
     Ptr arg2;
{
    list_t cur;

    if (listEmpty(l)) return ;
    for (cur = *l; cur; cur = cur->next)
	(*f)(cur->item, arg2);
}

/* Apply f with (item) to each item in l */
void listApply1(l, f)
     List l;
     F_VOID f;
{
    list_t cur;

    if (listEmpty(l)) return ;
    for (cur = *l; cur; cur = cur->next)
	(*f)(cur->item);
}

/* Empty l and if f is not NULL free all items with it. */
void listClear(l, f)
     List l;
     F_VOID f;
{
    list_t temp1, temp2;

    if (listEmpty(l)) return ;
    for (temp1 = *l; temp1; temp1 = temp2) {
	temp2 = temp1->next;
	listFreeNode(temp1, f);
    }
    (*l) = (list_t)NULL;
}

/* Free all memory used by l and if f is not NULL free all items with it. */
void listFree(l, f)
     List l;
     F_VOID f;
{
    listClear(l, f);
    memFreeBlock((Ptr)l, listAlloc);
}

/* Find the item in l that satisfies (*test)(item, arg2) or NULL if test
 * fails on all items. If test is NULL, it defaults to == */
Ptr listFind(l, test, arg2)
     List l;
     F_BOOLEAN test;
     Ptr arg2;
{
    list_t current;

    if (listEmpty(l)) return (Ptr)NULL;

    if (test == (F_BOOLEAN)NULL) {
	for (current = *l; current; current = current->next)
	    if (current->item == arg2) return arg2;
    }
    else {
	for (current = *l; current; current = current->next)
	    if ((*test)(current->item, arg2)) return current->item;
    }

    return (Ptr)NULL;
}

/* Find the sublist in l starting with the first item that satisfies
 * (*test)(item, arg2).  Return the sublist or NULL if test fails on all
 * items. If test is NULL, == is used. */
List listSublist(l, test, arg2)
     List l;
     F_BOOLEAN test;
     Ptr arg2;
{
    if (listEmpty(l)) return (List)NULL;

    if (test == (F_BOOLEAN)NULL) {
	for (; *l; l = &((*l)->next))
	    if ((*l)->item == arg2) break ;
    }
    else {
	for (; *l; l = &((*l)->next))
	    if ((*test)((*l)->item, arg2)) break ;
    }

    return l;
}

/* Remove the first item in l that satisfies (*test)(item, arg2).  If
 * the item is not found, nothing happens. */
void listDelete(l, test, arg2)
     List l;
     F_BOOLEAN test;
     Ptr arg2;
{
    List temp;

    if (temp = listSublist(l, test, arg2))
	(void)listPop(temp);
}

/* Add obj before the first item in l satisfying (*sort)(obj, item) or
 * at the end if no such items exist. */
void listAddSorted(obj, l, sort)
     Ptr obj;
     List l;
     F_BOOLEAN sort;
{
    if (listEmpty(l)) return ;

    for (; *l; l = &((*l)->next))
	if ((*sort)(obj, (*l)->item)) break ;
    listPush(obj, l);
}

/* Return the number of items in l */
int listLength(l)
     List l;
{
    list_t c;
    int len;

    if (listEmpty(l)) return 0;
    for (c = (*l)->next, len = 1; c; c = c->next, len++);
    return len;
}

/* Destructively reverse the order of items in l */
void listReverse(l)
     List l;
{
    list_t temp, prev, cur;

    prev = (list_t)NULL;
    cur = *l;
    while (cur) {
	temp = cur->next;
	cur->next = prev;
	prev = cur;
	cur = temp;
    }
    *l = prev;
}

		/* merge sort routines */

/* divide List l into Lists a and b.  num is the length of the list. l may
 * no longer be usable */
static void listDivide(l, a, b, num)
     List l, a, b;
     int num;
{
    list_t temp;

    for (num = num>>1, temp = *l; num > 1; num--, temp = temp->next);
    *a = *l;
    *b = temp->next;
    temp->next = (list_t)NULL;
}

static Boolean (*list_comp)();

/* Merge a and b into list l.  If list_comp(item from a, item from b) is
 * non-zero then the item from a is merged into l before the item from b. */
static void listMerge(a, b, l)
     List a, b, l;
{
    list_t i1, i2;
    List current;

    i1 = *a;
    i2 = *b;
    current = l;

    while (i1 && i2) {
	if ((*list_comp)(i1->item, i2->item)) {
	    *current = i1;
	    current = &(i1->next);
	    i1 = i1->next;
	}
	else {
	    *current = i2;
	    current = &(i2->next);
	    i2 = i2->next;
	}
    }
    /* either one or both lists have been fully processed */
    if (i1)
	*current = i1;
    else if (i2) *current = i2;
    else *current = (list_t)NULL;
}

/* apply merge sort to l (l's length is len) */
static List listMergeSort(l, len)
     List l;
     int len;
{
    list_t a, b;

    if (len <= 1) return l;
  
    listDivide(l, &a, &b, len);
    listMerge(listMergeSort(&a, len>>1), listMergeSort(&b, len - (len>>1)), l);
    return l;
}

/* Sort the items in l so that (*comp)(item, next item) is always TRUE. */
void listSort(l, comp)
     List l;
     F_BOOLEAN comp;
{
    list_comp = comp;
    (void)listMergeSort(l, listLength(l));
}
