/* math primitives */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#include "fools.h"
#include "utils.h"
#include "prim.h"
#include <math.h>

#ifndef lint
static char SccsId[] = "@(#)mathprim.c	1.10 3/4/90";
#endif

#define CHECKARG(x) typeCheck(argv[(x)], Number)

#define UNARY() ASSERT(argc == 1);CHECKARG(0)
#define BINARY() ASSERT(argc == 2);CHECKARG(0);CHECKARG(1)

DEFINE(primExactp)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Number);
    return checkCond(argv[0], EXACT) ? TrueSymb : NilSymb;
}

DEFINE(primInexactp)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Number);
    return checkCond(argv[0], EXACT) ? NilSymb : TrueSymb;
}

/* simple arithmetic */

/* if dbl is in the range of an integer return an integer value */
#define retInteger(dbl)\
if (fabs((dbl)) <= (double)0x7fffffff)\
    return newInteger(gcNew, (long)(dbl))

DEFINE(primAdd)
{
    Boolean ints = TRUE;
    Obj num;
    double dres;

    dres = (double)0;
    while (--argc >= 0) {
	num = *argv++;
	if (CLASS(num) == Number) ints = FALSE;
	else typeCheck(num, Integer);

	dres += objNum(num);
    }

    if (ints) retInteger(dres);
    return newNumber(gcNew, dres);
}

DEFINE(primMul)
{
    Boolean ints = TRUE;
    Obj num;
    double dres;

    dres = (double)1;
    while (--argc >= 0) {
	num = *argv++;
	if (CLASS(num) == Number) ints = FALSE;
	else typeCheck(num, Integer);

	dres *= objNum(num);
    }

    if (ints) retInteger(dres);
    return newNumber(gcNew, dres);
}

DEFINE(primDiv)
{
    Obj num = *argv++;
    double res;

    typeCheck(num, Number);
    if (argc == 1) res = (double)1 / objNum(num);
    else {
	res = objNum(num);
	while (--argc > 0) {
	    typeCheck(num = *(argv++), Number);
	    res /= objNum(num);
	}
    }

    return newNumber(gcNew, res);
}

DEFINE(primQuotient)
{
    ASSERT(argc == 2);

    typeCheck(argv[0], Integer);
    typeCheck(argv[1], Integer);
    return newInteger(gcNew, objInteger(argv[0]) / objInteger(argv[1]));
}

DEFINE(primRemainder)
{
    long n, d;

    ASSERT(argc == 2);

    typeCheck(argv[0], Integer);
    typeCheck(argv[1], Integer);
    n = objInteger(argv[0]);
    d = objInteger(argv[1]);

    return newInteger(gcNew, n - (n/d) * d);
}

DEFINE(primSub)
{
    Boolean ints = TRUE;
    Obj num;
    double dres;

    num = *argv++;
    if (CLASS(num) == Number) ints = FALSE;
    else typeCheck(num, Integer);

    dres = objNum(num);
    if (argc == 1)
	dres = -dres;
    else {
	while (--argc > 0) {
	    num = *argv++;
	    if (CLASS(num) == Number) ints = FALSE;
	    else typeCheck(num, Integer);

	    dres -= objNum(num);
	}
    }
    if (ints) retInteger(dres);
    return newNumber(gcNew, dres);
}

/* number comparison */

DEFINE(primEq)
{
    BINARY();
    return (objNum(argv[0]) == objNum(argv[1])) ? TrueSymb : FalseSymb;
}

DEFINE(primGt)
{
    BINARY();
    return (objNum(argv[0]) > objNum(argv[1])) ? TrueSymb : FalseSymb;
}

DEFINE(primGE)
{
    BINARY();
    return (objNum(argv[0]) >= objNum(argv[1])) ? TrueSymb : FalseSymb;
}

DEFINE(primLt)
{
    BINARY();
    return (objNum(argv[0]) < objNum(argv[1])) ? TrueSymb : FalseSymb;
}

DEFINE(primLE)
{
    BINARY();
    return (objNum(argv[0]) <= objNum(argv[1])) ? TrueSymb : FalseSymb;
}

/* trig functions */

DEFINE(primSin)
{
    UNARY();
    return newNumber(gcNew, sin(objNum(argv[0])));
}

DEFINE(primCos)
{
    UNARY();
    return newNumber(gcNew, cos(objNum(argv[0])));
}


DEFINE(primTan)
{
    UNARY();
    return newNumber(gcNew, tan(objNum(argv[0])));
}

DEFINE(primAsin)
{
    UNARY();
    return newNumber(gcNew, asin(objNum(argv[0])));
}

DEFINE(primAcos)
{
    UNARY();
    return newNumber(gcNew, acos(objNum(argv[0])));
}

DEFINE(primAtan)
{
    if (argc > 2)
	errorPrint(BadArgs, "to atan (expects 1 or 2)");
    else if (argc == 2) {
	BINARY();
	return newNumber(gcNew, atan2(objNum(argv[0]), objNum(argv[1])));
    }
    else if (argc == 1) {
	UNARY();
	return newNumber(gcNew, atan(objNum(argv[0])));
    }
}

/* more math functions */

extern long random();

DEFINE(primRandom)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Integer);
    return newInteger(gcNew, random() % objInteger(argv[0]));
}

DEFINE(primSrandom)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Integer);
    srandom((int)objInteger(argv[0]));
    return argv[0];
}

DEFINE(primFloor)
{
    if (objIsClass(argv[0], Integer)) return argv[0];
    typeCheck(argv[0], Number);
    return newNumber(gcNew, floor(objNum(argv[0])));
}

DEFINE(primCeiling)
{
    if (objIsClass(argv[0], Integer)) return argv[0];
    typeCheck(argv[0], Number);
    return newNumber(gcNew, ceil(objNum(argv[0])));
}

DEFINE(primAbs)
{
    if (objIsClass(argv[0], Integer)) {
	long val = objInteger(argv[0]);

	return val >= 0 ? argv[0] : newInteger(gcNew, -val);
    }
    typeCheck(argv[0], Number);
    return newNumber(gcNew, fabs(objNum(argv[0])));
}

DEFINE(primRound)
{
#if defined(sequent) || defined(DGUX)

    double num;

    if (objIsClass(argv[0], Integer)) return argv[0];
    typeCheck(argv[0], Number);
    num = objNum(argv[0]);
    return newNumber(gcNew, num < 0.0 ? ceil(num - 0.5) : floor(num + 0.5));

#else

    extern double rint();

    if (objIsClass(argv[0], Integer)) return argv[0];
    typeCheck(argv[0], Number);
    return newNumber(gcNew, rint(objNum(argv[0])));

#endif /* defined(sequent) || defined(DGUX) */
}

DEFINE(primPow)
{
    BINARY();
    return newNumber(gcNew, pow(objNum(argv[0]), objNum(argv[1])));
}

DEFINE(primExp)
{
    UNARY();
    return newNumber(gcNew, exp(objNum(argv[0])));
}

DEFINE(primLog)
{
    UNARY();
    return newNumber(gcNew, log(objNum(argv[0])));
}

DEFINE(primLog10)
{
    UNARY();
    return newNumber(gcNew, log10(objNum(argv[0])));
}

#if !defined(DGUX) && !defined(sequent)

DEFINE(primExpm1)
{
    extern double expm1();
    UNARY();
    return newNumber(gcNew, expm1(objNum(argv[0])));
}

DEFINE(primLog1p)
{
    extern double log1p();
    UNARY();
    return newNumber(gcNew, log1p(objNum(argv[0])));
}

#endif /* defined(DGUX) || defined(sequent) */

static struct {
    char *name;
    F_OBJ op;
    int mask, nargs;
} mathops[] = {
    /* name		op		mask	nargs */

    { "exact?",		primExactp,	0,	1 },
    { "inexact?",	primInexactp,	0,	1 },
    { "+",		primAdd,	OPTARG,	2 },
    { "*",		primMul,	OPTARG,	2 },
    { "/",		primDiv,	OPTARG,	1 },
    { "-",		primSub,	OPTARG,	1 },
    { "quotient",	primQuotient,	0,	2 },
    { "remainder",	primRemainder,	0,	2 },
    { "=",		primEq,		0,	2 },
    { ">",		primGt,		0,	2 },
    { ">=",		primGE,		0,	2 },
    { "<",		primLt,		0,	2 },
    { "<=",		primLE,		0,	2 },
    { "sin",		primSin,	0,	1 },
    { "asin",		primAsin,	0,	1 },
    { "cos",		primCos,	0,	1 },
    { "acos",		primAcos,	0,	1 },
    { "tan",		primTan,	0,	1 },
    { "atan",		primAtan,	OPTARG,	1 },
    { "floor",		primFloor,	0,	1 },
    { "ceiling",	primCeiling,	0,	1 },
    { "abs",		primAbs,	0,	1 },
    { "round",		primRound,	0,	1 },
    { "random",		primRandom,	0,	1 },
    { "srandom",	primSrandom,	0,	1 },
    { "expt",		primPow,	0,	2 },
    { "exp",		primExp,	0,	1 },
    { "ln",		primLog,	0,	1 },
    { "log",		primLog10,	0,	1 },
#if !defined(DGUX) && !defined(sequent)
    { "ln1+",		primLog1p,	0,	1 },
    { "exp-1",		primExpm1,	0,	1 },
#endif /* defined(DGUX) || defined(sequent) */
};

void mathInit()
{
    int i;

    for (i = 0; i < sizeof (mathops) / sizeof (*mathops); i++)
	(void)newPrim(gcNew, mathops[i].name, mathops[i].op, GlobalEnv,
		      mathops[i].mask, mathops[i].nargs);
}	
