/* ECOFF support written by George Hartzell
 * MACH-O support based on code by Chris Maeda (cmaeda@cs.cmu.edu)
 */

#include "scheme.h"

#ifdef CAN_LOAD_OBJ
#ifdef COFF
#  include <filehdr.h>
#  include <syms.h>
#  undef TYPE         /* ldfnc.h defines a TYPE macro. */
#  include <ldfcn.h>
#  undef TYPE
#  define TYPE(x) ((int)((x) >> TYPESHIFT))
#else
#ifdef ECOFF
#  include <filehdr.h>
#  include <aouthdr.h>
#  include <scnhdr.h>
#  include <syms.h>
#else
#ifdef MACH_O
#  include <rld.h>
#else
#  include <a.out.h>
#  include <sys/types.h>
#endif
#endif
#endif
#endif

static Object V_Load_Path, V_Load_Noisilyp, V_Load_Libraries;

#ifdef CAN_LOAD_OBJ
#  ifdef STD_LOAD_LIBS
#    define Default_Load_Libraries STD_LOAD_LIBS
#  else
#    define Default_Load_Libraries "-lc"
#  endif
#else
#  define Default_Load_Libraries ""
#endif

#if defined(CAN_DUMP) || defined(CAN_LOAD_OBJ)
char Loader_Input[20];
#endif

#ifdef CAN_LOAD_OBJ
static char Loader_Output[20];
#ifdef ECOFF
struct headers {
    struct filehdr fhdr;
    struct aouthdr aout;
    struct scnhdr section[3];    
};
#endif
#endif

Init_Load () {
    Define_Variable (&V_Load_Path, "load-path",
	Cons (Make_String (".", 1),
	Cons (Make_String (SCM_DIR, sizeof (SCM_DIR) - 1),
	Cons (Make_String (LIB_DIR, sizeof (LIB_DIR) - 1), Null))));
    Define_Variable (&V_Load_Noisilyp, "load-noisily?", False);
    Define_Variable (&V_Load_Libraries, "load-libraries", 
	Make_String (Default_Load_Libraries, sizeof Default_Load_Libraries-1));
}

static Object Is_O_File (name) Object name; {
    register char *p;
    register struct S_String *str;

    if (TYPE(name) == T_Symbol)
	name = SYMBOL(name)->name;
    str = STRING(name);
    p = str->data + str->size;
    return str->size >= 2 && *--p == 'o' && *--p == '.';
}

Check_Loadarg (x) Object x; {
    Object tail;
    register t = TYPE(x);

    if (t == T_Symbol || t == T_String)
	return;
    if (t != T_Pair)
	Wrong_Type_Combination (x, "string, symbol, or list");
    for (tail = x; !Nullp (tail); tail = Cdr (tail)) {
	Object f = Car (tail);
	if (TYPE(f) != T_Symbol && TYPE(f) != T_String)
	    Wrong_Type_Combination (f, "string or symbol");
	if (!Is_O_File (f))
	    Primitive_Error ("~s: not an object file", f);
    }
}

Object General_Load (what, env) Object what, env; {
    Object oldenv;
    GC_Node;

    Check_Type (env, T_Environment);
    oldenv = The_Environment;
    GC_Link (oldenv);
    Switch_Environment (env);
    Check_Loadarg (what);
    if (TYPE(what) == T_Pair)
#ifdef CAN_LOAD_OBJ
	Load_Object (what)
#endif
	;
    else if (Is_O_File (what))
#ifdef CAN_LOAD_OBJ
	Load_Object (Cons (what, Null))
#endif
	;
    else
	Load_Source (what);
    Switch_Environment (oldenv);
    GC_Unlink;
    return Void;
}

Object P_Load (argc, argv) register argc; register Object *argv; {
    return General_Load (argv[0], argc == 1 ? The_Environment : argv[1]);
}

Load_Source (name) Object name; {
    Object port, val;
    GC_Node;

    port = General_Open_File (name, P_INPUT, Var_Get (V_Load_Path));
    GC_Link (port);
    while (1) {
	val = General_Read (port, 1);
	if (TYPE(val) == T_End_Of_File)
	    break;
	val = Eval (val);
	if (Truep (Var_Get (V_Load_Noisilyp))) {
	    Print (val);
	    P_Newline (0);
	}
    }
    P_Close_Input_Port (port);
    GC_Unlink;
}

#ifdef CAN_LOAD_OBJ

#ifdef MACH_O
Load_Object (names) Object names; {
    long retval;
    struct mach_header *hdr;
    char **filenames;
    NXStream *err_stream;
    register i, n;
    Object port, tail, fullnames;
    GC_Node3;
    Alloca_Begin;

    port = tail = fullnames = Null;
    GC_Link3 (port, tail, fullnames);
    for (n = 0, tail = names; !Nullp (tail); n++, tail = Cdr (tail)) {
	port = General_Open_File (Car (tail), P_INPUT, Var_Get (V_Load_Path));
	fullnames = Cons (PORT(port)->name, fullnames);
	P_Close_Input_Port (port);
    }
    GC_Unlink;

    Alloca (filenames, char**, (n+1)*sizeof(char *));
    for (i = 0; i < n; i++, fullnames = Cdr (fullnames)) {
	register struct S_String *str = STRING(Car (fullnames));
	Alloca (filenames[i], char*, str->size+1);
	bcopy (str->data, filenames[i], str->size);
	filenames[i][str->size] = '\0';
    }
    filenames[i] = 0;

    Disable_Interrupts;
    /* Construct a stream for error logging:
     */
    if ((err_stream = NXOpenFile (fileno (stderr), NX_WRITEONLY)) == 0)
	Primitive_Error ("NXOpenFile failed");

    retval = rld_load (err_stream, /* report error messages here */
	&hdr,                      /* return header address here */
	filenames,                 /* load these */
	"/dev/null");              /* doesn't work if NULL?! */
    NXClose (err_stream);
    if (retval != 1)
	Primitive_Error ("rld_load() failed");

    /* Grab the symbol table from the just-loaded file:
     */
    if (The_Symbols)
	Free_Symbols (The_Symbols);
    The_Symbols = Snarf_Symbols (hdr);
    if (!Call_Initializers (The_Symbols, 0))
	Primitive_Error ("no initializers");
    Enable_Interrupts;
    Alloca_End;
}

#else /* MACH_O */

Load_Object (names) Object names; {
#ifdef ECOFF
    struct headers hdr;
#else
    struct exec hdr;
#endif
    register char *brk, *obrk, *buf, *lp, *li;
    register n, f, len, liblen;
    Object port, tail, fullnames, libs;
    FILE *fp;
    GC_Node3;
    Alloca_Begin;

    li = Loader_Input;
    if (li[0] == 0)
	li = A_Out_Name;
    strcpy (Loader_Output, "/tmp/ldXXXXXX");
    mktemp (Loader_Output);

    port = tail = fullnames = Null;
    GC_Link3 (port, tail, fullnames);
    for (len = 0, tail = names; !Nullp (tail); tail = Cdr (tail)) {
	port = General_Open_File (Car (tail), P_INPUT, Var_Get (V_Load_Path));
	fullnames = Cons (PORT(port)->name, fullnames);
	len += STRING(Car (fullnames))->size + 1;
	P_Close_Input_Port (port);
    }
    GC_Unlink;

    libs = Var_Get (V_Load_Libraries);
    if (TYPE(libs) == T_String) {
        liblen = STRING(libs)->size;
	lp = STRING(libs)->data;
    } else {
	liblen = 3; lp = "-lc";
    }

    Alloca (buf, char*, strlen (A_Out_Name) + len + liblen + 100);

    obrk = brk = sbrk (0);
    brk = (char *)((int)brk + 7 & ~7);

#ifdef XFLAG_BROKEN
    sprintf (buf, "/bin/ld -N -A %s -T %x -o %s ",
#else 
#ifdef hp9000s300
    sprintf (buf, "/bin/ld -N -x -A %s -R %x -o %s ",
#else
#ifdef USE_GNULD
    sprintf (buf, "/usr/local/lib/gcc-ld -N -x -A %s -T %x -o %s ",
#else
    sprintf (buf, "/bin/ld -N -x -A %s -T %x -o %s ",
#endif
#endif
#endif
	li, brk, Loader_Output);

    for (tail = fullnames; !Nullp (tail); tail = Cdr (tail)) {
	register struct S_String *str = STRING(Car (tail));
	strncat (buf, str->data, str->size);
	strcat (buf, " ");
    }
    strncat (buf, lp, liblen);

    if (Verbose)
	printf ("[%s]\n", buf);
    if (system (buf) != 0) {
	(void)unlink (Loader_Output);
	Primitive_Error ("system linker failed");
    }
    Disable_Interrupts;               /* To ensure that f gets closed */
    if ((f = open (Loader_Output, 0)) == -1) {
	(void)unlink (Loader_Output);
	Primitive_Error ("cannot open tempfile");
    }
    if (Loader_Input[0])
	(void)unlink(Loader_Input);
    strcpy (Loader_Input, Loader_Output);
    if (read (f, (char *)&hdr, sizeof (hdr)) != sizeof (hdr)) {
err:
	close (f);
	Primitive_Error ("corrupt tempfile (/bin/ld is broken)");
    }
#ifdef ECOFF
    n = hdr.aout.tsize + hdr.aout.dsize + hdr.aout.bsize;
#else
    n = hdr.a_text + hdr.a_data + hdr.a_bss;
#endif
    n += brk - obrk;
    if (sbrk (n) == (char *)-1) {
	close (f);
	Primitive_Error ("not enough memory to load object file");
    }
    bzero (obrk, n);
#ifdef ECOFF
    n -= hdr.aout.bsize;
    (void)lseek (f, (long)hdr.section[0].s_scnptr, 0);
#else
    n -= hdr.a_bss;
#endif
    if (read (f, brk, n) != n)
	goto err;
    if ((fp = fdopen (f, "r")) == NULL) {
	close (f);
	Primitive_Error ("cannot fdopen object file");
    }
    if (The_Symbols)
	Free_Symbols (The_Symbols);
    The_Symbols = Snarf_Symbols (fp, &hdr);
    fclose (fp);
    if (!Call_Initializers (The_Symbols, brk))
	Primitive_Error ("no initializers in object file(s)");
    Enable_Interrupts;
    Alloca_End;
}

Finit_Load () {
    if (Loader_Input[0])
	(void)unlink (Loader_Input);
}
#endif /* MACH_O */

#endif /* CAN_LOAD_OBJ */
