;==============================================================================

; file: "env.scm"

;------------------------------------------------------------------------------
;
; Environment manipulation and declaration handling package:
;
;------------------------------------------------------------------------------

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; Environment manipulation:
; ------------------------

; structure that represents variables:

(define (make-var

    name       ; symbol that denotes the variable
    bound      ; procedure node that binds the variable (#f if global)
    refs       ; set of nodes that reference this variable
    sets       ; set of nodes that assign a value to this variable
    source)    ; source where variable is first encountered

  (vector var-tag name bound refs sets source #f))

(define (var? x)
  (and (vector? x)
       (> (vector-length x) 0)
       (eq? (vector-ref x 0) var-tag)))

(define (var-name x)          (vector-ref x 1))
(define (var-bound x)         (vector-ref x 2))
(define (var-refs x)          (vector-ref x 3))
(define (var-sets x)          (vector-ref x 4))
(define (var-source x)        (vector-ref x 5))
(define (var-info x)          (vector-ref x 6))
(define (var-name-set! x y)   (vector-set! x 1 y))
(define (var-bound-set! x y)  (vector-set! x 2 y))
(define (var-refs-set! x y)   (vector-set! x 3 y))
(define (var-sets-set! x y)   (vector-set! x 4 y))
(define (var-source-set! x y) (vector-set! x 5 y))
(define (var-info-set! x y)   (vector-set! x 6 y))

(define var-tag (list 'VAR-TAG))

(define (var-copy var)
  (make-var (var-name var)
            #t
            (set-empty)
            (set-empty)
            (var-source var)))


; temporary variables are used to name intermediate values

(define (make-temp-var name)
  (make-var name #t (set-empty) (set-empty) #f))

(define (temp-var? var)
  (eq? (var-bound var) #t))

; special variable used to denote the return address of a procedure

(define ret-var (make-temp-var 'ret))
(define ret-var-set (set-singleton ret-var))

; special variable used to denote the pointer to the closed variables

(define closure-env-var (make-temp-var 'closure-env))

; special variable used to denote empty slots

(define empty-var (make-temp-var #f))


; structure that represents environments:

(define (make-empty-environment)
  (let ((global-env (list (make-var #f #f #f #f #f)))) ; dummy entry for global var.
    (vector global-env
            global-env
            '()  ; macro definitions
            '()  ; declarations
            0))) ; local counter to generate unique names

(define (lookup name env source)
  (cond ((eq? (var-name (car env)) name)
         (car env))
        ((null? (cdr env)) ; it's a new global variable
         (set-cdr! env
           (list (make-var name
                           #f
                           (set-empty)
                           (set-empty) 
                           source)))
         (cadr env))
        (else
         (lookup name (cdr env) source))))

(define (lookup-var name env source)
  (lookup name (vector-ref env 0) source))

(define (lookup-global-var name env)
  (lookup name (vector-ref env 1) #f))

(define (lookup-macro name env)
  (let ((x (assq name (vector-ref env 2))))
    (if x (cdr x) #f)))

(define (new-frame vars env)
  (vector (append vars (vector-ref env 0))
          (vector-ref env 1)
          (vector-ref env 2)
          (vector-ref env 3)))

(define (new-macro name proc env)
  (vector (vector-ref env 0)
          (vector-ref env 1)
          (cons (cons name proc) (vector-ref env 2))
          (vector-ref env 3)))

(define (new-decls decls env)
  (vector (vector-ref env 0)
          (vector-ref env 1)
          (vector-ref env 2)
          (append decls (vector-ref env 3))))

(define (env-declarations env)
  (vector-ref env 3))

(define (env-global-variables env)
  (cdr (vector-ref env 1)))


; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; Declarations:
; ------------
;
; A declaration has the form: (##declare <item1> <item2> ...)
;
; an <item> can be one of 5 types:
;
; - flag declaration           : (<id>)
; - parameterized declaration  : (<id> <parameter>)
; - boolean declaration        : (<id>)  or  (NOT <id>)
; - namable declaration        : (<id> <name>...)
; - namable boolean declaration: (<id> <name>...)  or  (NOT <id> <name>...)

; Declarations table (for parsing):

(define flag-declarations            '())
(define parameterized-declarations   '())
(define boolean-declarations         '())
(define namable-declarations         '())
(define namable-boolean-declarations '())

(define (define-flag-decl name type)
  (set! flag-declarations (cons (cons name type) flag-declarations))
  '())

(define (define-parameterized-decl name)
  (set! parameterized-declarations (cons name parameterized-declarations))
  '())

(define (define-boolean-decl name)
  (set! boolean-declarations (cons name boolean-declarations))
  '())

(define (define-namable-decl name type)
  (set! namable-declarations (cons (cons name type) namable-declarations))
  '())

(define (define-namable-boolean-decl name)
  (set! namable-boolean-declarations (cons name namable-boolean-declarations))
  '())

; Declaration constructors:

(define (flag-decl type val)
  (list type val))

(define (parameterized-decl id parm)
  (list id parm))

(define (boolean-decl id pos)
  (list id pos))

(define (namable-decl type val names)
  (cons type (cons val names)))

(define (namable-boolean-decl id pos names)
  (cons id (cons pos names)))

; Declaration querying:

(define (declaration-value declaration-name element default decls)
  (cond ((null? decls)
         default)
        ((and (eq? (car (car decls)) declaration-name)
              (or (null? (cddr (car decls)))
                  (memq element (cddr (car decls)))))
         (cadr (car decls)))
        (else
         (declaration-value declaration-name element default (cdr decls)))))

;==============================================================================
