#| -*-Scheme-*-

$Header: /scheme/src/edwin/RCS/bochsmod.scm,v 1.7 1992/05/12 17:51:19 mhwu Exp $

Copyright (c) 1991-92 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Bochser mode for edwin

(declare (usual-integrations))

(define (initialize-package!)
  (add-event-receiver!
   (ref-variable select-buffer-hook)
   (lambda (buffer)
     (if (eq? (buffer-major-mode buffer)
	      (ref-mode-object bochser))
	 (attach-bindings buffer #f))))
  (initialize-bochser!))

(define-command bochser-mode
  "Enter Bochser mode."
  ()
  (lambda ()
    (let ((cur-buffer (current-buffer)))
      (set-buffer-major-mode! cur-buffer (ref-mode-object bochser))
      (let* ((environment (evaluation-environment #f))
	     (bochs
	      (if (ref-variable bochsmod-create-env-buffers?)
		  (create-bochs environment)
		  (let ((bochs
			 (with-output-to-mark (current-point)
			   (lambda ()
			     (create-bochs-and-write environment)))))
		    (local-set-variable! bochsmod-bindings-buffer
					 (bochsmod-create-bindings-buffer
					  cur-buffer
					  environment))
		    (local-set-variable! bochsmod-default-scheme-environment
					 environment)
		    (local-set-variable! bochsmod-default-syntax-table
					 (ref-variable scheme-syntax-table))
		    (attach-bindings cur-buffer #f)
		    bochs))))
	(set-bochs/syntax-table! bochs (ref-variable scheme-syntax-table))
	(add-group-move-point-daemon!
	 (buffer-group cur-buffer)
	 (lambda (group new-point old-point)
	   (bochsmod-register-move-point! group new-point old-point)))
	(bochs/open bochs)
	(bochsmod-move-into-bochs bochs)))))

(define-major-mode bochser scheme "Bochser"
  "Major mode for Bochser style Scheme interaction. 
Commands are similar to Scheme mode except:

  \\[bochsmod-open] opens a bochs on the object before point.
  \\[bochsmod-close] closes the current bochs and returns you 
  to the surrounding bochs.
  \\[bochsmod-extract-code] copies the contents of a code bochs
  (around point) into the kill ring.
  \\[bochsmod-delete-window] deletes the bochsmod window and its corresponding
  environment window
  \\[bochsmod-switch-to-buffer] coalesces the bochsmod environment and
  bindings windows, selects a buffer and places it in the coalesced window."

  (local-set-variable! enable-transcript-buffer false)
  (local-set-variable! evaluation-input-recorder bochsmod-input-recorder)
  (local-set-variable! comint-input-ring
		       (make-ring (ref-variable comint-input-ring-size)))
  (local-set-variable! evaluation-output-receiver bochsmod-output-receiver)
  (local-set-variable! bochsmod-internal-bochs-table (make-empty-bochs-table)))

(define-variable-per-buffer bochs
  "The bochs object associated with this buffer. 
This is a read only variable."
  #f)

(define-variable-per-buffer bochsmod-internal-bochs-table
  "A table giving a mapping between marks (signifying the end of a
region containing a bochs) and bochses that are within this buffer.
This is for internal use only."
  #f)

(define-variable-per-buffer bochsmod-bindings-buffer
  "The bochser bindings buffer associated with this 
environment bochs buffer.
This is for internal use only."
  #f)

(define-variable-per-buffer bochsmod-bochs-buffer
  "The bochser buffer associated with this bindings buffer.
This is a read only variable."
  #f)

(define-variable-per-buffer bochsmod-default-scheme-environment
  "The scheme environment used for evaluations that are not within a bochs.
This is a read only variable"
  #f)

(define-variable-per-buffer bochsmod-default-syntax-table
  "The syntax table used for evaluations that are not within a bochs.
This is a read only variable"
  #f)

(define-variable-per-buffer bochsmod-current-environment-bochs
  "The nearest environment bochs surrounding the current point.
This is for internal use only."
  #f)

(define-variable bochsmod-smart-windows
  "Let bochser try to intelligently play with your bochser windows"
  #t)

(define-variable bochsmod-show-bindings
  "Create and show a list of bindings in a window attached to your
bochser windows"
  #t)

(define-variable bochsmod-create-env-buffers?
  "Open environment bochses in their own buffers. Otherwise, open
up a region in the current buffer."
  #f)

(define-variable bochsmod-buffers-in-screens?
  "Let bochsmod create new buffers in new screens. This will also
happen if USE-MULTIPLE-SCREENS is set."
  #f)

(let-syntax
    ((define-read-only
       (macro (var-name)
	 `(define-variable-value-validity-test (ref-variable-object ,var-name)
	    (lambda (variable)
	      (lambda (value)
		value			;don't care
		(if (variable-value variable)
		    (editor-error "The variable "
				  (string-upcase (symbol->string ',var-name))
				  " cannot be reset."))))))))
  (define-read-only bochs)
  (define-read-only bochsmod-bochs-buffer)
  (define-read-only bochsmod-default-scheme-environment))

(define (make-button-point-procedure receiver)
  (lambda ()
    (let ((button-event (current-button-event)))
      (let ((window (button-event/window button-event)))
	(receiver
	 (or (window-coordinates->mark window
				       (button-event/x button-event)
				       (button-event/y button-event))
	     (buffer-end (window-buffer window))))))))

(define open-bochs-before-point
  (lambda (point)
    (bochs/open (find-bochs-before-point
		 point
		 (lambda ()
		   (editor-error
		    "Point must be immediately after a bochs"))))))

(define-command bochsmod-open-bochs-before-point
  "Open a bochs on the object before point."
  "d"
  (lambda (point) (open-bochs-before-point point)))

(define-command bochsmod-open-bochs-before-mouse-point
  "Open a bochs on the object before the mouse point."
  ()
  (make-button-point-procedure open-bochs-before-point))

(define close-bochs-before-point
  (lambda (point)
    (let ((bochs (find-bochs-before-point point
					  (lambda () (ref-variable bochs)))))
      (if (not bochs)
	  (editor-error "No bochs found."))
      (if (bochs/open? bochs)
	  (bochs/close bochs)
	  (editor-error "Bochs is already closed.")))))

(define-command bochsmod-close-bochs-before-point
  "Close a bochs on the object before point. If there is no object
there then close the bochs associated with the buffer"
  "d"
 (lambda (point) (close-bochs-before-point point)))

(define-command bochsmod-close-bochs-before-mouse-point
  "Close a bochs on the object before point. If there is no object
there then close the bochs associated with the buffer"
  ()
  (make-button-point-procedure close-bochs-before-point))

(define close-bochs-around-point
  (lambda (point)
    (let ((bochs (find-bochs-around-point point
					  (lambda () (ref-variable bochs)))))
      (if (not bochs)
	  (editor-error "No bochs found."))
      (if (bochs/open? bochs)
	  (bochs/close bochs)
	  (editor-error "Bochs is already closed."))
      bochs)))

(define-command bochsmod-close-bochs-around-point
  "Close a bochs on the object before point. If there is no object
there then close the bochs associated with the buffer"
  "d"
  (lambda (point)
    (let ((bochs (close-bochs-around-point point)))
      (set-current-point! (region-end
			   (edit-object/region-or-buffer
			    (bochs/closed-edit-object bochs)))))))

(define-command bochsmod-close-bochs-around-mouse-point
  "Close a bochs on the object before the mouse. If there is no object
there then close the bochs associated with the buffer"
  ()
  (make-button-point-procedure close-bochs-around-point))

(define-command bochsmod-extract-code
  "Copy the contents of a code bochs (before point) into the kill ring."
  "d"
  (lambda (point)
    (let* ((bochs (find-bochs-before-point
		   point
		   (lambda ()
		     (editor-error
		      "Point must be immediately after a bochs"))))
	   (object (bochs/object bochs)))
      (if (lambda? object)
	  (copy-code-to-kill-ring object 0)
	  (editor-error "Point must be after a code bochs.")))))

(define-command bochsmod-kill-buffer
  "One arg, a string or a buffer.  Get rid of the specified bochsmod
buffer (and screen if BOCHSMOD-BUFFERS-IN-SCREENS? is true).
  If it's a bochsmod environment buffer, also kill its associated bindings
buffer and window."
  "bKill buffer"
  (lambda (buffer-name)
    (detach-bindings)
    (let ((bindings-buffer (ref-variable bochsmod-bindings-buffer))
	  (bochs (ref-variable bochs)))
      (kill-buffer-interactive (find-buffer buffer-name))
      (if bindings-buffer
	  (kill-buffer-interactive bindings-buffer))
      (if bochs
	  (begin
	    (bochs/close bochs)
	    (set-bochs/open-edit-object! bochs #f)))
      (if (and (ref-variable bochsmod-buffers-in-screens?)
	       (cdr (screen-list)))
	  (delete-screen! (selected-screen))))))

(define-command bochsmod-delete-window
  "Delete the bochsmod window and its corresponding
environment window"
  ()
  (lambda ()
    (detach-bindings)
    ((ref-command delete-window))))

(define-command bochsmod-delete-other-windows
  "Make the current bochsmod environment and bindings windows fill the
screen."
  ()
  (lambda ()
    ((ref-command delete-other-windows))
    (attach-bindings (current-buffer) #t)))

(define-command bochsmod-switch-to-buffer
  "Coalesces the bochsmod environment and bindings windows, selects a
buffer (possibly new) and places it in the coalesced window."
  (prompt-for-select-buffer "Switch to buffer")
  (lambda (buffer-name)
    (bochsmod-select-buffer (find-buffer buffer-name))))

(define-command down-bochs
  "Move forward down one level of bochses.
With argument, do this that many times.
A negative argument means move backward but still go down a level."
  "p"
  (lambda (argument)
    (move-thing forward-down-bochs argument)))

(define-command backward-down-bochs
  "Move backward down one level of bochses.
With argument, do this that many times.
A negative argument means move forward but still go down a level."
  "p"
  (lambda (argument)
    (move-thing backward-down-bochs argument)))

(define-command up-bochs
  "Move forward out one level of bochses.
With argument, do this that many times.
A negative argument means move backward but still to a less deep spot."
  "p"
  (lambda (argument)
    (move-thing forward-up-bochs argument)))

(define-command backward-up-bochs
  "Move backward out one level of bochses.
With argument, do this that many times.
A negative argument means move forward but still to a less deep spot."
  "p"
  (lambda (argument)
    (move-thing backward-up-bochs argument)))

(define-command bochsmod-attach-bindings
  "Attach a bindings buffer to the bochs buffer associated with the current
window"
  ()
  (lambda ()
    (if (not (ref-variable bochsmod-bindings-buffer))
	(local-set-variable! bochsmod-bindings-buffer
			     (bochsmod-create-bindings-buffer
			      (current-buffer)
			      (ref-variable scheme-environment))))
    (attach-bindings (current-buffer) #t)))


(define-structure (edit-object
		   (conc-name EDIT-OBJECT/)
		   (constructor %make-region-edit-object
				(region-or-buffer string string-offset))
		   (constructor %make-buffer-edit-object (region-or-buffer)))
  region-or-buffer			;Where the bochs is.
  (string false)			;If bochs is a region then
					;this is the variable contents
					;of the region. This is
					;neccessary because bad things
					;will happen to the region
					;when an enclosing bochs is
					;closed.
  string-offset				;The offset of the string from
					;the beginning of the bochs region.
  )

(define (editor-open-bochs bochs)

  (define (switch-to-buffer buffer)
    (bochsmod-select-buffer buffer)
    (local-set-variable! bochs bochs)
    (local-set-variable! scheme-environment
			 (bochs/environment bochs))
    (local-set-variable! scheme-syntax-table
			 (bochs/syntax-table bochs))
    (local-set-variable! bochsmod-default-scheme-environment
			 (bochs/environment bochs))
    (local-set-variable! bochsmod-default-syntax-table
			 (bochs/syntax-table bochs))
    (local-set-variable! bochsmod-current-environment-bochs bochs)
    buffer)

  (define (create-buffer in-new-screen?)
    (let ((buffer (bochsmod-create-buffer bochs)))
      (set-buffer-major-mode! buffer (bochs-find-mode bochs))
      (set-bochs/open-edit-object! bochs
				   (%make-buffer-edit-object buffer))
      (if in-new-screen?
	  (let ((new-screen (make-screen buffer)))
	    (select-screen new-screen)))
      (switch-to-buffer buffer)
      (local-set-variable! bochsmod-bindings-buffer
			   (bochsmod-create-bindings-buffer
			    buffer
			    (bochs/environment bochs)))
      (attach-bindings buffer #t)
      buffer))

  (define (find-or-create-buffer)
    (let ((buffer (bochsmod-find-buffer bochs)))
      (if buffer
	  (switch-to-buffer buffer)
	  (create-buffer (and (or (ref-variable bochsmod-buffers-in-screens?)
				  (ref-variable use-multiple-screens))
			      (multiple-screens?))))))

  (let ((object (bochs/object bochs)))
    (cond ((environment? object)
	   (if (ref-variable bochsmod-create-env-buffers?)
	       (find-or-create-buffer)
	       (begin
		 (bochsmod-open-env-region bochs)))
	   (bochsmod-move-into-bochs bochs))
	  ((procedure? object)
	   (bochs/open (proc-bochs/code-bochs bochs))
	   (bochs/open (proc-bochs/environment-bochs bochs))
	   (copy-code-to-kill-ring object (current-column)))
	  (else (bochsmod-expand bochs)))))

(define (bochsmod-open-env-region bochs)
  (let ((edit-object (bochs/open-edit-object bochs)))
    (if edit-object
	(begin
	  (bochsmod-expand bochs)
	  (update-interior-bochs-edit-objects! bochs))
	(replace-bochs-string bochs
			      (bochsmod-open-env-bochs-string bochs) #t))))

(define (bochsmod-open-env-bochs-string bochs)
  (let ((old-string (edit-object/string (bochs/closed-edit-object bochs))))
    (string-append old-string
		   " -------------- \n\n-------------")))

(define (editor-close-bochs bochs)
  (let ((object (bochs/object bochs)))
    (cond
     ((procedure? object)
      (let ((env-bochs (proc-bochs/environment-bochs bochs))
	    (code-bochs (proc-bochs/code-bochs bochs)))
	(if (bochs/open? env-bochs) (bochs/close env-bochs))
	(if (bochs/open? code-bochs) (bochs/close code-bochs))))
     (else 
      (let ((maybe-buffer (edit-object/region-or-buffer
			   (bochs/open-edit-object bochs))))
	(if (buffer? maybe-buffer)
	    (and (eq? maybe-buffer (current-buffer))
		 (bochsmod-select-buffer (bochsmod-find-nearest-buffer bochs)))
	    (begin
	      (bochsmod-unexpand bochs))))))))

(define (bochsmod-select-buffer buffer)
    (let ((bindings-buffer (ref-variable bochsmod-bindings-buffer)))
      (detach-bindings bindings-buffer)
      (select-buffer buffer)))
  
(define (bochs-find-mode bochs)
  (let ((object (bochs/object bochs)))
    (cond ((procedure? object) (ref-mode-object bochser))
	  ((environment? object) (ref-mode-object bochser)))))

(define (bochsmod-expand bochs)
  (let ((object (bochs/object bochs)))
    (cond ((lambda? object)
	   (lisp-indent-sexp
	    (replace-bochs-string bochs (code-string object) #t)))
	  ((environment? object)
	   (replace-bochs-string bochs
				 (edit-object/string
					(bochs/open-edit-object bochs))
				 #t))
	  (else unspecific))))

(define (bochsmod-unexpand bochs)
  (replace-bochs-string bochs
			(edit-object/string (bochs/closed-edit-object bochs))
			#f))

(define (replace-bochs-string bochs string expand?)
  (let* ((edit-object ((if expand?
			   bochs/closed-edit-object
			   bochs/open-edit-object)
		       bochs))
	 (bochs-region (edit-object/region-or-buffer edit-object))
	 (string-offset (edit-object/string-offset edit-object))
	 (start (mark-right-inserting-copy
		 (mark+ (region-start bochs-region)
			string-offset)))
	 (end (mark-permanent-copy
	       (mark- (region-end bochs-region)
		      bochs-suffix-length))))
    ((if expand?
	 set-bochs/closed-edit-object!
	 set-bochs/open-edit-object!)
     bochs
     (%make-region-edit-object
      bochs-region
      (region->string (make-region start end))
      string-offset))
    (delete-string start end)
    (insert-string string start)
    ((if expand?
	 set-bochs/open-edit-object!
	 set-bochs/closed-edit-object!)
     bochs
     (%make-region-edit-object bochs-region string string-offset))
    (indent-region (line-start end 0 'limit) end (bochs-start-column bochs))
    start))

(define (update-interior-bochs-edit-objects! bochs)
  (let* ((bochs-region (edit-object/region-or-buffer
			(or (bochs/open-edit-object bochs)
			    (bochs/closed-edit-object bochs))))
	 (start (and (region? bochs-region)
		     (mark+ (region-start bochs-region)
			    bochs-prefix-length)))
	 (end-of-bochs (and start (forward-bochs start))))
    (let loop ((bochs-end end-of-bochs))
      (let ((bochs-start
	     (and bochs-end (backward-bochs bochs-end))))
	(if (or (not bochs-start)
		(mark>= bochs-start (region-end bochs-region)))
	    unspecific
	    (let ((bochs
		   (find-bochs-immediately-after-point
		    bochs-start
		    (lambda () #f))))
	      (if  (not bochs)
		   (loop (forward-bochs (mark1+ bochs-start)))
		   (begin
		     (update-bochs-edit-object!
		      bochs
		      bochs-start
		      bochs-end)
		     (update-interior-bochs-edit-objects! bochs)
		     (loop (forward-bochs bochs-end))))))))))

(define (update-bochs-edit-object! bochs bochs-start bochs-end)
  (let ((open-edit-object (bochs/open-edit-object bochs))
	(closed-edit-object (bochs/closed-edit-object bochs))
	(bochs-region (make-region
		       (mark-permanent-copy bochs-start)
		       (mark-permanent-copy bochs-end))))
    (if open-edit-object
	(set-edit-object/region-or-buffer! open-edit-object bochs-region))
    (if closed-edit-object
	(set-edit-object/region-or-buffer! closed-edit-object bochs-region))))

(define (copy-code-to-kill-ring lambda-or-procedure indentation)
  (kill-ring-save (code-string lambda-or-procedure indentation) #t))

;;; The first character of BOCHS-PREFIX and the last character of
;;; BOCHS-SUFFIX  must be open and close syntax table objects because
;;; the FIND-xxx routines use FORWARD-SEXP and BACKWARD-SEXP.

(define bochs-prefix "{")
(define bochs-prefix-length (string-length bochs-prefix))
(define bochs-open-char (string-ref bochs-prefix 0))

(define bochs-number-indicator ":")
(define bochs-number-indicator-length (string-length bochs-number-indicator))

(define bochs-suffix "}")
(define bochs-suffix-length (string-length bochs-suffix))
(define bochs-close-char (string-ref bochs-suffix (-1+ bochs-suffix-length)))

(modify-syntax-entry! scheme-mode:syntax-table
		      (string-ref bochs-prefix 0)
		      (string #\( bochs-close-char))

(modify-syntax-entry! scheme-mode:syntax-table
		      bochs-close-char
		      (string #\) bochs-open-char))

(define bochs-only-syntax-table
  ;; The following modificiations should nullify the syntax table entries
  ;; of the non-bochser list delimiters
  (do ((entries
	(vector-copy (syntax-table/entries scheme-mode:syntax-table)))
       (i 0 (1+ i)))
      ((>= i (vector-length entries)) (%make-syntax-table entries))
    (let ((char (ascii->char i)))
      (if (or
	   (and (char=? ((ucode-primitive char->syntax-code)
			   (syntax-table/entries scheme-mode:syntax-table)
			   char)
			  #\()
		(not (char=? char bochs-open-char)))
	   (and (char=? ((ucode-primitive char->syntax-code)
			   (syntax-table/entries scheme-mode:syntax-table)
			   char)
			  #\))
		(not (char=? char bochs-close-char))))
	  (vector-set! entries
		       i
		       ((ucode-primitive string->syntax-entry) " "))))))

;; This will change when bochses are real edwin objects

(define (find-bochs-around-point point if-not-found)
  (find-bochs-immediately-after-point (backward-up-one-bochs point)
				      if-not-found))

(define (find-bochs-before-point point if-not-found)
  (find-bochs-immediately-after-point (backward-bochs point) if-not-found))

(define (find-bochs-immediately-after-point point if-not-found)
  (let* ((start point)
	 (end (and start (buffer-end (mark-buffer point)))))
    (or (and start
	     (let ((number-start
		    (search-forward bochs-number-indicator start end)))
	       (and number-start
		    (lookup-bochs
		     (region->string (make-region number-start end))))))
	(if-not-found))))

(define (lookup-bochs string)
  (let* ((bochs-number-end (substring-find-next-char-in-set
			    string
			    0
			    (string-length string)
			    (char-set-invert char-set:numeric))))
    (bochs/lookup (string->number (substring string
					     0
					     bochs-number-end)))))
(define (backward-bochs mark)
  (scan-bochs-backward mark 0))

(define (forward-bochs mark)
  (scan-bochs-forward mark 0))

(define (backward-up-one-bochs mark)
  (scan-bochs-backward mark 1))

(define (forward-up-one-bochs mark)
  (scan-bochs-forward mark 1))

(define (backward-down-one-bochs mark)
  (scan-bochs-backward mark -1))

(define (forward-down-one-bochs mark)
  (scan-bochs-forward mark -1))

(define forward-down-bochs)
(define backward-down-bochs)
(make-motion-pair forward-down-one-bochs backward-down-one-bochs
  (lambda (f b)
    (set! forward-down-bochs f)
    (set! backward-down-bochs b)
    unspecific))

(define forward-up-bochs)
(define backward-up-bochs)
(make-motion-pair forward-up-one-bochs backward-up-one-bochs
  (lambda (f b)
    (set! forward-up-bochs f)
    (set! backward-up-bochs b)
    unspecific))

(define (scan-bochs-forward mark depth)
  (let* ((group (mark-group mark))
	 (index
	  ((ucode-primitive scan-list-forward)
	   (syntax-table/entries bochs-only-syntax-table)
	   group
	   (mark-index mark)
	   (group-end-index group)
	   depth
	   #f
	   #f)))
    (and index (make-mark group index))))

(define (scan-bochs-backward mark depth)
  (let* ((group (mark-group mark))
	 (index
	  ((ucode-primitive scan-list-backward)
	   (syntax-table/entries bochs-only-syntax-table)
	   group
	   (mark-index mark)
	   (group-start-index group)
	   depth
	   #f
	   #f)))
    (and index (make-mark group index))))

#|
(define (find-bochs-before-point point if-not-found)
  (or (lookup-bochs point) (if-not-found)))

(define (lookup-bochs point)
  (bochs-table-lookup point))
|#
(define (make-empty-bochs-table) '())

(define mark-association (association-procedure mark= car))

(define (add-to-bochs-table! mark bochs)
  (local-set-variable! bochsmod-internal-bochs-table
		       (cons (cons mark bochs)
			     (ref-variable bochsmod-internal-bochs-table))))

(define (bochs-table-lookup mark)
  (let ((entry
	 (mark-association mark (ref-variable bochsmod-internal-bochs-table))))
    (and entry (cdr entry))))

(define (bochsmod-find-buffer bochs)
  (let* ((maybe-edit-object (bochs/open-edit-object bochs))
	 (maybe-buffer (and maybe-edit-object
			    (edit-object/region-or-buffer
			     maybe-edit-object))))
    (and maybe-buffer (buffer? maybe-buffer) maybe-buffer)))

(define (bochsmod-create-buffer bochs)
  (new-buffer (bochs/name bochs)))

(define (bochsmod-input-recorder region)
  (ring-push! (ref-variable comint-input-ring) (region->string region)))

(define (bochsmod-output-receiver value output-string)
  (if output-string
      (begin
	(let ((output-string-start
	       (mark-right-inserting-copy (current-point))))
	  (insert-string output-string)
	  (let* ((output-string-end (current-point))
		 (output-first-newline
		  (cond
		   ((char-search-forward
		     #\Newline
		     output-string-start
		     output-string-end)
		    => mark-permanent-copy)
		   (else false))))
	    (if output-first-newline
		(with-current-point output-first-newline
		  (lambda ()
		    (lisp-indent-line #f)
		    (let ((indent-start
			   (line-start output-first-newline 1 'LIMIT))
			  (indent-end
			   (line-start output-string-end 0 'LIMIT)))
		      (if (mark<= indent-start indent-end)
			  (let ((output-start-column (current-column)))
			    (indent-rigidly indent-start
					    indent-end
					    output-start-column)))))))))))
  (guarantee-newline)
  (lisp-indent-line #f)
  (insert-string (transcript-value-prefix-string value true))
  (if (not (undefined-value? value))
      (with-output-to-current-point
	  (lambda ()
	    (if (bochsable-object? value)
		(begin
		  (fresh-line)
		  (lisp-indent-line #f)
		  (create-bochs-and-write value))
		(write value)))))
  (guarantee-newlines 2)
  (lisp-indent-line #f)
  (let ((bind-buffer (ref-variable bochsmod-bindings-buffer)))
    (and bind-buffer
	 (update-bindings bind-buffer))))

;;; BOCHSMOD-WRITE should do something object-oriented. We should be
;;; able to easily define new bochs types with their own output
;;; procedure.

(define (create-bochs-and-write object)
  (cond
   ((procedure? object)
    (let ((env-bochs (create-bochs (procedure-environment object)))
	  (code-bochs (create-bochs (procedure-lambda object))))
      (let ((bochs (create-bochs object env-bochs code-bochs)))
	(write-bochs bochs write-procedure-bochs))))
   ((environment? object)
    (let ((bochs (create-bochs object)))
      (write-bochs bochs write-environment-bochs)))
   ((lambda? object)
    (let ((bochs (create-bochs object)))
      (write-bochs bochs write-code-bochs)))))

(define (write-bochs bochs bochs-writer)
  (with-values
      (lambda ()
	(bochs-writer bochs))
    (lambda (bochs-edit-object bochs-mark)
      (set-bochs/closed-edit-object! bochs bochs-edit-object)
      (add-to-bochs-table! bochs-mark bochs)
      (let* ((bochs-region
	      (edit-object/region-or-buffer bochs-edit-object))
	     (maybe-newline
	      (char-search-forward
	       #\Newline
	       (region-start bochs-region)
	       (region-end bochs-region))))
	(if maybe-newline
	    (indent-region
	     maybe-newline
	     (region-end bochs-region)
	     (+ (bochs-start-column bochs) 2))))
      bochs)))

(define (write-procedure-bochs bochs)
  (bochs-writer-wrapper bochs
   (lambda()
     (write-string "Proc"))     
   (lambda()
     (write-char #\Newline)
     (write-bochs (proc-bochs/code-bochs bochs) write-code-bochs)
     (write-char #\Newline)
     (write-bochs (proc-bochs/environment-bochs bochs)
		  write-environment-bochs))
   (lambda () unspecific)))

(define (write-environment-bochs bochs)
  (bochs-writer-wrapper bochs
   (lambda()
     (write-string "Env"))
   (lambda ()
     (display (bochs/name bochs)))
   (lambda () unspecific)))

(define (write-code-bochs bochs)
  (bochs-writer-wrapper bochs
   (lambda()
     (write-string "Code"))
   (lambda ()
     (display (bochs/name bochs)))
   (lambda () unspecific)))
  
(define (bochs-writer-wrapper bochs specific-prefix-writer
			      specific-bochs-writer specific-suffix-writer)
  (let ((bochs-string-region))
    (let ((bochs-region
	   (recording-output-region
	    (lambda ()
	      (write-string bochs-prefix)
	      (specific-prefix-writer)
	      (write-string bochs-number-indicator)
	      (write (bochs/lookup-number bochs))
	      (write-char #\space)
	      (set! bochs-string-region
		    (recording-output-region specific-bochs-writer))
	      (specific-suffix-writer)
	      (write-string bochs-suffix)))))
      (values
       (%make-region-edit-object (make-region
				  (mark-left-inserting-copy
				   (region-start bochs-region))
				   (region-end bochs-region))
				 (region->string bochs-string-region)
				 (- (mark-index
				     (region-start bochs-string-region))
				    (mark-index
				     (region-start bochs-region))))
       (mark-right-inserting-copy (region-end bochs-region))))))

(define (default-syntax-table)
  (let ((buffer (current-buffer)))
    (evaluation-syntax-table buffer (evaluation-environment buffer))))

(define (code-string procedure-or-lambda #!optional indentation)
  (with-output-to-string
    (lambda ()
      (pp
       (lambda-source-code 
	(if (procedure? procedure-or-lambda)
	    (procedure-lambda procedure-or-lambda)
	    procedure-or-lambda))
       (current-output-port)
       #t
       (if (default-object? indentation)
	   0
	   indentation)))))

(define-variable bindings-window-fraction
  "The maximum size of a bochser bindings window as a
fraction of its associated environment bochs window"
  (/ 1 3))

(define (attach-bindings bochs-buffer force?)
  (if (or force?
	  (ref-variable bochsmod-smart-windows)
	  (ref-variable bochsmod-show-bindings))
      (let ((bindings-buffer
	     (ref-variable bochsmod-bindings-buffer bochs-buffer)))
	(and bindings-buffer
	     (begin
	       (update-bindings bindings-buffer)
	       (let ((env-window (current-window)))
		 (and (not (buffer-visible? bindings-buffer))
		      (let ((bind-buffer-size
			     (region-count-lines
			      (buffer-region bindings-buffer)))
			    (env-window-size
			     (window-y-size env-window)))
			(window-split-vertically!
			 env-window
			 (let ((max-bindings-size
				(round
				 (* env-window-size
				    (ref-variable bindings-window-fraction)))))
			   (if (> bind-buffer-size max-bindings-size)
			       max-bindings-size
			       (1+ bind-buffer-size))))
			(select-buffer bindings-buffer)
			(select-window (other-window))))))))))

(define (detach-bindings #!optional bindings-buffer)
  (if (or (ref-variable bochsmod-smart-windows)
	  (ref-variable bochsmod-show-bindings))
      (let* ((current-window (current-window))
	     (window-above (and (window-has-up-neighbor? current-window)
				(window-1+ current-window))))
	(if (and window-above
		 (eq? (window-buffer window-above)
		      (if (default-object? bindings-buffer)
			  (ref-variable bochsmod-bindings-buffer)
			  bindings-buffer)))
	    (window-delete! window-above)))))

	     
(define (bochsmod-create-bindings-buffer bochs-buffer env)
  (let ((bind-buffer
	 (new-buffer (string-append
		      "Bindings for: "
		      (write-to-string (environment-name env))))))
    (set-buffer-read-only! bind-buffer)
    (set-variable-local-value!
     bind-buffer
     (ref-variable-object bochsmod-bochs-buffer)
     bochs-buffer)
    bind-buffer))

(define (update-bindings bind-buffer)
  (and bind-buffer
       (or (ref-variable bochsmod-smart-windows)
	   (ref-variable bochsmod-show-bindings))
       (begin
	 (set-buffer-writeable! bind-buffer)
	 (let ((start (buffer-start bind-buffer))
	       (bochs-buffer (ref-variable bochsmod-bochs-buffer bind-buffer)))
	   (delete-string start (buffer-end bind-buffer))
	   (with-output-to-mark start
	     (lambda ()
	       (let* ((nearest-env-bochs
		       (bochsmod-find-env-bochs-around-point
			(buffer-point bochs-buffer)
			(lambda () #f)))
		      (environment
		       (if nearest-env-bochs
			   (bochs/environment nearest-env-bochs)
			   (evaluation-environment #f))))
		 (write (environment-name environment))
		 (show-environment-bindings
		  environment
		  #t
		  (current-output-port))))))
	 (set-buffer-read-only! bind-buffer)
	 (set-buffer-point! bind-buffer (buffer-start bind-buffer)))))

(define region? pair?)

(define (bochsmod-find-env-bochs-around-point point if-not-found)
  (let loop ((mark point))
    (let ((bochs
	   (and mark (find-bochs-around-point mark (lambda () #f)))))
      (if (not bochs)
	  (if-not-found)
	  (if (environment? (bochs/object bochs))
	      bochs
	      (loop (backward-up-one-bochs mark)))))))

(define (bochsmod-find-nearest-buffer bochs)
  (let* ((edit-object (bochs/closed-edit-object bochs))
	 (region-or-buffer (and edit-object
				(edit-object/region-or-buffer edit-object))))
    (cond ((not region-or-buffer) false)
	  ((buffer? region-or-buffer) region-or-buffer)
	  ((region? region-or-buffer) (group-buffer
				       (region-group region-or-buffer)))
	  (else (error "Bochsmod-find-nearest-buffer: bad bochs." bochs)))))

(define (recording-output-region thunk)
  (let ((start (mark-right-inserting-copy (current-point))))
    (thunk)
    (make-region start
		 (mark-right-inserting-copy (current-point)))))

(define (bochsmod-register-move-point! group new-point old-point)
  old-point
  (let ((new-env-bochs (bochsmod-find-env-bochs-around-point
			new-point
			(lambda () #f)))
	;; The following is not strictly neccessary since we could compute
	;; it using bochsmod-find-env-bochs-around-point, but for
	;; efficiency sake we keep track of it explicitly
	(current-env-bochs (ref-variable bochsmod-current-environment-bochs)))
    (if (not (eq? new-env-bochs current-env-bochs))
	(begin
	  (local-set-variable! scheme-syntax-table
			       (or
				(and new-env-bochs
				     (bochs/syntax-table new-env-bochs))
				(ref-variable
				 bochsmod-default-syntax-table)))
	  (local-set-variable! bochsmod-current-environment-bochs
			       new-env-bochs)
	  (let ((new-env
		 (or (and new-env-bochs (bochs/environment new-env-bochs))
		     (ref-variable bochsmod-default-scheme-environment))))
	    (and (not (eq? new-env (ref-variable scheme-environment)))
		 (local-set-variable! scheme-environment new-env)
		 (let* ((current-buffer (group-buffer group))
			(bind-buffer (ref-variable
				      bochsmod-bindings-buffer
				      current-buffer)))
		   (update-bindings bind-buffer))))))))

(define (bochsmod-move-into-bochs bochs)
  (and (bochs/open? bochs)
       (set-current-point! (bochsmod-find-good-point-in-bochs bochs))))

(define (bochs-start-column bochs)
  (let ((region-or-buffer (edit-object/region-or-buffer
			   (bochs/closed-edit-object bochs))))
    (if (not (region? region-or-buffer))
	(editor-error "Bochs must be a region:" bochs))
    (mark-column (region-start region-or-buffer))))

(define (bochsmod-find-good-point-in-bochs bochs)
  (let* ((edit-object (bochs/open-edit-object bochs))
	 (region-or-buffer
	  (and edit-object
	       (edit-object/region-or-buffer edit-object)))
	 (region
	  (if (region? region-or-buffer)
	      region-or-buffer
	      (make-region (buffer-start region-or-buffer)
			   (buffer-end region-or-buffer)))))
    (let ((object (bochs/object bochs)))
      (cond
       ((environment? object)
	(let ((good-point
	       (mark-permanent-copy
		(or (char-search-backward
		     #\Newline
		     (region-end region)
		     (region-start region))
		    (region-end region)))))
	  (if (region? region-or-buffer)
	      (indent-region
	       (line-start good-point 0 'limit)
	       good-point
	       (+ (bochs-start-column bochs) 2)))
	  good-point))
       (else (region-end region))))))

(define-prefix-key 'bochser #\C-c 'prefix-key)
(define-key 'bochser '(#\C-c #\o) 'bochsmod-open-bochs-before-point)
(define-key 'bochser (make-modified-button 'control 0 'down)
  'bochsmod-open-bochs-before-mouse-point)
(define-key 'bochser '(#\c-c #\c) 'bochsmod-close-bochs-before-point)
(define-key 'bochser (make-modified-button 'control 1 'down)
  'bochsmod-close-bochs-before-mouse-point)
(define-key 'bochser '(#\c-c #\c-c) 'bochsmod-close-bochs-around-point)
(define-key 'bochser (make-modified-button 'control 2 'down)
  'bochsmod-close-bochs-around-mouse-point)
(define-key 'bochser '(#\C-c #\M-w) 'bochsmod-extract-code)
(define-key 'bochser '(#\c-x #\0) 'bochsmod-delete-window)
(define-key 'bochser '(#\c-x #\1) 'bochsmod-delete-other-windows)
(define-key 'bochser '(#\c-x #\b) 'bochsmod-switch-to-buffer)
(define-key 'bochser '(#\c-x #\k) 'bochsmod-kill-buffer)
(define-key 'bochser '(#\c-c #\c-d) 'down-bochs)
(define-key 'bochser '(#\c-c #\c-u) 'backward-up-bochs)

(define-key 'bochser #\M-p 'comint-previous-input)
(define-key 'bochser #\M-n 'comint-next-input)

(define-key 'bochser '(#\C-c #\C-r) 'comint-history-search-backward)
(define-key 'bochser '(#\C-c #\C-s) 'comint-history-search-forward)