#| -*-Scheme-*-

$Header: vmspar.scm,v 14.2 88/06/13 12:00:26 GMT cph Rel $

Copyright (c) 1988 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Pathname Parsing -- VMS
;;; package: (runtime pathname-parser)

(declare (usual-integrations))

(define (parse-pathname string receiver)
  (let ((start 0)
	(end (string-length string))
	(index)
	(host false)
	(device false)
	(directory '())
	(name false)
	(type false)
	(version false))

    (define-integrable (find-char char)
      (set! index (substring-find-next-char string start end char))
      index)

    (if (and (find-char #\:) (substring-prefix? "::" 0 2 string index end))
	(begin
	  (set! host (substring string start index))
	  (set! start (+ index 2))))
    (if (find-char #\:)
	(begin
	  (set! device (wildify string start index))
	  (set! start (1+ index))))
    (if (and (< start end)
	     (char=? #\[ (string-ref string start))
	     (find-char #\]))
	(begin
	  (set! directory (parse-directory string (1+ start) index))
	  (set! start (1+ index))))
    (if (< start end)
	(cond ((find-char #\.)
	       (set! name (wildify string start index))
	       (set! start (1+ index))
	       (if (not (find-char #\;))
		   (set! type (wildify string start end))
		   (begin
		     (set! version (parse-version string (1+ index) end))
		     (set! type
			   (wildify string start (if version index end))))))
	      ((find-char #\;)
	       (set! version (parse-version string (1+ index) end))
	       (set! name (wildify string start (if version index end))))
	      (else
	       (set! name (wildify string start end))
	       (set! start end))))
  (receiver host device (simplify-directory directory) name type version)))

(define (pathname-as-directory pathname)
  (make-pathname
   (pathname-host pathname)
   (pathname-device pathname)
   (let ((directory (pathname-directory pathname)))
     (let ((file (pathname-unparse-name (pathname-name pathname)
					(pathname-type pathname)
					(pathname-version pathname))))
       (if (string-null? file)
	   directory
	   (simplify-directory
	    (let ((file-components
		   (list (parse-directory-component file
						    0
						    (string-length file)))))
	      (cond ((or (null? directory) (eq? directory 'UNSPECIFIC))
		     file-components)
		    ((pair? directory) (append directory file-components))
		    (else (error "Illegal pathname directory" directory))))))))
   false
   false
   false))

(define (parse-directory string start end)
  (define (loop start)
    (let ((index (substring-find-next-char string start end #\.)))
      (if index
	  (cons (parse-directory-component string start index)
		(loop (1+ index)))
	  (list (parse-directory-component string start end)))))
  ;; This assumes that "dev:[]foo.scm" is absolute, while
  ;; "dev:foo.scm" is relative.  I'm not sure this is right.
  (cond ((= start end) (list 'ROOT))
	((char=? #\. (string-ref string start)) (loop (1+ start)))
	(else
	 (let ((components (loop start)))
	   (if (eq? 'UP (car components))
	       components
	       (cons 'ROOT components))))))

(define (parse-directory-component string start end)
  (cond ((substring=? string start end "*" 0 1) 'WILD)
	((substring=? string start end "-" 0 1) 'UP)
	(else (substring string start end))))

(define (parse-version string start end)
  (cond ((= start end) false)
	((substring=? string start end "*" 0 1) 'WILD)
	((substring=? string start end "-1" 0 2) 'NEWEST)
	((substring=? string start end "0" 0 1) false)
	(else
	 (digits->number (reverse! (substring->list string start end)) 1 0))))

(define (digits->number digits weight accumulator)
  (if (null? digits)
      accumulator
      (let ((value (char->digit (car digits) 10)))
	(and value
	     (digits->number (cdr digits)
			     (* weight 10)
			     (+ (* weight value) accumulator))))))

(define (wildify string start end)
  (if (substring=? string start end "*" 0 1)
      'WILD
      (substring string start end)))