 /*  -*- Mode: C;  -*-
 * File: except_.h
 * Author: Heinz Schmidt (hws@ICSI.Berkeley.EDU)
 * Copyright (C) International Computer Science Institute, 1991
 *
 * COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY
 * and is subject to the terms of the SATHER LIBRARY GENERAL PUBLIC
 * LICENSE contained in the file: "sather/doc/license.txt" of the Sather
 * distribution. The license is also available from ICSI, 1947 Center
 * St., Suite 600, Berkeley CA 94704, USA. 
 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ** FUNCTION: Exception handling based on catch and throw.
 **  This is written as a basis for several implementations including C and
 **  C++, so that the macros can be easily used where Sather is used to
 **  prototype C++ classes. Once the C++ exception handling is out there,
 **  the mechanism below can be adapted to run on top of that.
 **  Usage outside Sather see comment below.
 **
 ** RELATED PACKAGES: ./exc_test*.c
 **
 ** RCS: $Id: all_.h,v 0.22 1992/06/30 14:12:04 hws Exp $
 ** HISTORY:
 ** Last edited: Jul  1 21:58 1992 (hws)
 **  Jun 30 23:04 1992 (hws): EH* macros add $OB catcher, improve readability
 ** Created: Sun Jun 30 15:01:52 1991 (hws)
 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ 
 */

#ifndef EXCEPT_H
#define EXCEPT_H

#include <setjmp.h>

/* int setjmp(env)
 *     jmp_buf env;
 *     
 * void longjmp(env, val)
 *     jmp_buf env;
 *     int val;  
 */

/* 
 * One exception handler "frame" portion holds this, allocated on stack.
 * Per thread info, here global.
 */

struct EH_frame {
        jmp_buf exc_buf;
        int     exc_type;              /* mono > 0 ; poly < 0 */
        ptr     exc_prev;
};

ptr EH_last_frame;;

/* TEH1 = dynamic chaining + (pop_to_nxt_hdl_if_non_matching*; longjmp).
 *        rather than        (longjmp; pop_to_nxt_hdlr_if_non_matching)*
 * TEH2 = array of TEH1 chains, justified in process control where bad things
 * can get worse fast.
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * Usage:
 * EH_CATCH(TYPE,OBJ,
 *          TRY,             
 *          HDLR);           
 * EH_THROW(TYPE,OBJ);       
 *
 * TYPE is the tag to which the exception is raised, i.e., an int code.
 *      a positive tag matches just this type.
 *      a negative tag matches this type or one of its subtypes matches.
 * OBJ is a ptr to a user defined error object (describing the error), raised
 *      by EH_THROW, and received and used in HDLR code of the EH_CATCH.
 * TRY is the normal block of code.
 * HDLR the handler block of code that takes control when an error occurs in TRY.
 *
 * The dynamic handler chain is on the stack. Each frame pointing
 * to the previous handler and holding sufficient information to find the
 * right handler without longjmp. Only when a match is found we do the
 * expensive longjmp. Note that Sather code relies on type-safety of
 * of this code. In the C or C++ use of these macros the TYPE codes must
 * be chosen appropriately to guarantee similar type safety. We suggest
 * a table of EXCEPTION TYPES and a 2D BIT ARRAY representation of the
 * subtype relation for these few types (IS_A_DES_OF_).
 */

/* To be able to abort, we need to restore the system handler, then only raise
 * the exception. The final exit should never be reached, but who knows...
 */

void eh_restore_sigquit();

#define EH_ABORT(TYPE,OBJ,MSG) \
  { fprintf(stderr, "Unhandled exception \"%d\" of type \"%d\". %s.\n",OBJ,TYPE,MSG); \
    eh_restore_sigquit; \
    exit(1);\
   }

#define EH_CATCH(TYPE,OBJ,TRY,HDLR)\
      { struct EH_frame exc_cur;             \
        exc_cur.exc_type=TYPE;               \
        exc_cur.exc_prev=EH_last_frame;      \
        EH_last_frame=(ptr)&exc_cur;         \
        if (!(int)(OBJ=(ptr)setjmp((&exc_cur)->exc_buf))){   \
           {TRY;}                            \
	   EH_last_frame=exc_cur.exc_prev;   \
        } else {                             \
	   EH_last_frame=exc_cur.exc_prev;   \
           {HDLR;}                           \
        }; }

/* Note: for the the C++ use you may want to use EH_THROW_TYPE to pass
 * a tag explicitly. For Sather we want to allow polymorphic throws.
 */

#define EH_THROW_TYPE(TYPE,OBJ)              \
       {struct EH_frame *exc;                \
	  int exc_TYPE;                      \
        for (exc=(struct EH_frame *)EH_last_frame;\
             ((int)exc)!=0;                  \
             exc=(struct EH_frame *)(exc->exc_prev)){\
	   exc_TYPE = exc->exc_type;         \
           if (EH_MATCH((TYPE),exc_TYPE)){   \
              longjmp(exc->exc_buf,(int)(OBJ));\
           };\
        };\
        EH_ABORT(TYPE,OBJ,"Broken using EH_THROW_TYPE under program control"); }

/* Evaluate OBJ once only! */

#define EH_THROW(OBJ)  \
        { int EH_OBJ = (int)(OBJ); \
	    { if (EH_OBJ == 0) \
		{ EH_ABORT(0,0,"Broken using EH_THROW on void object."); \
		} else { \
		  int EH_TYPE = TYPE_((ptr)EH_OBJ); \
		  if (EH_TYPE == 0) \
		    { EH_ABORT(0,EH_OBJ,"Broken using EH_THROW on void type"); \
		    } else { \
		      EH_THROW_TYPE(EH_TYPE,EH_OBJ); \
		    }  \
		}      \
	    }          \
	}

#define EH_MONO(TYPE) (TYPE)>0
#define EH_ALL(TYPE) (TYPE)==-OB_ici
#define EH_DES_MATCH(T,ETYP) (T)==-(ETYP)?1:(IS_A_DES_OF_(T,-(ETYP)))
#define EH_MATCH(T,ETYP) ((T)==(ETYP)?1:EH_MONO(ETYP)?0:EH_ALL(ETYP)?1:EH_DES_MATCH(T,ETYP))

#endif
/* EOF except_.h */

